/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */

#include "nactv-debug.h"
#include "process.h"

#define _GNU_SOURCE

#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <dirent.h>
#include <ctype.h>
#include <stdio.h>


static int is_simple_number (const char* text)
{
	for (; *text!='\0'; text++)
		if (!isdigit(*text))
			break;
	return (*text=='\0');
}

static char *read_system_file (const char *path)
{
	char *contents = NULL;
	FILE *f;
	
	f = fopen(path, "r");
	if (f != NULL)
	{
		char *line;
		size_t line_size;
		ssize_t line_length;
	
		line_size = 128;
		line = malloc(line_size);
		line_length = getline(&line, &line_size, f);
		
		if (line_length > 0)
		{
			int i;
			for (i=0; i<line_length; i++)
				if (line[i] == '\0')
					line[i] = ' ';
			contents = g_strdup(line);
		}
		
		free(line);
		fclose(f);
	}
	return contents;
}

unsigned int get_running_processes (Process **processes)
{
	unsigned int nrprocesses = 0;
	DIR *procdir;
	*processes = NULL;
	
	procdir = opendir("/proc");
	if (procdir != NULL)
	{
		struct dirent dentry, *pdentry;
		GArray *aprocesses;
		
		aprocesses = g_array_new(FALSE, TRUE, sizeof(Process));
		
		while ( readdir_r(procdir, &dentry, &pdentry) == 0 )
		{
			Process process;
			char *exeLinkPath, *exeFilePath, *cmdlineFilePath;
			
			if (pdentry == NULL)
				break;
			if (pdentry->d_type != DT_DIR)
				continue;
			
			if (!is_simple_number(pdentry->d_name))
				continue;
			
			memset(&process, 0, sizeof(Process));
			
			process.pid = atol(pdentry->d_name);
			exeLinkPath = g_build_path("/", "/proc", pdentry->d_name, "exe", NULL);
			exeFilePath = g_file_read_link(exeLinkPath, NULL);
			process.name = (exeFilePath!=NULL) ? g_path_get_basename(exeFilePath) : NULL;

			cmdlineFilePath = g_build_path("/", "/proc", pdentry->d_name, "cmdline", NULL);
			process.commandline = read_system_file(cmdlineFilePath);
				
			g_array_append_val(aprocesses, process);
			
			if (exeFilePath != NULL)
				g_free(exeFilePath);
			g_free(exeLinkPath);
			g_free(cmdlineFilePath);
		}
		closedir (procdir);
		
		nrprocesses = aprocesses->len;
		*processes = (aprocesses->len > 0) ? (Process*)aprocesses->data : NULL;
		g_array_free(aprocesses, (*processes==NULL));
	}
	
	return nrprocesses;
}



#define PRG_SOCKET_PFX    "socket:["
#define PRG_SOCKET_PFXl (strlen(PRG_SOCKET_PFX))
#define PRG_SOCKET_PFX2   "[0000]:"
#define PRG_SOCKET_PFX2l  (strlen(PRG_SOCKET_PFX2))

static void extract_type_1_socket_inode (const char lname[], long * inode_p) {

    /* If lname is of the form "socket:[12345]", extract the "12345"
       as *inode_p.  Otherwise, return -1 as *inode_p.
       */

    if (strlen(lname) < PRG_SOCKET_PFXl+3) 
		*inode_p = -1;
    else if (memcmp(lname, PRG_SOCKET_PFX, PRG_SOCKET_PFXl)) 
		*inode_p = -1;
    else if (lname[strlen(lname)-1] != ']') 
		*inode_p = -1;
    else {
        char inode_str[strlen(lname + 1)];  /* e.g. "12345" */
        const int inode_str_len = strlen(lname) - PRG_SOCKET_PFXl - 1;
        char *serr;

        strncpy(inode_str, lname+PRG_SOCKET_PFXl, inode_str_len);
        inode_str[inode_str_len] = '\0';
        *inode_p = strtol(inode_str,&serr,0);
        if (!serr || *serr || *inode_p < 0 || *inode_p >= INT_MAX) 
            *inode_p = -1;
    }
}


static void extract_type_2_socket_inode (const char lname[], long * inode_p) {

    /* If lname is of the form "[0000]:12345", extract the "12345"
       as *inode_p.  Otherwise, return -1 as *inode_p.
       */

    if (strlen(lname) < PRG_SOCKET_PFX2l+1) *inode_p = -1;
    else if (memcmp(lname, PRG_SOCKET_PFX2, PRG_SOCKET_PFX2l)) *inode_p = -1;
    else {
        char *serr;

        *inode_p=strtol(lname + PRG_SOCKET_PFX2l,&serr,0);
        if (!serr || *serr || *inode_p < 0 || *inode_p >= INT_MAX)
            *inode_p = -1;
    }
}

unsigned int process_get_socket_inodes (long pid, long **inodes)
{
	char spid[48];
	char *sfddir;
	DIR *fddir;
	unsigned int nsockets = 0;
	*inodes = NULL;
	
	snprintf(spid, sizeof(spid), "%ld", pid);
	sfddir = g_build_path("/", "/proc", spid, "fd", NULL);
	
	fddir = opendir (sfddir);
	if (fddir != NULL)
	{
		struct dirent *fdentry, dentry;
		GArray *ainodes;
		
		ainodes = g_array_new(FALSE, TRUE, sizeof(long));
		
		while ( readdir_r (fddir, &dentry, &fdentry) == 0 )
		{
			char *file_name, *link_path;
			long inode = -1;
			
			if (fdentry==NULL)
				break;
			if (fdentry->d_type != DT_LNK)
				continue;
			link_path = g_build_path("/", sfddir, fdentry->d_name, NULL);
			file_name = g_file_read_link(link_path, NULL);
			
			if (file_name != NULL)
			{
				extract_type_1_socket_inode(file_name, &inode);
				if (inode < 0)
					extract_type_2_socket_inode(file_name, &inode);
			}
			if (inode > 0)
			{
				g_array_append_val(ainodes, inode);
			}
			
			if (file_name != NULL)
				g_free(file_name);
			g_free(link_path);
		}
		closedir(fddir);
		
		nsockets = ainodes->len;
		*inodes = (ainodes->len > 0) ? (long*)ainodes->data : NULL;
		g_array_free(ainodes, (*inodes==NULL));
	}
	
	g_free(sfddir);
	return nsockets;	
}


void process_delete_contents (Process *process)
{
	if (process != NULL)
	{
		if (process->name != NULL)
			g_free(process->name);
		if (process->commandline != NULL)
			g_free(process->commandline);
	}
}

void free_processes (Process *processes, unsigned int nprocesses)
{
	unsigned int i;
	if (processes != NULL)
	{
		for (i=0; i<nprocesses; i++)
			process_delete_contents(processes + i);
		g_free(processes);
	}
}

