/*
 * SOURCE:  applic.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: application and option routines
 *
 * UPDATES: 08/15/1991 - major rewrite
 *          10/21/1991 - version 1.2
 *                       customize menu
 *          10/23/1991 - flags for custom menu
 *          10/25/1991 - bug in chdir() fixed
 *          12/05/1991 - lots of options added
 *          20/01/1992 - Gadget-Expander added
 *
 * (c)M.Schollmeyer
 */

#include "main.h"

static BOOL _boxwarning( char * );

extern struct Editor ed;
extern struct MemoryHeader mh;

/*
 *    Name: exe_tex
 *  Return: void
 * Purpose: This routine launches TeX from within the editor.
 *
 *
 *
 * (c)M.Schollmeyer
 */
void exe_tex( void ) {

    char buffer[BUFSIZE];

    if( chkdir(0) != TRUE )
        return;

    /* save name of this file beeing executed */
    if (!smakec( buffer, "%p%b.%e", NULL)) return;
    PutProfileString( PR_TEXLASTFILE, buffer );
    /* save current path for error checking */
    getcwd( buffer, BUFSIZE );
    PutProfileString( PR_TEXCWD, buffer );

    /* reset viewerrors(), so next time we scan for errors we start again */
    viewerrors( VIEW_CLEAR );

    if (smakec( buffer, GetProfileString(PR_TEXCOMMAND, DEF_TEXCMD), NULL ))
        LaunchOverlay( buffer, 0 );
}


/*
 *    Name: opt_tex
 *  Return: void
 * Purpose: this routine opens a window and prompts for the LaTeX
 *          command line.
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH  52
#define WIN_HEIGHT 23
void opt_tex( void ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "TeX Options", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };


    struct IntuiText *ita[31];
    struct IntuiText it[30];
    static struct Request CODEBASED req = {
        NULL,
        NULL, MULTISELECT,
    };
    static struct IntuiText CODEBASED it_drive = {
        NULL, NULL, "&Default Drive:", -14, 0
    };
    static struct Gadget CODEBASED gad_pop = {
        &g_ok, &it_drive, 18, 17, 9, 1, POPGADGET,
        (APTR)&req,
    };

    static struct IntuiText CODEBASED it_ems = {
        NULL, NULL, "Disable E&MS", 4, 0
    };
    static struct Gadget CODEBASED g_ems = {
        &gad_pop, &it_ems, 30, 15, 15, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_scroll = {
        NULL, NULL, "&Scroll Mode", 4, 0
    };
    static struct Gadget CODEBASED g_scroll = {
        &g_ems, &it_scroll, 30, 14, 15, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_nonstop = {
        NULL, NULL, "&Nonstop Mode", 4, 0
    };
    static struct Gadget CODEBASED g_nonstop = {
        &g_scroll, &it_nonstop, 30, 13, 16, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_errstop = {
        NULL, NULL, "&Error Stop Mode", 4, 0
    };
    static struct Gadget CODEBASED g_errstop = {
        &g_nonstop, &it_errstop, 30, 12, 19, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_batch = {
        NULL, NULL, "&Batch Mode", 4, 0
    };
    static struct Gadget CODEBASED g_batch = {
        &g_errstop, &it_batch, 30, 11, 14, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_clear = {
        NULL, NULL, "Clear Bit &7", 4, 0
    };
    static struct Gadget CODEBASED g_clear = {
        &g_batch, &it_clear, 30, 10, 15, 1, CHECKGADGET | TOGGLESELECT,
    };

    /* INITEX options */
    static struct IntuiText CODEBASED it_control = {
        NULL, NULL, "Enable &Control Chars", 4, 0
    };
    static struct Gadget CODEBASED g_control = {
        &g_clear, &it_control, 3, 14, 24, 1, CHECKGADGET | TOGGLESELECT,
    };
    static struct IntuiText CODEBASED it_8bitout = {
        NULL, NULL, "8-Bit for &Output", 4, 0
    };
    static struct Gadget CODEBASED g_8bitout = {
        &g_control, &it_8bitout, 3, 13, 20, 1, CHECKGADGET | TOGGLESELECT,
    };
    static struct IntuiText CODEBASED it_8bitin = {
        NULL, NULL, "8-Bit for &Input", 4, 0
    };
    static struct Gadget CODEBASED g_8bitin = {
        &g_8bitout, &it_8bitin, 3, 12, 19, 1, CHECKGADGET | TOGGLESELECT,
    };
    static struct IntuiText CODEBASED it_enable = {
        NULL, NULL, "Enable INITE&X", 4, 0
    };
    static struct Gadget CODEBASED g_enable = {
        &g_8bitin, &it_enable, 3, 11, 17, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct Border CODEBASED b_initex = {
        NULL, "INITEX", 2, 10, 26, 6 };


    static struct IntuiText CODEBASED it_lplaing = {
        NULL, NULL, "LaTeX (&german)", 4, 0
    };
    static struct Gadget CODEBASED g_lplaing = {
        &g_enable, &it_lplaing, 27, 7, 18, 1, OPTIONGADGET|TOGGLESELECT|NODESELECT,
        NULL, 1L
    };

    static struct IntuiText CODEBASED it_lplain = {
        NULL, NULL, "&LaTeX", 4, 0
    };
    static struct Gadget CODEBASED g_lplain = {
        &g_lplaing, &it_lplain, 27, 6, 9, 1, OPTIONGADGET|TOGGLESELECT|NODESELECT,
        NULL, 1L
    };

    static struct IntuiText CODEBASED it_plaing = {
        NULL, NULL, "pl&ain TeX (german)", 4, 0
    };
    static struct Gadget CODEBASED g_plaing = {
        &g_lplain, &it_plaing, 3, 7, 22, 1, OPTIONGADGET|TOGGLESELECT|NODESELECT,
        NULL, 1L
    };

    static struct IntuiText CODEBASED it_plain = {
        NULL, NULL, "&plain TeX", 4, 0
    };
    static struct Gadget CODEBASED g_plain = {
        &g_plaing, &it_plain, 3, 6, 13, 1, OPTIONGADGET|TOGGLESELECT|NODESELECT,
        NULL, 1L
    };

    static struct Border CODEBASED b_format = {
        NULL, "Preload Format", 2, 5, 45, 4 };

    static struct IntuiText CODEBASED it_strgad = {
        NULL, NULL, "&Command Name:", 0, -1
    };
    static struct StringInfo CODEBASED si_strgad = {
        NULL, BUFSIZE-1, 0L
    };
    static struct Gadget CODEBASED g_str = {
        &g_plain, &it_strgad, 2, 3, 46, 1, TEXTGADGET | BORDERLESS | ACTIVATED,
        (APTR)&si_strgad,
    };

    struct Gadget *g;
    int i, maxdrives, numdrives, drive;
    char *drivenames, *cp;
    unsigned long flags;

    if( ! ( si_strgad.Buffer = AllocHeap(BUFSIZE) ) ) return;
    strncpy( si_strgad.Buffer, GetProfileString( PR_TEXCOMMAND, DEF_TEXCMD ), BUFSIZE );

    maxdrives = GetNumDrives();
    drive = GetCurrDisk();
    drivenames = AllocHeap( maxdrives * 2 );

    req.Items = ita;

    it[0].Next = NULL;
    it[0].TextBorder = NULL;
    it[0].Text = "none";
    it[0].Left = it[0].Top = 0;
    it[0].Width = 0;
    it[0].Flags = 0L;
    ita[0] = &it[0];

    cp = GetProfileString( PR_TEXDEFDRIVE, NULL );
    req.CurrIt = 0;

    for( i = 0, numdrives = 1; i < maxdrives; ++i ) {
        SelectDisk( i );
        if( i == GetCurrDisk() ) {
            it[numdrives].Next = NULL;
            it[numdrives].TextBorder = NULL;
            drivenames[2*i] = i + 'a';
            drivenames[2*i+1] = '\0';
            it[numdrives].Text = &drivenames[2*i];
            if (cp) {
                if (strcmp(it[numdrives].Text, cp)==0)
                    req.CurrIt = numdrives;
            }

            it[numdrives].Left = it[i].Top = 0;
            it[numdrives].Width = 0;
            it[numdrives].Flags = 0L;
            ita[numdrives] = &it[numdrives];
            ++numdrives;
        }
    }

    SelectDisk( drive );
    ita[numdrives] = NULL;

    flags = GetProfileLong( PR_TEXFLAGS );
    if( flags & TEX_CLEAR7 )      SETBIT( g_clear.Flags, SELECTED );
    else                            CLEARBIT( g_clear.Flags, SELECTED );
    if( flags & TEX_8BITIN )      SETBIT( g_8bitin.Flags, SELECTED );
    else                            CLEARBIT( g_8bitin.Flags, SELECTED );
    if( flags & TEX_8BITOUT )     SETBIT( g_8bitout.Flags, SELECTED );
    else                            CLEARBIT( g_8bitout.Flags, SELECTED );
    if( flags & TEX_BATCHMODE )   SETBIT( g_batch.Flags, SELECTED );
    else                            CLEARBIT( g_batch.Flags, SELECTED );
    if( flags & TEX_NOEMS )       SETBIT( g_ems.Flags, SELECTED );
    else                            CLEARBIT( g_ems.Flags, SELECTED );
    if( flags & TEX_ERRSTOPMODE ) SETBIT( g_errstop.Flags, SELECTED );
    else                            CLEARBIT( g_errstop.Flags, SELECTED );
    if( flags & TEX_INITEX )      SETBIT( g_enable.Flags, SELECTED );
    else                            CLEARBIT( g_enable.Flags, SELECTED );
    if( flags & TEX_NONSTOPMODE ) SETBIT( g_nonstop.Flags, SELECTED );
    else                            CLEARBIT( g_nonstop.Flags, SELECTED );
    if( flags & TEX_CTRLCHARS )   SETBIT( g_control.Flags, SELECTED );
    else                            CLEARBIT( g_control.Flags, SELECTED );
    if( flags & TEX_SCROLLMODE )  SETBIT( g_scroll.Flags, SELECTED );
    else                            CLEARBIT( g_scroll.Flags, SELECTED );


    if( flags & TEX_PLAIN ) {
        SETBIT( g_plain.Flags, SELECTED );
        SETBIT( g_plaing.Flags, SELECTED );
        CLEARBIT( g_lplain.Flags, SELECTED );
        CLEARBIT( g_lplaing.Flags, SELECTED );
    } else {
        SETBIT( g_lplain.Flags, SELECTED );
        SETBIT( g_lplaing.Flags, SELECTED );
        CLEARBIT( g_plain.Flags, SELECTED );
        CLEARBIT( g_plaing.Flags, SELECTED );
    }
    if( flags & TEX_GERMAN ) {
        CLEARBIT( g_plain.Flags, SELECTED );
        CLEARBIT( g_lplain.Flags, SELECTED );
    } else {
        CLEARBIT( g_plaing.Flags, SELECTED );
        CLEARBIT( g_lplaing.Flags, SELECTED );
    }


    OpenWindow( &win, HLP_LATEXCMD );
    DrawBorder( &b_initex, win.Left, win.Top );
    DrawBorder( &b_format, win.Left, win.Top );


    do {
        g = OpenGadget( &g_str, win.Left, win.Top, 0L );
        if( g == &g_help ) help( -1 );
    } while( g == &g_help );

    if( g == NULL || g == &g_cancel ) goto done;

    if( packstr( si_strgad.Buffer ) == -1 )
        strcpy( si_strgad.Buffer, DEF_TEXCMD );
    PutProfileString( PR_TEXCOMMAND, si_strgad.Buffer );

    flags = 0L;
    if( g_clear.Flags & SELECTED ) SETBIT( flags, TEX_CLEAR7 );
    if( g_8bitin.Flags & SELECTED ) SETBIT( flags, TEX_8BITIN );
    if( g_8bitout.Flags & SELECTED ) SETBIT( flags, TEX_8BITOUT );
    if( g_batch.Flags & SELECTED ) SETBIT( flags, TEX_BATCHMODE );
    if( g_ems.Flags & SELECTED ) SETBIT( flags, TEX_NOEMS );
    if( g_errstop.Flags & SELECTED ) SETBIT( flags, TEX_ERRSTOPMODE );
    if( g_enable.Flags & SELECTED ) SETBIT( flags, TEX_INITEX );
    if( g_nonstop.Flags & SELECTED ) SETBIT( flags, TEX_NONSTOPMODE );
    if( g_control.Flags & SELECTED ) SETBIT( flags, TEX_CTRLCHARS );
    if( g_scroll.Flags & SELECTED ) SETBIT( flags, TEX_SCROLLMODE );

    if( (g_plaing.Flags|g_lplaing.Flags) & SELECTED )
        SETBIT( flags, TEX_GERMAN );
    if( (g_plain.Flags|g_plaing.Flags) & SELECTED )
        SETBIT( flags, TEX_PLAIN );

    PutProfileLong( PR_TEXFLAGS, flags );

    if (req.CurrIt)
        PutProfileString( PR_TEXDEFDRIVE, it[req.CurrIt].Text );
    else
        ClearProfileData( PR_TEXDEFDRIVE );

done:
    free( drivenames );
    free( si_strgad.Buffer );
    CloseWindow( &win );
}
#undef WIN_WIDTH
#undef WIN_HEIGHT



/*
 *    Name: opt_driver
 *  Return: void
 * Purpose: this routine sets the options for exe_print() and exe_preview()
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 50
#define WIN_HEIGHT 22
void opt_driver( void ) {

    struct Gadget *expand_libs( struct Gadget *, struct Gadget *, unsigned, unsigned, unsigned long );

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "Driver Options", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_add = {
        NULL, NULL, "&Additional Options:", 0, -1
    };
    static struct StringInfo CODEBASED si_add = {
        NULL, BUFSIZE-1, 0L
    };
    static struct Gadget CODEBASED g_add = {
        &g_ok, &it_add, 2, 16, 46, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_add,
    };

    static struct IntuiText CODEBASED it_fresy = {
        NULL, NULL, "y:", -2, 0
    };
    static struct StringInfo CODEBASED si_fresy = {
        NULL, 5, 0L
    };
    static struct Gadget CODEBASED g_fresy = {
        &g_add, &it_fresy, 32, 12, 8, 1, TEXTGADGET | BORDERLESS | LONGINT,
        (APTR)&si_fresy,
    };

    static struct IntuiText CODEBASED it_fresx = {
        NULL, NULL, "Font &resolution x:", -18, 0
    };
    static struct StringInfo si_fresx = {
        NULL, 5, 0L
    };
    static struct Gadget CODEBASED g_fresx = {
        &g_fresy, &it_fresx, 21, 12, 8, 1, TEXTGADGET | BORDERLESS | LONGINT,
        (APTR)&si_fresx,
    };

    static struct IntuiText CODEBASED it_resy = {
        NULL, NULL, "y:", -2, 0
    };
    static struct StringInfo CODEBASED si_resy = {
        NULL, 5, 0L
    };
    static struct Gadget CODEBASED g_resy = {
        &g_fresx, &it_resy, 32, 11, 8, 1, TEXTGADGET | BORDERLESS | LONGINT,
        (APTR)&si_resy,
    };

    static struct IntuiText CODEBASED it_resx = {
        NULL, NULL, "Devi&ce resolution x:", -20, 0
    };
    static struct StringInfo CODEBASED si_resx = {
        NULL, 5, 0L
    };
    static struct Gadget CODEBASED g_resx = {
        &g_resy, &it_resx, 21, 11, 8, 1, TEXTGADGET | BORDERLESS | LONGINT,
        (APTR)&si_resx,
    };

    static struct IntuiText CODEBASED it_scale = {
        NULL, NULL, "Font &scaling:", -13, 0
    };
    static struct StringInfo CODEBASED si_scale = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_scale = {
        &g_resx, &it_scale, 21, 9, 8, 1, TEXTGADGET | BORDERLESS | LONGINT,
        (APTR)&si_scale,
    };


    static struct IntuiText CODEBASED it_limit = {
        NULL, NULL, "F&ont load limit:", -16, 0
    };
    static struct StringInfo CODEBASED si_limit = {
        NULL, 10, 0L
    };
    static struct Gadget CODEBASED g_limit = {
        &g_scale, &it_limit, 21, 8, 8, 1, TEXTGADGET | BORDERLESS | LONGINT,
        (APTR)&si_limit,
    };

    static struct IntuiText CODEBASED it_graph = {
        NULL, NULL, "&graph path:", -11, 0
    };
    static struct StringInfo CODEBASED si_graph = {
        NULL, BUFSIZE-1, 0L
    };
    static struct Gadget CODEBASED g_graph = {
        &g_limit, &it_graph, 13, 6, 35, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_graph,
    };


    static struct IntuiText CODEBASED it_dvi = {
        NULL, NULL, "&dvi path:", -9, 0
    };
    static struct StringInfo CODEBASED si_dvi = {
        NULL, BUFSIZE-1, 0L
    };
    static struct Gadget CODEBASED g_dvi = {
        &g_graph, &it_dvi, 13, 5, 35, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_dvi,
    };

    static struct IntuiText CODEBASED it_virt = {
        NULL, NULL, "&virt path:", -10, 0
    };
    static struct StringInfo CODEBASED si_virt = {
        NULL, BUFSIZE-1, 0L
    };
    static struct Gadget CODEBASED g_virt = {
        &g_dvi, &it_virt, 13, 4, 35, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_virt,
    };

    static struct IntuiText CODEBASED it_lib = {
        NULL, NULL, "&lib path:", -9, 0
    };
    static struct StringInfo CODEBASED si_lib = {
        NULL, BUFSIZE-1, 0L
    };
    static struct Gadget CODEBASED g_lib = {
        &g_virt, &it_lib, 13, 3, 35, 1, DOEXPAND | TEXTGADGET | BORDERLESS,
        (APTR)&si_lib,
    };

    static struct IntuiText CODEBASED it_font = {
        NULL, NULL, "&font path:", -10, 0
    };
    static struct StringInfo CODEBASED si_font = {
        NULL, BUFSIZE-1, 0L
    };
    static struct Gadget CODEBASED g_font = {
        &g_lib, &it_font, 13, 2, 35, 1, TEXTGADGET | BORDERLESS | ACTIVATED,
        (APTR)&si_font,
    };

    struct Gadget *g;

    if( ! ( si_font.Buffer = AllocHeap( BUFSIZE ) ) ) goto done;
    if( ! ( si_lib.Buffer  = AllocHeap( BUFSIZE ) ) ) goto done;
    if( ! ( si_virt.Buffer = AllocHeap( BUFSIZE ) ) ) goto done;
    if( ! ( si_dvi.Buffer  = AllocHeap( BUFSIZE ) ) ) goto done;
    if( ! ( si_graph.Buffer= AllocHeap( BUFSIZE ) ) ) goto done;

    if( ! ( si_add.Buffer = AllocHeap( BUFSIZE ) ) ) goto done;

    strncpy( si_font.Buffer,  GetProfileString( PR_DRIVEROPTS(0), &profnull ), BUFSIZE );
    strncpy( si_lib.Buffer,   GetProfileString( PR_DRIVEROPTS(1), &profnull ), BUFSIZE );
    strncpy( si_virt.Buffer,  GetProfileString( PR_DRIVEROPTS(2), &profnull ), BUFSIZE );
    strncpy( si_dvi.Buffer,   GetProfileString( PR_DRIVEROPTS(3), &profnull ), BUFSIZE );
    strncpy( si_graph.Buffer, GetProfileString( PR_DRIVEROPTS(4), &profnull ), BUFSIZE );

    strncpy( si_add.Buffer,   GetProfileString( PR_DRIVEROPTS(5), &profnull ), BUFSIZE );

    si_limit.LongInt   = (long)GetProfileInt( PR_DRIVERINTS(0) );
    si_scale.LongInt   = (long)GetProfileInt( PR_DRIVERINTS(1) );
    si_resx.LongInt    = (long)GetProfileInt( PR_DRIVERINTS(2) );
    si_resy.LongInt    = (long)GetProfileInt( PR_DRIVERINTS(3) );
    si_fresx.LongInt   = (long)GetProfileInt( PR_DRIVERINTS(4) );
    si_fresy.LongInt   = (long)GetProfileInt( PR_DRIVERINTS(5) );

    OpenWindow( &win, HLP_DRIVER );
    do {
        g = OpenGadget( &g_font, win.Left, win.Top, expand_libs );
        if( g == &g_help ) help( -1 );
    } while( g == &g_help );
    CloseWindow( &win );

    if( g == &g_cancel || g == NULL )
        goto done;

    packstr( si_font.Buffer );
    PutProfileString( PR_DRIVEROPTS(0), si_font.Buffer );

    packstr( si_lib.Buffer );
    PutProfileString( PR_DRIVEROPTS(1), si_lib.Buffer );

    packstr( si_virt.Buffer );
    PutProfileString( PR_DRIVEROPTS(2), si_virt.Buffer );

    packstr( si_dvi.Buffer );
    PutProfileString( PR_DRIVEROPTS(3), si_dvi.Buffer );

    packstr( si_graph.Buffer );
    PutProfileString( PR_DRIVEROPTS(4), si_graph.Buffer );

    packstr( si_add.Buffer );
    PutProfileString( PR_DRIVEROPTS(5), si_add.Buffer );

    PutProfileInt( PR_DRIVERINTS(0), (int)si_limit.LongInt);
    PutProfileInt( PR_DRIVERINTS(1), (int)si_scale.LongInt);
    PutProfileInt( PR_DRIVERINTS(2), (int)si_resx.LongInt);
    PutProfileInt( PR_DRIVERINTS(3), (int)si_resy.LongInt);
    PutProfileInt( PR_DRIVERINTS(4), (int)si_fresx.LongInt);
    PutProfileInt( PR_DRIVERINTS(5), (int)si_fresy.LongInt);

done:
    free( si_font.Buffer  );
    free( si_lib.Buffer   );
    free( si_virt.Buffer  );
    free( si_dvi.Buffer   );
    free( si_graph.Buffer );

    free( si_add.Buffer );

}
#undef WIN_WIDTH
#undef WIN_HEIGHT

struct Gadget *expand_libs( struct Gadget *f, struct Gadget *g,
unsigned left_off, unsigned top_off, unsigned long flags ) {

    struct DiskTransferArea _far *dta;
    char *cp;
    int i;
    static char CODEBASED semicolon[] = ";";

    if(!( flags & CG_F4 ) )
        return NULL;

    cp = ((struct StringInfo *)g->SpecialInfo)->Buffer;
    if( !*cp || !SetCurrDir( cp ) ) {
        DoErrorBox( HLP_EXLIB1,
                "You must specify a valid\ndirectory before expanding" );
        return NULL;
    }

    dta = GetDTAAddress();
    if( FindFirstFile( "*.fli", 0 ) ) {
        if( cp[strlen(cp)-1] != ';' )
            strcat( cp, semicolon );

        do {
            cp += strlen(cp);
            i = strlen(dta->Name)-4;
            if( i < 0 ) i = 0;
            strncpy( cp, dta->Name, i );
            cp[i] = '\0';
            strlow( cp );
            strcat( cp, semicolon );
        } while( FindNextFile() );

        cp[strlen(cp)-1] = '\0';

        return NULL;
    }
    DoErrorBox( HLP_EXLIB2, "No font libraries found in directory\n'%s'",
                cp );
}


/*
 *    Name: opt_pages
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 67
#define WIN_HEIGHT 20
void opt_pages( void ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "Pages", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, 27, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_preview = {
        NULL, NULL, "&Use Page Options with PreView", 4, 0
    };
    static struct Gadget CODEBASED g_preview = {
        &g_ok, &it_preview, 3, 14, 28, 1, CHECKGADGET | TOGGLESELECT,
    };

    /*
     *  Range
     */
    static struct IntuiText CODEBASED it_num = {
        NULL, NULL, "&Number of Pages", -16, 0
    };
    static struct StringInfo CODEBASED si_num = {
        NULL, BUFSIZE-1, 0L
    };
    static struct Gadget CODEBASED g_num = {
        &g_preview, &it_num, 53, 11, 10, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_num,
    };

    static struct IntuiText CODEBASED it_skip = {
        NULL, NULL, "&Skip", -5, 0
    };
    static struct StringInfo CODEBASED si_skip = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_skip = {
        &g_num, &it_skip, 53, 10, 10, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_skip,
    };

    static struct IntuiText CODEBASED it_end = {
        NULL, NULL, "En&d", -4, 0
    };
    static struct StringInfo CODEBASED si_end = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_end = {
        &g_skip, &it_end, 53, 8, 10, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_end,
    };

    static struct IntuiText CODEBASED it_begin = {
        NULL, NULL, "Be&gin", -6, 0
    };
    static struct StringInfo CODEBASED si_begin = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_begin = {
        &g_end, &it_begin, 53, 7, 10, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_begin,
    };

    /*
     * Combine Pages group
     */

    static struct IntuiText CODEBASED it_index = {
        NULL, NULL, "&Index", -6, 0
    };
    static struct StringInfo CODEBASED si_index = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_index = {
        &g_begin, &it_index, 14, 12, 20, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_index,
    };

    static struct IntuiText CODEBASED it_pages = {
        NULL, NULL, "&Pages", -6, 0
    };
    static struct StringInfo CODEBASED si_pages = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_pages = {
        &g_index, &it_pages, 29, 10, 5, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_pages,
    };

    static struct IntuiText CODEBASED it_first = {
        NULL, NULL, "&First", -6, 0
    };
    static struct StringInfo CODEBASED si_first = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_first = {
        &g_pages, &it_first, 14, 10, 5, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_first,
    };

    /*
     * Selection group
     */
    static struct IntuiText CODEBASED it_back = {
        NULL, NULL, "&Backwards", 4, 0
    };
    static struct Gadget CODEBASED g_back = {
        &g_first, &it_back, 37, 3, 13, 1, CHECKGADGET | TOGGLESELECT,
    };
    static struct IntuiText CODEBASED it_all = {
        NULL, NULL, "&All", 4, 0
    };
    static struct Gadget CODEBASED g_all = {
        &g_back, &it_all, 56, 2, 7, 1, OPTIONGADGET | TOGGLESELECT,
        NULL, 1
    };
    static struct IntuiText CODEBASED it_even = {
        NULL, NULL, "&Even", 4, 0
    };
    static struct Gadget CODEBASED g_even = {
        &g_all, &it_even, 46, 2, 8, 1, OPTIONGADGET | TOGGLESELECT,
        NULL, 1
    };
    static struct IntuiText CODEBASED it_odd = {
        NULL, NULL, "&Odd", 4, 0
    };
    static struct Gadget CODEBASED g_odd = {
        &g_even, &it_odd, 37, 2, 7, 1, OPTIONGADGET | TOGGLESELECT,
        NULL, 1
    };


    /*
     * Layout group
     */
    static struct IntuiText CODEBASED it_trans = {
        NULL, NULL, "T&ransformation", -15, 0
    };
    static struct StringInfo CODEBASED si_trans = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_trans = {
        &g_odd, &it_trans, 29, 7, 5, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_trans,
    };

    static struct IntuiText CODEBASED it_tadj = {
        NULL, NULL, "Adjust", -7, 0
    };
    static struct StringInfo CODEBASED si_tadj = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_tadj = {
        &g_trans, &it_tadj, 29, 5, 5, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_tadj,
    };

    static struct IntuiText CODEBASED it_tmargin = {
        NULL, NULL, "Top &Margin", -11, 0
    };
    static struct StringInfo CODEBASED si_tmargin = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_tmargin = {
        &g_tadj, &it_tmargin, 16, 5, 5, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_tmargin,
    };

    static struct IntuiText CODEBASED it_ladj = {
        NULL, NULL, "Adjust", -7, 0
    };
    static struct StringInfo CODEBASED si_ladj = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_ladj = {
        &g_tmargin, &it_ladj, 29, 4, 5, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_ladj,
    };

    static struct IntuiText CODEBASED it_lmargin = {
        NULL, NULL, "&Left Margin", -12, 0
    };
    static struct StringInfo CODEBASED si_lmargin = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_lmargin = {
        &g_ladj, &it_lmargin, 16, 4, 5, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_lmargin,
    };

    static struct IntuiText CODEBASED it_width = {
        NULL, NULL, "&Width", -6, 0
    };
    static struct StringInfo CODEBASED si_width = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_width = {
        &g_lmargin, &it_width, 16, 3, 5, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_width,
    };

    static struct IntuiText CODEBASED it_height = {
        NULL, NULL, "Heigh&t", -7, 0
    };
    static struct StringInfo CODEBASED si_height = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_height = {
        &g_width, &it_height, 16, 2, 5, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_height,
    };

    static struct Border CODEBASED b_range = { NULL, "Range", 35, 5, 31, 9 };
    static struct Border CODEBASED b_combine = { &b_range, "Combine Pages", 1, 9, 34, 5 };
    static struct Border CODEBASED b_select = { &b_combine, "Selection", 35, 1, 31, 4 };
    static struct Border CODEBASED b_layout = { &b_select, "Layout", 1, 1, 34, 8 };

    struct Gadget *g;
    struct StringInfo *si;

    int i;
    unsigned long flags;

    for( i = 0, g = &g_height; g; g = g->Next ) {
        if( g->Flags & (CHECKGADGET|OPTIONGADGET|BOOLGADGET) )
            continue;

        si = (struct StringInfo *)g->SpecialInfo;
        si->Buffer = AllocHeap( 100 );
        if( si->Buffer == NULL )
            goto done;
        strcpy( si->Buffer, GetProfileString( PR_P_HEIGHT+i, &profnull ) );
        ++i;
    }

    flags = GetProfileLong( PR_P_FLAGS );
    if( flags & PAGE_ODD )   SETBIT( g_odd.Flags, SELECTED );
    else                   CLEARBIT( g_odd.Flags, SELECTED );
    if( flags & PAGE_EVEN )  SETBIT( g_even.Flags, SELECTED );
    else                   CLEARBIT( g_even.Flags, SELECTED );
    if( flags & PAGE_ALL )   SETBIT( g_all.Flags, SELECTED );
    else                   CLEARBIT( g_all.Flags, SELECTED );
    if( flags & PAGE_BACK )  SETBIT( g_back.Flags, SELECTED );
    else                   CLEARBIT( g_back.Flags, SELECTED );
    if( flags & PAGE_PREVIEW )  SETBIT( g_preview.Flags, SELECTED );
    else                        CLEARBIT( g_preview.Flags, SELECTED );



    if( OpenWindow( &win, HLP_PAGES ) ) {
        DrawBorder( &b_layout, win.Left, win.Top );
        do {
            g = OpenGadget( &g_height, win.Left, win.Top, NULL );
            if( g == &g_help ) help( -1 );
        } while( g == &g_help );
        CloseWindow( &win );
    }

    if( g && g != &g_cancel ) {
        for( i = 0, g = &g_height; g; g = g->Next ) {
            if( g->Flags & (CHECKGADGET|OPTIONGADGET|BOOLGADGET) )
                continue;

            si = (struct StringInfo *)g->SpecialInfo;
            if( packstr( si->Buffer ) == -1 )
                ClearProfileData( PR_P_HEIGHT+i );
            else
                PutProfileString( PR_P_HEIGHT+i, si->Buffer );
            ++i;
        }
        flags = 0L;
        if( g_odd.Flags & SELECTED ) SETBIT( flags, PAGE_ODD );
        if( g_even.Flags & SELECTED ) SETBIT( flags, PAGE_EVEN );
        if( g_all.Flags & SELECTED ) SETBIT( flags, PAGE_ALL );
        if( g_back.Flags & SELECTED ) SETBIT( flags, PAGE_BACK );
        if( g_preview.Flags & SELECTED ) SETBIT( flags, PAGE_PREVIEW );
        PutProfileLong( PR_P_FLAGS, flags );

    }
done:
    for( g = &g_height; g; g = g->Next ) {
        if( g->Flags & (CHECKGADGET|OPTIONGADGET|BOOLGADGET) )
            continue;

        si = (struct StringInfo *)g->SpecialInfo;
        free( si->Buffer );
    }
}
#undef WIN_WIDTH
#undef WIN_HEIGHT


/*
 *    Name: exe_print
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 52
#define WIN_HEIGHT 15
void exe_print( void ) {

    struct Gadget *expand_pfile( struct Gadget *, struct Gadget *, unsigned, unsigned, unsigned long );

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "Print", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };

    static struct IntuiText CODEBASED it_ok = {
        NULL, NULL, "&OK", 1, 0
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, &it_ok, (WIN_WIDTH-12)/2-3, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText CODEBASED it_print = {
        NULL, NULL, "Print", 1, 0
    };
    static struct Gadget CODEBASED g_print = {
        &g_ok, &it_print, 2, WIN_HEIGHT-4,
        9, 3, BOOLGADGET | ENDGADGET | ACTIVATED,
    };

    static struct IntuiText CODEBASED it_add = {
        NULL, NULL, "&Additional Options:", 0, -1
    };
    static struct StringInfo CODEBASED si_add = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_add = {
        &g_print, &it_add, 2, 10, 47, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_add,
    };

    static struct IntuiText CODEBASED it_pages = {
        NULL, NULL, "Pa&ges...", 1, 0
    };
    static struct Gadget CODEBASED g_pages = {
        &g_add, &it_pages, 36, 6,
        12, 3, BOOLGADGET | ENDGADGET,
    };

    static struct StringInfo CODEBASED si_file = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_file = {
        &g_pages, NULL, 6, 7, 23, 1, DOEXPAND | TEXTGADGET | BORDERLESS,
        (APTR)&si_file,
    };

    static struct IntuiText CODEBASED it_file = {
        NULL, NULL, "Print stand alone &file:", 4, 0
    };
    static struct Gadget CODEBASED g_filecheck = {
        &g_file, &it_file, 2, 6, 27, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_qual = {
        NULL, NULL, "High &Quality", 4, 0
    };
    static struct Gadget CODEBASED g_qual = {
        &g_filecheck, &it_qual, 33, 4, 16, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_post = {
        NULL, NULL, "&Post Repeat", -12, 0
    };
    static struct StringInfo CODEBASED si_post = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_post = {
        &g_qual, &it_post, 38, 3, 11, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_post,
    };

    static struct IntuiText CODEBASED it_repeat = {
        NULL, NULL, "&Repeat", -7, 0
    };
    static struct StringInfo CODEBASED si_repeat = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_repeat = {
        &g_post, &it_repeat, 38, 2, 11, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_repeat,
    };


    static struct IntuiText CODEBASED it_output = {
        NULL, NULL, "Ou&tput:", -7, 0
    };
    static struct StringInfo CODEBASED si_output = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_output = {
        &g_repeat, &it_output, 9, 2, 12, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_output,
    };

    int fptr;
    char buffer[BUFSIZE];
    char tmpfilename[BUFSIZE];
    char *cp;

    struct Gadget *g;
    int oddeven;
    unsigned long flags;

    if( ! ( si_output.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    strcpy( si_output.Buffer, GetProfileString( PR_PRINTOUTPUT, &profnull ) );

    if( ! ( si_add.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    strcpy( si_add.Buffer, GetProfileString( PR_PRINTADD, &profnull ) );

    if( ! ( si_file.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    strcpy( si_file.Buffer, GetProfileString( PR_PRINTSTANDALONE, &profnull ) );

    if( ! ( si_repeat.Buffer = AllocHeap( 100 ) ) ) goto done;
    strcpy( si_repeat.Buffer, GetProfileString( PR_PRINTREPEAT, &profnull ) );

    if( ! ( si_post.Buffer = AllocHeap( 100 ) ) ) goto done;
    strcpy( si_post.Buffer, GetProfileString( PR_PRINTPOSTREP, &profnull ) );

    flags = GetProfileLong( PR_PRINTFLAGS );
    if( flags & PRINT_HIGHQUAL ) SETBIT( g_qual.Flags, SELECTED );
    else                         CLEARBIT( g_qual.Flags, SELECTED );

    OpenWindow( &win, HLP_PRINT );

    for ever {
        g = OpenGadget( &g_output, win.Left, win.Top, expand_pfile );

        if( g == &g_help ) help( -1 );
        else if( g == &g_pages ) opt_pages();
        else break;
    }

    CloseWindow( &win );

    if( g == &g_cancel || g == NULL )
        goto done;

    flags = GetProfileLong( PR_PRINTFLAGS );
    if( g_qual.Flags & SELECTED )
        SETBIT( flags, PRINT_HIGHQUAL );
    else
        CLEARBIT( flags, PRINT_HIGHQUAL );
    PutProfileLong( PR_PRINTFLAGS, flags );

    if( packstr( si_output.Buffer ) == -1 )
        strcpy( si_output.Buffer, DEF_PRINTOUT );
    PutProfileString( PR_PRINTOUTPUT, si_output.Buffer );
    PutProfileString( PR_PRINTREPEAT, si_repeat.Buffer );
    PutProfileString( PR_PRINTPOSTREP, si_post.Buffer );
    PutProfileString( PR_PRINTADD, si_add.Buffer );
    PutProfileString( PR_PRINTSTANDALONE, si_file.Buffer );

    if( g == &g_ok )
        goto done;

    if( g_filecheck.Flags & SELECTED ) {
        if( chkdir(1) != TRUE )
            return;
        cp = si_file.Buffer;
    } else {
        if( chkdir(0) != TRUE )
            return;
        cp = NULL;
    }

    if (smakec( buffer, GetProfileString( PR_PRINTCOMMAND, DEF_PRINTCMD ), cp))
        LaunchOverlay( buffer, 0 );

done:
    free( si_add.Buffer );
    free( si_output.Buffer );
    free( si_file.Buffer );
    free( si_repeat.Buffer );
    free( si_post.Buffer );

    ClearProfileData( PR_RESPONSEFILE );
}
#undef WIN_WIDTH
#undef WIN_HEIGHT

struct Gadget *expand_pfile( struct Gadget *f, struct Gadget *g,
    unsigned left_off, unsigned top_off, unsigned long flags ) {

    char *cp;

    if(!( flags & CG_F4 ) )
        return NULL;

    cp = ((struct StringInfo *)g->SpecialInfo)->Buffer;
    DOS_Request( "Print Stand Alone File", cp, cp, -1 );
    return NULL;
}

/*
 *    Name: opt_print
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 50
#define WIN_HEIGHT 13
void opt_print( void ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "Print Options", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_xonxoff = {
        NULL, NULL, "&XON/XOFF-Protocol", 4, 0
    };
    static struct Gadget CODEBASED g_xonxoff = {
        &g_ok, &it_xonxoff, 26, 7, 21, 1, CHECKGADGET | TOGGLESELECT,
    };
    static struct IntuiText CODEBASED it_qual = {
        NULL, NULL, "High &Quality", 4, 0
    };
    static struct Gadget CODEBASED g_qual = {
        &g_xonxoff, &it_qual, 26, 6, 16, 1, CHECKGADGET | TOGGLESELECT,
    };
    static struct IntuiText CODEBASED it_init = {
        NULL, NULL, "No &Init Printer", 4, 0
    };
    static struct Gadget CODEBASED g_init = {
        &g_qual, &it_init, 26, 5, 19, 1, CHECKGADGET | TOGGLESELECT,
    };
    static struct IntuiText CODEBASED it_offset = {
        NULL, NULL, "&Job Offset", 4, 0
    };
    static struct Gadget CODEBASED g_offset = {
        &g_init, &it_offset, 6, 7, 14, 1, CHECKGADGET | TOGGLESELECT,
    };
    static struct IntuiText CODEBASED it_descjet = {
        NULL, NULL, "&Descjet", 4, 0
    };
    static struct Gadget CODEBASED g_descjet = {
        &g_offset, &it_descjet, 6, 6, 11, 1, CHECKGADGET | TOGGLESELECT,
        NULL, 1L
    };
    static struct IntuiText CODEBASED it_kyocera = {
        NULL, NULL, "&Kyocera", 4, 0
    };
    static struct Gadget CODEBASED g_kyocera = {
        &g_descjet, &it_kyocera, 6, 5, 11, 1, CHECKGADGET | TOGGLESELECT,
        NULL, 1L
    };

    static struct IntuiText CODEBASED it_strgad = {
        NULL, NULL, "&Command Line:", 0, -1
    };
    static struct StringInfo CODEBASED si_strgad = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_str = {
        &g_kyocera, &it_strgad, 2, 3, 46, 1, TEXTGADGET | BORDERLESS | ACTIVATED,
        (APTR)&si_strgad,
    };

    struct Gadget *g;
    unsigned long flags;

    if( ! ( si_strgad.Buffer = AllocHeap( BUFSIZE+1 ) ) ) return;
    strcpy( si_strgad.Buffer,
        GetProfileString( PR_PRINTCOMMAND, DEF_PRINTCMD ) );

    flags = GetProfileLong( PR_PRINTFLAGS );
    if( flags & PRINT_XONXOFF  ) SETBIT( g_xonxoff.Flags, SELECTED );
    else                         CLEARBIT( g_xonxoff.Flags, SELECTED );
    if( flags & PRINT_HIGHQUAL ) SETBIT( g_qual.Flags, SELECTED );
    else                         CLEARBIT( g_qual.Flags, SELECTED );
    if( flags & PRINT_INITPRINT) SETBIT( g_init.Flags, SELECTED );
    else                         CLEARBIT( g_init.Flags, SELECTED );
    if( flags & PRINT_JOBOFFSET) SETBIT( g_offset.Flags, SELECTED );
    else                         CLEARBIT( g_offset.Flags, SELECTED );
    if( flags & PRINT_DESCJET  ) SETBIT( g_descjet.Flags, SELECTED );
    else                         CLEARBIT( g_descjet.Flags, SELECTED );
    if( flags & PRINT_KYOCERA  ) SETBIT( g_kyocera.Flags, SELECTED );
    else                         CLEARBIT( g_kyocera.Flags, SELECTED );

    if( OpenWindow( &win, HLP_PRINTCMD ) ) {
        do {
            g = OpenGadget( &g_str, win.Left, win.Top, 0L );
            if( g == &g_help ) help( -1 );
        } while( g == &g_help );

        CloseWindow( &win );

        if( g && g != &g_cancel ) {
            if( packstr( si_strgad.Buffer ) == -1 )
                strcpy( si_strgad.Buffer, DEF_PRINTCMD );
            PutProfileString( PR_PRINTCOMMAND, si_strgad.Buffer );

            flags = 0L;
            if( g_xonxoff.Flags & SELECTED ) SETBIT( flags, PRINT_XONXOFF );
            if( g_qual.Flags & SELECTED ) SETBIT( flags, PRINT_HIGHQUAL );
            if( g_init.Flags & SELECTED ) SETBIT( flags, PRINT_INITPRINT );
            if( g_offset.Flags & SELECTED ) SETBIT( flags, PRINT_JOBOFFSET );
            if( g_descjet.Flags & SELECTED ) SETBIT( flags, PRINT_DESCJET );
            if( g_kyocera.Flags & SELECTED ) SETBIT( flags, PRINT_KYOCERA );
            PutProfileLong( PR_PRINTFLAGS, flags );
        }
        free( si_strgad.Buffer );
    }
}
#undef WIN_WIDTH
#undef WIN_HEIGHT

/*
 *    Name: exe_pscript
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 52
#define WIN_HEIGHT 15
void exe_pscript( void ) {

    struct Gadget *expand_pfile( struct Gadget *, struct Gadget *, unsigned, unsigned, unsigned long );

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "Postscript", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };

    static struct IntuiText CODEBASED it_ok = {
        NULL, NULL, "&OK", 1, 0
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, &it_ok, (WIN_WIDTH-12)/2-3, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText CODEBASED it_print = {
        NULL, NULL, "Print", 1, 0
    };
    static struct Gadget CODEBASED g_print = {
        &g_ok, &it_print, 2, WIN_HEIGHT-4,
        9, 3, BOOLGADGET | ENDGADGET | ACTIVATED,
    };

    static struct IntuiText CODEBASED it_add = {
        NULL, NULL, "&Additional Options:", 0, -1
    };
    static struct StringInfo CODEBASED si_add = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_add = {
        &g_print, &it_add, 2, 10, 47, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_add,
    };

    static struct IntuiText CODEBASED it_pages = {
        NULL, NULL, "Pa&ges...", 1, 0
    };
    static struct Gadget CODEBASED g_pages = {
        &g_add, &it_pages, 36, 6,
        12, 3, BOOLGADGET | ENDGADGET,
    };

    static struct StringInfo CODEBASED si_file = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_file = {
        &g_pages, NULL, 6, 7, 23, 1, DOEXPAND | TEXTGADGET | BORDERLESS,
        (APTR)&si_file,
    };

    static struct IntuiText CODEBASED it_file = {
        NULL, NULL, "Print stand alone &file:", 4, 0
    };
    static struct Gadget CODEBASED g_filecheck = {
        &g_file, &it_file, 2, 6, 27, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_manual = {
        NULL, NULL, "&Manual Feed", 4, 0
    };
    static struct Gadget CODEBASED g_manual = {
        &g_filecheck, &it_manual, 33, 4, 16, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_post = {
        NULL, NULL, "&Post Repeat", -12, 0
    };
    static struct StringInfo CODEBASED si_post = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_post = {
        &g_manual, &it_post, 38, 3, 11, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_post,
    };

    static struct IntuiText CODEBASED it_repeat = {
        NULL, NULL, "&Repeat", -7, 0
    };
    static struct StringInfo CODEBASED si_repeat = {
        NULL, 100, 0L
    };
    static struct Gadget CODEBASED g_repeat = {
        &g_post, &it_repeat, 38, 2, 11, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_repeat,
    };


    static struct IntuiText CODEBASED it_output = {
        NULL, NULL, "Ou&tput:", -7, 0
    };
    static struct StringInfo CODEBASED si_output = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_output = {
        &g_repeat, &it_output, 9, 2, 12, 1, TEXTGADGET|BORDERLESS,
        (APTR)&si_output,
    };

    int fptr;
    char buffer[BUFSIZE];
    char tmpfilename[BUFSIZE];
    char *cp;

    struct Gadget *g;
    int oddeven;
    unsigned long flags;

    if( ! ( si_output.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    strcpy( si_output.Buffer, GetProfileString( PR_PSCRIPTOUTPUT, &profnull ) );

    if( ! ( si_add.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    strcpy( si_add.Buffer, GetProfileString( PR_PSCRIPTADD, &profnull ) );

    if( ! ( si_file.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    strcpy( si_file.Buffer, GetProfileString( PR_PRINTSTANDALONE, &profnull ) );

    if( ! ( si_repeat.Buffer = AllocHeap( 100 ) ) ) goto done;
    strcpy( si_repeat.Buffer, GetProfileString( PR_PRINTREPEAT, &profnull ) );

    if( ! ( si_post.Buffer = AllocHeap( 100 ) ) ) goto done;
    strcpy( si_post.Buffer, GetProfileString( PR_PRINTPOSTREP, &profnull ) );

    flags = GetProfileLong( PR_PRINTFLAGS );
    if( flags & PRINT_MANUAL )   SETBIT( g_manual.Flags, SELECTED );
    else                         CLEARBIT( g_manual.Flags, SELECTED );

    OpenWindow( &win, HLP_PRINT );

    for ever {
        g = OpenGadget( &g_output, win.Left, win.Top, expand_pfile );

        if( g == &g_help ) help( -1 );
        else if( g == &g_pages ) opt_pages();
        else break;
    }

    CloseWindow( &win );

    if( g == &g_cancel || g == NULL )
        goto done;

    flags = GetProfileLong( PR_PRINTFLAGS );
    if( g_manual.Flags & SELECTED )
        SETBIT( flags, PRINT_MANUAL );
    else
        CLEARBIT( flags, PRINT_MANUAL );
    PutProfileLong( PR_PRINTFLAGS, flags );

    if( packstr( si_output.Buffer ) == -1 )
        strcpy( si_output.Buffer, DEF_PRINTOUT );
    PutProfileString( PR_PSCRIPTOUTPUT, si_output.Buffer );
    PutProfileString( PR_PRINTREPEAT, si_repeat.Buffer );
    PutProfileString( PR_PRINTPOSTREP, si_post.Buffer );
    PutProfileString( PR_PRINTADD, si_add.Buffer );
    PutProfileString( PR_PRINTSTANDALONE, si_file.Buffer );

    if( g == &g_ok )
        goto done;

    if( g_filecheck.Flags & SELECTED ) {
        if( chkdir(1) != TRUE )
            return;
        cp = si_file.Buffer;
    } else {
        if( chkdir(0) != TRUE )
            return;
        cp = NULL;
    }

    if(smakec( buffer, GetProfileString( PR_PSCRIPTCMD, DEF_PSCRIPTCMD ), cp))
        LaunchOverlay( buffer, 0 );

done:
    free( si_add.Buffer );
    free( si_output.Buffer );
    free( si_file.Buffer );
    free( si_repeat.Buffer );
    free( si_post.Buffer );

}
#undef WIN_WIDTH
#undef WIN_HEIGHT

/*
 *    Name: opt_pscript
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 50
#define WIN_HEIGHT 9
void opt_pscript( void ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "Postscript Options", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_strgad = {
        NULL, NULL, "&Command Line:", 0, -1
    };
    static struct StringInfo CODEBASED si_strgad = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_str = {
        &g_ok, &it_strgad, 2, 3, 46, 1, TEXTGADGET | BORDERLESS | ACTIVATED,
        (APTR)&si_strgad,
    };

    struct Gadget *g;

    if( ! ( si_strgad.Buffer = AllocHeap( BUFSIZE+1 ) ) ) return;
    strcpy( si_strgad.Buffer,
        GetProfileString( PR_PSCRIPTCMD, DEF_PSCRIPTCMD ) );

    if( OpenWindow( &win, HLP_DEFAULT ) ) {
        do {
            g = OpenGadget( &g_str, win.Left, win.Top, 0L );
            if( g == &g_help ) help( -1 );
        } while( g == &g_help );

        CloseWindow( &win );

        if( g && g != &g_cancel ) {
            if( packstr( si_strgad.Buffer ) == -1 )
                strcpy( si_strgad.Buffer, DEF_PSCRIPTCMD );
            PutProfileString( PR_PSCRIPTCMD, si_strgad.Buffer );

        }
        free( si_strgad.Buffer );
    }
}
#undef WIN_WIDTH
#undef WIN_HEIGHT


/*
 *    Name: exe_preview
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
void exe_preview( void ) {

    char buffer[BUFSIZE];

    if( chkdir(0) != TRUE )
        return;

    if (smakec( buffer, GetProfileString( PR_PREVIEWCOMMAND, DEF_PREVIEWCMD ), NULL))
        LaunchOverlay( buffer, 0 );

    ClearProfileData( PR_RESPONSEFILE );
}

/*
 *    Name: opt_preview
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 65
#define WIN_HEIGHT 19
// #pragma optimize( "elg", off )
void opt_preview( void ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "PreView Options", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_add = {
        NULL, NULL, "&Additional Options:", 0, -1
    };
    static struct StringInfo CODEBASED si_add = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_add = {
        &g_ok, &it_add, 2, 13, 61, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_add,
    };

    static struct IntuiText CODEBASED it_unit = {
        NULL, NULL, "&Unit of Measure:", -16, 0
    };
    static struct StringInfo CODEBASED si_unit = {
        NULL, 10, 0L
    };
    static struct Gadget CODEBASED g_unit = {
        &g_add, &it_unit, 46, 9, 6, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_unit,
    };

    static struct IntuiText CODEBASED it_scaley = {
        NULL, NULL, "y:", -2, 0
    };
    static struct StringInfo CODEBASED si_scaley = {
        NULL, 10, 0L
    };
    static struct Gadget CODEBASED g_scaley = {
        &g_unit, &it_scaley, 57, 8, 6, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_scaley,
    };

    static struct IntuiText CODEBASED it_scalex = {
        NULL, NULL, "&Scaling x:", -10, 0
    };
    static struct StringInfo CODEBASED si_scalex = {
        NULL, 10, 0L
    };
    static struct Gadget CODEBASED g_scalex = {
        &g_scaley, &it_scalex, 46, 8, 6, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_scalex,
    };

    static struct IntuiText CODEBASED it_grayy = {
        NULL, NULL, "y:", -2, 0
    };
    static struct StringInfo CODEBASED si_grayy = {
        NULL, 10, 0L
    };
    static struct Gadget CODEBASED g_grayy = {
        &g_scalex, &it_grayy, 57, 7, 6, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_grayy,
    };

    static struct IntuiText CODEBASED it_grayx = {
        NULL, NULL, "&Gray Scaling x:", -15, 0
    };
    static struct StringInfo CODEBASED si_grayx = {
        NULL, 10, 0L
    };
    static struct Gadget CODEBASED g_grayx = {
        &g_grayy, &it_grayx, 46, 7, 6, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_grayx,
    };

    static struct IntuiText CODEBASED it_homey = {
        NULL, NULL, "y:", -2, 0
    };
    static struct StringInfo CODEBASED si_homey = {
        NULL, 10, 0L
    };
    static struct Gadget CODEBASED g_homey = {
        &g_grayx, &it_homey, 57, 6, 6, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_homey,
    };

    static struct IntuiText CODEBASED it_homex = {
        NULL, NULL, "Home &Position x:", -16, 0
    };
    static struct StringInfo CODEBASED si_homex = {
        NULL, 10, 0L
    };
    static struct Gadget CODEBASED g_homex = {
        &g_homey, &it_homex, 46, 6, 6, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_homex,
    };

    static struct IntuiText CODEBASED it_nostat = {
        NULL, NULL, "&No Status Line", 4, 0
    };
    static struct Gadget CODEBASED g_nostat = {
        &g_homex, &it_nostat, 6, 9, 18, 1, CHECKGADGET | TOGGLESELECT,
        NULL,
    };
    static struct IntuiText CODEBASED it_ruler = {
        NULL, NULL, "Show &Ruler", 4, 0
    };
    static struct Gadget CODEBASED g_ruler = {
        &g_nostat, &it_ruler, 6, 8, 18, 1, CHECKGADGET | TOGGLESELECT,
        NULL,
    };
    static struct IntuiText CODEBASED it_white = {
        NULL, NULL, "&White on Black", 4, 0
    };
    static struct Gadget CODEBASED g_white = {
        &g_ruler, &it_white, 6, 7, 18, 1, CHECKGADGET | TOGGLESELECT,
        NULL,
    };

    static struct Border CODEBASED b_opt = {
        NULL, "Options", 4, 6, 22, 5 };

    static struct StringInfo CODEBASED si_video = {
        NULL, 49, 0L
    };
    static struct Gadget CODEBASED g_video = {
        &g_white, NULL, 53, 4, 10, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_video,
    };

    static char *CODEBASED modes[] = {
        "Default",
        "CGA 640x200",
        "EGA 640x350 or 640x200",
        "VGA 640x480",
        "HGC 720x348",
        "Genoa SuperEGA 800x600",
        "Paradise VGA 800x600",
        "Video Seven VGA 800x600",
        "Tseng Labs EVA 800x600",
        "HP Vectra 640x400",
        "MDS Genius and PRIMUS II 728x1008"
        "EGA 640x200, 16 colors",
        "EGA 640x350, 16 colors",
        "EGA 640x350, 16 of 64 colors",
        "VGA 640x480, 16 colors",
        "VGA 640x480, 16 of 64 colors",
        "VGA 640x480, 16 of 262144 colors",
        "EGA 640x350, 2 colors",
        "VGA 640x480, 2 colors",
        "Olivetti M24, 640x400",
        "Customize",
    };

    struct IntuiText *ita[22];
    struct IntuiText it[22];
    static struct Request CODEBASED req = {
        NULL, NULL, MULTISELECT };

    static struct IntuiText CODEBASED it_adapter = {
        NULL, NULL, "&Video Adapter:", -14, 0
    };
    static struct Gadget CODEBASED gad_pop = {
        &g_video, &it_adapter, 15, 4, 37, 1, POPGADGET,
        (APTR)&req,
    };

    static struct IntuiText CODEBASED it_cmd = {
        NULL, NULL, "&Command Line:", -13, 0
    };
    static struct StringInfo CODEBASED si_cmd = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_cmd = {
        &gad_pop, &it_cmd, 15, 2, 48, 1, TEXTGADGET | BORDERLESS | ACTIVATED,
        (APTR)&si_cmd,
    };

    struct Gadget *g;
    int i;
    unsigned long flags;

    req.Items = ita;

    if( ! (    si_add.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    if( ! (    si_cmd.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    if( ! (  si_video.Buffer = AllocHeap(       50 ) ) ) goto done;
    if( ! (  si_homex.Buffer = AllocHeap(       50 ) ) ) goto done;
    if( ! (  si_homey.Buffer = AllocHeap(       50 ) ) ) goto done;
    if( ! ( si_scalex.Buffer = AllocHeap(       50 ) ) ) goto done;
    if( ! ( si_scaley.Buffer = AllocHeap(       50 ) ) ) goto done;
    if( ! (  si_grayx.Buffer = AllocHeap(       50 ) ) ) goto done;
    if( ! (  si_grayy.Buffer = AllocHeap(       50 ) ) ) goto done;
    if( ! (   si_unit.Buffer = AllocHeap(       50 ) ) ) goto done;


    strcpy( si_add.Buffer,
        GetProfileString( PR_PREVIEWADD, &profnull ) );
    strcpy( si_cmd.Buffer,
        GetProfileString( PR_PREVIEWCOMMAND, DEF_PREVIEWCMD ) );
    strcpy( si_video.Buffer,
        GetProfileString( PR_PREVIEWADAPTER, &profnull ) );
    strcpy( si_homex.Buffer,
        GetProfileString( PR_PREVIEWHP, &profnull ) );
    strcpy( si_homey.Buffer,
        GetProfileString( PR_PREVIEWVP, &profnull ) );
    strcpy( si_grayx.Buffer,
        GetProfileString( PR_PREVIEWGX, &profnull ) );
    strcpy( si_grayy.Buffer,
        GetProfileString( PR_PREVIEWGY, &profnull ) );
    strcpy( si_scalex.Buffer,
        GetProfileString( PR_PREVIEWSX, &profnull ) );
    strcpy( si_scaley.Buffer,
        GetProfileString( PR_PREVIEWSY, &profnull ) );
    strcpy( si_unit.Buffer,
        GetProfileString( PR_PREVIEWUNIT, &profnull ) );

    i = atoi( si_video.Buffer );
    if( i <= 19 )
        req.CurrIt = i;
    else
        req.CurrIt = 20;

    for( i = 0; i < 20; ++i ) {
        it[i].Next = NULL;
        it[i].TextBorder = NULL;
        it[i].Text = modes[i];
        it[i].Left = it[i].Top = 0;
        it[i].Width = 0;
        it[i].Flags = 0L;
        ita[i] = &it[i];
    }

    ita[i] = NULL;

    flags = GetProfileLong( PR_PREVIEWFLAGS );
    if( flags & PREVIEW_WHITEONBLACK ) SETBIT( g_white.Flags, SELECTED );
    else                               CLEARBIT( g_white.Flags, SELECTED );
    if( flags & PREVIEW_RULER )        SETBIT( g_ruler.Flags, SELECTED );
    else                               CLEARBIT( g_ruler.Flags, SELECTED );
    if( flags & PREVIEW_NOSTATUS )     SETBIT( g_nostat.Flags, SELECTED );
    else                               CLEARBIT( g_nostat.Flags, SELECTED );

    OpenWindow( &win, HLP_PREVIEWCMD );
    DrawBorder( &b_opt, win.Left, win.Top );

    do {
        g = OpenGadget( &g_cmd, win.Left, win.Top, NULL );
        if( g == &g_help ) help( -1 );
    } while( g == &g_help );
    CloseWindow( &win );

    if( g && g != &g_cancel ) {
        if( packstr( si_cmd.Buffer ) == -1 )
            strcpy( si_cmd.Buffer, DEF_PREVIEWCMD );
        PutProfileString( PR_PREVIEWADD, si_add.Buffer );
        PutProfileString( PR_PREVIEWCOMMAND, si_cmd.Buffer );
        if( req.CurrIt != 20 )
            sprintf( si_video.Buffer, "%d", req.CurrIt );
        PutProfileString( PR_PREVIEWADAPTER, si_video.Buffer );

        PutProfileString( PR_PREVIEWHP, si_homex.Buffer );
        PutProfileString( PR_PREVIEWVP, si_homey.Buffer );
        PutProfileString( PR_PREVIEWGX, si_grayx.Buffer );
        PutProfileString( PR_PREVIEWGY, si_grayy.Buffer );
        PutProfileString( PR_PREVIEWSX, si_scalex.Buffer );
        PutProfileString( PR_PREVIEWSY, si_scaley.Buffer );
        PutProfileString( PR_PREVIEWUNIT, si_unit.Buffer );

        flags = 0L;
        if( g_white.Flags & SELECTED )  SETBIT( flags, PREVIEW_WHITEONBLACK );
        if( g_ruler.Flags & SELECTED )  SETBIT( flags, PREVIEW_RULER );
        if( g_nostat.Flags & SELECTED ) SETBIT( flags, PREVIEW_NOSTATUS );
        PutProfileLong( PR_PREVIEWFLAGS, flags );
   }

done:
    free( si_cmd.Buffer    );
    free( si_add.Buffer    );
    free( si_video.Buffer  );
    free( si_homex.Buffer  );
    free( si_homey.Buffer  );
    free( si_scalex.Buffer );
    free( si_scaley.Buffer );
    free( si_grayx.Buffer  );
    free( si_grayy.Buffer  );
    free( si_unit.Buffer   );
}
// #pragma optimize( "", on )
#undef WIN_WIDTH
#undef WIN_HEIGHT


/*
 *    Name: chkdir
 *  Return: TRUE if successful
 *          FALSE routine aborted (e.g. PSEUDOFILE)
 *          CANCEL aborted by user
 *
 * Purpose: checks if the current directory equals the project directory
 *
 *
 *
 * (c)M.Schollmeyer
 */
int chkdir( BOOL flags ) {

    char project[BUFSIZE], current[BUFSIZE];
    unsigned int ret;
    int currdisk;

    if( ed.Flags & PSEUDOFILE && !flags ) {
        DoMessageBox( DMB_OK, HLP_PSEUDOFILES,
                  "Warning", "Pseudo files may not be handeled" );
        return FALSE;
    }

    /* get the project directory */
    getpd( project, BUFSIZE );

    /* first, check if drives are equal. This avoids accessing
       drives wich are not ready... */
    currdisk = GetCurrDisk();
    if( (tolower(project[0])-'a') == (char)currdisk ) {

        getcwd( current, BUFSIZE );

        if( *current && current[strlen(current)-1] != '\\' )
            strcat( current, "\\" );

        if( strcmp( current, project ) == 0 )
            return TRUE;
    }

    ret = DoMessageBox( DMB_YES|DMB_NO|DMB_CANCEL, HLP_PROJDIR,
                  "Warning",
  "Current directory is not the project directory\nChange to project directory?" );

    if( ret == DMB_CANCEL )
        return CANCEL;

    if( ret == DMB_YES ) {
        ret = SetCurrDir( project );
        if( !ret ) {
            ret = DoMessageBox( DMB_YES|DMB_CANCEL, HLP_PROJDIR2,
                                "Warning",
                                "Could not set the current directory to\n'%s'\nContinue?",
                                project );
            if( ret == DMB_CANCEL )
                return CANCEL;
        }
    }
    return TRUE;
}

char *getpd( char *buf, int len ) {

    char *cp, *src, *end;

    cp = buf;
    src = GetProfileString( PR_FILENAME(ed.CurrFile), &profnull );
    end = src + strlen(src);
    while( end > src ) {
        if( *end == '\\' ) {
            ++end;
            break;
        }
        --end;
    }
    while( src < end ) {
        *cp = toupper( *src );
        ++cp;
        ++src;
    }
    *cp = '\0';
    return buf;
}

#define WIN_WIDTH 50
#define WIN_HEIGHT 9
void dos_command( void ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "DOS Command", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_strgad = {
        NULL, NULL, "&Command Line:", 0, -1
    };
    static struct StringInfo CODEBASED si_strgad = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_str = {
        &g_ok, &it_strgad, 2, 3, 46, 1, TEXTGADGET | BORDERLESS | ACTIVATED,
        (APTR)&si_strgad,
    };

    struct Gadget *g;

    char buffer[BUFSIZE];

    if( ! ( si_strgad.Buffer = AllocHeap( BUFSIZE+1 ) ) ) return;
    strcpy( si_strgad.Buffer, GetProfileString( PR_DOSCOMMAND, &profnull        ) );

    OpenWindow( &win, HLP_DOSCMD );
    do {
        g = OpenGadget( &g_str, win.Left, win.Top, 0L );
        if( g == &g_help ) help( -1 );
    } while( g == &g_help );
    CloseWindow( &win );

    if( g == &g_ok || g == &g_str ) {
        PutProfileString( PR_DOSCOMMAND, si_strgad.Buffer );
        free( si_strgad.Buffer );

        if (smakec( buffer, GetProfileString( PR_DOSCOMMAND, &profnull ), NULL))
            LaunchOverlay( buffer, 0 );
    }
    free( si_strgad.Buffer );
}
#undef WIN_WIDTH
#undef WIN_HEIGHT

static char umlaut_upper[] = "";
static char umlaut_lower[] = "";

int strlow( char *str ) {

    register int len = 0;
    char *cp;

    while( *str ) {
        if (cp=strchr(umlaut_upper, *str))
            *str = umlaut_lower[cp-umlaut_upper];
        else
            *str = tolower( *str );
        ++len;
        ++str;
    }

    return len;
}

int strupper( char *str ) {

    register int len = 0;
    char *cp;

    while( *str ) {
        if (cp=strchr(umlaut_lower, *str))
            *str = umlaut_upper[cp-umlaut_lower];
        else
            *str = toupper( *str );
        ++len;
        ++str;
    }

    return len;
}


static char *CODEBASED cust_cmd[PR_MAXCUSTOM];
static char *CODEBASED cust_opt[PR_MAXCUSTOM];
static char *CODEBASED cust_name[PR_MAXCUSTOM];
static unsigned long CODEBASED cust_flags[PR_MAXCUSTOM+1];

#define WIN_WIDTH  46
#define WIN_HEIGHT 17
// #pragma optimize( "elg", off )
void customizemenu( void ) {

    static void _removeentry( int, int * );
    static void _moveentry( int, int );
    static BOOL _editcustomgad( int, char * );

    static struct Window CODEBASED win = {

        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "Customize Menu", 0L

    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };

    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };
    static struct IntuiText CODEBASED it_move = {
        NULL, NULL, "&Move", 2, 0
    };
    static struct Gadget CODEBASED g_move = {
        &g_ok, &it_move, 34, 6,
        10, 3, BOOLGADGET | ENDGADGET,
    };
    static struct IntuiText CODEBASED it_edit = {
        NULL, NULL, "&Edit", 2, 0
    };
    static struct Gadget CODEBASED g_edit = {
        &g_move, &it_edit, 34, 3,
        10, 3, BOOLGADGET | ENDGADGET,
    };
    static struct IntuiText CODEBASED it_rem = {
        NULL, NULL, "&Remove", 1, 0
    };
    static struct Gadget CODEBASED g_rem = {
        &g_edit, &it_rem, 24, 6,
        10, 3, BOOLGADGET | ENDGADGET,
    };
    static struct IntuiText CODEBASED it_add = {
        NULL, NULL, "&Add", 2, 0
    };
    static struct Gadget CODEBASED g_add = {
        &g_rem, &it_add, 24, 3,
        10, 3, BOOLGADGET | ENDGADGET,
    };

    struct IntuiText *ita[PR_MAXCUSTOM+1];
    struct IntuiText it[PR_MAXCUSTOM];
    static struct Request CODEBASED req = {
        NULL, NULL, MULTISELECT };

    static struct Gadget CODEBASED gad_req = {
        &g_add, NULL, 2, 2, 21, 9, REQGADGET | ACTIVATED,
        (APTR)&req,

    };

    struct Gadget *g;
    int i, numslots, j;
    char *cp;
    int move_reg = -1;

    req.Items = ita;

    SETBIT( req.Flags, RESETREQUEST );

    for( i = 0; i < PR_MAXCUSTOM; ++i ) {
        cust_cmd[i] = cust_opt[i] = cust_name[i] = NULL;
        cust_flags[i] = 0L;
    }

    for( i = 0; i < PR_MAXCUSTOM; ++i ) {
        cp = GetProfileString( PR_CUSTOMCMD(i), NULL );
        if( cp ) {
            if( !( cp = HeapStrDup( cp ) ) ) goto done;
            cust_cmd[i] = cp;
        }
        cp = GetProfileString( PR_CUSTOMOPT(i), NULL );
        if( cp ) {
            if( !( cp = HeapStrDup( cp ) ) ) goto done;
            cust_opt[i] = cp;
        }
        cp = GetProfileString( PR_CUSTOMNAME(i), NULL );
        if( cp ) {
            if( !( cp = HeapStrDup( cp ) ) ) goto done;
            cust_name[i] = cp;
        }
        cust_flags[i] = GetProfileLong( PR_CUSTOMFLAGS(i) );
    }

    OpenWindow( &win, HLP_CUSTOMMENU );
    do {
        for( i = 0; i < PR_MAXCUSTOM; ++i ) {
            if( cust_name[i] == NULL )
                break;

            ita[i] = &it[i];
            it[i].Next = NULL;
            it[i].TextBorder = NULL;
            if( cust_flags[i] & CM_SECTION )
                it[i].Text = "---------------------";
            else
                it[i].Text = cust_name[i];
            it[i].Left = 0;
            it[i].Top = 0;
            it[i].Flags = 0;
        }

        ita[i] = NULL;
        numslots = i;
        if( numslots < 2 )
            SETBIT( g_move.Flags, DISABLED );
        else
            CLEARBIT( g_move.Flags, DISABLED );

        if( numslots == 0 )
            SETBIT( g_edit.Flags, DISABLED );
        else
            CLEARBIT( g_edit.Flags, DISABLED );

        g = OpenGadget( &gad_req, win.Left, win.Top, 0L );
        SETBIT( req.Flags, RESETREQUEST );

        if( g == &g_help )
            help( -1 );
        else if( g == &g_add ) {
            if( numslots < PR_MAXCUSTOM ) {
                if( _editcustomgad( numslots, "Customize Menu/Add" ) )
                    req.CurrIt = numslots++;
            } else {
                DoErrorBox( HLP_CUSTOM2,
  "There are no more free slots.\nChoose 'Remove' to remove one\nor more items and try again...\n" );
            }
        } else if( g == &g_rem ) {
            SetMSGadget( &req );
            SETBIT( req.Flags, RESETREQUEST );

            for( i = numslots; i > 0; --i ) {
                if( req.Items[i-1]->Flags & IT_SELECTED ) {
                    _removeentry( i-1, &numslots );
                }
            }

        } else if( g == &g_edit ) {
            _editcustomgad( req.CurrIt, "Customize Menu/Edit" );
        } else if( g == &g_ok ) {
ok_done:
            for( i = 0; i < PR_MAXCUSTOM; ++i ) {
                PutProfileString( PR_CUSTOMNAME(i), cust_name[i] );
                PutProfileString( PR_CUSTOMCMD(i), cust_cmd[i] );
                PutProfileString( PR_CUSTOMOPT(i), cust_opt[i] );
                PutProfileLong( PR_CUSTOMFLAGS(i), cust_flags[i] );
            }
            break;
        } else if( g == &gad_req ) {
            if( move_reg == -1 ) {
                /* currently not moving */
                if( req.RetVal == RETURNKEY )
                    goto ok_done;
                _editcustomgad( req.CurrIt, "Customize Menu/Edit" );
            } else {
                _moveentry( move_reg, req.CurrIt );
                DrawGadget( &gad_req, win.Left, win.Top );
                while( GetInput()->buttons & MBUTTON_LEFT );
            }
            move_reg = -1;
            CLEARBIT( gad_req.Flags, SINGLESELECT );
        }

        if( g == &g_move ) {
            SETBIT( gad_req.Flags, SINGLESELECT );
            move_reg = req.CurrIt;
        } else
            move_reg = -1;

    } while( g != &g_cancel && g );
    CloseWindow( &win );
    updatemenu();

done:
    for( i = 0; i < PR_MAXCUSTOM; ++i ) {
        if( cust_cmd[i] ) free( cust_cmd[i] );
        if( cust_opt[i] ) free( cust_opt[i] );
        if( cust_name[i] ) free( cust_name[i] );
    }

}
#undef WIN_WIDTH
#undef WIN_HEIGHT
// #pragma optimize( "", on )

static void _removeentry( int slot, int *numslots ) {

    if( cust_cmd[slot] ) free( cust_cmd[slot] );
    if( cust_opt[slot] ) free( cust_opt[slot] );
    if( cust_name[slot] ) free( cust_name[slot] );
    cust_cmd[slot] = cust_opt[slot] = cust_name[slot] = NULL;
    cust_flags[slot] = 0L;

    _moveentry( slot, *numslots-1 );
    --*numslots;
}


static void _moveentry( int from, int to ) {

    int i, direction;

    if( from < to )
        direction = 1;
    else if( from > to )
        direction = -1;
    else
        return;

    for( i = from; i != to; i += direction ) {
        swapmem( (char *)&cust_cmd[i],   (char *)&cust_cmd[i+direction], sizeof( char * ) );
        swapmem( (char *)&cust_opt[i],   (char *)&cust_opt[i+direction], sizeof( char * ) );
        swapmem( (char *)&cust_name[i],  (char *)&cust_name[i+direction], sizeof( char * ) );
        swapmem( (char *)&cust_flags[i], (char *)&cust_flags[i+direction], sizeof( long ) );
    }

}


void swapmem( char *cp1, char *cp2, int bytes ) {

    char c;

    while( bytes ) {
        c = *cp1;
        *cp1++ = *cp2;
        *cp2++ = c;
        --bytes;
    }
}

#define WIN_WIDTH 52
#define WIN_HEIGHT 19
static BOOL _editcustomgad( int slot, char *title ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, NULL, 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_sec = {
        NULL, NULL, "&Menu Section", 4, 0
    };
    static struct Gadget CODEBASED g_sec = {
        &g_ok, &it_sec, 2, 13,
        16, 1, CHECKGADGET|TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_check = {
        NULL, NULL, "Check &Project Directory", 4, 0
    };
    static struct Gadget CODEBASED g_check = {
        &g_sec, &it_check, 2, 12,
        27, 1, CHECKGADGET|TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_return = {
        NULL, NULL, "&Request On Return", 4, 0
    };
    static struct Gadget CODEBASED g_return = {
        &g_check, &it_return, 2, 11,
        21, 1, CHECKGADGET|TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_add = {
        NULL, NULL, "&Help Message:", 0, -1
    };
    static struct StringInfo CODEBASED si_add = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_add = {
        &g_return, &it_add, 2, 9, 47, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_add,
    };

    static struct IntuiText CODEBASED it_cmd = {
        NULL, NULL, "&Command Line:", 0, -1
    };
    static struct StringInfo CODEBASED si_cmd = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_cmd = {
        &g_add, &it_cmd, 2, 6, 47, 1, TEXTGADGET | BORDERLESS,
        (APTR)&si_cmd,
    };

    static struct IntuiText CODEBASED it_name = {
        NULL, NULL, "Command &Name:", 0, -1
    };
    static struct StringInfo CODEBASED si_name = {
        NULL, 47, 0L
    };
    static struct Gadget CODEBASED g_name = {
        &g_cmd, &it_name, 2, 3, 47, 1, TEXTGADGET | BORDERLESS | ACTIVATED,
        (APTR)&si_name,
    };

    struct Gadget *g;
    BOOL ret = FALSE;
    unsigned long cflags;
    int i;
    char *cp;

    win.Title = title;

    if( slot >= PR_MAXCUSTOM || slot < 0 )
        goto done;

    /* clear pointers */
    si_add.Buffer = si_cmd.Buffer = si_name.Buffer = NULL;

    if( ! ( si_add.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    if( ! ( si_cmd.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;
    if( ! ( si_name.Buffer = AllocHeap( BUFSIZE+1) ) ) goto done;

    if( cust_opt[slot] )
        strcpy( si_add.Buffer, cust_opt[slot] );
    else
        si_add.Buffer[0] = '\0';
    if( cust_cmd[slot] )
        strcpy( si_cmd.Buffer, cust_cmd[slot] );
    else
        si_cmd.Buffer[0] = '\0';
    if( cust_name[slot] )
        strcpy( si_name.Buffer, cust_name[slot] );
    else
        si_name.Buffer[0] = '\0';

    cflags = cust_flags[slot];

    if( cflags & CM_CHECKDIR )
        SETBIT( g_check.Flags, SELECTED );
    else
        CLEARBIT( g_check.Flags, SELECTED );
    if( cflags & CM_RETURN )
        SETBIT( g_return.Flags, SELECTED );
    else
        CLEARBIT( g_return.Flags, SELECTED );
    if( cflags & CM_SECTION )
        SETBIT( g_sec.Flags, SELECTED );
    else
        CLEARBIT( g_sec.Flags, SELECTED );

    OpenWindow( &win, HLP_CUSTOMMENU );
    do {
        g = OpenGadget( &g_name, win.Left, win.Top, 0L );
        if( g == &g_help ) help( -1 );
    } while( g == &g_help );
    CloseWindow( &win );

    if( g && g != &g_cancel ) {
        packstr( si_cmd.Buffer );

        cp = HeapStrDup( si_add.Buffer );
        if( cp == NULL ) goto done;
        free( cust_opt[slot] );
        cust_opt[slot] = cp;

        cp = HeapStrDup( si_cmd.Buffer );
        if( cp == NULL ) goto done;
        free( cust_cmd[slot] );
        cust_cmd[slot] = cp;

        cp = HeapStrDup( si_name.Buffer );
        if( cp == NULL ) goto done;
        free( cust_name[slot] );
        cust_name[slot] = cp;

        cflags = 0L;
        if( g_check.Flags & SELECTED )
            cflags |= CM_CHECKDIR;
        if( g_return.Flags & SELECTED )
            cflags |= CM_RETURN;
        if( g_sec.Flags & SELECTED )
            cflags |= CM_SECTION;
        cust_flags[slot] = cflags;

        ret = TRUE;
    }

done:
    free( si_cmd.Buffer );
    free( si_add.Buffer );
    free( si_name.Buffer );

    return ret;
}
#undef WIN_WIDTH
#undef WIN_HEIGHT


void exe_custom( int slot ) {

    char buffer[BUFSIZE];
    unsigned long cflags;

    cflags = GetProfileLong( PR_CUSTOMFLAGS( slot ) );

    if( slot >= PR_MAXCUSTOM || slot < 0 )
        return;

    if( cflags & CM_CHECKDIR ) {
        if( chkdir(0) != TRUE )
            return;
    }

    if (smakec( buffer, GetProfileString( PR_CUSTOMCMD(slot), &profnull ), NULL)) {
        if( GetProfileLong( PR_CUSTOMFLAGS(slot) ) & CM_RETURN )
            LaunchOverlay( buffer, LO_WAIT );
        else
            LaunchOverlay( buffer, 0 );
    }
}

/* end of file applic.c */
