/* Copyright (c) 2017-2023 hors<horsicq@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifndef SPECABSTRACT_H
#define SPECABSTRACT_H

#ifndef USE_ARCHIVE
#define USE_ARCHIVE
#endif
#ifndef USE_DEX
#define USE_DEX
#endif
#ifndef USE_PDF
#define USE_PDF
#endif

#include "xformats.h"

class SpecAbstract : public QObject {
    Q_OBJECT

public:
    enum RECORD_TYPE {
        RECORD_TYPE_UNKNOWN = 0,
        RECORD_TYPE_APKOBFUSCATOR,
        RECORD_TYPE_APKTOOL,
        RECORD_TYPE_CERTIFICATE,
        RECORD_TYPE_COMPILER,
        RECORD_TYPE_CONVERTER,
        RECORD_TYPE_CRYPTER,
        RECORD_TYPE_DATABASE,
        RECORD_TYPE_DEBUGDATA,
        RECORD_TYPE_DONGLEPROTECTION,
        RECORD_TYPE_DOSEXTENDER,
        RECORD_TYPE_FORMAT,
        RECORD_TYPE_GENERIC,
        RECORD_TYPE_IMAGE,
        RECORD_TYPE_INSTALLER,
        RECORD_TYPE_INSTALLERDATA,
        RECORD_TYPE_JAROBFUSCATOR,
        RECORD_TYPE_JOINER,
        RECORD_TYPE_LANGUAGE,
        RECORD_TYPE_LIBRARY,
        RECORD_TYPE_LINKER,
        RECORD_TYPE_NETCOMPRESSOR,
        RECORD_TYPE_NETOBFUSCATOR,
        RECORD_TYPE_OPERATIONSYSTEM,
        RECORD_TYPE_PACKER,
        RECORD_TYPE_PETOOL,
        RECORD_TYPE_PROTECTION,
        RECORD_TYPE_PROTECTOR,
        RECORD_TYPE_PROTECTORDATA,
        RECORD_TYPE_SFX,
        RECORD_TYPE_SFXDATA,
        RECORD_TYPE_SIGNTOOL,
        RECORD_TYPE_SOURCECODE,
        RECORD_TYPE_STUB,
        RECORD_TYPE_TOOL,
        RECORD_TYPE_VIRTUALMACHINE
        // TODO more
    };

    enum RECORD_NAME {
        RECORD_NAME_UNKNOWN = 0,
        RECORD_NAME_12311134,
        RECORD_NAME_1337EXECRYPTER,
        RECORD_NAME_32LITE,
        RECORD_NAME_7Z,
        RECORD_NAME_AASE,
        RECORD_NAME_ABCCRYPTOR,
        RECORD_NAME_ACPROTECT,
        RECORD_NAME_ACTIVEMARK,
        RECORD_NAME_ACTUALINSTALLER,
        RECORD_NAME_ADVANCEDINSTALLER,
        RECORD_NAME_ADVANCEDUPXSCRAMMBLER,
        RECORD_NAME_AESOBFUSCATOR,
        RECORD_NAME_AFFILLIATEEXE,
        RECORD_NAME_AGAINNATIVITYCRYPTER,
        RECORD_NAME_AGILENET,
        RECORD_NAME_AHPACKER,
        RECORD_NAME_AHTEAMEPPROTECTOR,
        RECORD_NAME_AINEXE,
        RECORD_NAME_AIX,
        RECORD_NAME_ALCHEMYMINDWORKS,
        RECORD_NAME_ALEXPROTECTOR,
        RECORD_NAME_ALIASOBJ,
        RECORD_NAME_ALIBABACLANG,
        RECORD_NAME_ALIBABAPROTECTION,
        RECORD_NAME_ALIENYZE,
        RECORD_NAME_ALIPAYCLANG,
        RECORD_NAME_ALIPAYOBFUSCATOR,
        RECORD_NAME_ALLATORIOBFUSCATOR,
        RECORD_NAME_ALLOY,
        RECORD_NAME_ALPINECLANG,
        RECORD_NAME_ALPINELINUX,
        RECORD_NAME_ANDPAKK2,
        RECORD_NAME_ANDROID,
        RECORD_NAME_ANDROIDAPKSIGNER,
        RECORD_NAME_ANDROIDARSC,
        RECORD_NAME_ANDROIDCLANG,
        RECORD_NAME_ANDROIDGRADLE,
        RECORD_NAME_ANDROIDJETPACK,
        RECORD_NAME_ANDROIDMAVENPLUGIN,
        RECORD_NAME_ANDROIDNDK,
        RECORD_NAME_ANDROIDSDK,
        RECORD_NAME_ANDROIDSIGNAPK,
        RECORD_NAME_ANDROIDXML,
        RECORD_NAME_ANSKYAPOLYMORPHICPACKER,
        RECORD_NAME_ANSLYMPACKER,
        RECORD_NAME_ANTIDOTE,
        RECORD_NAME_ANTILVL,
        RECORD_NAME_APACHEANT,
        RECORD_NAME_APACK,
        RECORD_NAME_APKEDITOR,
        RECORD_NAME_APKENCRYPTOR,
        RECORD_NAME_APKMODIFIERSIGNAPK,
        RECORD_NAME_APKPROTECT,
        RECORD_NAME_APKPROTECTOR,
        RECORD_NAME_APKS,
        RECORD_NAME_APKSIGNATURESCHEME,
        RECORD_NAME_APKSIGNER,
        RECORD_NAME_APKTOOLPLUS,
        RECORD_NAME_APK_SIGNER,
        RECORD_NAME_APPGUARD,
        RECORD_NAME_APPIMAGE,
        RECORD_NAME_APPLEJDK,
        RECORD_NAME_APPLELLVM,
        RECORD_NAME_APPORTABLECLANG,
        RECORD_NAME_APPSOLID,
        RECORD_NAME_AR,
        RECORD_NAME_ARCRYPT,
        RECORD_NAME_ARJ,
        RECORD_NAME_ARMADILLO,
        RECORD_NAME_ARMASSEMBLER,
        RECORD_NAME_ARMC,
        RECORD_NAME_ARMCCPP,
        RECORD_NAME_ARMLINKER,
        RECORD_NAME_ARMNEONCCPP,
        RECORD_NAME_ARMPROTECTOR,
        RECORD_NAME_ARMTHUMBCCPP,
        RECORD_NAME_ARMTHUMBMACROASSEMBLER,
        RECORD_NAME_AROS,
        RECORD_NAME_ARXAN,
        RECORD_NAME_ASDPACK,
        RECORD_NAME_ASMGUARD,
        RECORD_NAME_ASPACK,
        RECORD_NAME_ASPLINUX,
        RECORD_NAME_ASPROTECT,
        RECORD_NAME_ASSCRYPTER,
        RECORD_NAME_ASSEMBLER,
        RECORD_NAME_ASSEMBLYINVOKE,
        RECORD_NAME_AU,
        RECORD_NAME_AUTOIT,
        RECORD_NAME_AVASTANTIVIRUS,
        RECORD_NAME_AVERCRYPTOR,
        RECORD_NAME_AVI,
        RECORD_NAME_AVPACK,
        RECORD_NAME_AZPROTECT,
        RECORD_NAME_BABELNET,
        RECORD_NAME_BACKDOORPECOMPRESSPROTECTOR,
        RECORD_NAME_BAIDUPROTECTION,
        RECORD_NAME_BAIDUSIGNATUREPLATFORM,
        RECORD_NAME_BAMBAM,
        RECORD_NAME_BANGCLEPROTECTION,
        RECORD_NAME_BASIC,
        RECORD_NAME_BASIC4ANDROID,
        RECORD_NAME_BAT2EXEC,
        RECORD_NAME_BEAWEBLOGIC,
        RECORD_NAME_BEROEXEPACKER,
        RECORD_NAME_BIOHAZARDCRYPTER,
        RECORD_NAME_BITMAPINFOHEADER,
        RECORD_NAME_BITROCKINSTALLER,
        RECORD_NAME_BITSHAPEPECRYPT,
        RECORD_NAME_BLADEJOINER,
        RECORD_NAME_BORLANDCCPP,
        RECORD_NAME_BORLANDCPP,
        RECORD_NAME_BORLANDCPPBUILDER,
        RECORD_NAME_BORLANDDELPHI,
        RECORD_NAME_BORLANDDELPHIDOTNET,
        RECORD_NAME_BORLANDOBJECTPASCALDELPHI,
        RECORD_NAME_BORLANDOSSERVICES,
        RECORD_NAME_BREAKINTOPATTERN,
        RECORD_NAME_BRIDGEOS,
        RECORD_NAME_BRIDGEOSSDK,
        RECORD_NAME_BTWORKSCODEGUARD,
        RECORD_NAME_BUNDLETOOL,
        RECORD_NAME_BURNEYE,
        RECORD_NAME_BYTEDANCESECCOMPILER,
        RECORD_NAME_BYTEGUARD,
        RECORD_NAME_BZIP2,
        RECORD_NAME_C,
        RECORD_NAME_CAB,
        RECORD_NAME_CARBON,
        RECORD_NAME_CAUSEWAY,
        RECORD_NAME_CCBYUNIHACKERS,
        RECORD_NAME_CCBYVORONTSOV,
        RECORD_NAME_CCPP,
        RECORD_NAME_CELESTYFILEBINDER,
        RECORD_NAME_CEXE,
        RECORD_NAME_CHROMIUMCRASHPAD,
        RECORD_NAME_CIGICIGICRYPTER,
        RECORD_NAME_CIL,
        RECORD_NAME_CLANG,
        RECORD_NAME_CLICKTEAM,
        RECORD_NAME_CLISECURE,
        RECORD_NAME_COCOA,
        RECORD_NAME_CODEGEARCPP,
        RECORD_NAME_CODEGEARCPPBUILDER,
        RECORD_NAME_CODEGEARDELPHI,
        RECORD_NAME_CODEGEAROBJECTPASCALDELPHI,
        RECORD_NAME_CODESIGN,
        RECORD_NAME_CODEVEIL,
        RECORD_NAME_CODEWALL,
        RECORD_NAME_COFF,
        RECORD_NAME_COMEXSIGNAPK,
        RECORD_NAME_COMPOUNDFILEBINARYFORMAT,
        RECORD_NAME_CONFUSER,
        RECORD_NAME_CONFUSEREX,
        RECORD_NAME_COPYMINDER,
        RECORD_NAME_CPP,
        RECORD_NAME_CREATEINSTALL,
        RECORD_NAME_CRINKLER,
        RECORD_NAME_CRUNCH,
        RECORD_NAME_CRYEXE,
        RECORD_NAME_CRYPTABLESEDUCATION,
        RECORD_NAME_CRYPTCOM,
        RECORD_NAME_CRYPTDISMEMBER,
        RECORD_NAME_CRYPTER,
        RECORD_NAME_CRYPTIC,
        RECORD_NAME_CRYPTOCRACKPEPROTECTOR,
        RECORD_NAME_CRYPTOOBFUSCATORFORNET,
        RECORD_NAME_CRYPTORBYDISMEMBER,
        RECORD_NAME_CRYPTOZ,
        RECORD_NAME_CRYPTRROADS,
        RECORD_NAME_CSHARP,
        RECORD_NAME_CVTOMF,
        RECORD_NAME_CVTPGD,
        RECORD_NAME_CVTRES,
        RECORD_NAME_CWSDPMI,
        RECORD_NAME_CYGWIN,
        RECORD_NAME_D,
        RECORD_NAME_D2JAPKSIGN,
        RECORD_NAME_DALKRYPT,
        RECORD_NAME_DALVIK,
        RECORD_NAME_DBPE,
        RECORD_NAME_DCRYPTPRIVATE,
        RECORD_NAME_DEB,
        RECORD_NAME_DEBIANCLANG,
        RECORD_NAME_DEBIANLINUX,
        RECORD_NAME_DEEPSEA,
        RECORD_NAME_DEPACK,
        RECORD_NAME_DEPLOYMASTER,
        RECORD_NAME_DEX,
        RECORD_NAME_DEX2JAR,
        RECORD_NAME_DEXGUARD,
        RECORD_NAME_DEXLIB,
        RECORD_NAME_DEXLIB2,
        RECORD_NAME_DEXMERGE,
        RECORD_NAME_DEXPROTECTOR,
        RECORD_NAME_DIET,
        RECORD_NAME_DINGBAOZENGNATIVEOBFUSCATOR,
        RECORD_NAME_DIRTYCRYPTOR,
        RECORD_NAME_DJVU,
        RECORD_NAME_DMD,
        RECORD_NAME_DMD32,
        RECORD_NAME_DNGUARD,
        RECORD_NAME_DOS16M,
        RECORD_NAME_DOS4G,
        RECORD_NAME_DOTBJFNT,
        RECORD_NAME_DOTFIXNICEPROTECT,
        RECORD_NAME_DOTFUSCATOR,
        RECORD_NAME_DOTNET,
        RECORD_NAME_DOTNETREACTOR,
        RECORD_NAME_DOTNETSHRINK,
        RECORD_NAME_DOTNETSPIDER,
        RECORD_NAME_DOTNETZ,
        RECORD_NAME_DOTOOLSSIGNAPK,
        RECORD_NAME_DRAGONARMOR,
        RECORD_NAME_DROPBOX,
        RECORD_NAME_DVCLAL,
        RECORD_NAME_DX,
        RECORD_NAME_DXSHIELD,
        RECORD_NAME_DYAMAR,
        RECORD_NAME_DYNASM,
        RECORD_NAME_EASYPROTECTOR,
        RECORD_NAME_EAZFUSCATOR,
        RECORD_NAME_ECLIPSE,
        RECORD_NAME_ECMASCRIPT,
        RECORD_NAME_ELECKEY,
        RECORD_NAME_EMBARCADEROCPP,
        RECORD_NAME_EMBARCADEROCPPBUILDER,
        RECORD_NAME_EMBARCADERODELPHI,
        RECORD_NAME_EMBARCADERODELPHIDOTNET,
        RECORD_NAME_EMBARCADEROOBJECTPASCALDELPHI,
        RECORD_NAME_EMPTYFILE,
        RECORD_NAME_ENCRYPTPE,
        RECORD_NAME_ENIGMA,
        RECORD_NAME_ENIGMAVIRTUALBOX,
        RECORD_NAME_EPEXEPACK,
        RECORD_NAME_EPROT,
        RECORD_NAME_EXCELSIORJET,
        RECORD_NAME_EXE32PACK,
        RECORD_NAME_EXECRYPT,
        RECORD_NAME_EXECRYPTOR,
        RECORD_NAME_EXEFOG,
        RECORD_NAME_EXEJOINER,
        RECORD_NAME_EXEMPLARINSTALLER,
        RECORD_NAME_EXEPACK,
        RECORD_NAME_EXEPASSWORDPROTECTOR,
        RECORD_NAME_EXESAX,
        RECORD_NAME_EXESHIELD,
        RECORD_NAME_EXESTEALTH,
        RECORD_NAME_EXPORT,
        RECORD_NAME_EXPRESSOR,
        RECORD_NAME_EXPRESSOR_KERNEL32,
        RECORD_NAME_EXPRESSOR_USER32,
        RECORD_NAME_EZIP,
        RECORD_NAME_FAKESIGNATURE,
        RECORD_NAME_FAKUSCRYPTOR,
        RECORD_NAME_FASM,
        RECORD_NAME_FASTFILECRYPT,
        RECORD_NAME_FASTPROXY,
        RECORD_NAME_FEARZCRYPTER,
        RECORD_NAME_FEARZPACKER,
        RECORD_NAME_FENIXOS,
        RECORD_NAME_FILESHIELD,
        RECORD_NAME_FISHNET,
        RECORD_NAME_FISHPEPACKER,
        RECORD_NAME_FISHPESHIELD,
        RECORD_NAME_FLASHVIDEO,
        RECORD_NAME_FLEXLM,
        RECORD_NAME_FLEXNET,
        RECORD_NAME_FORTRAN,
        RECORD_NAME_FOUNDATION,
        RECORD_NAME_FPC,
        RECORD_NAME_FREEBSD,
        RECORD_NAME_FREECRYPTOR,
        RECORD_NAME_FSG,
        RECORD_NAME_GCC,
        RECORD_NAME_GENERIC,
        RECORD_NAME_GENERICLINKER,
        RECORD_NAME_GENTEEINSTALLER,
        RECORD_NAME_GENTOOLINUX,
        RECORD_NAME_GHAZZACRYPTER,
        RECORD_NAME_GHOSTINSTALLER,
        RECORD_NAME_GIF,
        RECORD_NAME_GIXPROTECTOR,
        RECORD_NAME_GKRIPTO,
        RECORD_NAME_GKSETUPSFX,
        RECORD_NAME_GNUASSEMBLER,
        RECORD_NAME_GNULINKER,
        RECORD_NAME_GO,
        RECORD_NAME_GOASM,
        RECORD_NAME_GOATSPEMUTILATOR,
        RECORD_NAME_GOLD,
        RECORD_NAME_GOLIATHNET,
        RECORD_NAME_GOLINK,
        RECORD_NAME_GOOGLE,
        RECORD_NAME_GOOGLEPLAY,
        RECORD_NAME_GPINSTALL,
        RECORD_NAME_GUARDIANSTEALTH,
        RECORD_NAME_GZIP,
        RECORD_NAME_H4CKY0UORGCRYPTER,
        RECORD_NAME_HACCREWCRYPTER,
        RECORD_NAME_HACKSTOP,
        RECORD_NAME_HALVCRYPTER,
        RECORD_NAME_HANCOMLINUX,
        RECORD_NAME_HDUS_WJUS,
        RECORD_NAME_HIAPKCOM,
        RECORD_NAME_HIDEANDPROTECT,
        RECORD_NAME_HIDEPE,
        RECORD_NAME_HIKARIOBFUSCATOR,
        RECORD_NAME_HMIMYSPACKER,
        RECORD_NAME_HMIMYSPROTECTOR,
        RECORD_NAME_HOODLUM,
        RECORD_NAME_HOUNDHACKCRYPTER,
        RECORD_NAME_HPUX,
        RECORD_NAME_HTML,
        RECORD_NAME_HXS,
        RECORD_NAME_HYPERTECHCRACKPROOF,
        RECORD_NAME_IBMJDK,
        RECORD_NAME_IBMPCPASCAL,
        RECORD_NAME_ICE,
        RECORD_NAME_ICRYPT,
        RECORD_NAME_IJIAMI,
        RECORD_NAME_IJIAMILLVM,
        RECORD_NAME_IKVMDOTNET,
        RECORD_NAME_IL2CPP,
        RECORD_NAME_ILASM,
        RECORD_NAME_IMPORT,
        RECORD_NAME_INFCRYPTOR,
        RECORD_NAME_INNOSETUP,
        RECORD_NAME_INQUARTOSOBFUSCATOR,
        RECORD_NAME_INSTALL4J,
        RECORD_NAME_INSTALLANYWHERE,
        RECORD_NAME_INSTALLSHIELD,
        RECORD_NAME_IOS,
        RECORD_NAME_IOSSDK,
        RECORD_NAME_IPA,
        RECORD_NAME_IPADOS,
        RECORD_NAME_IPHONEOS,
        RECORD_NAME_IPBPROTECT,
        RECORD_NAME_IRIX,
        RECORD_NAME_ISO9660,
        RECORD_NAME_JACK,
        RECORD_NAME_JAM,
        RECORD_NAME_JAR,
        RECORD_NAME_JAVA,
        RECORD_NAME_JAVACOMPILEDCLASS,
        RECORD_NAME_JDK,
        RECORD_NAME_JDPACK,
        RECORD_NAME_JETBRAINS,
        RECORD_NAME_JIAGU,
        RECORD_NAME_JPEG,
        RECORD_NAME_JSCRIPT,
        RECORD_NAME_JVM,
        RECORD_NAME_KAOSPEDLLEXECUTABLEUNDETECTER,
        RECORD_NAME_KBYS,
        RECORD_NAME_KCRYPTOR,
        RECORD_NAME_KGBCRYPTER,
        RECORD_NAME_KIAMSCRYPTOR,
        RECORD_NAME_KIRO,
        RECORD_NAME_KIWIVERSIONOBFUSCATOR,
        RECORD_NAME_KKRUNCHY,
        RECORD_NAME_KOTLIN,
        RECORD_NAME_KRATOSCRYPTER,
        RECORD_NAME_KRYPTON,
        RECORD_NAME_KUR0KX2TO,
        RECORD_NAME_LAMECRYPT,
        RECORD_NAME_LARP64,
        RECORD_NAME_LAYHEYFORTRAN90,
        RECORD_NAME_LAZARUS,
        RECORD_NAME_LCCLNK,
        RECORD_NAME_LCCWIN,
        RECORD_NAME_LDC,
        RECORD_NAME_LGLZ,
        RECORD_NAME_LHA,
        RECORD_NAME_LHASSFX,
        RECORD_NAME_LIAPP,
        RECORD_NAME_LIGHTNINGCRYPTERPRIVATE,
        RECORD_NAME_LIGHTNINGCRYPTERSCANTIME,
        RECORD_NAME_LINUX,
        RECORD_NAME_LLD,
        RECORD_NAME_LOCKTITE,
        RECORD_NAME_LSCRYPRT,
        RECORD_NAME_LUACOMPILED,
        RECORD_NAME_LUCYPHER,
        RECORD_NAME_LZEXE,
        RECORD_NAME_LZFSE,
        RECORD_NAME_MACHOFAT,
        RECORD_NAME_MAC_OS,
        RECORD_NAME_MAC_OS_X,
        RECORD_NAME_MACOS,
        RECORD_NAME_MACOSSDK,
        RECORD_NAME_MACROBJECT,
        RECORD_NAME_MALPACKER,
        RECORD_NAME_MANDRAKELINUX,
        RECORD_NAME_MASKPE,
        RECORD_NAME_MASM,
        RECORD_NAME_MASM32,
        RECORD_NAME_MAXTOCODE,
        RECORD_NAME_MCLINUX,
        RECORD_NAME_MEDUSAH,
        RECORD_NAME_MEW10,
        RECORD_NAME_MEW11SE,
        RECORD_NAME_MFC,
        RECORD_NAME_MICROSOFTACCESS,
        RECORD_NAME_MICROSOFTC,
        RECORD_NAME_MICROSOFTCOMPILEDHTMLHELP,
        RECORD_NAME_MICROSOFTCOMPOUND,
        RECORD_NAME_MICROSOFTCPP,
        RECORD_NAME_MICROSOFTDOTNETFRAMEWORK,
        RECORD_NAME_MICROSOFTEXCEL,
        RECORD_NAME_MICROSOFTINSTALLER,
        RECORD_NAME_MICROSOFTLINKER,
        RECORD_NAME_MICROSOFTLINKERDATABASE,
        RECORD_NAME_MICROSOFTOFFICE,
        RECORD_NAME_MICROSOFTOFFICEWORD,
        RECORD_NAME_MICROSOFTPHOENIX,
        RECORD_NAME_MICROSOFTVISIO,
        RECORD_NAME_MICROSOFTVISUALSTUDIO,
        RECORD_NAME_MICROSOFTWINHELP,
        RECORD_NAME_MINGW,
        RECORD_NAME_MINIX,
        RECORD_NAME_MINKE,
        RECORD_NAME_MKFPACK,
        RECORD_NAME_MOBILETENCENTPROTECT,
        RECORD_NAME_MODESTO,
        RECORD_NAME_MODGUARD,
        RECORD_NAME_MOLD,
        RECORD_NAME_MOLEBOX,
        RECORD_NAME_MOLEBOXULTRA,
        RECORD_NAME_MONEYCRYPTER,
        RECORD_NAME_MORPHNAH,
        RECORD_NAME_MORTALTEAMCRYPTER,
        RECORD_NAME_MORTALTEAMCRYPTER2,
        RECORD_NAME_MORUKCREWCRYPTERPRIVATE,
        RECORD_NAME_MOTODEVSTUDIOFORANDROID,
        RECORD_NAME_MP3,
        RECORD_NAME_MP4,
        RECORD_NAME_MPACK,
        RECORD_NAME_MPRESS,
        RECORD_NAME_MRUNDECTETABLE,
        RECORD_NAME_MSDOS,
        RECORD_NAME_MSLRH,
        RECORD_NAME_MSYS,
        RECORD_NAME_MSYS2,
        RECORD_NAME_MZ0OPE,
        RECORD_NAME_NAGAINLLVM,
        RECORD_NAME_NAGAPTPROTECTION,
        RECORD_NAME_NAKEDPACKER,
        RECORD_NAME_NASM,  // The Netwide Assembler
        RECORD_NAME_NATIVECRYPTORBYDOSX,
        RECORD_NAME_NCODE,
        RECORD_NAME_NEOLITE,
        RECORD_NAME_NETBSD,
        RECORD_NAME_NETEASEAPKSIGNER,
        RECORD_NAME_NIDHOGG,
        RECORD_NAME_NIM,
        RECORD_NAME_NJOINER,
        RECORD_NAME_NJOY,
        RECORD_NAME_NME,
        RECORD_NAME_NOOBYPROTECT,
        RECORD_NAME_NOODLECRYPT,
        RECORD_NAME_NORTHSTARPESHRINKER,
        RECORD_NAME_NOSINSTALLER,
        RECORD_NAME_NOSTUBLINKER,
        RECORD_NAME_NOXCRYPT,
        RECORD_NAME_NPACK,
        RECORD_NAME_NQSHIELD,
        RECORD_NAME_NSIS,
        RECORD_NAME_NSK,
        RECORD_NAME_NSPACK,
        RECORD_NAME_OBFUSCAR,
        RECORD_NAME_OBFUSCATORLLVM,
        RECORD_NAME_OBFUSCATORNET2009,
        RECORD_NAME_OBJECTIVEC,
        RECORD_NAME_OBJECTPASCAL,
        RECORD_NAME_OBJECTPASCALDELPHI,
        RECORD_NAME_OBSIDIUM,
        RECORD_NAME_OLLVMTLL,
        RECORD_NAME_ONESPANPROTECTION,  // till 2018 Vasco !
        RECORD_NAME_OPENBSD,
        RECORD_NAME_OPENDOCUMENT,
        RECORD_NAME_OPENJDK,
        RECORD_NAME_OPENSOURCECODECRYPTER,
        RECORD_NAME_OPENVMS,
        RECORD_NAME_OPENVOS,
        RECORD_NAME_OPERA,
        RECORD_NAME_ORACLESOLARISLINKEDITORS,
        RECORD_NAME_OREANSCODEVIRTUALIZER,
        RECORD_NAME_ORIEN,
        RECORD_NAME_OS2,
        RECORD_NAME_OSCCRYPTER,
        RECORD_NAME_OS_X,
        RECORD_NAME_P0KESCRAMBLER,
        RECORD_NAME_PACKMAN,
        RECORD_NAME_PACKWIN,
        RECORD_NAME_PANDORA,
        RECORD_NAME_PANGXIE,
        RECORD_NAME_PCGUARD,
        RECORD_NAME_PCOM,
        RECORD_NAME_PCSHRINK,
        RECORD_NAME_PDB,
        RECORD_NAME_PDBFILELINK,
        RECORD_NAME_PDF,
        RECORD_NAME_PEARMOR,
        RECORD_NAME_PEBUNDLE,
        RECORD_NAME_PECOMPACT,
        RECORD_NAME_PECRYPT32,
        RECORD_NAME_PEDIMINISHER,
        RECORD_NAME_PEENCRYPT,
        RECORD_NAME_PELOCK,
        RECORD_NAME_PELOCKNT,
        RECORD_NAME_PENGUINCRYPT,
        RECORD_NAME_PEPACK,
        RECORD_NAME_PEPACKSPROTECT,
        RECORD_NAME_PEQUAKE,
        RECORD_NAME_PERL,
        RECORD_NAME_PESHIELD,
        RECORD_NAME_PESPIN,
        RECORD_NAME_PETITE,
        RECORD_NAME_PETITE_KERNEL32,
        RECORD_NAME_PETITE_USER32,
        RECORD_NAME_PEX,
        RECORD_NAME_PFECX,
        RECORD_NAME_PGMPAK,
        RECORD_NAME_PHOENIXPROTECTOR,
        RECORD_NAME_PHP,
        RECORD_NAME_PICRYPTOR,
        RECORD_NAME_PKLITE,
        RECORD_NAME_PKLITE32,
        RECORD_NAME_PKZIPMINISFX,
        RECORD_NAME_PLAIN,
        RECORD_NAME_PLEXCLANG,
        RECORD_NAME_PMODEW,
        RECORD_NAME_PNG,
        RECORD_NAME_POKECRYPTER,
        RECORD_NAME_POLYCRYPTPE,
        RECORD_NAME_POSIX,
        RECORD_NAME_POWERBASIC,
        RECORD_NAME_PRIVATEEXEPROTECTOR,
        RECORD_NAME_PROGUARD,
        RECORD_NAME_PROPACK,
        RECORD_NAME_PROTECTEXE,
        RECORD_NAME_PSEUDOAPKSIGNER,
        RECORD_NAME_PUBCRYPTER,
        RECORD_NAME_PUNISHER,
        RECORD_NAME_PUREBASIC,
        RECORD_NAME_PUSSYCRYPTER,
        RECORD_NAME_PYINSTALLER,
        RECORD_NAME_PYTHON,
        RECORD_NAME_QDBH,
        RECORD_NAME_QIHOO360PROTECTION,
        RECORD_NAME_QNX,
        RECORD_NAME_QRYPT0R,
        RECORD_NAME_QT,
        RECORD_NAME_QTINSTALLER,
        RECORD_NAME_QUICKPACKNT,
        RECORD_NAME_R8,
        RECORD_NAME_RADIALIX,
        RECORD_NAME_RAR,
        RECORD_NAME_RCRYPTOR,
        RECORD_NAME_RDGTEJONCRYPTER,
        RECORD_NAME_REDHATLINUX,
        RECORD_NAME_RELPACK,
        RECORD_NAME_RENETPACK,
        RECORD_NAME_RESOURCE,
        RECORD_NAME_RESOURCE_CURSOR,
        RECORD_NAME_RESOURCE_DIALOG,
        RECORD_NAME_RESOURCE_ICON,
        RECORD_NAME_RESOURCE_MENU,
        RECORD_NAME_RESOURCE_STRINGTABLE,
        RECORD_NAME_RESOURCE_VERSIONINFO,
        RECORD_NAME_REVPROT,
        RECORD_NAME_RJCRUSH,
        RECORD_NAME_RLP,
        RECORD_NAME_RLPACK,
        RECORD_NAME_ROGUEPACK,
        RECORD_NAME_ROSASM,
        RECORD_NAME_RTF,
        RECORD_NAME_RUBY,
        RECORD_NAME_RUST,
        RECORD_NAME_SAFEENGINELLVM,
        RECORD_NAME_SAFEENGINESHIELDEN,
        RECORD_NAME_SANDHOOK,
        RECORD_NAME_SCOBFUSCATOR,
        RECORD_NAME_SCPACK,
        RECORD_NAME_SCRNCH,
        RECORD_NAME_SDPROTECTORPRO,
        RECORD_NAME_SECNEO,
        RECORD_NAME_SECSHELL,
        RECORD_NAME_SECURESHADE,
        RECORD_NAME_SECUROM,
        RECORD_NAME_SERGREENAPPACKER,
        RECORD_NAME_SETUPFACTORY,
        RECORD_NAME_SEXECRYPTER,
        RECORD_NAME_SHELL,
        RECORD_NAME_SHRINKER,
        RECORD_NAME_SIGNATORY,
        RECORD_NAME_SIGNUPDATE,
        RECORD_NAME_SIMBIOZ,
        RECORD_NAME_SIMCRYPTER,
        RECORD_NAME_SIMPLECRYPTER,
        RECORD_NAME_SIMPLEPACK,
        RECORD_NAME_SINGLEJAR,
        RECORD_NAME_SIXXPACK,
        RECORD_NAME_SKATER,
        RECORD_NAME_SMARTASSEMBLY,
        RECORD_NAME_SMARTINSTALLMAKER,
        RECORD_NAME_SMOKESCREENCRYPTER,
        RECORD_NAME_SNAPDRAGONLLVMARM,
        RECORD_NAME_SNAPPROTECT,
        RECORD_NAME_SNOOPCRYPT,
        RECORD_NAME_SOFTDEFENDER,
        RECORD_NAME_SOFTSENTRY,
        RECORD_NAME_SOFTWARECOMPRESS,
        RECORD_NAME_SOFTWAREZATOR,
        RECORD_NAME_SOLARIS,
        RECORD_NAME_SOURCERYCODEBENCH,
        RECORD_NAME_SOURCERYCODEBENCHLITE,
        RECORD_NAME_SPICESNET,
        RECORD_NAME_SPIRIT,
        RECORD_NAME_SPOONINSTALLER,
        RECORD_NAME_SPOONSTUDIO,
        RECORD_NAME_SPOONSTUDIO2011,
        RECORD_NAME_SQUEEZSFX,
        RECORD_NAME_SQUIRRELINSTALLER,
        RECORD_NAME_STARFORCE,
        RECORD_NAME_STARTOSLINUX,
        RECORD_NAME_STASFODIDOCRYPTOR,
        RECORD_NAME_STONESPEENCRYPTOR,  // TODO Check name from .Stone Section // TODO EP !!!
        RECORD_NAME_SUNOS,
        RECORD_NAME_SUNWORKSHOP,
        RECORD_NAME_SUNWORKSHOPCOMPILERS,
        RECORD_NAME_SUSELINUX,
        RECORD_NAME_SVKPROTECTOR,
        RECORD_NAME_SYLLABLE,
        RECORD_NAME_SWF,
        RECORD_NAME_SWIFT,
        RECORD_NAME_TAR,
        RECORD_NAME_TARMAINSTALLER,
        RECORD_NAME_TELOCK,
        RECORD_NAME_TENCENTLEGU,
        RECORD_NAME_TENCENTPROTECTION,
        RECORD_NAME_TGRCRYPTER,
        RECORD_NAME_THEBESTCRYPTORBYFSK,
        RECORD_NAME_THEMIDAWINLICENSE,
        RECORD_NAME_THEZONECRYPTER,
        RECORD_NAME_THINSTALL,
        RECORD_NAME_THUMBC,
        RECORD_NAME_TIFF,
        RECORD_NAME_TINYC,
        RECORD_NAME_TINYPROG,
        RECORD_NAME_TINYSIGN,
        RECORD_NAME_TOTALCOMMANDERINSTALLER,
        RECORD_NAME_TPPPACK,
        RECORD_NAME_TRU64,
        RECORD_NAME_TSTCRYPTER,
        RECORD_NAME_TTF,
        RECORD_NAME_TTPROTECT,
        RECORD_NAME_TURBOBASIC,
        RECORD_NAME_TURBOC,
        RECORD_NAME_TURBOCPP,
        RECORD_NAME_TURBOLINKER,
        RECORD_NAME_TURBOLINUX,
        RECORD_NAME_TURBOSTUDIO,
        RECORD_NAME_TURKISHCYBERSIGNATURE,
        RECORD_NAME_TURKOJANCRYPTER,
        RECORD_NAME_TVOS,
        RECORD_NAME_TVOSSDK,
        RECORD_NAME_UBUNTUCLANG,
        RECORD_NAME_UBUNTULINUX,
        RECORD_NAME_UCEXE,
        RECORD_NAME_UNDERGROUNDCRYPTER,
        RECORD_NAME_UNDOCRYPTER,
        RECORD_NAME_UNICODE,
        RECORD_NAME_UNICOMSDK,
        RECORD_NAME_UNILINK,
        RECORD_NAME_UNITY,
        RECORD_NAME_UNIVERSALTUPLECOMPILER,
        RECORD_NAME_UNIX,
        RECORD_NAME_UNKOWNCRYPTER,
        RECORD_NAME_UNK_UPXLIKE,
        RECORD_NAME_UNOPIX,
        RECORD_NAME_UPX,
        RECORD_NAME_UTF8,
        RECORD_NAME_VALVE,
        RECORD_NAME_VBNET,
        RECORD_NAME_VBSTOEXE,
        RECORD_NAME_VCASMPROTECTOR,
        RECORD_NAME_VCL,
        RECORD_NAME_VCLPACKAGEINFO,
        RECORD_NAME_VDOG,
        RECORD_NAME_VERACRYPT,
        RECORD_NAME_VINELINUX,
        RECORD_NAME_VIRBOXPROTECTOR,
        RECORD_NAME_VIRTUALIZEPROTECT,
        RECORD_NAME_VIRTUALPASCAL,
        RECORD_NAME_VISE,
        RECORD_NAME_VISUALBASIC,
        RECORD_NAME_VISUALCCPP,
        RECORD_NAME_VISUALCSHARP,
        RECORD_NAME_VISUALOBJECTS,
        RECORD_NAME_VMPROTECT,
        RECORD_NAME_VMUNPACKER,
        RECORD_NAME_VMWARE,
        RECORD_NAME_VPACKER,
        RECORD_NAME_WALLE,
        RECORD_NAME_WANGZEHUALLVM,
        RECORD_NAME_WATCHOS,
        RECORD_NAME_WATCHOSSDK,
        RECORD_NAME_WATCOMC,
        RECORD_NAME_WATCOMCCPP,
        RECORD_NAME_WATCOMLINKER,
        RECORD_NAME_WAV,
        RECORD_NAME_WDOSX,
        RECORD_NAME_WEBP,
        RECORD_NAME_WHITELLCRYPT,
        RECORD_NAME_WINACE,
        RECORD_NAME_WINAUTH,
        RECORD_NAME_WINDOFCRYPT,
        RECORD_NAME_WINDOWS,
        RECORD_NAME_WINDOWSBITMAP,
        RECORD_NAME_WINDOWSCE,
        RECORD_NAME_WINDOWSCURSOR,
        RECORD_NAME_WINDOWSICON,
        RECORD_NAME_WINDOWSINSTALLER,
        RECORD_NAME_WINDOWSMEDIA,
        RECORD_NAME_WINDRIVERLINUX,
        RECORD_NAME_WINGSCRYPT,
        RECORD_NAME_WINKRIPT,
        RECORD_NAME_WINRAR,
        RECORD_NAME_WINUPACK,
        RECORD_NAME_WINZIP,
        RECORD_NAME_WISE,
        RECORD_NAME_WIXTOOLSET,
        RECORD_NAME_WLCRYPT,
        RECORD_NAME_WLGROUPCRYPTER,
        RECORD_NAME_WOUTHRSEXECRYPTER,
        RECORD_NAME_WWPACK,
        RECORD_NAME_WWPACK32,
        RECORD_NAME_WXWIDGETS,
        RECORD_NAME_X86ASSEMBLER,
        RECORD_NAME_XAR,
        RECORD_NAME_XBOX,
        RECORD_NAME_XCODE,
        RECORD_NAME_XCOMP,
        RECORD_NAME_XENOCODE,
        RECORD_NAME_XENOCODEPOSTBUILD,
        RECORD_NAME_XENOCODEPOSTBUILD2009FORDOTNET,
        RECORD_NAME_XENOCODEPOSTBUILD2010FORDOTNET,
        RECORD_NAME_XENOCODEVIRTUALAPPLICATIONSTUDIO2009,
        RECORD_NAME_XENOCODEVIRTUALAPPLICATIONSTUDIO2010,
        RECORD_NAME_XENOCODEVIRTUALAPPLICATIONSTUDIO2010ISVEDITION,
        RECORD_NAME_XENOCODEVIRTUALAPPLICATIONSTUDIO2012ISVEDITION,
        RECORD_NAME_XENOCODEVIRTUALAPPLICATIONSTUDIO2013ISVEDITION,
        RECORD_NAME_XML,
        RECORD_NAME_XPACK,
        RECORD_NAME_XTREAMLOK,
        RECORD_NAME_XTREMEPROTECTOR,
        RECORD_NAME_XVOLKOLAK,
        RECORD_NAME_XZ,
        RECORD_NAME_YANDEX,
        RECORD_NAME_YANO,
        RECORD_NAME_YIDUN,
        RECORD_NAME_YODASCRYPTER,
        RECORD_NAME_YODASPROTECTOR,
        RECORD_NAME_YZPACK,
        RECORD_NAME_ZELDACRYPT,
        RECORD_NAME_ZIG,
        RECORD_NAME_ZIP,
        RECORD_NAME_ZLIB,
        RECORD_NAME_ZPROTECT,
        RECORD_NAME_UNKNOWN0,
        RECORD_NAME_UNKNOWN1,
        RECORD_NAME_UNKNOWN2,
        RECORD_NAME_UNKNOWN3,
        RECORD_NAME_UNKNOWN4,
        RECORD_NAME_UNKNOWN5,
        RECORD_NAME_UNKNOWN6,
        RECORD_NAME_UNKNOWN7,
        RECORD_NAME_UNKNOWN8,
        RECORD_NAME_UNKNOWN9
    };

    // TODO flags(static scan/emul/heur) ? Check
    struct SCAN_STRUCT {
        bool bIsHeuristic;
        XBinary::SCANID id;
        XBinary::SCANID parentId;
        RECORD_TYPE type;
        RECORD_NAME name;
        QString sVersion;
        QString sInfo;
    };

    enum DETECTTYPE {
        DETECTTYPE_UNKNOWN = 0,
        DETECTTYPE_ARCHIVE,
        DETECTTYPE_CODESECTION,
        DETECTTYPE_DEXSTRING,
        DETECTTYPE_DEXTYPE,
        DETECTTYPE_ENTRYPOINT,
        DETECTTYPE_ENTRYPOINTSECTION,
        DETECTTYPE_HEADER,
        DETECTTYPE_IMPORTHASH,
        DETECTTYPE_NETANSISTRING,
        DETECTTYPE_NETUNICODESTRING,
        DETECTTYPE_OVERLAY,
        DETECTTYPE_RESOURCES,
        DETECTTYPE_RICH,
        DETECTTYPE_SECTIONNAME
    };

    struct DETECT_RECORD {
        qint64 nOffset;  // memory scan
        XBinary::FILEPART filepart;
        DETECTTYPE detectType;
        QString sValue;  // mb TODO variant
        quint32 nVariant;
        XBinary::FT fileType;
        RECORD_TYPE type;
        RECORD_NAME name;
        QString sVersion;
        QString sInfo;
    };

    struct SCAN_RESULT {
        qint64 nScanTime;
        QString sFileName;
        QList<SCAN_STRUCT> listRecords;
        QList<DETECT_RECORD> listHeurs;
    };

    struct _SCANS_STRUCT {
        qint64 nOffset;
        quint32 nVariant;
        XBinary::FT fileType;
        RECORD_TYPE type;
        RECORD_NAME name;
        QString sVersion;
        QString sInfo;
        bool bIsHeuristic;
        QVariant varExtra;
    };

    struct SCAN_RECORD {
        XBinary::FT fileType;
        RECORD_TYPE type;
        RECORD_NAME name;
        QString sVersion;
        QString sInfo;
    };

    struct BASIC_PE_INFO {
        quint32 nEntryPoint;
    };

    struct BASIC_INFO {
        qint64 nElapsedTime;
        XBinary::SCANID parentId;
        XBinary::SCANID id;
        QString sHeaderSignature;
        XBinary::_MEMORY_MAP memoryMap;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapHeaderDetects;
        QList<SCAN_STRUCT> listDetects;
        bool bIsDeepScan;
        bool bIsHeuristicScan;
        bool bIsVerbose;
        bool bShowDetects;
        bool bIsUnknown;
        bool bIsTest;
        QList<DETECT_RECORD> listHeurs;
    };

    struct BINARYINFO_STRUCT {
        BASIC_INFO basic_info;

        bool bIsPlainText;
        bool bIsUTF8;
        XBinary::UNICODE_TYPE unicodeType;
        QString sHeaderText;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapTextHeaderDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTexts;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultArchives;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCertificates;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultDebugData;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultInstallerData;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultSFXData;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultFormats;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultDatabases;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultImages;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultProtectorData;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraryData;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultResources;

        QList<SCAN_STRUCT> listRecursiveDetects;
    };

    struct DEXINFO_STRUCT {
        BASIC_INFO basic_info;

        XDEX_DEF::HEADER header;
        QList<XDEX_DEF::MAP_ITEM> mapItems;
        QList<QString> listStrings;
        QList<QString> listTypeItemStrings;
        QList<XDEX_DEF::FIELD_ITEM_ID> listFieldIDs;
        QList<XDEX_DEF::METHOD_ITEM_ID> listMethodIDs;
        bool bIsStringPoolSorted;
        bool bIsOverlayPresent;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapStringDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapTypeDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLinkers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCompilers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultProtectors;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;
    };

    struct ZIPINFO_STRUCT {
        BASIC_INFO basic_info;

        QList<XArchive::RECORD> listArchiveRecords;

        bool bIsJAR;
        bool bIsAPK;
        bool bIsIPA;
        bool bIsAPKS;
        bool bIsJava;
        bool bIsKotlin;

        DEXINFO_STRUCT dexInfoClasses;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapArchiveDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapMetainfosDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultSigntools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultArchives;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultFormats;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultAPKProtectors;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCompilers;

        QList<SCAN_STRUCT> listRecursiveDetects;
    };

    struct MACHOFATINFO_STRUCT {
        BASIC_INFO basic_info;

        QList<XArchive::RECORD> listArchiveRecords;

        QList<SCAN_STRUCT> listRecursiveDetects;
    };

    struct COMINFO_STRUCT {
        BASIC_INFO basic_info;
        qint64 nEntryPointOffset;
        QString sEntryPointSignature;
        QString sOverlaySignature;
        qint64 nOverlayOffset;
        qint64 nOverlaySize;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapEntryPointDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultPackers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultProtectors;

        QList<SCAN_STRUCT> listRecursiveDetects;
    };

    struct MSDOSINFO_STRUCT {
        BASIC_INFO basic_info;
        qint64 nEntryPointOffset;
        QString sEntryPointSignature;
        QString sOverlaySignature;
        qint64 nOverlayOffset;
        qint64 nOverlaySize;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapEntryPointDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultDosExtenders;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLinkers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCompilers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultProtectors;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultPackers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultSFX;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;

        QList<SCAN_STRUCT> listRecursiveDetects;
    };

    struct ELFINFO_STRUCT {
        BASIC_INFO basic_info;
        QString sEntryPointSignature;
        bool bIs64;
        bool bIsBigEndian;  // TODO move to basic
        QList<XELF::TAG_STRUCT> listTags;
        QList<QString> listLibraries;
        QList<QString> listComments;
        QList<XELF_DEF::Elf_Shdr> listSectionHeaders;
        QList<XELF_DEF::Elf_Phdr> listProgramHeaders;
        QList<XELF::SECTION_RECORD> listSectionRecords;
        QList<XELF::NOTE> listNotes;

        qint32 nCommentSection;
        qint32 nStringTableSection;
        QByteArray baStringTable;

        XBinary::OFFSETSIZE osCommentSection;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapCommentSectionDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapEntryPointDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLinkers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCompilers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultPackers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultProtectors;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;
    };

    struct LEINFO_STRUCT {
        BASIC_INFO basic_info;
        QString sEntryPointSignature;
        QString sOverlaySignature;
        qint64 nOverlayOffset;
        qint64 nOverlaySize;

        QList<XMSDOS::MS_RICH_RECORD> listRichSignatures;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapEntryPointDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLinkers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCompilers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;

        QList<SCAN_STRUCT> listRecursiveDetects;
    };

    struct LXINFO_STRUCT {
        BASIC_INFO basic_info;
        QString sEntryPointSignature;
        QString sOverlaySignature;
        qint64 nOverlayOffset;
        qint64 nOverlaySize;

        QList<XMSDOS::MS_RICH_RECORD> listRichSignatures;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapEntryPointDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLinkers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCompilers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;

        QList<SCAN_STRUCT> listRecursiveDetects;
    };

    struct NEINFO_STRUCT {
        BASIC_INFO basic_info;
        QString sEntryPointSignature;
        QString sOverlaySignature;
        qint64 nOverlayOffset;
        qint64 nOverlaySize;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapEntryPointDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLinkers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCompilers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;

        QList<SCAN_STRUCT> listRecursiveDetects;
    };

    struct MACHOINFO_STRUCT {
        BASIC_INFO basic_info;
        QString sEntryPointSignature;
        bool bIs64;
        bool bIsBigEndian;
        QList<XMACH::COMMAND_RECORD> listCommandRecords;
        QList<XMACH::LIBRARY_RECORD> listLibraryRecords;
        QList<XMACH::SEGMENT_RECORD> listSegmentRecords;
        QList<XMACH::SECTION_RECORD> listSectionRecords;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapEntryPointDetects;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLinkers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCompilers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultProtectors;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultSigntools;
    };

    struct PEINFO_STRUCT {
        BASIC_INFO basic_info;
        qint64 nEntryPointOffset;
        QString sEntryPointSignature;
        QString sOverlaySignature;
        qint64 nOverlayOffset;
        qint64 nOverlaySize;
        XMSDOS_DEF::IMAGE_DOS_HEADEREX dosHeader;
        XPE_DEF::IMAGE_FILE_HEADER fileHeader;
        union OPTIONAL_HEADER {
            XPE_DEF::IMAGE_OPTIONAL_HEADER32 optionalHeader32;
            XPE_DEF::IMAGE_OPTIONAL_HEADER64 optionalHeader64;
        } optional_header;
        QList<XPE_DEF::IMAGE_SECTION_HEADER> listSectionHeaders;
        QList<XPE::SECTION_RECORD> listSectionRecords;
        QList<QString> listSectionNames;
        QList<XPE::IMPORT_HEADER> listImports;
        QList<XPE::IMPORT_RECORD> listImportRecords;
        quint64 nImportHash64;
        quint32 nImportHash32;
        QList<quint32> listImportPositionHashes;
        XPE::EXPORT_HEADER exportHeader;
        QList<QString> listExportFunctionNames;
        QList<XPE::RESOURCE_RECORD> listResources;
        QList<XMSDOS::MS_RICH_RECORD> listRichSignatures;
        QString sResourceManifest;
        XPE::RESOURCES_VERSION resVersion;
        XPE::CLI_INFO cliInfo;

        QMap<RECORD_NAME, _SCANS_STRUCT> mapOverlayDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapEntryPointDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapImportDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapExportDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapDotAnsiStringsDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapDotUnicodeStringsDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapCodeSectionDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapEntryPointSectionDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapSectionNamesDetects;
        //        QMap<RECORD_NAME,_SCANS_STRUCT> mapRichDetects;
        QMap<RECORD_NAME, _SCANS_STRUCT> mapResourcesDetects;

        qint32 nEntryPointSection;
        qint32 nResourcesSection;
        qint32 nImportSection;
        qint32 nCodeSection;
        qint32 nDataSection;
        qint32 nConstDataSection;
        qint32 nRelocsSection;
        qint32 nTLSSection;
        QString sEntryPointSectionName;
        XADDR nEntryPointAddress;
        XADDR nImageBaseAddress;
        quint8 nMinorLinkerVersion;
        quint8 nMajorLinkerVersion;
        quint16 nMinorImageVersion;
        quint16 nMajorImageVersion;
        bool bIs64;
        bool bIsNetPresent;
        bool bIsTLSPresent;

        XBinary::OFFSETSIZE osHeader;
        XBinary::OFFSETSIZE osEntryPointSection;
        XBinary::OFFSETSIZE osCodeSection;
        XBinary::OFFSETSIZE osDataSection;
        XBinary::OFFSETSIZE osConstDataSection;
        XBinary::OFFSETSIZE osImportSection;
        XBinary::OFFSETSIZE osResourcesSection;

        QMap<RECORD_NAME, SCAN_STRUCT> mapResultOperationSystems;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLinkers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultCompilers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLibraries;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultTools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultPETools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultSigntools;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultProtectors;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultJoiners;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultPackers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultInstallers;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultSFX;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultNETObfuscators;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultNETCompressors;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultDongleProtection;
        QMap<RECORD_NAME, SCAN_STRUCT> mapResultLanguages;

        QList<SCAN_STRUCT> listRecursiveDetects;
    };

    struct SCAN_OPTIONS {
        //        bool bEmulate; // TODO Check
        bool bIsRecursiveScan;
        bool bIsDeepScan;
        bool bIsHeuristicScan;
        bool bIsVerbose;
        bool bAllTypesScan;
        bool bShowDetects;
        bool bResultAsXML;
        bool bResultAsJSON;
        bool bResultAsCSV;
        bool bResultAsTSV;
        bool bSubdirectories;
        bool bIsImage;
        bool bIsTest;
        XBinary::FT fileType;            // Optional
        XBinary::FILEPART initFilePart;  // Optional
        QVariant varInfo;                // Optional
    };

    // TODO Check
    //    struct UNPACK_OPTIONS {
    //        // PE/PE+
    //        bool bCopyOverlay;  // In
    //    };

    struct _BASICINFO {
        quint32 nVariant;
        const XBinary::FT fileType;
        const RECORD_TYPE type;
        const RECORD_NAME name;
        const char *pszVersion;
        const char *pszInfo;
    };

    struct SIGNATURE_RECORD {
        _BASICINFO basicInfo;
        const char *pszSignature;
    };

    struct STRING_RECORD {
        _BASICINFO basicInfo;
        const char *pszString;
    };

    struct PE_RESOURCES_RECORD {
        _BASICINFO basicInfo;
        bool bIsString1;
        const char *pszName1;
        quint32 nID1;
        bool bIsString2;
        const char *pszName2;
        quint32 nID2;
    };

    struct CONST_RECORD {
        _BASICINFO basicInfo;
        quint64 nConst1;
        quint64 nConst2;
    };

    struct MSRICH_RECORD {
        _BASICINFO basicInfo;
        quint16 nID;
        quint32 nBuild;
    };

    struct VCL_STRUCT {
        quint32 nValue;
        qint64 nOffset;
        bool bIs64;
    };

    struct VCL_PACKAGEINFO_MODULE {
        quint8 nFlags;
        quint8 nHashCode;
        QString sName;
    };

    struct VCL_PACKAGEINFO {
        quint32 nFlags;
        quint32 nUnknown;
        quint32 nRequiresCount;
        QList<VCL_PACKAGEINFO_MODULE> listModules;
    };

    struct VI_STRUCT {
        bool bIsValid;
        QString sVersion;
        QString sInfo;
        quint64 nValue;
    };

    explicit SpecAbstract(QObject *pParent = nullptr);

    static void scan(QIODevice *pDevice, SpecAbstract::SCAN_RESULT *pScanResult, qint64 nOffset, qint64 nSize, XBinary::SCANID parentId,
                     SpecAbstract::SCAN_OPTIONS *pOptions, bool bInit, XBinary::PDSTRUCT *pPdStruct = nullptr);

    static QString append(const QString &sResult, const QString &sString);

    static QString recordTypeIdToString(RECORD_TYPE id);
    static QString recordNameIdToString(RECORD_NAME id);
    static QString heurTypeIdToString(DETECTTYPE id);

    //    static SpecAbstract::UNPACK_OPTIONS getPossibleUnpackOptions(QIODevice *pDevice,SpecAbstract::SCAN_OPTIONS *pOptions); // TODO Check

    static QString _SCANS_STRUCT_toString(const _SCANS_STRUCT *pScanStruct, bool bShowType = true);

    static QString createResultString(const SCAN_STRUCT *pScanStruct);
    static QString createResultString2(const SCAN_STRUCT *pScanStruct);
    static QString createFullResultString(const SCAN_STRUCT *pScanStruct);
    static QString createFullResultString2(const SCAN_STRUCT *pScanStruct);
    static QString createTypeString(const SCAN_STRUCT *pScanStruct);

    static QString createShortResultString(const SCAN_RESULT scanResult);

    static SCAN_STRUCT createHeaderScanStruct(const SCAN_STRUCT *pScanStruct);

    static BINARYINFO_STRUCT getBinaryInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset,
                                           XBinary::PDSTRUCT *pPdStruct);
    static COMINFO_STRUCT getCOMInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, XBinary::PDSTRUCT *pPdStruct);
    static MSDOSINFO_STRUCT getMSDOSInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset,
                                         XBinary::PDSTRUCT *pPdStruct);
    static ELFINFO_STRUCT getELFInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, XBinary::PDSTRUCT *pPdStruct);
    static MACHOINFO_STRUCT getMACHOInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset,
                                         XBinary::PDSTRUCT *pPdStruct);
    static LEINFO_STRUCT getLEInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, XBinary::PDSTRUCT *pPdStruct);
    static LXINFO_STRUCT getLXInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, XBinary::PDSTRUCT *pPdStruct);
    static NEINFO_STRUCT getNEInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, XBinary::PDSTRUCT *pPdStruct);
    static PEINFO_STRUCT getPEInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, XBinary::PDSTRUCT *pPdStruct);
    static DEXINFO_STRUCT getDEXInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, XBinary::PDSTRUCT *pPdStruct);
    static ZIPINFO_STRUCT getZIPInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, XBinary::PDSTRUCT *pPdStruct);
    static MACHOFATINFO_STRUCT getMACHOFATInfo(QIODevice *pDevice, XBinary::SCANID parentId, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset,
                                               XBinary::PDSTRUCT *pPdStruct);

    static _SCANS_STRUCT getScansStruct(quint32 nVariant, XBinary::FT fileType, RECORD_TYPE type, RECORD_NAME name, const QString &sVersion, const QString &sInfo,
                                        qint64 nOffset);

    static void PE_handle_import(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);  // TODO remove !!!
    static void PE_handle_OperationSystems(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_Protection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_VMProtect(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_VProtect(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo,
                                   XBinary::PDSTRUCT *pPdStruct);                                                       // TODO move to protection
    static void PE_handle_TTProtect(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);  // TODO move to protection
    static void PE_handle_SafeengineShielden(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_tElock(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_Armadillo(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_Obsidium(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_Themida(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_StarForce(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_Petite(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_NETProtection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_Microsoft(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_Borland(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_Watcom(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_Tools(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_PETools(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_wxWidgets(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_GCC(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, SpecAbstract::PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_Signtools(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_Installers(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_SFX(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_PolyMorph(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_DongleProtection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_NeoLite(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_PrivateEXEProtector(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void PE_handle_VisualBasicCryptors(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handle_DelphiCryptors(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);

    static void PE_handle_Joiners(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);

    static bool PE_isProtectionPresent(PEINFO_STRUCT *pPEInfo);
    static void PE_handle_UnknownProtection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);

    static void PE_handle_FixDetects(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_handleLanguages(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);

    static void PE_handle_Recursive(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);

    static void Binary_handle_Texts(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo, XBinary::PDSTRUCT *pPdStruct);
    static void COM_handle_Protection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, COMINFO_STRUCT *pCOMInfo);
    static void Binary_handle_Archives(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo, XBinary::PDSTRUCT *pPdStruct);
    static void Binary_handle_Certificates(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handle_DebugData(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handle_Formats(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handle_Databases(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handle_Images(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handle_InstallerData(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handle_SFXData(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handle_ProtectorData(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handle_LibraryData(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handle_Resources(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);

    static void Binary_handle_FixDetects(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);
    static void Binary_handleLanguages(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, BINARYINFO_STRUCT *pBinaryInfo);

    static void MSDOS_handle_OperationSystems(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MSDOSINFO_STRUCT *pMSDOSInfo);
    static void MSDOS_handle_Tools(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MSDOSINFO_STRUCT *pMSDOSInfo);
    static void MSDOS_handle_Borland(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MSDOSINFO_STRUCT *pMSDOSInfo, XBinary::PDSTRUCT *pPdStruct);
    static void MSDOS_handle_Protection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MSDOSINFO_STRUCT *pMSDOSInfo, XBinary::PDSTRUCT *pPdStruct);
    static void MSDOS_handle_SFX(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MSDOSINFO_STRUCT *pMSDOSInfo);
    static void MSDOS_handle_DosExtenders(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MSDOSINFO_STRUCT *pMSDOSInfo, XBinary::PDSTRUCT *pPdStruct);

    static void MSDOS_handleLanguages(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MSDOSINFO_STRUCT *pMSDOSInfo, XBinary::PDSTRUCT *pPdStruct);

    static void MSDOS_handle_Recursive(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MSDOSINFO_STRUCT *pMSDOSInfo, XBinary::PDSTRUCT *pPdStruct);

    static void ELF_handle_OperationSystems(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ELFINFO_STRUCT *pELFInfo);
    static void ELF_handle_CommentSection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ELFINFO_STRUCT *pELFInfo, XBinary::PDSTRUCT *pPdStruct);
    static void ELF_handle_Tools(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ELFINFO_STRUCT *pELFInfo, XBinary::PDSTRUCT *pPdStruct);
    static void ELF_handle_GCC(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ELFINFO_STRUCT *pELFInfo);
    static void ELF_handle_Protection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ELFINFO_STRUCT *pELFInfo, XBinary::PDSTRUCT *pPdStruct);
    static void ELF_handle_UnknownProtection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ELFINFO_STRUCT *pELFInfo, XBinary::PDSTRUCT *pPdStruct);

    static void ELF_handle_FixDetects(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ELFINFO_STRUCT *pELFInfo);
    static void ELF_handleLanguages(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ELFINFO_STRUCT *pELFInfo, XBinary::PDSTRUCT *pPdStruct);

    static void MACHO_handle_Tools(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MACHOINFO_STRUCT *pMACHInfo, XBinary::PDSTRUCT *pPdStruct);
    static void MACHO_handle_Protection(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MACHOINFO_STRUCT *pMACHInfo);
    static void MACHO_handle_FixDetects(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MACHOINFO_STRUCT *pMACHInfo, XBinary::PDSTRUCT *pPdStruct);

    static void MACHO_handleLanguages(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, MACHOINFO_STRUCT *pMACHInfo, XBinary::PDSTRUCT *pPdStruct);

    static void LE_handle_OperationSystems(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, LEINFO_STRUCT *pLEInfo);
    static void LE_handle_Microsoft(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, LEINFO_STRUCT *pLEInfo, XBinary::PDSTRUCT *pPdStruct);
    static void LE_handle_Borland(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, LEINFO_STRUCT *pLEInfo);

    static void LE_handleLanguages(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, LEINFO_STRUCT *pLEInfo, XBinary::PDSTRUCT *pPdStruct);

    static void LX_handle_OperationSystems(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, LXINFO_STRUCT *pLXInfo);
    static void LX_handle_Microsoft(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, LXINFO_STRUCT *pLXInfo, XBinary::PDSTRUCT *pPdStruct);
    static void LX_handle_Borland(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, LXINFO_STRUCT *pLXInfo);

    static void LX_handleLanguages(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, LXINFO_STRUCT *pLXInfo, XBinary::PDSTRUCT *pPdStruct);

    static void NE_handle_OperationSystems(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, NEINFO_STRUCT *pNEInfo);
    static void NE_handle_Borland(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, NEINFO_STRUCT *pNEInfo);

    static void NE_handleLanguages(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, NEINFO_STRUCT *pNEInfo, XBinary::PDSTRUCT *pPdStruct);

    static void DEX_handle_Tools(QIODevice *pDevice, SCAN_OPTIONS *pOptions, DEXINFO_STRUCT *pDEXInfo, XBinary::PDSTRUCT *pPdStruct);
    static void DEX_handle_Dexguard(QIODevice *pDevice, DEXINFO_STRUCT *pDEXInfo, XBinary::PDSTRUCT *pPdStruct);
    static void DEX_handle_Protection(QIODevice *pDevice, DEXINFO_STRUCT *pDEXInfo, XBinary::PDSTRUCT *pPdStruct);

    static void DEX_handleLanguages(QIODevice *pDevice, DEXINFO_STRUCT *pDEXInfo, XBinary::PDSTRUCT *pPdStruct);

    static void Zip_handle_Microsoftoffice(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct);
    static void Zip_handle_OpenOffice(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct);
    static void Zip_handle_Metainfos(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct);
    static void Zip_handle_JAR(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct);
    static void Zip_handle_APK(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct);
    static void Zip_handle_IPA(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct);
    static void Zip_handle_Recursive(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct);
    static void Zip_handle_FixDetects(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct);
    static void Zip_handleLanguages(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct);

    static DEXINFO_STRUCT Zip_scan_DEX(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, ZIPINFO_STRUCT *pZipInfo, XBinary::PDSTRUCT *pPdStruct,
                                       const QString &sFileName);

    static void updateVersion(QMap<RECORD_NAME, SCAN_STRUCT> *pMap, RECORD_NAME name, const QString &sVersion);
    static void updateInfo(QMap<RECORD_NAME, SCAN_STRUCT> *pMap, RECORD_NAME name, const QString &sInfo);
    static void updateVersionAndInfo(QMap<RECORD_NAME, SCAN_STRUCT> *pMap, RECORD_NAME name, const QString &sVersion, const QString &sInfo);

    static bool isScanStructPresent(QList<SpecAbstract::SCAN_STRUCT> *pListScanStructs, XBinary::FT fileType, RECORD_TYPE type = RECORD_TYPE_UNKNOWN,
                                    RECORD_NAME name = RECORD_NAME_UNKNOWN, const QString &sVersion = "", const QString &sInfo = "");

    static VI_STRUCT get_UPX_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::FT fileType,
                                XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT _get_UPX_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::FT fileType);
    static VI_STRUCT get_GCC_vi1(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_GCC_vi2(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_Nim_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_Zig_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_PyInstaller_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT _get_GCC_string(const QString &sString);
    static VI_STRUCT get_WindowsInstaller_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_gold_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_TurboLinker_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions);
    static VI_STRUCT get_Enigma_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_DeepSea_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_SmartAssembly_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_R8_marker_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_Go_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_Rust_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT get_ObfuscatorLLVM_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT _get_ObfuscatorLLVM_string(const QString &sString);
    static VI_STRUCT get_AndroidClang_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, XBinary::PDSTRUCT *pPdStruct);
    static VI_STRUCT _get_AndroidClang_string(const QString &sString);
    static VI_STRUCT _get_AlipayClang_string(const QString &sString);
    static VI_STRUCT _get_AlpineClang_string(const QString &sString);
    static VI_STRUCT _get_AlibabaClang_string(const QString &sString);
    static VI_STRUCT _get_PlexClang_string(const QString &sString);
    static VI_STRUCT _get_UbuntuClang_string(const QString &sString);
    static VI_STRUCT _get_DebianClang_string(const QString &sString);
    static VI_STRUCT _get_AlipayObfuscator_string(const QString &sString);
    static VI_STRUCT _get_wangzehuaLLVM_string(const QString &sString);
    static VI_STRUCT _get_ByteGuard_string(const QString &sString);
    static VI_STRUCT _get_TencentObfuscation_string(const QString &sString);
    static VI_STRUCT _get_AppImage_string(const QString &sString);
    static VI_STRUCT _get_HikariObfuscator_string(const QString &sString);
    static VI_STRUCT _get_SnapProtect_string(const QString &sString);
    static VI_STRUCT _get_ByteDanceSecCompiler_string(const QString &sString);
    static VI_STRUCT _get_DingbaozengNativeObfuscator_string(const QString &sString);
    static VI_STRUCT _get_SafeengineLLVM_string(const QString &sString);
    static VI_STRUCT _get_NagainLLVM_string(const QString &sString);
    static VI_STRUCT _get_iJiami_string(const QString &sString);
    static VI_STRUCT _get_AppleLLVM_string(const QString &sString);
    static VI_STRUCT _get_ApportableClang_string(const QString &sString);
    static VI_STRUCT _get_ARMAssembler_string(const QString &sString);
    static VI_STRUCT _get_ARMLinker_string(const QString &sString);
    static VI_STRUCT _get_ARMC_string(const QString &sString);
    static VI_STRUCT _get_ARMCCPP_string(const QString &sString);
    static VI_STRUCT _get_ARMNEONCCPP_string(const QString &sString);
    static VI_STRUCT _get_ARMThumbCCPP_string(const QString &sString);
    static VI_STRUCT _get_ARMThumbMacroAssembler_string(const QString &sString);
    static VI_STRUCT _get_ThumbC_string(const QString &sString);
    static VI_STRUCT _get_clang_string(const QString &sString);
    static VI_STRUCT _get_DynASM_string(const QString &sString);
    static VI_STRUCT _get_Delphi_string(const QString &sString);
    static VI_STRUCT _get_LLD_string(const QString &sString);
    static VI_STRUCT _get_mold_string(const QString &sString);
    static VI_STRUCT _get_OracleSolarisLinkEditors_string(const QString &sString);
    static VI_STRUCT _get_SunWorkShop_string(const QString &sString);
    static VI_STRUCT _get_SunWorkShopCompilers_string(const QString &sString);
    static VI_STRUCT _get_SnapdragonLLVMARM_string(const QString &sString);
    static VI_STRUCT _get_NASM_string(const QString &sString);
    static VI_STRUCT _get_TencentLegu_string(const QString &sString);
    static VI_STRUCT _get_OllvmTll_string(const QString &sString);
    static VI_STRUCT _get_DelphiVersionFromCompiler(const QString &sString);
    static VI_STRUCT _get_SourceryCodeBench_string(const QString &sString);

    static bool PE_isValid_UPX(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);
    static void PE_x86Emul(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo, XBinary::PDSTRUCT *pPdStruct);

    static VI_STRUCT PE_get_PECompact_vi(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, PEINFO_STRUCT *pPEInfo);

    static QList<VCL_STRUCT> PE_getVCLstruct(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize, bool bIs64,
                                             XBinary::PDSTRUCT *pPdStruct);
    static VCL_PACKAGEINFO PE_getVCLPackageInfo(QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, QList<XPE::RESOURCE_RECORD> *pListResources,
                                                XBinary::PDSTRUCT *pPdStruct);

    static SCAN_STRUCT scansToScan(BASIC_INFO *pBasicInfo, _SCANS_STRUCT *pScansStruct);

    static QByteArray _BasicPEInfoToArray(BASIC_PE_INFO *pInfo);
    static BASIC_PE_INFO _ArrayToBasicPEInfo(const QByteArray *pbaArray);

    static void memoryScan(QMap<RECORD_NAME, _SCANS_STRUCT> *pMapRecords, QIODevice *pDevice, SpecAbstract::SCAN_OPTIONS *pOptions, qint64 nOffset, qint64 nSize,
                           SpecAbstract::SIGNATURE_RECORD *pRecords, qint32 nRecordsSize, XBinary::FT fileType1, XBinary::FT fileType2, BASIC_INFO *pBasicInfo,
                           DETECTTYPE detectType, XBinary::PDSTRUCT *pPdStruct);
    static void signatureScan(QMap<RECORD_NAME, _SCANS_STRUCT> *pMapRecords, const QString &sSignature, SIGNATURE_RECORD *pRecords, qint32 nRecordsSize,
                              XBinary::FT fileType1, XBinary::FT fileType2, BASIC_INFO *pBasicInfo, DETECTTYPE detectType, XBinary::PDSTRUCT *pPdStruct);
    static void PE_resourcesScan(QMap<RECORD_NAME, _SCANS_STRUCT> *pMapRecords, QList<XPE::RESOURCE_RECORD> *pListResources, PE_RESOURCES_RECORD *pRecords,
                                 qint32 nRecordsSize, XBinary::FT fileType1, XBinary::FT fileType2, BASIC_INFO *pBasicInfo, DETECTTYPE detectType,
                                 XBinary::PDSTRUCT *pPdStruct);
    static void stringScan(QMap<RECORD_NAME, _SCANS_STRUCT> *pMapRecords, QList<QString> *pListStrings, STRING_RECORD *pRecords, qint32 nRecordsSize,
                           XBinary::FT fileType1, XBinary::FT fileType2, BASIC_INFO *pBasicInfo, DETECTTYPE detectType, XBinary::PDSTRUCT *pPdStruct);
    static void constScan(QMap<RECORD_NAME, _SCANS_STRUCT> *pMapRecords, quint64 nCost1, quint64 nCost2, CONST_RECORD *pRecords, qint32 nRecordsSize,
                          XBinary::FT fileType1, XBinary::FT fileType2, BASIC_INFO *pBasicInfo, DETECTTYPE detectType, XBinary::PDSTRUCT *pPdStruct);
    static void MSDOS_richScan(QMap<RECORD_NAME, _SCANS_STRUCT> *pMapRecords, quint16 nID, quint32 nBuild, quint32 nCount, MSRICH_RECORD *pRecords, qint32 nRecordsSize,
                               XBinary::FT fileType1, XBinary::FT fileType2, BASIC_INFO *pBasicInfo, DETECTTYPE detectType, XBinary::PDSTRUCT *pPdStruct);
    static void archiveScan(QMap<RECORD_NAME, _SCANS_STRUCT> *pMapRecords, QList<XArchive::RECORD> *pListArchiveRecords, STRING_RECORD *pRecords, qint32 nRecordsSize,
                            XBinary::FT fileType1, XBinary::FT fileType2, BASIC_INFO *pBasicInfo, DETECTTYPE detectType, XBinary::PDSTRUCT *pPdStruct);
    static void archiveExpScan(QMap<RECORD_NAME, _SCANS_STRUCT> *pMapRecords, QList<XArchive::RECORD> *pListArchiveRecords, STRING_RECORD *pRecords, qint32 nRecordsSize,
                               XBinary::FT fileType1, XBinary::FT fileType2, BASIC_INFO *pBasicInfo, DETECTTYPE detectType, XBinary::PDSTRUCT *pPdStruct);

    static void signatureExpScan(XBinary *pXBinary, XBinary::_MEMORY_MAP *pMemoryMap, QMap<RECORD_NAME, _SCANS_STRUCT> *pMapRecords, qint64 nOffset,
                                 SIGNATURE_RECORD *pRecords, qint32 nRecordsSize, XBinary::FT fileType1, XBinary::FT fileType2, BASIC_INFO *pBasicInfo,
                                 DETECTTYPE detectType, XBinary::PDSTRUCT *pPdStruct);

    static QList<_SCANS_STRUCT> MSDOS_richScan(quint16 nID, quint32 nBuild, quint32 nCount, MSRICH_RECORD *pRecords, qint32 nRecordsSize, XBinary::FT fileType1,
                                               XBinary::FT fileType2, BASIC_INFO *pBasicInfo, DETECTTYPE detectType, XBinary::PDSTRUCT *pPdStruct);

    static QByteArray serializeScanStruct(const SCAN_STRUCT &scanStruct, bool bIsHeader = false);
    static SCAN_STRUCT deserializeScanStruct(const QByteArray &baData, bool *pbIsHeader = nullptr);

    static void getLanguage(QMap<RECORD_NAME, SCAN_STRUCT> *pMapDetects, QMap<RECORD_NAME, SCAN_STRUCT> *pMapLanguages, XBinary::PDSTRUCT *pPdStruct);
    static void fixLanguage(QMap<RECORD_NAME, SCAN_STRUCT> *pMapLanguages);

    static _SCANS_STRUCT getScansStructFromOsInfo(const XBinary::OSINFO &osInfo);
    static QString getMsRichString(quint16 nId, quint16 nBuild, quint32 nCount, XBinary::PDSTRUCT *pPdStruct);

    static QList<XBinary::SCANSTRUCT> convert(QList<SCAN_STRUCT> *pListScanStructs);

private:
    static bool MSDOS_compareRichRecord(_SCANS_STRUCT *pResult, MSRICH_RECORD *pRecord, quint16 nID, quint32 nBuild, quint32 nCount, XBinary::FT fileType1,
                                        XBinary::FT fileType2);
    static void filterResult(QList<SCAN_STRUCT> *pListRecords, const QSet<RECORD_TYPE> &stRecordTypes, XBinary::PDSTRUCT *pPdStruct);
    static void _fixRichSignatures(QList<_SCANS_STRUCT> *pListRichSignatures, qint32 nMajorLinkerVersion, qint32 nMinorLinkerVersion, XBinary::PDSTRUCT *pPdStruct);

protected:
    void _errorMessage(const QString &sErrorMessage);
    void _infoMessage(const QString &sInfoMessage);

signals:
    void errorMessage(const QString &sErrorMessage);
    void infoMessage(const QString &sInfoMessage);
};

#endif  // SPECABSTRACT_H
