/*** -*- Mode: Javascript; tab-width: 2;

The contents of this file are subject to the Mozilla Public
License Version 1.1 (the "License"); you may not use this file
except in compliance with the License. You may obtain a copy of
the License at http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS
IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
implied. See the License for the specific language governing
rights and limitations under the License.

The Original Code is jslib team code.
The Initial Developer of the Original Code is jslib team.

Portions created by jslib team are
Copyright (C) 2000 jslib team.  All
Rights Reserved.

Original Author: Pete Collins <pete@mozdevgroup.com>
Contributor(s): 

***/

if (typeof(JS_LIB_LOADED)=='boolean') 
{
  /****************** Globals **********************/
  const JS_ZIP_LOADED       = true;
  const JS_ZIP_FILE         = "zip.js";
  const JS_ZIP_CID          = "@mozilla.org/libjar/zip-reader;1";
  const JS_ZIP_I_ZIP_READER = "nsIZipReader";
  const JS_ZIP_FILE_CID     = "@mozilla.org/file/local;1";
  const JS_ZIP_I_LOCAL_FILE = "nsILocalFile";
  const JS_ZIP_FILE_INIT    = "initWithPath";
  
  function Zip (aZipFile) 
  {
    if (!aZipFile)
      return jslibErrorMsg("NS_ERROR_XPC_NOT_ENOUGH_ARGS"); 

    // sanity check for nsIFile obj, File obj, or string path arg
    if (jslibTypeIsObj(aZipFile)) {
      // see if it is a jslib obj or nsIFile
      if (!jslibTypeIsUndef(aZipFile.nsIFile))
        this.mZipFile = aZipFile.nsIFile;
      else
        this.mZipFile = aZipFile;
    } else if (jslibTypeIsStr(aZipFile)) {
      this.mZipFile = new this.ZipPath(aZipFile);
    }
 
    if (!this.mZipFile.exists())
      return jslibErrorMsg("NS_ERROR_FILE_NOT_FOUND", this.mZipFile.path);

    this.init();
  }
  
  Zip.prototype = {
  
  ZipReader : new C.Constructor(JS_ZIP_CID, JS_ZIP_I_ZIP_READER),
  ZipPath   : new C.Constructor(JS_ZIP_FILE_CID, JS_ZIP_I_LOCAL_FILE, 
                                         JS_ZIP_FILE_INIT),
  DIRECTORY : 0x01, // 1
  mPerm     : 0755,
  FINISHED  : "*** FINISHED . . .  .",
  mInit     : false,
  mZipFile  : null,
  mZip      : null,
  mIsOpen   : false,
  
  // initialize the zip reader
  // *********************************
  init : function ()
  {
    try {
      this.mZip = new this.ZipReader();
      this.mZip.init(this.mZipFile);
      this.mInit = true;
      this.open();
    } catch (e) { return jslibError(e); }
  }, 
  // *********************************

  // open zip file
  // *********************************
  open : function ()
  {
    if (!this.mInit)
      this.init();

    if (this.mIsOpen) return;

    try { this.mZip.open(); } 
    catch (e) { return jslibError(e); }

    this.mIsOpen = true;
  },
  // *********************************
  
  // close zip file
  // *********************************
  close : function ()
  {
    if (!this.mInit)
      return jslibErrorMsg("NS_ERROR_NOT_INITIALIZED");

    if (!this.mIsOpen) return;

    try { this.mZip.close(); } 
    catch (e) { return jslibError(e); }

    this.mInit = false;
    this.mIsOpen = false;
  },
  // *********************************
  
  // extract zip archive
  // *********************************
  extract : function (aDest, aDomEl) 
  {
    if (!this.mInit)
      return jslibErrorMsg("NS_ERROR_NOT_INITIALIZED");

    if (arguments.length < 1) 
      return jslibErrorMsg("NS_ERROR_INVALID_ARG");
    else if (!aDest)
      return jslibErrorMsg("NS_ERROR_FILE_TARGET_DOES_NOT_EXIST");
  
    var rv = JS_LIB_OK;
    var dest;

    // sanity check for nsIFile objects string path args
    if (jslibTypeIsObj(aDest))
      dest = aDest;
    else if (jslibTypeIsStr(aDest))
      dest = new this.ZipPath(aDest);

    var entry;
    var newDir;
    try {
      if (!dest.exists() || !dest.isDirectory())
        dest.create( this.DIRECTORY, this.mPerm);
  
      jslibDebug("\nExtracting:\t"+this.mZipFile.path+"\nTo:\t\t"+dest.path+" . . . \n\n");
  
      var entries = this.findEntries("*");
      var destbase = new this.ZipPath(dest.path);
      while (entries.hasMoreElements()) {
        entry = jslibQI(entries.getNext(), "nsIZipEntry");
        dest  = new this.ZipPath(dest.path);
        dest.setRelativeDescriptor(destbase, entry.name);

        // create if entry is a dir
        if (!dest.exists() && entry.name.length - 1 == entry.name.lastIndexOf("/")) {
          dest.create(this.DIRECTORY, this.mPerm);
          continue;
        } else if (!dest.parent.exists()) {
          dest.parent.create(this.DIRECTORY, this.mPerm);
        } else if (dest.exists()) {
          continue;
        }

        if (aDomEl)
          aDomEl.value = "extracting: "+dest.leafName;

        jslibDebug("extracting: ["+entry.name+"] To: ["+dest.path+"]");
        this.mZip.extract(entry.name, dest);
      }
    } catch (e) { rv = jslibError(e); }
  
    jslibDebug("\n\n"+this.FINISHED+"\n\n");
  
    if (aDomEl)
      aDomEl.value = this.FINISHED;
  
    return rv;
  }, 
  // *********************************

  // returns simple enumerator whose elements are of type nsIZipEntry
  // *********************************
  findEntries : function (aPattern) 
  {
    if (!aPattern)
      return jslibErrorMsg("NS_ERROR_INVALID_ARG");

    if (!this.mInit)
      return jslibErrorMsg("NS_ERROR_NOT_INITIALIZED");

    var rv = null;
    try {
      rv = this.mZip.findEntries(aPattern);
    } catch (e) { rv = jslibError(e); }
      
    return rv;
  }, 
  // *********************************

  // returns array of entries found w/ aPattern
  // *********************************
  getEntries : function (aPattern) 
  {
    if (!aPattern)
      return jslibErrorMsg("NS_ERROR_INVALID_ARG");

    if (!this.mInit)
      return jslibErrorMsg("NS_ERROR_NOT_INITIALIZED");

    var rv = new Array();
    try {
      var entries = this.findEntries(aPattern);
      while (entries.hasMoreElements()) {
        var entry = jslibQI(entries.getNext(), "nsIZipEntry");
        rv.push(entry.name);
      }
    } catch (e) { jslibError(e); }
      
    return rv;
  }, 
  // *********************************
  
  // returns nsIFile object of initialized zip file
  // *********************************
  get file ()
  {
    if (!this.mInit)
      return jslibErrorMsg("NS_ERROR_NOT_INITIALIZED");

    return this.mZip.file;
  },
  // *********************************

  // reads a zip entry
  // *********************************
  readEntry : function (aZipEntry)
  {
    if (!aZipEntry)
      return jslibErrorMsg("NS_ERROR_INVALID_ARG");

    if (!this.mInit)
      return jslibErrorMsg("NS_ERROR_NOT_INITIALIZED");

    var rv;
    try {
      var sis = jslibCreateInstance("@mozilla.org/scriptableinputstream;1", 
                                    "nsIScriptableInputStream");
      sis.init(this.getInputStream(aZipEntry));
      rv = sis.read(sis.available());
      sis.close();
    } catch (e) { rv = jslibError(e); }

    return rv;
  },
  // *********************************

  // returns input stream containing contents of specified zip entry
  // *********************************
  getInputStream : function (aZipEntry)
  {
    if (!aZipEntry)
      return jslibErrorMsg("NS_ERROR_INVALID_ARG");

    if (!this.mInit)
      return jslibErrorMsg("NS_ERROR_NOT_INITIALIZED");

    var rv;
    try {
      rv = this.mZip.getInputStream(aZipEntry);
    } catch (e) { rv = jslibError(e); }

    return rv;
  }
  // *********************************
} // END Zip CLASS

jslibDebug('*** load: '+JS_ZIP_FILE+' OK');

} // END BLOCK JS_LIB_LOADED CHECK

// If jslib base library is not loaded, jslibDebug this error.
else {
    dump("JS_FILE library not loaded:\n"                                +
         " \tTo load use: chrome://jslib/content/jslib.js\n"            +
         " \tThen: include(jslib_zip);\n\n");
}
