/*
 * License: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Gesso, an extension for Mozilla Firefox
 *
 * The Initial Developer of the Original Code is Ningjie (Jim) Chen.
 * Portions created by the Initial Developer are Copyright (C) 2007
 * the Initial Developer. All Rights Reserved.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the License, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the License, the GPL or the LGPL.
 *
 * See MPL.txt for terms of the Mozilla Public License Version 1.1
 * See GPL.txt for terms of the GNU General Public License Version 2.0
 * See LGPL.txt for terms of the GNU Lesser General Public License Version 2.1
 */

#include "stdafx.h"
#include "igesso.h"
#include "gesso.h"
#include "gessoutil.h"
const nsIID nsISupportsIID = NS_ISUPPORTS_IID;

class CGessoFormattedText : public CComObjectRoot, public IDataObject {

	BEGIN_COM_MAP(CGessoFormattedText)
		COM_INTERFACE_ENTRY(IDataObject)
	END_COM_MAP();
	DECLARE_PROTECT_FINAL_CONSTRUCT();

protected:
	typedef CSimpleMap<CLIPFORMAT, nsString> FormattedTextMap;
	FormattedTextMap m_map;

	nsString &GetFormat(FORMATETC *pformat, HRESULT &hr) {
		if (pformat->dwAspect == DVASPECT_CONTENT) {
			if (pformat->lindex == -1) {
				if (pformat->tymed & TYMED_HGLOBAL) {
					int i = m_map.FindKey(pformat->cfFormat);
					if (i >= 0) {
						hr = S_OK;
						return m_map.GetValueAt(i);
					} else hr = DV_E_FORMATETC;
				} else hr = DV_E_TYMED;
			} else hr = DV_E_LINDEX;
		} else hr = DV_E_DVASPECT;
		return *reinterpret_cast<nsString*>(NULL);
	}
	HRESULT FillMedium(nsString &data, STGMEDIUM *pmedium) {
		switch (pmedium->tymed) {
			case TYMED_HGLOBAL:	{
					size_t iSize = GlobalSize(pmedium->hGlobal) / sizeof(WCHAR) - 1;
					if (iSize < data.Length()) return STG_E_MEDIUMFULL;
					PWSTR pStr = reinterpret_cast<PWSTR>(GlobalLock(pmedium->hGlobal));
					if (pStr) {
						CopyMemory(pStr, data.BeginReading(), sizeof(WCHAR) * data.Length());
						*(pStr + data.Length()) = NULL; // null-terminated
						GlobalUnlock(pmedium->hGlobal);
						return S_OK;
					}
				} break;
			case TYMED_ISTREAM: {
					ULONG ulWritten; WCHAR wcNull = NULL;
					if (FAILED(pmedium->pstm->Write(data.BeginReading(), sizeof(WCHAR) * data.Length(), &ulWritten) ||
						FAILED(pmedium->pstm->Write(&wcNull, sizeof(WCHAR), &ulWritten)))) return STG_E_MEDIUMFULL;
					return S_OK;
				}
			case TYMED_FILE:
				break;
			case TYMED_ISTORAGE:
				break;
		}
		return E_INVALIDARG;
	}
	
public:
	BOOL Add(CLIPFORMAT f, nsAString &d) {
		return m_map.Add(f, nsString(d));
	}
	STDMETHODIMP GetData(FORMATETC *pformatetcIn, STGMEDIUM *pmedium) {
		HRESULT hr;
		nsString &text = GetFormat(pformatetcIn, hr);
		if (FAILED(hr)) return hr;
		pmedium->hGlobal = GlobalAlloc(GMEM_MOVEABLE, sizeof(WCHAR) * (text.Length() + 1));
		if (!pmedium->hGlobal) return STG_E_MEDIUMFULL;
		pmedium->tymed = TYMED_HGLOBAL;
		pmedium->pUnkForRelease = NULL;
		return FillMedium(text, pmedium);
	}
	STDMETHODIMP GetDataHere(FORMATETC *pformatetcIn, STGMEDIUM *pmedium) {
		HRESULT hr;
		nsString &text = GetFormat(pformatetcIn, hr);
		if (FAILED(hr)) return hr;
		return FillMedium(text, pmedium);
	}
	STDMETHODIMP QueryGetData(FORMATETC *pformatetc) {
		HRESULT hr;
		GetFormat(pformatetc, hr);
		return hr;
	}
	STDMETHODIMP GetCanonicalFormatEtc(FORMATETC *pformatetcIn, FORMATETC *pformatetcOut) {
		if (pformatetcIn->lindex != -1) return DV_E_LINDEX;
		if (m_map.FindKey(pformatetcIn->cfFormat) < 0) return DV_E_FORMATETC;
		pformatetcOut->cfFormat = pformatetcIn->cfFormat;
		pformatetcOut->lindex = -1;
		pformatetcOut->dwAspect = DVASPECT_CONTENT;
		pformatetcOut->ptd = NULL;
		return S_OK;
	}
	STDMETHODIMP SetData(FORMATETC *, STGMEDIUM *, BOOL) {
		return E_NOTIMPL;
	}
	STDMETHODIMP EnumFormatEtc(DWORD dwDirection, IEnumFORMATETC **penumFormatEtc) {
		if (dwDirection != DATADIR_GET) return E_NOTIMPL;
		FORMATETC *pFormat = new FORMATETC[m_map.GetSize()];
		if (!pFormat) return E_OUTOFMEMORY;
		for (int i = 0; i < m_map.GetSize(); ++i) {
			pFormat[i].cfFormat = m_map.GetKeyAt(i);
			pFormat[i].dwAspect = DVASPECT_CONTENT;
			pFormat[i].lindex = -1;
			pFormat[i].ptd = 0;
			pFormat[i].tymed = TYMED_HGLOBAL;
		}
		HRESULT hr = ::CreateFormatEnumerator(m_map.GetSize(), pFormat, penumFormatEtc);
		delete [] pFormat;
		return hr;
	}
	STDMETHODIMP DAdvise(FORMATETC *, DWORD, IAdviseSink *, DWORD *) {
		return OLE_E_ADVISENOTSUPPORTED;
	}
	STDMETHODIMP DUnadvise(DWORD) {
		return OLE_E_ADVISENOTSUPPORTED;
	}
	STDMETHODIMP EnumDAdvise(IEnumSTATDATA **) {
		return OLE_E_ADVISENOTSUPPORTED;
	}
};
typedef CComPtr<CComObject<CGessoFormattedText> > CGessoFormattedTextPtr;



HRESULT CGessoTextStore::FinalConstruct() {
	m_dwLock = 0;
	m_dwSinkMask = 0;
	return S_OK;
}

STDMETHODIMP CGessoTextStore::AdviseSink(REFIID riid, IUnknown *punk, DWORD dwMask) {
	if (punk == NULL || !IsEqualIID(riid, IID_ITextStoreACPSink)) 
		return E_INVALIDARG;
	else if (!m_pSink) {
		if (FAILED(punk->QueryInterface(riid, reinterpret_cast<LPVOID*>(&m_pSink))))
			return E_UNEXPECTED;
	}
	else if (!m_pSink.IsEqualObject(punk)) 
		return CONNECT_E_ADVISELIMIT; // another sink installed
	m_dwSinkMask = dwMask;
	return S_OK;
}

STDMETHODIMP CGessoTextStore::UnadviseSink(IUnknown *punk) {
	if (punk == NULL || !m_pSink.IsEqualObject(punk)) return E_INVALIDARG;
	m_pSink = NULL;
	m_dwSinkMask = 0;
	return S_OK;
}

STDMETHODIMP CGessoTextStore::RequestLock(DWORD dwLockFlags, HRESULT *phrSession) {
	if (m_dwLock == 0) {
		m_dwLock = dwLockFlags & (~TS_LF_SYNC);
		m_ulQueuedLocksCount = 0;
		m_pDoc->Lock(TRUE);
		if (m_pSink && (FAILED(m_pSink->OnLockGranted(dwLockFlags & (~TS_LF_SYNC)))))	// grant lock
			OutputDebugString(_T("OnLockGranted Failed"));
		if (m_ulQueuedLocksCount > 0)	// queued requests
			for (unsigned int i = 0; i < m_ulQueuedLocksCount; ++i)
				m_pSink->OnLockGranted(m_dwQueuedLocks[i]);
		m_pDoc->Lock(FALSE);
		m_dwLock = 0;	// release lock
		*phrSession = S_OK;
	}
	else if ((dwLockFlags & TS_LF_SYNC) != 0)
		*phrSession = TS_E_SYNCHRONOUS;	// cannot grant sync lock
	else {
		*phrSession = TS_S_ASYNC;	// cannot grant async lock, queue request
		if (m_ulQueuedLocksCount < MAX_QUEUED_LOCKS)
            m_dwQueuedLocks[m_ulQueuedLocksCount++] = dwLockFlags & (~TS_LF_SYNC);
	}
	return S_OK;
}

STDMETHODIMP CGessoTextStore::GetStatus(TS_STATUS *pdcs) {
	return NS_TO_HRESULT(m_pDoc->GetStatus(CGessoStatus(pdcs)));
}

STDMETHODIMP CGessoTextStore::QueryInsert(LONG acpTestStart, LONG acpTestEnd, ULONG cch, LONG *pacpResultStart, LONG *pacpResultEnd) {
	return NS_TO_HRESULT(m_pDoc->QueryInsert(acpTestStart, acpTestEnd, cch, CGessoOutLONG(pacpResultStart), CGessoOutLONG(pacpResultEnd)));
}

STDMETHODIMP CGessoTextStore::GetSelection(ULONG ulIndex, ULONG ulCount, TS_SELECTION_ACP *pSelection, ULONG *pcFetched) {
	ulCount;
	// only input selection supported
	if ((m_dwLock & TS_LF_READ) != TS_LF_READ) return TS_E_NOLOCK;
	if (ulIndex != TF_DEFAULT_SELECTION) return TS_E_NOSELECTION;
	nsresult nr;
	if (NS_SUCCEEDED(nr = m_pDoc->GetSelection(0, CGessoSelection(pSelection))))
		*pcFetched = 1;
	return NS_TO_HRESULT(nr);
}

STDMETHODIMP CGessoTextStore::SetSelection(ULONG ulCount, const TS_SELECTION_ACP *pSelection) {
	ulCount;
	// only input selection supported
	if ((m_dwLock & TS_LF_READWRITE) != TS_LF_READWRITE) return TS_E_NOLOCK;
	return NS_TO_HRESULT(m_pDoc->SetSelection(0, CGessoSelection(pSelection)));
}

STDMETHODIMP CGessoTextStore::GetText(LONG acpStart, LONG acpEnd, WCHAR *pchPlain, ULONG cchPlainReq, ULONG *pcchPlainOut, TS_RUNINFO *prgRunInfo, ULONG ulRunInfoReq, ULONG *pulRunInfoOut, LONG *pacpNext) {
	if ((m_dwLock & TS_LF_READ) != TS_LF_READ) return TS_E_NOLOCK;
	nsresult nr = NS_OK;
	CGessoRunInfo *pRunInfos = new CGessoRunInfo[ulRunInfoReq];
	iGessoRunInfo **ppRunInfos = new iGessoRunInfo*[ulRunInfoReq];
	if (pRunInfos && ppRunInfos) {
		for (ULONG i = 0; i < ulRunInfoReq; ++i) {
			pRunInfos[i].m_obj.m_ptr = &prgRunInfo[i];
			memset(&prgRunInfo[i], 0, sizeof(TS_RUNINFO));
			ppRunInfos[i] = pRunInfos[i];
		}
		*pcchPlainOut = 0;
		nsString strPlain;
		if (NS_SUCCEEDED(nr = m_pDoc->GetText(acpStart, acpEnd, CGessoOutAString(strPlain),
				pchPlain == NULL || cchPlainReq == 0 ? FALSE : TRUE, ppRunInfos, ulRunInfoReq, CGessoOutULONG(pulRunInfoOut)))) {
			int iLen = min(strPlain.Length(), cchPlainReq - 1);
			memcpy(pchPlain, strPlain.BeginReading(), iLen * sizeof(WCHAR));
			*pcchPlainOut = iLen; pchPlain[iLen] = NULL;
			*pacpNext = acpStart + iLen;
		}
	} else nr = NS_ERROR_OUT_OF_MEMORY;
	if (ppRunInfos) delete [] ppRunInfos;
	if (pRunInfos) delete [] pRunInfos;
	return NS_TO_HRESULT(nr);
}

STDMETHODIMP CGessoTextStore::SetText(DWORD dwFlags, LONG acpStart, LONG acpEnd, const WCHAR *pchText, ULONG cch, TS_TEXTCHANGE *pChange) {
	dwFlags;
	if ((m_dwLock & TS_LF_READWRITE) != TS_LF_READWRITE) return TS_E_NOLOCK;
	HRESULT hr;
	TS_SELECTION_ACP acpSel = {acpStart, acpEnd, {TS_AE_END, FALSE}};
	if (FAILED(hr = SetSelection(1, &acpSel))) return hr;
	hr = InsertTextAtSelection(0, pchText, cch, &acpSel.acpStart, &acpSel.acpEnd, pChange);
	return hr;
}

STDMETHODIMP CGessoTextStore::GetFormattedText(LONG acpStart, LONG acpEnd, IDataObject **ppDataObject) {
	if ((m_dwLock & TS_LF_READWRITE) != TS_LF_READWRITE) return TS_E_NOLOCK;
	HRESULT hr; ULONG ulCount;
	if (FAILED(hr = NS_TO_HRESULT(m_pDoc->GetFormattedTextCount(acpStart, acpEnd, CGessoOutULONG(&ulCount))))) return hr;
	CComObject<CGessoFormattedText> *pRaw; 
	if (FAILED(hr = CComObject<CGessoFormattedText>::CreateInstance(&pRaw))) return hr;
	CGessoFormattedTextPtr pText = pRaw; pRaw = NULL;
	for (ULONG i = 0; i < ulCount; ++i) {
		WORD wFormat; nsString strFormat;
		if (FAILED(hr = NS_TO_HRESULT(m_pDoc->GetFormattedText(i, CGessoOutWORD(&wFormat), CGessoOutAString(strFormat)))))
			return hr;
		if (FAILED(pText->Add(wFormat, strFormat))) return E_OUTOFMEMORY;
	}
	*ppDataObject = pText; (*ppDataObject)->AddRef();
	return S_OK;
}

STDMETHODIMP CGessoTextStore::GetEmbedded(LONG /*acpPos*/, REFGUID /*rguidService*/, REFIID /*riid*/, IUnknown ** /*ppunk*/) {
	// zero support for embedded objects
	// what a hassle would it be if we actually have to implement all these methods
	return E_NOTIMPL;
}

STDMETHODIMP CGessoTextStore::QueryInsertEmbedded(const GUID * /*pguidService*/, const FORMATETC * /*pFormatEtc*/, BOOL * /*pfInsertable*/) {
	return E_NOTIMPL;
}

STDMETHODIMP CGessoTextStore::InsertEmbedded(DWORD /*dwFlags*/, LONG /*acpStart*/, LONG /*acpEnd*/, IDataObject * /*pDataObject*/, TS_TEXTCHANGE * /*pChange*/) {
	return E_NOTIMPL;
}

STDMETHODIMP CGessoTextStore::RequestSupportedAttrs(DWORD dwFlags, ULONG cFilterAttrs, const TS_ATTRID *paFilterAttrs) {
	return NS_TO_HRESULT(m_pDoc->RequestSupportedAttrs(dwFlags, cFilterAttrs, paFilterAttrs));
}

STDMETHODIMP CGessoTextStore::RequestAttrsAtPosition(LONG acpPos, ULONG cFilterAttrs, const TS_ATTRID *paFilterAttrs, DWORD dwFlags) {
	return NS_TO_HRESULT(m_pDoc->RequestAttrsAtPosition(acpPos, cFilterAttrs, paFilterAttrs, dwFlags));
}

STDMETHODIMP CGessoTextStore::RequestAttrsTransitioningAtPosition(LONG acpPos, ULONG cFilterAttrs, const TS_ATTRID *paFilterAttrs, DWORD dwFlags) {
	return NS_TO_HRESULT(m_pDoc->RequestAttrsTransitioningAtPosition(acpPos, cFilterAttrs, paFilterAttrs, dwFlags));
}

STDMETHODIMP CGessoTextStore::FindNextAttrTransition(LONG acpStart, LONG acpHalt, ULONG cFilterAttrs, const TS_ATTRID *paFilterAttrs, DWORD dwFlags, LONG *pacpNext, BOOL *pfFound, LONG *plFoundOffset) {
	return NS_TO_HRESULT(m_pDoc->FindNextAttrTransition(acpStart, acpHalt, cFilterAttrs, paFilterAttrs, dwFlags, CGessoOutLONG(pacpNext), CGessoOutBOOL(pfFound), CGessoOutLONG(plFoundOffset)));
}

STDMETHODIMP CGessoTextStore::RetrieveRequestedAttrs(ULONG ulCount, TS_ATTRVAL *paAttrVals, ULONG *pcFetched) {
	nsresult nr = NS_OK;
	CGessoAttrVal *pVals = new CGessoAttrVal[ulCount];
	iGessoAttrVal **ppVals = new iGessoAttrVal* [ulCount];
	if (pVals && ppVals) {
		for (ULONG i = 0; i < ulCount; ++i) {
			pVals[i].m_obj.m_ptr = &paAttrVals[i];
			memset(&paAttrVals[i], 0, sizeof(TS_ATTRVAL));
			ppVals[i] = pVals[i];
		}
		nr = m_pDoc->RetrieveRequestedAttrs(ulCount, ppVals, CGessoOutULONG(pcFetched));
	} else nr = NS_ERROR_OUT_OF_MEMORY;
	if (ppVals) delete [] ppVals;
	if (pVals) delete [] pVals;
	return NS_TO_HRESULT(nr);
}

STDMETHODIMP CGessoTextStore::GetEndACP(LONG *pacp) {
	if ((m_dwLock & TS_LF_READ) != TS_LF_READ) return TS_E_NOLOCK;
	return NS_TO_HRESULT(m_pDoc->GetEndACP(CGessoOutLONG(pacp)));
}

STDMETHODIMP CGessoTextStore::GetActiveView(TsViewCookie *pvcView) {
	return NS_TO_HRESULT(m_pDoc->GetActiveView(CGessoOutDWORD(pvcView)));
}

STDMETHODIMP CGessoTextStore::GetACPFromPoint(TsViewCookie vcView, const POINT *pt, DWORD dwFlags, LONG *pacp) {
	return NS_TO_HRESULT(m_pDoc->GetACPFromPoint(vcView, CGessoPoint(pt), dwFlags, CGessoOutLONG(pacp)));
}

STDMETHODIMP CGessoTextStore::GetTextExt(TsViewCookie vcView, LONG acpStart, LONG acpEnd, RECT *prc, BOOL *pfClipped) {
	if ((m_dwLock & TS_LF_READ) != TS_LF_READ) return TS_E_NOLOCK;
	return NS_TO_HRESULT(m_pDoc->GetTextExt(vcView, acpStart, acpEnd, CGessoRect(prc), CGessoOutBOOL(pfClipped)));
}

STDMETHODIMP CGessoTextStore::GetScreenExt(TsViewCookie vcView, RECT *prc) {
	return NS_TO_HRESULT(m_pDoc->GetScreenExt(vcView, CGessoRect(prc)));
}

STDMETHODIMP CGessoTextStore::GetWnd(TsViewCookie vcView, HWND *phwnd) {
	nsIAccessNodePtr pNode; nsIAccessibleDocumentPtr pDoc;
	if (NS_FAILED(m_pDoc->GetAccessNode(vcView, getter_AddRefs(pNode))) ||
		NS_FAILED(pNode->GetAccessibleDocument(getter_AddRefs(pDoc))) ||
		(!pDoc) ||
		NS_FAILED(pDoc->GetWindowHandle(reinterpret_cast<void**>(phwnd))))
		*phwnd = NULL;
	return S_OK;
	/**/
}

STDMETHODIMP CGessoTextStore::InsertTextAtSelection(DWORD dwFlags, const WCHAR *pchText, ULONG cch, LONG *pacpStart, LONG *pacpEnd, TS_TEXTCHANGE *pChange) {
	if ((m_dwLock & TS_LF_READWRITE) != TS_LF_READWRITE) return TS_E_NOLOCK;
	return NS_TO_HRESULT(m_pDoc->InsertTextAtSelection(dwFlags, nsDependentString(pchText, cch), 
		CGessoOutLONG(pacpStart), CGessoOutLONG(pacpEnd), CGessoTextChange(pChange)));
}

STDMETHODIMP CGessoTextStore::InsertEmbeddedAtSelection(DWORD /*dwFlags*/, IDataObject * /*pDataObject*/, LONG * /*pacpStart*/, LONG * /*pacpEnd*/, TS_TEXTCHANGE * /*pChange*/) {
	return E_NOTIMPL;
}


NS_IMETHODIMP CGessoTextStore::RemoveDocument()
{
	m_pDoc = NULL;
	// destroy all editing contexts (including primary)
	HRESULT hr = H_TO_NSRESULT(m_pDocMgr->Pop(TF_POPF_ALL));
	m_pDocMgr = NULL;
	return hr;
}

NS_IMETHODIMP CGessoTextStore::Focus()
{
	return H_TO_NSRESULT(CGesso::GetThreadMgr()->SetFocus(m_pDocMgr));
}

NS_IMETHODIMP CGessoTextStore::Unfocus()
{
	CGesso::GetThreadMgr()->SetFocus(NULL);
    return NS_OK;
}

/* void onAttrsChange (in LONG acpStart, in LONG acpEnd, in ULONG cAttrs, [array, size_is (cAttrs), const] in iGessoAttrID paAttrs); */
NS_IMETHODIMP CGessoTextStore::OnAttrsChange(LONG acpStart, LONG acpEnd, ULONG cAttrs, const iGessoAttrID *paAttrs)
{
	if (m_pSink && m_dwSinkMask & TS_AS_ATTR_CHANGE) 
		return H_TO_NSRESULT(m_pSink->OnAttrsChange(acpStart, acpEnd, cAttrs, paAttrs));
    return NS_OK;
}

/* void onEndEditTransaction (); */
NS_IMETHODIMP CGessoTextStore::OnEndEditTransaction()
{
	if (m_pSink) return H_TO_NSRESULT(m_pSink->OnEndEditTransaction());
	return NS_OK;
}

/* void onLayoutChange (in LONG lcode, in DWORD vcView); */
NS_IMETHODIMP CGessoTextStore::OnLayoutChange(LONG lcode, DWORD vcView)
{
	if (m_pSink && m_dwSinkMask & TS_AS_LAYOUT_CHANGE) 
		return H_TO_NSRESULT(m_pSink->OnLayoutChange(static_cast<TsLayoutCode>(lcode), vcView));
	return NS_OK;
}

/* void onSelectionChange (); */
NS_IMETHODIMP CGessoTextStore::OnSelectionChange()
{
	if (m_pSink && m_dwSinkMask & TS_AS_SEL_CHANGE)
		return H_TO_NSRESULT(m_pSink->OnSelectionChange());
	return NS_OK;
}

/* void onStartEditTransaction (); */
NS_IMETHODIMP CGessoTextStore::OnStartEditTransaction()
{
	if (m_pSink) return H_TO_NSRESULT(m_pSink->OnStartEditTransaction());
	return NS_OK;
}

/* void onStatusChange (in DWORD dwFlags); */
NS_IMETHODIMP CGessoTextStore::OnStatusChange(DWORD dwFlags)
{
	if (m_pSink && m_dwSinkMask & TS_AS_STATUS_CHANGE)
		return H_TO_NSRESULT(m_pSink->OnStatusChange(dwFlags));
	return NS_OK;
}

/* void onTextChange (in DWORD dwFlags, in iGessoTextChange pChange); */
NS_IMETHODIMP CGessoTextStore::OnTextChange(DWORD dwFlags, iGessoTextChange *pChange)
{
	if (m_pSink && m_dwSinkMask & TS_AS_TEXT_CHANGE) {
		TS_TEXTCHANGE change = {0}; LONG ltemp;
		if (NS_SUCCEEDED(pChange->GetStart(&ltemp))) change.acpStart = ltemp;
		if (NS_SUCCEEDED(pChange->GetOldEnd(&ltemp))) change.acpOldEnd = ltemp;
		if (NS_SUCCEEDED(pChange->GetNewEnd(&ltemp))) change.acpNewEnd = ltemp;
		return H_TO_NSRESULT(m_pSink->OnTextChange(dwFlags, &change));
	}
	return NS_OK;
}

