/*$Id: ActiveRuleSet.h,v 1.3 2006/10/06 19:31:40 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#ifndef ACTIVERULESSET_H
#define ACTIVERULESSET_H

#include "Rules.h"

/*
  Holds all rules in use by a single HTTP request. 
  Every HTTP request uses its own copy of
  this structure, because rules can be removed from it, when it
  is certain they don't match request (ex. different URL then
  one specified by the rule)
*/
class ActiveRuleSet
{
	friend int urlContentMatched(void *id, int index, void* ctx);
	friend int headersContentMatched(void *id, int index, void* ctx);
	friend int bodyContentMatched(void *id, int index, void* ctx);
	
 private:	
	/*
	  Set of rules that don't have any *_content options or with all
	  *_content options matched.
	  Patterns specified with *_re options are checked only when
	  there is no *_content option left to match.
	 */
	RuleSet noContent;
	
	/*maps Rule* to set of all url_content options of this rule
	  that already matched*/
	map<Rule *, set<PatternMatchData *> > urlMatchedParts;
	/*same as above but for headers_content options*/
	map<Rule *, set<PatternMatchData *> > headersMatchedParts;
	/*same as above but for body_content options*/
	map<Rule *, set<PatternMatchData *> > bodyMatchedParts;
	
	/*Set of rules that have all url_content matched*/
	RuleSet urlContentCorrect;
	/*Set of rules that have all headers_content matched*/
	RuleSet headersContentCorrect;
	/*Set of rules that have all body_content matched*/
	RuleSet bodyContentCorrect;
	
	/*Set of rules that for sure not match*/
	set<const Rule *> noMatch;
	
	/*set of REs that matched body*/
	set<const RE *> bodyMatchedREs;
	
	/*Version of rules in this ActiveRuleSet, to check if these rules are
	  still valid*/
	unsigned int rulesVersion;

	/*
	  Rules object from which this ActiveRuleSet was initialized.
	 */
	Rules *rules;
	
	/* url of a HTTP request and headers of a response*/
	char *url, *headers;
	int headerslen, urllen;
	
	inline bool match(const RE &re_info, const char *s, int len);

	void checkUrlSet(set<Rule *> &rules, list<const Rule *> &result);
	
	void checkHeadersSet(set<Rule *> &rules, list<const Rule *> &result);

	void checkBodySet(set<Rule *> &rules, const char *body, int bodylen,
			  list<const Rule *> &result);
	
 public:
	ActiveRuleSet(Rules *initRules);
	~ActiveRuleSet();


	//void urlContentMatched(PatternMatchData *match);
	
	list<const Rule*> 
		checkURL(const char *url, int urllen);
	list<const Rule*> 
		checkHeaders(const char *headers, int headerslen);
	list<const Rule*> 
		checkBody(const char *body, int bodylen);

};

#endif
