/*$Id: Firekeeper.cpp,v 1.25 2007/02/13 00:11:14 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "nsCOMPtr.h"
#include "nsXPCOM.h"
#include "nsMemory.h"
#include "nsCRT.h"
#include "nsStringAPI.h"
#include "nsEmbedString.h"

#include "nsIHttpProtocolHandler.h"
#include "nsIComponentManager.h"

#include "nsIServiceManager.h"

#include "Firekeeper.h"
#include "fkHttpHandlerFactory.h"
#include "Error.h"
#include "Common.h"


static const nsIID kIComponentRegistrarIID = NS_ICOMPONENTREGISTRAR_IID;
static const nsCID kfkHttpHandlerCID  = FK_HTTPHANDLER_CID;
static const nsCID kfkHttpsHandlerCID = FK_HTTPSHANDLER_CID;

/*Loaded rules*/
Rules *rules = nsnull;

nsCOMPtr<fkIJudge> fkJudge = nsnull;

/*TODO: this should be configurable*/
const char *rulesOrder[]={"pass", "drop", "alert"};

Firekeeper::Firekeeper()
{
	this->httpHandler = nsnull;
	this->httpsHandler = nsnull;
	TRACE("constructor %08x", this);
}

Firekeeper::~Firekeeper()
{
	TRACE("destructor %08x", this);
	FK_ASSERT(false);
	//firekeeper = nsnull;
}

NS_IMPL_ISUPPORTS2(Firekeeper, fkIFirekeeper,
		   nsIObserver)
	
/*
  Return global Firekeeper object (singleton)
 */
Firekeeper*
Firekeeper::GetInstance()
{
	static Firekeeper *firekeeper;
	
	TRACE("");
	if (firekeeper == nsnull){
		firekeeper = new Firekeeper();
		if (!firekeeper)
			return nsnull;
		
		/*don't destroy firekeeper*/
		NS_ADDREF(firekeeper);	
			
		nsresult rv = firekeeper->init();
		if (NS_FAILED(rv)){
			//TODO: delete firekeeper; ...
			return nsnull;
		}
	}
	
	NS_IF_ADDREF(firekeeper);
	return firekeeper;
}
 

/*
  Init Firekeeper. Replace oryginal Mozilla's HTTP and HTTPS handlers
  with new ones, able to trace and check traffic.
 */
NS_IMETHODIMP 
Firekeeper::init()
{
	TRACE("Firekeeper init() called");
	rules = nsnull;	
	rules = new Rules();
	if (!rules || rules->init() < 0)
		return NS_ERROR_OUT_OF_MEMORY;
	
	nsCOMPtr<nsIServiceManager> servman; 
	nsresult rv = NS_GetServiceManager(getter_AddRefs(servman)); 
	if (NS_FAILED(rv)){
		TRACE("Can't get ServiceManager %08x", rv);
		return rv;
	}

	nsCOMPtr<nsIComponentManager> compmgr; 
	rv = NS_GetComponentManager(getter_AddRefs(compmgr)); 
	if (NS_FAILED(rv)){
		TRACE("Can't get ComponentManager");
		return rv;
	}

	nsCOMPtr<nsIComponentRegistrar> compreg;	
	rv = compmgr->QueryInterface(kIComponentRegistrarIID, 
				     getter_AddRefs(compreg));
	if (NS_FAILED(rv)){
		TRACE("Can't get ComponentRegistrar");
		return rv;
	}
	
	/*register new HTTP handler*/
	nsCOMPtr<nsIHttpProtocolHandler> oldHandler;
	rv = servman->
		GetServiceByContractID(HTTPHANDLER_CONTRACTID,
				       NS_GET_IID(nsIHttpProtocolHandler),
				       getter_AddRefs(oldHandler));
	
	if (NS_FAILED(rv))
		return rv;
	
	if (!oldHandler){
		TRACE("oldHandler == NULL");
		return NS_ERROR_FAILURE;
	}
	TRACE("got old http handler %08x", &*oldHandler);
	
	httpHandler = new fkHttpHandler();
	if (!httpHandler){
		TRACE("Can't create httpHandler");
		return NS_ERROR_OUT_OF_MEMORY;
	}
	
	rv = httpHandler->init(oldHandler);
	if (NS_FAILED(rv)){
		TRACE("Can't init http handler");
		return rv;
	}	
	
	NS_ADDREF(httpHandler);
	
	fkHttpHandlerFactory *httpHandlerFactory = 
		new fkHttpHandlerFactory(httpHandler);
	
	if (!httpHandlerFactory){
		TRACE("Can't create httpHandlerFactory");
		return NS_ERROR_OUT_OF_MEMORY;
	}
	NS_ADDREF(httpHandlerFactory);	
	
	
	
	TRACE("registering %s", HTTPHANDLER_CONTRACTID);
	rv = compreg->RegisterFactory(kfkHttpHandlerCID,
				      "fkHttpHandler",
				      HTTPHANDLER_CONTRACTID,
				      httpHandlerFactory);	
	if (NS_FAILED(rv)){
		TRACE("Can't register factory");
		return rv;
	}

	TRACE("Http handler replaced. Getting https handler: %s", HTTPSHANDLER_CONTRACTID);

	
	/*register new HTTPS handler*/
	nsCOMPtr<nsIHttpProtocolHandler> oldHttpsHandler;
	rv = servman->
		GetServiceByContractID(HTTPSHANDLER_CONTRACTID, 
				       NS_GET_IID(nsIHttpProtocolHandler),
				       getter_AddRefs(oldHttpsHandler));
	
	TRACE("got https handler %08x", &*oldHttpsHandler);
	if (NS_FAILED(rv)){
		return rv;
	}
	
	if (!oldHttpsHandler){
		TRACE("oldHttpsHandler == NULL");
		return NS_ERROR_FAILURE;
	}	
	
	TRACE("creating new fkHttpsHandler");
	httpsHandler = new fkHttpHandler();
	if (!httpsHandler){
		TRACE("Can't create httpsHandler");
		return NS_ERROR_OUT_OF_MEMORY;
	}
	TRACE("initializing HttpsHandler");
	rv = httpsHandler->init(oldHttpsHandler);
	if (NS_FAILED(rv)){
		TRACE("Can't init https handler");
		return rv;
	}
	
	NS_ADDREF(httpsHandler);
	
	fkHttpHandlerFactory *httpsHandlerFactory =
		new fkHttpHandlerFactory(httpsHandler);
	if (!httpsHandlerFactory){
		TRACE("Can't create httpsHandlerFactory");
		return NS_ERROR_OUT_OF_MEMORY;
	}
	NS_ADDREF(httpsHandlerFactory);
	
	rv = compreg->RegisterFactory(kfkHttpsHandlerCID,
				      "fkHttpsHandler",
				      HTTPSHANDLER_CONTRACTID,
				      httpsHandlerFactory);
	if (NS_FAILED(rv)){
		TRACE("Can't register factory");
		return rv;
	}

	TRACE("https handler replaced");
	return NS_OK;
}

//-----------------------------------------------------------------------------
// Firekeeper::fkIFirekeeper
//-----------------------------------------------------------------------------


NS_IMETHODIMP 
Firekeeper::AddRule(const char *rule, char **errmsg)
{	
	char *error = nsnull;
	TRACE("");
	if (!errmsg)
		return NS_ERROR_NULL_POINTER;
	
	/*TODO: don't create parser here*/
	RulesParser p;
	
	if (p.init(&error) < 0){
		err_msg(error);
		return NS_ERROR_FAILURE;
	}
	Rule *newRule =  p.parseRule(rule, &error);
	
	if (error){
		TRACE("error %s", error);
		*errmsg = nsCRT::strdup(error);	
		if (!*errmsg)
			return NS_ERROR_OUT_OF_MEMORY;

		/*Can't return NS_ERROR_FAILURE here, it would
		  make errmsg variable inaccessible in javascript*/
		return NS_OK;
	}

	rules->addRule(newRule);	
	
	return NS_OK;
}

NS_IMETHODIMP 
Firekeeper::AddRules(const char *rulesbuf, char **errmsg)
{
	char *error;
	TRACE("");
	if (!errmsg)
		return NS_ERROR_NULL_POINTER;
	
	RulesParser p;
	if (p.init(&error)){
		err_msg(error);
		return NS_ERROR_FAILURE;
	}
	p.parseRules(rules, rulesbuf, &error);
	
	if (error){
		TRACE("error %s", error);
		*errmsg = nsCRT::strdup(error);	
		if (!*errmsg)
			return NS_ERROR_OUT_OF_MEMORY;
		
		/*even when there were parsing errors return NS_OK*/
	}	
	
	return NS_OK;
}

NS_IMETHODIMP 
Firekeeper::SetJudge(fkIJudge *judge)
{
	TRACE("");
	fkJudge = judge;
	return NS_OK;
}

NS_IMETHODIMP 
Firekeeper::Enable(PRBool tracingEnabled)
{
	TRACE("");
	if (!httpHandler){
		TRACE("null httpHandler");
		return NS_ERROR_FAILURE;
	}
	httpHandler->enableTracing(tracingEnabled);
	return NS_OK;
}

NS_IMETHODIMP 
Firekeeper::IsEnabled(PRBool *tracingEnabled)
{
	TRACE("");
	if (!tracingEnabled)
		return NS_ERROR_NULL_POINTER;
	if (!httpHandler)
		return NS_ERROR_FAILURE;
	*tracingEnabled = httpHandler->isTracingEnabled();
	return NS_OK;
}

NS_IMETHODIMP 
Firekeeper::ResetRules()
{
	TRACE("");
	if (rules == nsnull)
		return NS_ERROR_FAILURE;
	
	rules->resetRules();
	TRACE("done");
	return NS_OK;
}


//-----------------------------------------------------------------------------
// Firekeeper::nsIObserver
//-----------------------------------------------------------------------------
NS_IMETHODIMP 
Firekeeper::Observe(nsISupports *subject, const char* topic,
		    const PRUnichar* data) 
{
	TRACE("");
	if (!topic || strncmp(topic, "xpcom-startup", 13) != 0){
		return NS_OK;	
	}
	return NS_OK;
}
