/*$Id: fkStreamListener.cpp,v 1.28 2007/09/24 20:06:48 jwrobel Exp $*/

/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006,2007 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "fkStreamListener.h"
#include "nsIServiceManager.h"
#include "nsIURI.h"

#include "nsXPCOM.h"
#include "nsMemory.h"
#include "nsStringAPI.h"
#include "nsEmbedString.h"

#include "Firekeeper.h"
#include "fkRule.h"
#include "fkSentence.h"
#include "Error.h"
#include "Common.h"
#include "nsIByteArrayInputStream.h"

/*
  Firekeeper divides response body into fragments of random size.
  Each fragment is tested separately and passed to a browser if it
  doesn't break any rule (this is also how Snort handles TCP stream
  reassembly). Drawback of this solution is that it isn't 100%
  accurate. But because fragments have a random size, it is impossible for
  an attacker to guess where is a border of a fragment and to format a
  response in such a way that suspicious data is always spread between two
  fragments and undetected.
  Following improvement further increases accuracy: part
  of a data from the end of a fragment (also of a random size) is
  appended to the beginning of a next fragment.
  It is possible to bypass this system only for attacks which detection
  requires processing of a large amount of data. But essentially
  attacks have short fingerprints and can be easily detected
  with this approach.
*/
/*Minimum size of a body fragment that is cached and checked*/
#define FLUSH_POINT_MIN 2000
/*Maximum size of a body fragment that is cached and checked*/
#define FLUSH_POINT_MAX 4000

/*Minimum number of bytes that are appended from the end of a previous
  fragment to the beginning of a next one*/
#define NO_FLUSH_POINT_MIN 100
#define NO_FLUSH_POINT_MAX 200

#define REQUEST request

NS_IMPL_ISUPPORTS4(fkStreamListener, nsIStreamListener,
		   nsIRequestObserver,
		   nsIHttpHeaderVisitor,
		   fkIStreamListenerProxy);

/*
  fkStreamListener constructor
  @origListener
          Original listener to which not dangerous data is passed
  @channel
          fkHttpChannel that created this StreamListener
  @connRules
          ActiveRuleSet used to check data received via thia stream
 */
fkStreamListener::fkStreamListener(nsIStreamListener *origListener, 
				   fkHttpChannel *channel,
				   fkISentence *sentence,
				   ActiveRuleSet *connRules):
	body(nsnull), headers(nsnull), url(nsnull), body_len(0),
	headers_len(0), body_max(BODY_INIT_SIZE),
	headers_max(HEADERS_INIT_SIZE), offset(0), debugNoHeaders(true),
	flush_point(0), no_flush_point(0)
{
	TRACE("constructor %08x origListener = %08x",
	      this, origListener);

	this->origListener = origListener;
	this->sentence = sentence;
	this->connRules = connRules;
	this->channel = channel;
	NS_ADDREF(this->channel);	

	drawFlushPoint();
	drawNoFlushPoint();
}

NS_IMETHODIMP 
fkStreamListener::Init()
{
	body = (char *)nsMemory::Alloc(body_max);
	if (body == nsnull){
		body_max = headers_max = 0;
		return NS_ERROR_OUT_OF_MEMORY;
	}

	headers = (char *)nsMemory::Alloc(headers_max);
	if (headers == nsnull){
		body_max = headers_max = 0;
		return NS_ERROR_OUT_OF_MEMORY;
	}
	
	return NS_OK;
}

fkStreamListener::~fkStreamListener()
{
	TRACE("destructor %08x", this);
	NS_RELEASE(this->channel);

	if (body != nsnull)
		nsMemory::Free(body);
	if (headers != nsnull)
		nsMemory::Free(headers);
}

/*
  Draw flush_point value between FLUSH_POINT_MIN and FLUSH_POINT_MAX
 */
void fkStreamListener::drawFlushPoint()
{
	srand(flush_point + time(0) + (int)this);
	flush_point = FLUSH_POINT_MIN +
		rand() % (FLUSH_POINT_MAX - FLUSH_POINT_MIN);
	TRACE("new flush point %d", flush_point);
}

/*
  Draw no_flush_point value between NO_FLUSH_POINT_MIN and NO_FLUSH_POINT_MAX
 */
void fkStreamListener::drawNoFlushPoint()
{
	srand(no_flush_point + time(0) + (int)this);
	no_flush_point = NO_FLUSH_POINT_MIN +
		rand() % (NO_FLUSH_POINT_MAX - NO_FLUSH_POINT_MIN);

	TRACE("new no flush point %d", no_flush_point);
}

/*
  Get URL from which this listener receives data.
 */
const char*
fkStreamListener::getURL()
{
	if (!url.Length()){
		nsCOMPtr<nsIURI> uri;
		channel->GetURI(getter_AddRefs(uri));
		uri->GetSpec(url);
	}
	return url.get();
}

/*
  Append string to headers buffer
 */
NS_IMETHODIMP
fkStreamListener::headers_append(const char *src, PRUint32 len)
{
	if (headers == nsnull)
		return NS_ERROR_OUT_OF_MEMORY;

	if (expandBuf(&headers, len, headers_len, headers_max) != NS_OK)
		return NS_ERROR_OUT_OF_MEMORY;

	memcpy(headers + headers_len, src, len);
	headers_len += len;
	return NS_OK;
}

/*
  Expand buffer if needed.
  @buf
      buffer
  @len
      number of needed bytes
  @end
      number of bytes currently in use
  @max_size
      size of a buffer
 */
NS_IMETHODIMP
fkStreamListener::expandBuf(char **buf, PRUint32 len, PRUint32 end, PRUint32 &max_size)
{
	while(max_size  - end < len){
		max_size *= 2;
		char *tmp = (char *)nsMemory::Realloc(*buf, max_size);
		if (tmp == nsnull){
			max_size /= 2;
			return NS_ERROR_OUT_OF_MEMORY;
		}
		*buf = tmp;
	}
	TRACE("%08x buffer expanded to %d body_max = %d",
	      this, max_size, body_max);
	return NS_OK;
}


NS_IMETHODIMP
fkStreamListener::checkBody()
{
	list<const Rule *> match = connRules->checkBody(body, body_len);
	list<const Rule *>::iterator it;
	for(it = match.begin(); it != match.end(); ++it){
		TRACE("calling judge %08x hmm", &*fkJudge);
		fkRule *rule_raw = new fkRule();
		if (!rule_raw){
			return NS_ERROR_OUT_OF_MEMORY;
		}
		if (NS_FAILED(rule_raw->Init(*it))){
			delete rule_raw;
			return NS_ERROR_OUT_OF_MEMORY;
		}
		nsCOMPtr<fkIRule> rule = do_QueryInterface(rule_raw);
		
		fkResponseDetails *details_raw = new fkResponseDetails(headers, headers_len,
								       body, body_len, offset);
		if (!details_raw){
			return NS_ERROR_OUT_OF_MEMORY;
		}
		nsCOMPtr<fkIResponseDetails> details = do_QueryInterface(details_raw);

		nsresult rv = fkJudge->Judge(getURL(), rule, sentence, details);
		//FK_ASSERT(!NS_FAILED(rv));
		
		int action;
		sentence->GetAction(&action);
		TRACE("action = %d", action);
		if (action == sentence->BLOCK){
			TRACE("aborting request");
			channel->Cancel(NS_ERROR_ABORT);
			return NS_ERROR_ABORT;
		}
		else if (action == sentence->DONT_AUDIT){
			TRACE("passing request");
			no_flush_point = 0;
			break;
		}
	}
	return NS_OK;
}


/*
  Check body fragment and pass it to the original listener if it is ok.
  @buffer
         body fragment to check
 */
NS_IMETHODIMP
fkStreamListener::do_OnDataAvailable(nsIRequest *request, nsISupports *aContext)
{
	nsresult rv;
	TRACE("calling checkBody len =%d", body_len);
	rv = checkBody();
	if (NS_FAILED(rv)){
		TRACE("check body failed");
		return rv;
	}
	TRACE("check body ok");

	nsCOMPtr<nsIByteArrayInputStream> convertedStreamSup;

	int data_len = body_len - no_flush_point;
	char *lBuf = (char *) nsMemory::Alloc(data_len);
	if (!lBuf)
		return NS_ERROR_OUT_OF_MEMORY;
	memcpy(lBuf, body, data_len);

	rv = NS_NewByteArrayInputStream(getter_AddRefs(convertedStreamSup),
					lBuf, data_len);
	if (NS_FAILED(rv))
		return rv;

	nsCOMPtr<nsIInputStream> convertedStream =
		do_QueryInterface(convertedStreamSup, &rv);
	if (NS_FAILED(rv))
		return rv;

	if (no_flush_point){
		/*Copy no_flush_point bytes from the end of previous body fragment
		  to the beginning of a new one*/
		memmove(body, body + data_len, no_flush_point);
		body_len = no_flush_point;
	}else{
		body_len = 0;
	}

	rv = origListener->OnDataAvailable(tryGetRequestProxy(request), aContext,
					   convertedStream,
					   offset, data_len);
	offset += data_len;;
	return rv;
}

/*
    Chack last fragment of a response body and pass it to the oryginal listener
    if it is ok. Free body buffer.
 */
NS_IMETHODIMP
fkStreamListener::do_LastOnDataAvailable(nsIRequest *request, nsISupports* acontext)
{
	nsresult rv;
	FK_ASSERT(body != nsnull);
	TRACE("calling checkBody");

	rv = checkBody();
	if (NS_FAILED(rv))
		return rv;

	nsCOMPtr<nsIByteArrayInputStream> convertedStreamSup;

	/*this stream becomes owner of a body buffer*/
	rv = NS_NewByteArrayInputStream(getter_AddRefs(convertedStreamSup),
					body, body_max);
	if (NS_FAILED(rv))
		return rv;

	TRACE("%08x", body);

	body = nsnull;
	nsCOMPtr<nsIInputStream> convertedStream =
		do_QueryInterface(convertedStreamSup, &rv);
	if (NS_FAILED(rv))
		return rv;

	TRACE("OK offset = %d len %d", offset, body_len);
	TRACE("%08x %08x %08x %08x", &*convertedStreamSup, &*convertedStream, acontext, request);

	rv = origListener->OnDataAvailable(tryGetRequestProxy(request), acontext,
					   convertedStream, offset,
					   body_len);
	body_len = 0;
	return rv;
}

//-----------------------------------------------------------------------------
// fkStreamListener::fkIStreamListenerProxy
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkStreamListener::GetOrigStreamListener(nsIStreamListener **aStreamListener)
{
	TRACE("fkStreamListener::fkIStreamListenerProxy");
	NS_ENSURE_ARG_POINTER(aStreamListener);
	FK_ASSERT(origListener);

	NS_IF_ADDREF(*aStreamListener = origListener);
	return NS_OK;
}

//-----------------------------------------------------------------------------
// fkStreamListener::nsIStreamListener
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkStreamListener::OnDataAvailable(nsIRequest *request, nsISupports *ctxt,
				  nsIInputStream *input,
				  PRUint32 off, PRUint32 count)
{
	TRACE("got data count = %d offset = %d request = %08x origchannel = %08x ctx = %08x",
	      count, off, request, &*channel, ctxt);
	FK_ASSERT(debugNoHeaders == false);

	int action;
	nsresult rv;

	sentence->GetAction(&action);

	/*Whitelisted. Don't check it*/
	if (action == sentence->DONT_AUDIT){
		TRACE("whitelisted");
		return origListener->OnDataAvailable(tryGetRequestProxy(request), ctxt, input,
						     off, count);
	}
	FK_ASSERT(action != sentence->BLOCK);
	
	if (expandBuf(&body, count, body_len, body_max) != NS_OK){
		TRACE("Out of memory, can't expand buffer");
		return NS_ERROR_OUT_OF_MEMORY;
	}

	PRUint32 res;
	input->Read(body + body_len, count, &res);
	FK_ASSERT(res == count);
	body_len += count;
	
	/*
	  Not whole body fragment is filled.
	 */
	if (body_len < flush_point){
		TRACE("fragment not filled");
		return NS_OK;
	}

	/*Check current body fragment and pass it to the original listener*/
	rv = do_OnDataAvailable(request, ctxt);
	if (NS_FAILED(rv)){
		TRACE("do_OnDataAvailable FAILED");
		return rv;
	}

	drawFlushPoint();
	drawNoFlushPoint();
	return rv;
}


//-----------------------------------------------------------------------------
// fkStreamListener::nsIRequestObserver
//-----------------------------------------------------------------------------

/*request = 095b6fc4 origchannel = 095a4d08 proxy 095b73c8
  Request started. Get and check headers.
 */
NS_IMETHODIMP
fkStreamListener::OnStartRequest(nsIRequest *request, nsISupports* acontext)
{
	TRACE("Request started %08x request = %08x origchannel = %08x ctx = %08x ", this,
	      request, channel, acontext);
	nsresult rv = channel->VisitResponseHeaders(this);
	if (NS_FAILED(rv)){
		TRACE("Visit response headers failed");
		debugNoHeaders = true;
		return rv;
	}
	debugNoHeaders = false;

	/*(rv = headers_append("", 1);
		 if (NS_FAILED(rv)){
		 return rv;
			 }
		 TRACE("headers:\n %s", headers);*/

	list<const Rule *> match = connRules->checkHeaders(headers,
							   headers_len);
	list<const Rule *>::iterator it;
	for(it = match.begin(); it != match.end(); ++it){
		TRACE("calling judge");
		fkRule *rule_raw = new fkRule();
		if (!rule_raw){
			return NS_ERROR_OUT_OF_MEMORY;
		}
		if (NS_FAILED(rv = rule_raw->Init(*it))){
			delete rule_raw;
			return NS_ERROR_OUT_OF_MEMORY;
		}
		nsCOMPtr<fkIRule>
			rule(do_QueryInterface(rule_raw));

		fkResponseDetails *details_raw = new fkResponseDetails(headers, headers_len,
								       nsnull, 0, 0);
		if (!details_raw){
			return NS_ERROR_OUT_OF_MEMORY;
		}
		nsCOMPtr<fkIResponseDetails> details = do_QueryInterface(details_raw);
		
		fkJudge->Judge(getURL(), rule, sentence, details_raw);
		int action;
		sentence->GetAction(&action);
		TRACE("action = %d", action);
		if (action == sentence->BLOCK){
			TRACE("aborting request");
			return NS_ERROR_ABORT;
		}
		else if (action == sentence->DONT_AUDIT){
			TRACE("passing request");
			break;
		}
	}

	TRACE("call");
	rv = origListener->OnStartRequest(tryGetRequestProxy(request), acontext);
	TRACE("returned");
	return rv;
}

/*
  Request stopped. Check what is left in body buffer
*/
NS_IMETHODIMP
fkStreamListener::OnStopRequest(nsIRequest *request, nsISupports *acontext,
				nsresult astatus)
{
	TRACE("Request stopped %08x status = %d request = %08x origchannel = %08x ctx = %08x",
	      this, astatus, request, channel, acontext);
	nsresult rv;
	int action;
	sentence->GetAction(&action);

	if (body_len && action != sentence->DONT_AUDIT &&
	    action != sentence->BLOCK){
		TRACE("passing remaining body fragment");
		no_flush_point = 0;
		rv = do_LastOnDataAvailable(request, acontext);
		if (NS_FAILED(rv)){
			TRACE("do_LastOnDataAvailable failed %d", rv);
			astatus = rv;
		}
	}

	return origListener->OnStopRequest(tryGetRequestProxy(request), acontext, astatus);

}


//-----------------------------------------------------------------------------
// fkStreamListener::nsIHttpHeaderVisitor
//-----------------------------------------------------------------------------

/*Get single header and append it to headers buffer*/
NS_IMETHODIMP
fkStreamListener::VisitHeader(const nsACString &header,
			      const nsACString &value)
{

	nsresult rv;
	PRBool null_ter;
	const char *h, *v;
	int hlen, vlen;
	hlen = NS_CStringGetData(header, &h, &null_ter);
	FK_ASSERT(null_ter == PR_TRUE);

	vlen = NS_CStringGetData(value, &v, &null_ter);
	FK_ASSERT(null_ter == PR_TRUE);

	TRACE("h = %s v = %s", h, v);
	rv = headers_append(h, hlen);
	if (NS_FAILED(rv))
		return rv;
	rv = headers_append(": ", 2);
	if (NS_FAILED(rv))
		return rv;

	rv = headers_append(v, vlen);
	if (NS_FAILED(rv))
		return rv;

	rv = headers_append("\n", 1);
	if (NS_FAILED(rv))
		return rv;
	return NS_OK;
}
