/****************************************************************************
**  DTBServer (aka MUSE - Melhorn's Universal Server Emulator)
**  Copyright (c) 2003  Charles Melhorn
**
**  This file is part of DerTandemBrowser.
**
**  DerTandemBrowser is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  DerTandemBrowser is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
**  Or see the GNU web site at: http://www.gnu.org
**
****************************************************************************/ 

#include <stdio.h>
#include <stdlib.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <time.h>

#define DEFAULT_PORT 4444

#define TRUE  1
#define FALSE 0

#define MAXMSG       512
#define MAX_CONNECTS 8

#define M_PASSIVE     0  // Operating modes
#define M_INTERACTIVE 1
#define M_ECHO        2
#define M_PROXY       3
#define M_TEE         4

char * gModeStr_arr[] = {"PASSIVE", "INTERACTIVE", "ECHO", "PROXY", "TEE"}; 

char *       gVersion = "version 0.3";
uint16_t     gPort = DEFAULT_PORT;
int          gMode = M_PROXY;
int          gWriteDelay = 5;


/* flags */
int          bVerbose = TRUE;   



int make_socket (uint16_t port)
{
  int sock;
  struct sockaddr_in name;

  /* Create the socket. */
  sock = socket (PF_INET, SOCK_STREAM, 0);
  if (sock < 0)
    {
      perror ("socket");
      exit (EXIT_FAILURE);
    }

  /* Give the socket a name. */
  name.sin_family = AF_INET;
  name.sin_port = htons (port);
  name.sin_addr.s_addr = htonl (INADDR_ANY);
  if (bind (sock, (struct sockaddr *) &name, sizeof (name)) < 0)
    {
      perror ("bind");
      exit (EXIT_FAILURE);
    }

  return sock;
}


int read_from_client (int filedes, char* buffer, int buffsize)
{
  int nbytes;
  int j;


  nbytes = read (filedes, buffer, buffsize);
  buffer[nbytes] = '\0';

  if (nbytes < 0)
    {
      /* Read error. */
      perror ("read");
      return -1;
    }
  else if (nbytes == 0)
    {
      /* End-of-file. */
      printf("Read end-of-file\n");
      return -1;
    }
  else
    {
      /* Data read. */
      printf ("Server: got message of %d bytes: %s\n", nbytes, buffer);
      return 0;
    }
}


void write_to_client (int filedes, char* message, int msg_length)
{
  int nbytes;

  nbytes = write (filedes, message, msg_length);
  if (nbytes < 0)
    {
      perror ("write");
//      exit (EXIT_FAILURE);
    }
  else if (bVerbose) printf("Server: wrote %d bytes\n", nbytes);
}


int ProcessArguments(int m_argc, char* m_argv[])
{
  int a;
  int portcount = 0;
  int rv = 0;        // normal return value

  for (a = 1; a < m_argc; a++)
   {
     if (m_argv[a][0] != '-')
      {
        if (++portcount > 1)
         {
           printf("usage: dtbserver [port] [[-option]...]");
           rv = 1;
           break;
         }
        else
           gPort = atoi(m_argv[a]);
      }
     else // argument must be a flag
      {
        if (strcmp(m_argv[a], "-h") == 0)
         {
           printf("\n");
           printf("usage: dtbserver [port] [[-option]...]\n\n");
           printf("options:\n\n");
           printf("  -h: help\n");
           printf("  -v: version\n");
           printf("  -d: discreet, do not display received data on stdout\n");
           printf("  -q: quiet, no informational output sent to stdout\n");
           printf("  -e: echo received data back to sender\n");
           printf("  -p: proxy, data from first connection sent to all subsequent\n");
           printf("      connectors, data from any subsequent sent to first connector\n");
           printf("  -t: tee, data from any connector sent to all other connectors\n");
           printf("  -i: interactive, prompts for response after data is received\n");
           printf("  -w: write delay (usage: -wN, where N is delay in .1 msec increments)\n");
           printf("  -l: log received data to file (usage: -lfilename)\n");
           printf("\n");
           printf("Echo, proxy, tee, and interactive modes are mutually exclusive.\n");
           rv = 1;
           break;
         }
        else if (strcmp(m_argv[a], "-v") == 0)
         {
           printf("\n");
           printf("dtbserver %s\n", gVersion);
           rv = 1;
           break;
         }
        else if (strcmp(m_argv[a], "-q") == 0)
         {
           bVerbose = FALSE;
         }
        else if (strcmp(m_argv[a], "-e") == 0)
         {
           gMode = M_ECHO;
         }
        else if (strcmp(m_argv[a], "-d") == 0)
         {
           printf("\n");
           printf("option not implemented\n");
           rv = 1;
           break;
         }
        else if (strcmp(m_argv[a], "-p") == 0)
         {
           gMode = M_PROXY;
         }
        else if (strcmp(m_argv[a], "-t") == 0)
         {
           printf("\n");
           printf("option not implemented\n");
           rv = 1;
           break;
         }
        else if (strcmp(m_argv[a], "-i") == 0)
         {
           gMode = M_INTERACTIVE;
         }
        else if (strncmp(m_argv[a], "-w", 2) == 0)
         {
           gWriteDelay = atoi(&m_argv[a][2]);
           printf("write delay: %d", gWriteDelay);
         }
        else if (strncmp(m_argv[a], "-l", 2) == 0)
         {
           printf("\n");
           printf("option not implemented\n");
           rv = 1;
           break;
         }
        else
         {
           printf("\n");
           printf("usage: dtbserver [port] [[-option]...]\n\n");
           printf("options:\n\n");
           printf("  -h: help\n");
           printf("  -v: version\n");
           printf("  -d: discreet, do not display received data on stdout\n");
           printf("  -q: quiet, no informational output sent to stdout\n");
           printf("  -e: echo received data back to sender\n");
           printf("  -p: proxy, data from first connection sent to all subsequent\n");
           printf("      connectors, data from any subsequent sent to first connector\n");
           printf("  -t: tee, data from any connector sent to all other connectors\n");
           printf("  -i: interactive, prompts for response after data is received\n");
           printf("  -w: write delay (usage: -wN, where N is delay in .1 msec increments)\n");
           printf("  -l: log received data to file (usage: -lfilename)\n");
           printf("\n");
           printf("Echo, proxy, tee, and interactive modes are mutually exclusive.\n");
           rv = 1;
           break;
         }
      }
   } // end for loop

  return rv;
}


struct hostent *dtbhostinfo;
int            myaddress;
struct in_addr my_address;

main(int argc, char* argv[])
{
  int    sock;
  fd_set active_fd_set, read_fd_set;
  int    i, j, k;
  struct sockaddr_in clientname;
  size_t size;
  int    num_connections = 0;
  int    connection[MAX_CONNECTS];
  char   read_buffer[MAXMSG];
  char   response_msg[MAXMSG];

  struct timespec req_time, rem_time;


  if (argc >= 2)
    if (ProcessArguments(argc, argv) != 0)
      exit(0);

  if (bVerbose) printf("\n%s %s\nRunning in %s mode...\n",
                       argv[0], gVersion, gModeStr_arr[gMode]);   
  
  /* Create the socket and set it up to accept connections. */
  sock = make_socket (gPort);

  if (bVerbose) printf("Listening for connection on port %d ...\n", gPort);
  if (listen (sock, 1) < 0)
   {
     perror ("listen");
     exit (EXIT_FAILURE);
   }
      
  /* Initialize the set of active sockets. */
  FD_ZERO (&active_fd_set);
  FD_SET (sock, &active_fd_set);


  while (TRUE)
   {
     /* Block until input arrives on one or more active sockets. */
     read_fd_set = active_fd_set;
     if (select (FD_SETSIZE, &read_fd_set, NULL, NULL, NULL) < 0)
      {
        perror ("select");
        exit (EXIT_FAILURE);
      }
         
      /* Service all the sockets with input pending. */
      for (i = 0; i < FD_SETSIZE; ++i)
       if (FD_ISSET (i, &read_fd_set))
        {
         if (i == sock)
          {
             /* Connection request on original socket. */
             int new;
             size = sizeof (clientname);
             new = accept (sock,
                           (struct sockaddr *) &clientname,
                           &size);
             if (new < 0)
               {
                 perror ("accept");
                 exit (EXIT_FAILURE);
               }

             printf ("Server: connect from host %s, port %hd.\n",
                      inet_ntoa (clientname.sin_addr),
                      ntohs (clientname.sin_port));

             if (num_connections == MAX_CONNECTS)
              {
                printf("Maximum number of connections exceeded - closed connection");
                close (new);
              }
             else
              {
                connection[num_connections++] = new;
                FD_SET (new, &active_fd_set);
              }
          }
         else
          {
             /* Data arriving on an already-connected socket.    */
             /* If problem reading from socket, close connection */
             /* and remove from connections array. Otherwise,    */
             /* respond to message according to operating mode.  */

             if (read_from_client (i, read_buffer, MAXMSG) < 0)
               {
                 if (bVerbose) printf("Read Error: closing socket\n");
                 close (i);
                 FD_CLR (i, &active_fd_set);
                 
                 j = 0;
                 while (i != connection[j])
                   j++;

                 for (k = j; k < num_connections - 1; k++)
                   connection[k] = connection[k+1];

                 num_connections--; 
               }
             else
              {
                if (gMode == M_INTERACTIVE)
                 {
                   printf("? "); 
                   gets(response_msg);
                   printf("sent response: %s\n", response_msg);
                   write_to_client (i, response_msg, strlen(response_msg) + 1);
                 }
                else if (gMode == M_ECHO)
                 {
                   write_to_client (i, read_buffer, strlen(read_buffer) + 1);
                 }
                else if (gMode == M_PROXY)
                 {
                   if (i == connection[0])
                     for (j = 1; j < num_connections; j++)
                        write_to_client (connection[j], read_buffer, strlen(read_buffer) + 1);
                   else
                     write_to_client (connection[0], read_buffer, strlen(read_buffer) + 1);

                   // ??? delay to rate limit writes to clients. This
                   // is inefficient and should be replaced by explicit
                   // command buffers and check of last_command_sent time
                   // for each connection.
                   req_time.tv_sec = 0;
                   req_time.tv_nsec = gWriteDelay * 100000; //  gWriteDelay * .1 msec
                   nanosleep(&req_time, &rem_time); 
                 }
              }
          }
        
        } // end if select item set


   } // end while loop

}
