/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System.IO;
using System.Collections.Generic;

namespace LocaleInpector
{

internal class PropertiesFile : NameValueFile
{

public PropertiesFile(string path) : base(path)
{
}

public override void LoadContent()
{
    m_content = this.LoadText();
}

public override Merge GetMerge()
{
    return new MergeProperties(this);
}

public override Dictionary<string, string> LoadHashtableLogged()
{
    Dictionary<string, string> table = this.LoadHashtableCore(true);
    return table;
}

public override Dictionary<string, string> LoadHashtable()
{
    Dictionary<string, string> table = this.LoadHashtableCore(false);
    return table;
}

public override Dictionary<string, string> LoadHashtableDecoded()
{
    Dictionary<string, string> table = this.LoadHashtableCore(false);
    return table;
}

private Dictionary<string, string> m_hashtableCache = null;
private Escape                     m_esc            = new Escape();

public string[] GetDecodingErrors()
{
    Dictionary<string, string> t = LoadHashtableCore(false);
    return m_esc.GetErrors();
}

private Dictionary<string, string> LoadHashtableCore(bool logError)
{
    if (logError) {
        return this.LoadHashtableCore2(logError);
    }

    if (m_hashtableCache == null) {
        m_hashtableCache = this.LoadHashtableCore2(logError);
    }
    return m_hashtableCache;
}

private Dictionary<string, string> LoadHashtableCore2(bool logError)
{
    string section = string.Empty;
    List<string> lines = Strings.SplitLines(m_content);

    List<PoEntry> arr = new List<PoEntry>();
    PoEntry previousEntry = null;

    for (int idx = 0; idx < lines.Count; idx++) {

        string[] keyValue = ParsePropertyLine(lines[idx]);
        bool isNameValue = keyValue.Length == 2;

        if (previousEntry != null) {
            // new comment, key=value or wrapped value
            if (isNameValue) {
                if (keyValue[0][0] == '#') {
                    previousEntry.AppendComment(keyValue[1]);
                    continue;
                }
                if (previousEntry.References == null) {
                    // new value
                    previousEntry.AppendReference(keyValue[0]);
                    this.AddKey(previousEntry, keyValue[1], logError);
                } else {
                    this.AddKey(previousEntry, lines[idx], logError);
                }

            } else {
                // wrapped value?
                if (previousEntry.References == null) {
                    // malformed file or blank line
                    continue;
                }
                this.AddKey(previousEntry, lines[idx], logError);
            }

        } else {
            if (isNameValue) {
                previousEntry = new PoEntry();
                arr.Add(previousEntry);
                if (keyValue[0][0] == '#') {
                    previousEntry.AppendComment(keyValue[1]);
                    continue;
                }
                previousEntry.AppendReference(keyValue[0]);
                this.AddKey(previousEntry, keyValue[1], logError);
            } else {
                // blank line or malformed file
                continue;
            }
        }

        if (isNameValue) {
            if (keyValue[1].EndsWith(@"\") == false) {
                previousEntry = null;
            }
        } else {
            if (keyValue[0].EndsWith(@"\") == false) {
                previousEntry = null;
            }
        }
    }


    Dictionary<string, string> table = new Dictionary<string, string>(arr.Count);
    for (int idx = 0; idx < arr.Count; idx++) {
        PoEntry poEntry = arr[idx];
        string id = poEntry.References;
        if (id == null) {
            // comment at the end of file?
            continue;
        }
        if (id.Length == 0) {
            throw new System.NotSupportedException();
        }

        if (m_isTranslated) {
            NameValueFile.AddToHashtable(table, id, poEntry.Translated, logError ? m_path : null);
        } else {
            NameValueFile.AddToHashtable(table, id, poEntry.Original,   logError ? m_path : null);
        }
    }
    return table;
}

private void AddKey(PoEntry po, string val, bool logError)
{
    if (val.EndsWith(@"\")) {
        val = val.Substring(0, val.Length - 1);
    }

    val = m_esc.UnEscape(logError ? po.References : null, val);

    if (m_isTranslated) {
        po.AppendTranslated(val);
    } else {
        po.AppendOriginal(val);
    }
}

public static string[] ParsePropertyLine(string line)
{
    line = line.TrimStart();

    if (line.Length == 0) {
        return new string[] { string.Empty };
    }


    if (line[0] == '#') {
        if (line.Length == 1) {
            return new string[] { "#", string.Empty };
        } else {
            return new string[] { "#", line.Substring(1) };
        }
    }

    string[] keyValue = line.Split(new char[] { '=' }, 2);
    if (keyValue.Length < 2) {
        return new string[] { line };
    }
    if (keyValue[0].Length == 0) {
        return new string[] { line };
    }

    keyValue[0] = keyValue[0].Trim();
    keyValue[1] = keyValue[1].TrimStart();
    return keyValue;
}

}//class
}//ns
