/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System.IO;
using System.Collections.Generic;

namespace LocaleInpector
{

internal class IniFile : NameValueFile
{

public IniFile(string path) : base(path)
{
}

protected override System.Text.Encoding FindEncoding()
{
    System.Text.Encoding enc = BPIO.UTF8;
    string fileName;
    try {
        //fileName = System.IO.Path.GetFileName(m_path.Replace('|', System.IO.Path.DirectorySeparatorChar));
        fileName = System.IO.Path.GetFileName(m_path);
    } catch {
        return enc;
    }
    string mk = m_path.Substring(0, m_path.Length - fileName.Length) + "charset.mk";
    if (BPIO.IsFile(mk) == false) {
        return enc;
    }

    PropertiesFile prop = new PropertiesFile(mk);
    Dictionary<string, string> charset = prop.LoadHashtable();
    if (charset.ContainsKey("WIN_INSTALLER_CHARSET")) {
        enc = null;
        try {
            string cs = charset["WIN_INSTALLER_CHARSET"];
            if (cs.Length > 1) {
                int cp = int.Parse(cs.Substring(2));
                enc = System.Text.Encoding.GetEncoding(cp);
            }
        } catch {
            enc = null;
        }
    }
    if (enc == null) {
        System.Console.WriteLine();
        System.Console.WriteLine(@"ERROR: Codepage not supported: ""{0}"". Reading as UTF-8.", charset["WIN_INSTALLER_CHARSET"]);
        System.Console.WriteLine("FILE:  " + m_path);
        enc = BPIO.UTF8;
    }

    return enc;
}

public override void LoadContent()
{
    m_content = this.LoadText();
}

// Returns: "General", "Module 1", "Module 2"


public List<string> LoadSections()
{
    List<string> keys = new List<string>();
    List<string> lines = this.LoadLines();
    for (int idx = 0; idx < lines.Count; idx++) {
        string[] keyValue = ParseIniLine(lines[idx]);
        if (keyValue != null) {
            if (keyValue.Length == 1) {
                if (keys.Contains(keyValue[0])) {
                    System.Console.WriteLine("OPTIONS ERROR: duplicated section {0}", keyValue[0]);
                    //return null;
                    keys.Clear();
                    return keys;
                }
                keys.Add(keyValue[0]);
            }
        }
    }

    return keys;
}

public override Merge GetMerge()
{
    return new MergeIni(this);
}

public override Dictionary<string, string> LoadHashtableLogged()
{
    return this.LoadHashtableCore(true);
}

public override Dictionary<string, string> LoadHashtable()
{
    return this.LoadHashtableCore(false);
}


private Dictionary<string, string> LoadHashtableCore(bool logError)
{
    string section = string.Empty;
    Dictionary<string, string> table = new Dictionary<string, string>();
    List<string> lines = this.LoadLines();
    for (int idx = 0; idx < lines.Count; idx++) {
        string[] keyValue = ParseIniLine(lines[idx]);
        if (keyValue != null) {
            if (keyValue.Length == 2) {
                NameValueFile.AddToHashtable(table, section + "->" + keyValue[0], keyValue[1], logError ? m_path : null);
            } else {
                section = keyValue[0];
            }
        }
    }

    return table;
}

public List<string> LoadLines()
{
    return Strings.SplitLines(m_content);
}

public static string[] ParseIniLine(string line)
{
    if (line == null) {
        return null;
    }

    line = line.TrimStart();

    if (line.Length == 0) {
        return null;
    }

    if (line[0] == ';') {
        return null;
    }

    if (line[0] == '[') {
        line = line.TrimEnd();
        if (line[line.Length - 1] == ']') {
            return new string[] { line.Substring(1, line.Length - 2) };
        } else {
            return null;
        }
    }

    string[] keyValue = line.Split(new char[] { '=' }, 2);
    if (keyValue.Length < 2) {
        return null;
    }

    keyValue[0] = keyValue[0].Trim();
    return keyValue;
}

}//class
}//ns
