/* xpm2wico - .xpm to Windows .ico format converter
 *
 * Copyright (C) 2001 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <X11/xpm.h>

#include "xpm2wico.h"

#define display_warning(msg) \
	fprintf (stderr, "Warning: %s", msg)
#define display_error(msg) \
	fprintf (stderr, "*** Error: %s\nPlease wait for the next" \
		 " version or send me a patch.", msg)

typedef struct _ColorMap ColorMap;
typedef struct _DrawData DrawData;
typedef struct _IconData IconData;
typedef void (*ColorConverter) (XpmColor *color, RGBQuad *wincolor);

struct _ColorMap
{
  DWord nbr_colors;
  RGBQuad *color_list;
};

struct _DrawData
{
  Byte *data;
  DWord width;
  DWord height;
};

struct _IconData
{
  IconDir *header;
  IconDirEntry *icon_dir_entry;
  BitmapInfoHeader *bm_ih;
  ColorMap *color_map;
  Byte *xor_mask;
  Byte *and_mask;
  DWord xor_size;
  DWord and_size;
};

DWord xpm_col_tbl_size;
char *xpm_col_codes;

const char *package = PACKAGE;


void
write_byte (FILE *file, Byte byte)
{
  fwrite (&byte, sizeof (Byte), 1, file);
}

void
write_word (FILE *file, Word word)
{
  fwrite (&word, sizeof (Word), 1, file);
}

void
write_dword (FILE *file, DWord dword)
{
  fwrite (&dword, sizeof (DWord), 1, file);
}

void
write_long (FILE *file, Long long_val)
{
  fwrite (&long_val, sizeof (Long), 1, file);
}

int
color_size (char *cstring)
{
  int len, size;

  len = strlen (cstring + 1);

  size = len / 3;

  return size;
}

int hex_value (char hexchar)
{
  int value;

  if (hexchar > '9')
    {
      if (hexchar > 'F')
	value = hexchar - 87;
      else
	value = hexchar - 55;
    }
  else
    value = hexchar - 48;

  return value;
}

Word bytes_per_pixel (Word ncolors)
{
  Word bypp;

  if (ncolors < 256)
    bypp = 1; /* 8 bpp */
  else
    bypp = 3; /* > 8 bpp */

  return bypp;
}

DWord
rounded_and (int nb)
{
  DWord r_and;

  r_and = nb + ((8 - (nb % 8)) & ~8);

  return r_and;
}

Word
color_len (char *color)
{
  Word len;

  len = strlen (color + 1) / 3;

  return len;
}

Byte
norm_color (Word raw_color, Word len)
{
  int count, factor;
  Byte color;

  if (len == 2)
    color = raw_color;
  else if (len == 1)
    color = raw_color * 16;
  else
    {
      factor = 1;

      for (count = 0; count < (len - 2) ; count++)
	factor *= 16;

      color = raw_color / factor;
    }

  return color;
}

Byte
interp_color (char *color_str, Word len)
{
  Byte color;
  Word raw_color;
  int multipl, pos;

  raw_color = 0;
  multipl = 1;
  pos = len - 1;

  while (pos > -1)
    {
      raw_color += hex_value(*(color_str + pos)) * multipl;
      multipl *= 16;
      pos--;
    }

  color = norm_color (raw_color, len);

  return color;
}

void
color_convert (XpmColor *xpm_color, RGBQuad *wincolor)
{
  Word clen;
  char *char_ptr;

  char_ptr = xpm_color->c_color + 1;
  clen = color_len (xpm_color->c_color);

  wincolor->rgbRed = interp_color (char_ptr, clen);
  char_ptr += clen;
  wincolor->rgbGreen = interp_color (char_ptr, clen);
  char_ptr += clen;
  wincolor->rgbBlue = interp_color (char_ptr, clen);
  wincolor->rgbReserved = 0;  
}

void
gray_convert (XpmColor *xpm_color, RGBQuad *wincolor)
{
  Word clen;
  Byte gray_value;

  clen = color_len (xpm_color->g_color);
  gray_value = interp_color (xpm_color->g_color + 1, clen);
  wincolor->rgbRed = gray_value;
  wincolor->rgbGreen = gray_value;
  wincolor->rgbBlue = gray_value;
  wincolor->rgbReserved = 0;
}

void
gray4_convert (XpmColor *xpm_color, RGBQuad *wincolor)
{
  display_error ("No 4-level grayscale color handling at this time.");
}

void
mono_convert (XpmColor *xpm_color, RGBQuad *wincolor)
{
  display_error ("No monochrome color handling at this time.");
}

void
sym_convert (XpmColor *xpm_color, RGBQuad *wincolor)
{
  display_error ("No symbolic color handling at this time.");
}

/* Not a real converter */
void
transp_convert (XpmColor *xpm_color, RGBQuad *wincolor)
{
  wincolor->rgbRed = 0;
  wincolor->rgbGreen = 0;
  wincolor->rgbBlue = 0;
  wincolor->rgbReserved = 255;
}

/* This function is certainly buggy. The problem is that I don't have
   any xpm spec underhand to tell me how those different color schemes
   are supposed to work. Everything is based on X11/xpm.h. */

ColorConverter
choose_converter (XpmColor *xpm_color)
{
  ColorConverter cconverter;
  char *cstring;

  cconverter = NULL;

  if (xpm_color->c_color) /* colored color */
    {
      cstring = xpm_color->c_color;
      cconverter = color_convert;
    }
  else if (xpm_color->g_color) /* grayscaled color */
    {
      cstring = xpm_color->g_color;
      cconverter = gray_convert;
    }
  else if (xpm_color->g4_color) /* 4 level grayscaled color */
    {
      cstring = xpm_color->g4_color;
      cconverter = gray4_convert;
    }
  else if (xpm_color->m_color) /* monochrome */
    {
      cstring = xpm_color->m_color;
      cconverter = mono_convert;
    }
  else if (xpm_color->symbolic) /* symbolic color (see rgb.txt) */
    {
      cstring = xpm_color->symbolic;
      cconverter = sym_convert;
    }
  else
    {
      display_error ("unconvertible color found\n");
      fprintf (stderr, "The culprit has \"%s\" as its symbol.\n",
	       xpm_color->string);
      cstring = NULL;
      cconverter = NULL;
    }

  /* Transparency */
  if (cstring)
    if (!strcmp (cstring, "None"))
      cconverter = transp_convert;

  return cconverter;
}

RGBQuad
convert_color (XpmColor *xpm_color)
{
  RGBQuad win_color;
  char *cstring;
  ColorConverter cconverter;

  /* This has to be redone each time convert_color is invoked. It is
     written nowhere that xpm files are using the same color scheme
     for every colors. */
  cconverter = choose_converter (xpm_color);

  if (cconverter)
    cconverter (xpm_color, &win_color);
  else
    {
      win_color.rgbRed = 0;
      win_color.rgbGreen = 0;
      win_color.rgbBlue = 0;
      win_color.rgbReserved = 0;
      display_warning ("unmanaged color, converted to black\n");
    }

  return win_color;
}

int
is_valid_xpm_image (XpmImage *icon)
{
  int ret_code;
  char *err_msg;

  if (icon->width != icon->height)
    {
      err_msg = "Image width or height are not equal.\n"
	        "Can't currently scale image dimensions.";
      ret_code = EXIT_FAILURE;
    }
  else if (icon->width != 16
	   && icon->width != 32
	   && icon->width != 64)
    {
      fprintf (stderr,
	       "This image's dimensions are %dx%d while they should"
	       " be either 16x16, 32x32 or 64x64.\n"
	       "I can't currently scale images so you will have"
	       " to do this manually and try again afterwards...\n",
	       icon->width, icon->height);
      err_msg = NULL;
      ret_code = EXIT_FAILURE;
    }
  else
    {
      err_msg = NULL;
      ret_code = EXIT_SUCCESS;
    }

  if (ret_code == EXIT_FAILURE && err_msg)
    display_error (err_msg);

  return ret_code;
}

IconDir *
new_icon_header ()
{
  IconDir *header;

  header = malloc (sizeof (IconDir));
  header->idReserved = 0;
  header->idType = 1;
  header->idCount = 1;

  return header;
}

IconDirEntry *
new_icon_dir_entry ()
{
  IconDirEntry *icon_dir_entry;

  icon_dir_entry = malloc (sizeof (IconDirEntry));
  icon_dir_entry->bColorCount = 0;
  icon_dir_entry->bReserved = 0;
  icon_dir_entry->wPlanes = 0;
  icon_dir_entry->wBitCount = 0;
  icon_dir_entry->dwImageOffset = 22;

  return icon_dir_entry;
}

IconDirEntry *
create_dir_entry (XpmImage *icon,
		  IconData *icon_data,
		  DWord ncolors)
{
  IconDirEntry *icon_dir_entry;

  icon_dir_entry = new_icon_dir_entry ();
  icon_dir_entry->bWidth = icon->width;
  icon_dir_entry->bHeight = icon->height;
  if (ncolors <= 256)
    icon_dir_entry->dwBytesInRes =
      40 + ncolors * 4
      + icon_data->xor_size
      + icon_data->and_size;
  else
    icon_dir_entry->dwBytesInRes =
      40 + icon_data->xor_size
         + icon_data->and_size;

  return icon_dir_entry;
}

BitmapInfoHeader *
create_bm_ih (XpmImage *icon, Word bpp,
	      DWord image_size, DWord nbr_colors)
{
  BitmapInfoHeader *bm_ih;

  bm_ih = malloc (sizeof (BitmapInfoHeader));
  bm_ih->biSize = 40; /* This is a strange manner of ensuring data
			 consistency, particularily when the system
			 this struct was made for is still so
			 unreliable. I am doubtful... */
  bm_ih->biWidth = icon->width;
  bm_ih->biHeight = icon->height * 2;
  bm_ih->biPlanes = 1;
  bm_ih->biBitCount = bpp;
  bm_ih->biCompression = 0;
  bm_ih->biSizeImage = image_size;
  bm_ih->biXPelsPerMeter = 0;
  bm_ih->biYPelsPerMeter = 0;
  if (nbr_colors == 256)
    bm_ih->biClrUsed = nbr_colors;
  else
    bm_ih->biClrUsed = 0;
  bm_ih->biClrImportant = 0;

  return bm_ih;
}

DWord
real_nbr_colors (DWord nbr_colors)
{
  DWord nbr;

  if (nbr_colors < 257)
    nbr = 256;
  else
    nbr = 65536;

  return nbr;
}

int
white_present (ColorMap *color_map)
{
  RGBQuad *color_quad;
  int ccount, found;

  ccount = 0;
  found = 0;

  while (!found && ccount < color_map->nbr_colors)
    {
      color_quad = color_map->color_list + ccount;
      if ((color_quad->rgbRed == 255)
	  && (color_quad->rgbGreen == 255)
	  && (color_quad->rgbBlue == 255))
	found = !found;
      else
	ccount++;
    }

  return found;
}

void
add_white (ColorMap *color_map, unsigned int clr_count)
{
  RGBQuad *white;

  printf ("Adding one color slot for white, since it was not"
	  " previously found and is needed for transparency.\n");
  white = color_map->color_list + clr_count;
  white->rgbRed = 255;
  white->rgbGreen = 255;
  white->rgbBlue = 255;
  white->rgbReserved = 0;
}

ColorMap *
convert_color_map (XpmImage *icon, DWord ncolors)
{
  unsigned int clr_count;
  ColorMap *color_map;

  if (ncolors < 257)
    {
      color_map = malloc (sizeof (ColorMap));
      color_map->nbr_colors = ncolors;
      color_map->color_list = malloc (sizeof (RGBQuad) * ncolors);
      
      for (clr_count = 0 ; clr_count < icon->ncolors; clr_count++)
	*(color_map->color_list + clr_count) =
	  convert_color (icon->colorTable + clr_count);
      
      if (!white_present (color_map))
	{
	  if (clr_count < ncolors)
	    {
	      add_white (color_map, clr_count);
	      clr_count++;
	    }
	  else
	    fprintf (stderr,
		     "Could not add a white color to the colormap, icon may"
		     " not be displayed properly.\n");
	}

      printf ("The resulting .ico file will contain %u colors (%u max."
	      " colors).\n", clr_count, (unsigned int) ncolors);
    }
  else
    {
      color_map = NULL;
      printf ("The image is encoded on 24bpp. There is thus no need"
	      " for a color map.\n");
    }

  return color_map;
}

void
put_24bit_color (DrawData *draw_data,
		 Word x, Word y,
		 RGBQuad *color)
{
  Byte *ptr;

  ptr = draw_data->data
    + ((draw_data->height - y - 1)
       * draw_data->width
       + x) * 3;

  *ptr = color->rgbBlue;
  *(ptr + 1) = color->rgbGreen;
  *(ptr + 2) = color->rgbRed;
}

void
put_8bit_color (DrawData *draw_data,
		Word x, Word y,
		Byte val)
{
  Byte *ptr;

  ptr = draw_data->data
    + (draw_data->height - y - 1)
       * draw_data->width
    + x;

  *ptr = val;
}

void
set_bit (DrawData *draw_data, DWord x, DWord y)
{
  Byte *byte;
  Word bit_num;

  bit_num = (draw_data->height - y - 1) * draw_data->width + x;
  byte = draw_data->data + (Word) (bit_num / 8);
  
  *byte |= 128 >> (bit_num % 8);
}

Byte *
make_xor_mask (XpmImage *icon)
{
  DWord mask_size, x, y;
  DWord *uint_ptr;
  Word bypp;
  RGBQuad color; /* for bpp == 24 */
  DrawData *draw_data;
  Byte *mask;

  bypp = bytes_per_pixel (icon->ncolors); 

  mask_size = (DWord) icon->width * (DWord) icon->height * bypp;

  draw_data = malloc (sizeof (DrawData));
  draw_data->data = malloc (mask_size);
  draw_data->width = (DWord) icon->width;
  draw_data->height = (DWord) icon->height;

  uint_ptr = (DWord *) icon->data;

  for (y = 0; y < icon->height; y++)
    for (x = 0; x < icon->width; x++)
      {
	if (bypp == 1)
	  put_8bit_color (draw_data, x, y, *uint_ptr);
	else
	  {
	    color = convert_color (icon->colorTable + *uint_ptr);
	    put_24bit_color (draw_data, x, y, &color);
	  }
	
	uint_ptr++;
      }
  
  mask = draw_data->data;
  free (draw_data);

  return mask;
}

Byte *
make_and_mask (XpmImage *icon, ColorMap *color_map)
{
  DWord mask_size, x, y;
  DWord *uint_ptr;
  DrawData *draw_data;
  RGBQuad color; /* for bpp >= 16 */
  Byte *mask;

  mask_size = icon->width * icon->height / 8;

  draw_data = malloc (sizeof (DrawData));
  draw_data->data = calloc (mask_size, sizeof (Byte));
  draw_data->width = icon->width;
  draw_data->height = icon->height;

  uint_ptr = (DWord *) icon->data;

  for (y = 0; y < icon->height; y++)
    for (x = 0; x < icon->width; x++)
      {
	if (icon->ncolors <= 256)
	  color = *(color_map->color_list + *uint_ptr);
	else
	  color = convert_color (icon->colorTable + *uint_ptr);

	if (color.rgbReserved)
	  set_bit (draw_data, x, y);
	
	uint_ptr++;
      }
  
  mask = draw_data->data;
  free (draw_data);

  return mask;
}

void
save_header (FILE *ico_file, IconDir *header)
{
  write_word (ico_file, header->idReserved);
  write_word (ico_file, header->idType);
  write_word (ico_file, header->idCount);
}

void
save_dir_entry (FILE *ico_file, IconDirEntry *icon_dir_entry)
{
  write_byte (ico_file, icon_dir_entry->bWidth);
  write_byte (ico_file, icon_dir_entry->bHeight);
  write_byte (ico_file, icon_dir_entry->bColorCount);
  write_byte (ico_file, icon_dir_entry->bReserved);
  write_word (ico_file, icon_dir_entry->wPlanes);
  write_word (ico_file, icon_dir_entry->wBitCount);
  write_dword (ico_file, icon_dir_entry->dwBytesInRes);
  write_dword (ico_file, icon_dir_entry->dwImageOffset);
}

void
save_bm_ih (FILE *ico_file, BitmapInfoHeader *bm_ih)
{
  write_dword (ico_file, bm_ih->biSize);
  write_long (ico_file, bm_ih->biWidth);
  write_long (ico_file, bm_ih->biHeight);
  write_word (ico_file, bm_ih->biPlanes);
  write_word (ico_file, bm_ih->biBitCount);
  write_dword (ico_file, bm_ih->biCompression);
  write_dword (ico_file, bm_ih->biSizeImage);
  write_long (ico_file, bm_ih->biXPelsPerMeter);
  write_long (ico_file, bm_ih->biYPelsPerMeter);
  write_dword (ico_file, bm_ih->biClrUsed);
  write_dword (ico_file, bm_ih->biClrImportant);
}

void save_RGBQuad (FILE *file, RGBQuad *rgb_quad)
{
  write_byte (file, rgb_quad->rgbBlue);
  write_byte (file, rgb_quad->rgbGreen);
  write_byte (file, rgb_quad->rgbRed);
  write_byte (file, 0);
}

void
save_color_map (FILE *ico_file, ColorMap *color_map)
{
  DWord count;
  RGBQuad *cur_quad;

  cur_quad = color_map->color_list;

  for (count = 0; count < color_map->nbr_colors; count++)
    {
      save_RGBQuad (ico_file, cur_quad);
      cur_quad++;
    }
}

/*  void */
/*  show_fpos (FILE *file) */
/*  { */
/*    fpos_t pos; */

/*    fgetpos (file, &pos); */
/*    printf ("file_pos = %d\n", pos); */
/*  } */

inline void
save_mask (FILE *ico_file, DWord size, Byte *mask)
{
  fwrite (mask, sizeof (Byte), size, ico_file);
}

int save_icon (char *file_name, IconData *icon_data)
{
  int ret_code;
  FILE *ico_file;

  /* The 'b' flag seems important to avoid screwing our icon because
   * otherwise our data is automatically (read "unpredictably")
   * aligned... */
  ico_file = fopen (file_name, "wb+");

  if (!ico_file)
    {
      perror (file_name);
      ret_code = EXIT_FAILURE;
    }
  else
    {
      printf ("Saving icon to \"%s\"\n", file_name);

      save_header (ico_file, icon_data->header);
      save_dir_entry (ico_file, icon_data->icon_dir_entry);
      save_bm_ih (ico_file, icon_data->bm_ih);

      if (icon_data->color_map)
	save_color_map (ico_file, icon_data->color_map);

      save_mask (ico_file, icon_data->xor_size, icon_data->xor_mask);
      save_mask (ico_file, icon_data->and_size, icon_data->and_mask);

      fclose (ico_file);
      ret_code = EXIT_SUCCESS;
    }

  return ret_code;
}

int convert_icon (char *ico_file_name,
		  XpmInfo *icon_info,
		  XpmImage *icon)
{
  int ret_code;
  IconData *icon_data;
  DWord ncolors;
  Word bypp;

  ret_code = is_valid_xpm_image (icon);

  if (ret_code == EXIT_SUCCESS)
    {
      icon_data = malloc (sizeof (IconData));

      ncolors = real_nbr_colors (icon->ncolors);

      icon_data->color_map = convert_color_map (icon, ncolors);

      icon_data->xor_mask = make_xor_mask (icon);
      icon_data->and_mask = make_and_mask (icon,
					   icon_data->color_map);

      bypp = bytes_per_pixel (icon->ncolors);     

      icon_data->xor_size = icon->width * icon->height * bypp;
      icon_data->and_size = icon->width * icon->height / 8;
 
      icon_data->icon_dir_entry =
	create_dir_entry (icon, icon_data, ncolors);
      icon_data->header = new_icon_header ();
      icon_data->bm_ih = create_bm_ih (icon, bypp * 8,
				       icon_data->and_size
				       + icon_data->xor_size,
				       ncolors);

      ret_code = save_icon (ico_file_name, icon_data);
    }

  return ret_code;
}

/* As I am not sure every xpm file use the same symbol table, I feel
   better creating one dynamically... */
void
make_xpm_col_table (XpmImage *icon)
{
  char *cur_color;
  DWord max_colors, ccount, found;

  max_colors = icon->ncolors;
  ccount = 1;
  found = 0;

  while (ccount < max_colors && !found)
    {
      cur_color = (icon->colorTable + ccount)->string;

      if (*cur_color == *(icon->colorTable->string))
	found = !found;
      else
	ccount++;
    }
  
  max_colors = ccount;

  xpm_col_codes = malloc (max_colors + 1);
  xpm_col_tbl_size = ccount;
 
  for (ccount = 0; ccount < max_colors; ccount++)
    {
      cur_color = (icon->colorTable + ccount)->string;
      *(xpm_col_codes + ccount) = *cur_color;
    }

  *(xpm_col_codes + max_colors) = 0;
}

int test_exist (char *file_name)
{
  int ret_code;
  char answer[2];

  if (!access(file_name, F_OK))
    {
      printf ("It happens a file named \"%s\" already exists.\n"
	      "If you wish to overwrite it, please confirm by"
	      " typing 'Y' or 'y': ", file_name);
      scanf ("%1s", answer);

      if (*answer == 'y'
	  || *answer == 'Y')
	ret_code = 0;
      else
	{
	  fprintf (stderr, "Conversion has been aborted.\n");
	  ret_code = 1;
	}
    }
  else
    ret_code = 0;

  return ret_code;
}

void
display_icon_info (XpmImage *icon)
{
  printf ("Informations on input icon:\n"
	  "  width: %d, height: %d\n"
	  "  number of colors: %d\n",
	  icon->width,
	  icon->height,
	  icon->ncolors);
}

int
process_icon (char *xpm_file_name, char *ico_file_name)
{
  int x_ret_code, ret_code;
  XpmInfo *icon_info;
  XpmImage *icon;

  if (!test_exist (ico_file_name))
    {
      icon_info = malloc (sizeof (XpmInfo));
      icon = malloc (sizeof (XpmImage));

      printf ("Reading icon from xpm file \"%s\"\n", xpm_file_name);
      x_ret_code = XpmReadFileToXpmImage (xpm_file_name, icon, icon_info);

      if (!x_ret_code)
	{
	  make_xpm_col_table (icon);
	  display_icon_info (icon);
	  ret_code = convert_icon (ico_file_name, icon_info, icon);
	}
      else
	{
	  perror (xpm_file_name);
	  ret_code = EXIT_FAILURE;
	}
      
      free (icon);
      free (icon_info);
    }
  else
    ret_code = EXIT_FAILURE;

  return ret_code;
}

char *
deduce_ico_name (char *file_name)
{
  char *ret_name, *char_ptr, *last_slash_ptr;
  int len, nlen, ext, delta;

  len = strlen (file_name);
  char_ptr = strrchr (file_name, '.');
#ifdef __WIN32__
  last_slash_ptr = strrchr (file_name, '\\');  
#else
  last_slash_ptr = strrchr (file_name, '/');
#endif

  if ((DWord) last_slash_ptr < (DWord) char_ptr)
    {
      ext = 1;
      delta = ((DWord) char_ptr - (DWord) file_name);
      nlen = delta + 5;
    }
  else
    {
      ext = 0;
      delta = 0;
      nlen = len + 5;
    }

  ret_name = malloc (nlen);

  if (ext)
    {
      strncpy (ret_name, file_name, delta);
      sprintf (ret_name + delta, ".ico");
    }
  else
    sprintf (ret_name, "%s%s", file_name, ".ico");

  return ret_name;
}

void
copyright ()
{
  printf ("%s v. %s\n"
	  "Copyright (C) 2001 Wolfgang Sourdeau\n\n",
	  package, VERSION);
}

void
basic_usage ()
{
  printf ("Usage: %s XPM_IN_FILE [ICO_OUT_FILE]\n",
	  package);
}

void
help_and_version (char *param)
{
  if (!strcmp (param, "--version")
      || !strcmp (param, "-V"))
    {
      printf ("This is free software; see the source for copying"
	      " conditions. There is NO\n"
	      "warranty; not even for MERCHANTABILITY or FITNESS"
	      " FOR A PARTICULAR PURPOSE.\n");
    }
  else if (!strcmp (param, "--usage")
	   || !strcasecmp (param, "-h"))
    {
      basic_usage ();
      printf ("Try `%s --help' for more information.\n",
	      package);
    }
  else if (!strcmp (param, "--help")
	   || !strcmp (param, "-H"))
    {
      printf ("This utility converts an xpm icon to a Windows ico file.\n");
      basic_usage ();
      printf ("\nOptional parameters:\n"
	      "  -H, --help			display this help"
	      " message\n"
	      "  -h, --usage			display a short usage"
	      " information\n"
	      "  -V, --version			display version and"
	      " licensing informations\n"
	      "                         	(Please read them"
	      " carefully)\n\n"
	      "Example: %s ghfaxviewer-icon.xpm ghfv.ico\n",
	      package);
    }
  else
    {
      basic_usage ();
      printf ("Option `%s' not recognized.\n", package, param);
    }
}

int main (int argc, char *argv[])
{
  int ret_code;
  char *xpm_file_name, *ico_file_name;

  copyright ();

  if (argc == 2)
    {
      xpm_file_name = argv[1];

      if (*argv[1] != '-')
	{
	  ico_file_name = deduce_ico_name (argv[1]);      
	  ret_code = process_icon (xpm_file_name, ico_file_name);
	}
      else
	{
	  help_and_version (argv[1]);
	  ret_code = EXIT_SUCCESS;
	}
    }
  else if (argc == 3)
    {
      xpm_file_name = argv[1];
      ico_file_name = argv[2];
      ret_code = process_icon (xpm_file_name, ico_file_name);
    }
  else
    {
      fprintf (stderr, "%s needs a xpm filename as argument\n",
	       PACKAGE);
      ret_code = EXIT_FAILURE;
    }

  return ret_code;
}
