/**
 * Copyright (c) 2014,2019 Contributors to the Eclipse Foundation
 * 
 * See the NOTICE file(s) distributed with this work for additional
 * information regarding copyright ownership.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.smarthome.model.rule.jvmmodel;

import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smarthome.core.items.Item;
import org.eclipse.smarthome.core.items.ItemRegistry;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingRegistry;
import org.eclipse.smarthome.core.thing.events.ChannelTriggeredEvent;
import org.eclipse.smarthome.core.types.Command;
import org.eclipse.smarthome.core.types.State;
import org.eclipse.smarthome.core.types.Type;
import org.eclipse.smarthome.model.rule.rules.ChangedEventTrigger;
import org.eclipse.smarthome.model.rule.rules.CommandEventTrigger;
import org.eclipse.smarthome.model.rule.rules.EventEmittedTrigger;
import org.eclipse.smarthome.model.rule.rules.EventTrigger;
import org.eclipse.smarthome.model.rule.rules.GroupMemberChangedEventTrigger;
import org.eclipse.smarthome.model.rule.rules.GroupMemberCommandEventTrigger;
import org.eclipse.smarthome.model.rule.rules.GroupMemberUpdateEventTrigger;
import org.eclipse.smarthome.model.rule.rules.Rule;
import org.eclipse.smarthome.model.rule.rules.RuleModel;
import org.eclipse.smarthome.model.rule.rules.ThingStateChangedEventTrigger;
import org.eclipse.smarthome.model.rule.rules.UpdateEventTrigger;
import org.eclipse.smarthome.model.rule.rules.VariableDeclaration;
import org.eclipse.smarthome.model.script.jvmmodel.ScriptJvmModelInferrer;
import org.eclipse.smarthome.model.script.scoping.StateAndCommandProvider;
import org.eclipse.smarthome.model.script.script.Script;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.xbase.jvmmodel.IJvmDeclaredTypeAcceptor;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>Infers a JVM model from the source model.</p>
 * 
 * <p>The JVM model should contain all elements that would appear in the Java code
 * which is generated from the source model. Other models link against the JVM model rather than the source model.</p>
 * 
 * @author Oliver Libutzki - Xtext 2.5.0 migration
 */
@SuppressWarnings("all")
public class RulesJvmModelInferrer extends ScriptJvmModelInferrer {
  private final Logger logger = LoggerFactory.getLogger(RulesJvmModelInferrer.class);
  
  /**
   * Variable name for the item in a "state triggered" or "command triggered" rule
   */
  public final static String VAR_TRIGGERING_ITEM = "triggeringItem";
  
  /**
   * Variable name for the previous state of an item in a "changed state triggered" rule
   */
  public final static String VAR_PREVIOUS_STATE = "previousState";
  
  /**
   * Variable name for the received command in a "command triggered" rule
   */
  public final static String VAR_RECEIVED_COMMAND = "receivedCommand";
  
  /**
   * Variable name for the received event in a "trigger event" rule
   */
  public final static String VAR_RECEIVED_EVENT = "receivedEvent";
  
  /**
   * conveninence API to build and initialize JvmTypes and their members.
   */
  @Inject
  @Extension
  private JvmTypesBuilder _jvmTypesBuilder;
  
  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;
  
  @Inject
  private ItemRegistry itemRegistry;
  
  @Inject
  private ThingRegistry thingRegistry;
  
  @Inject
  private StateAndCommandProvider stateAndCommandProvider;
  
  /**
   * Is called for each instance of the first argument's type contained in a resource.
   * 
   * @param element - the model to create one or more JvmDeclaredTypes from.
   * @param acceptor - each created JvmDeclaredType without a container should be passed to the acceptor in order get attached to the
   *                   current resource.
   * @param isPreLinkingPhase - whether the method is called in a pre linking phase, i.e. when the global index isn't fully updated. You
   *        must not rely on linking using the index if iPrelinkingPhase is <code>true</code>
   */
  protected void _infer(final RuleModel ruleModel, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    String _firstUpper = StringExtensions.toFirstUpper(IterableExtensions.<String>head(((Iterable<String>)Conversions.doWrapArray(ruleModel.eResource().getURI().lastSegment().split("\\.")))));
    final String className = (_firstUpper + "Rules");
    final Procedure1<JvmGenericType> _function = (JvmGenericType it) -> {
      EList<JvmMember> _members = it.getMembers();
      final Function1<VariableDeclaration, JvmField> _function_1 = (VariableDeclaration it_1) -> {
        String _name = it_1.getName();
        JvmTypeReference _type = it_1.getType();
        JvmTypeReference _cloneWithProxies = null;
        if (_type!=null) {
          _cloneWithProxies=this._jvmTypesBuilder.cloneWithProxies(_type);
        }
        JvmField _field = this._jvmTypesBuilder.toField(it_1, _name, _cloneWithProxies);
        final Procedure1<JvmField> _function_2 = (JvmField field) -> {
          field.setStatic(true);
          boolean _isWriteable = it_1.isWriteable();
          boolean _not = (!_isWriteable);
          field.setFinal(_not);
          this._jvmTypesBuilder.setInitializer(field, it_1.getRight());
        };
        return ObjectExtensions.<JvmField>operator_doubleArrow(_field, _function_2);
      };
      List<JvmField> _map = ListExtensions.<VariableDeclaration, JvmField>map(ruleModel.getVariables(), _function_1);
      this._jvmTypesBuilder.<JvmMember>operator_add(_members, _map);
      final Set<String> fieldNames = CollectionLiterals.<String>newHashSet();
      final Iterable<Type> types = this.stateAndCommandProvider.getAllTypes();
      final Consumer<Type> _function_2 = (Type type) -> {
        final String name = type.toString();
        boolean _add = fieldNames.add(name);
        if (_add) {
          EList<JvmMember> _members_1 = it.getMembers();
          final Procedure1<JvmField> _function_3 = (JvmField it_1) -> {
            it_1.setStatic(true);
          };
          JvmField _field = this._jvmTypesBuilder.toField(ruleModel, name, this._jvmTypesBuilder.newTypeRef(ruleModel, type.getClass()), _function_3);
          this._jvmTypesBuilder.<JvmField>operator_add(_members_1, _field);
        } else {
          this.logger.warn("Duplicate field: \'{}\'. Ignoring \'{}\'.", name, type.getClass().getName());
        }
      };
      types.forEach(_function_2);
      Collection<Item> _items = null;
      if (this.itemRegistry!=null) {
        _items=this.itemRegistry.getItems();
      }
      if (_items!=null) {
        final Consumer<Item> _function_3 = (Item item) -> {
          final String name = item.getName();
          boolean _add = fieldNames.add(name);
          if (_add) {
            EList<JvmMember> _members_1 = it.getMembers();
            final Procedure1<JvmField> _function_4 = (JvmField it_1) -> {
              it_1.setStatic(true);
            };
            JvmField _field = this._jvmTypesBuilder.toField(ruleModel, item.getName(), this._jvmTypesBuilder.newTypeRef(ruleModel, item.getClass()), _function_4);
            this._jvmTypesBuilder.<JvmField>operator_add(_members_1, _field);
          } else {
            this.logger.warn("Duplicate field: \'{}\'. Ignoring \'{}\'.", item.getName(), item.getClass().getName());
          }
        };
        _items.forEach(_function_3);
      }
      Collection<Thing> _all = null;
      if (this.thingRegistry!=null) {
        _all=this.thingRegistry.getAll();
      }
      final Collection<Thing> things = _all;
      if (things!=null) {
        final Consumer<Thing> _function_4 = (Thing thing) -> {
          final String name = thing.getUID().toString();
          boolean _add = fieldNames.add(name);
          if (_add) {
            EList<JvmMember> _members_1 = it.getMembers();
            final Procedure1<JvmField> _function_5 = (JvmField it_1) -> {
              it_1.setStatic(true);
            };
            JvmField _field = this._jvmTypesBuilder.toField(ruleModel, name, this._jvmTypesBuilder.newTypeRef(ruleModel, thing.getClass()), _function_5);
            this._jvmTypesBuilder.<JvmField>operator_add(_members_1, _field);
          } else {
            this.logger.warn("Duplicate field: \'{}\'. Ignoring \'{}\'.", name, thing.getClass().getName());
          }
        };
        things.forEach(_function_4);
      }
      EList<JvmMember> _members_1 = it.getMembers();
      final Function1<Rule, JvmOperation> _function_5 = (Rule rule) -> {
        String _name = rule.getName();
        String _plus = ("_" + _name);
        final Procedure1<JvmOperation> _function_6 = (JvmOperation it_1) -> {
          it_1.setStatic(true);
          if (((this.containsCommandTrigger(rule) || this.containsStateChangeTrigger(rule)) || this.containsStateUpdateTrigger(rule))) {
            final JvmTypeReference itemTypeRef = this._jvmTypesBuilder.newTypeRef(ruleModel, Item.class);
            EList<JvmFormalParameter> _parameters = it_1.getParameters();
            JvmFormalParameter _parameter = this._jvmTypesBuilder.toParameter(rule, RulesJvmModelInferrer.VAR_TRIGGERING_ITEM, itemTypeRef);
            this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
          }
          boolean _containsCommandTrigger = this.containsCommandTrigger(rule);
          if (_containsCommandTrigger) {
            final JvmTypeReference commandTypeRef = this._jvmTypesBuilder.newTypeRef(ruleModel, Command.class);
            EList<JvmFormalParameter> _parameters_1 = it_1.getParameters();
            JvmFormalParameter _parameter_1 = this._jvmTypesBuilder.toParameter(rule, RulesJvmModelInferrer.VAR_RECEIVED_COMMAND, commandTypeRef);
            this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter_1);
          }
          if ((this.containsStateChangeTrigger(rule) && (!this.containsParam(it_1.getParameters(), RulesJvmModelInferrer.VAR_PREVIOUS_STATE)))) {
            final JvmTypeReference stateTypeRef = this._jvmTypesBuilder.newTypeRef(ruleModel, State.class);
            EList<JvmFormalParameter> _parameters_2 = it_1.getParameters();
            JvmFormalParameter _parameter_2 = this._jvmTypesBuilder.toParameter(rule, RulesJvmModelInferrer.VAR_PREVIOUS_STATE, stateTypeRef);
            this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _parameter_2);
          }
          boolean _containsEventTrigger = this.containsEventTrigger(rule);
          if (_containsEventTrigger) {
            final JvmTypeReference eventTypeRef = this._jvmTypesBuilder.newTypeRef(ruleModel, ChannelTriggeredEvent.class);
            EList<JvmFormalParameter> _parameters_3 = it_1.getParameters();
            JvmFormalParameter _parameter_3 = this._jvmTypesBuilder.toParameter(rule, RulesJvmModelInferrer.VAR_RECEIVED_EVENT, eventTypeRef);
            this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_3, _parameter_3);
          }
          if ((this.containsThingStateChangedEventTrigger(rule) && (!this.containsParam(it_1.getParameters(), RulesJvmModelInferrer.VAR_PREVIOUS_STATE)))) {
            final JvmTypeReference stateTypeRef_1 = this._jvmTypesBuilder.newTypeRef(ruleModel, State.class);
            EList<JvmFormalParameter> _parameters_4 = it_1.getParameters();
            JvmFormalParameter _parameter_4 = this._jvmTypesBuilder.toParameter(rule, RulesJvmModelInferrer.VAR_PREVIOUS_STATE, stateTypeRef_1);
            this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_4, _parameter_4);
          }
          this._jvmTypesBuilder.setBody(it_1, rule.getScript());
        };
        return this._jvmTypesBuilder.toMethod(rule, _plus, this._jvmTypesBuilder.newTypeRef(ruleModel, Void.TYPE), _function_6);
      };
      List<JvmOperation> _map_1 = ListExtensions.<Rule, JvmOperation>map(ruleModel.getRules(), _function_5);
      this._jvmTypesBuilder.<JvmMember>operator_add(_members_1, _map_1);
    };
    acceptor.<JvmGenericType>accept(this._jvmTypesBuilder.toClass(ruleModel, className)).initializeLater(_function);
  }
  
  private boolean containsParam(final EList<JvmFormalParameter> params, final String param) {
    final Function1<JvmFormalParameter, String> _function = (JvmFormalParameter it) -> {
      return it.getName();
    };
    return ListExtensions.<JvmFormalParameter, String>map(params, _function).contains(param);
  }
  
  private boolean containsCommandTrigger(final Rule rule) {
    EList<EventTrigger> _eventtrigger = rule.getEventtrigger();
    for (final EventTrigger trigger : _eventtrigger) {
      {
        if ((trigger instanceof CommandEventTrigger)) {
          return true;
        }
        if ((trigger instanceof GroupMemberCommandEventTrigger)) {
          return true;
        }
      }
    }
    return false;
  }
  
  private boolean containsStateChangeTrigger(final Rule rule) {
    EList<EventTrigger> _eventtrigger = rule.getEventtrigger();
    for (final EventTrigger trigger : _eventtrigger) {
      {
        if ((trigger instanceof ChangedEventTrigger)) {
          return true;
        }
        if ((trigger instanceof GroupMemberChangedEventTrigger)) {
          return true;
        }
      }
    }
    return false;
  }
  
  private boolean containsStateUpdateTrigger(final Rule rule) {
    EList<EventTrigger> _eventtrigger = rule.getEventtrigger();
    for (final EventTrigger trigger : _eventtrigger) {
      {
        if ((trigger instanceof UpdateEventTrigger)) {
          return true;
        }
        if ((trigger instanceof GroupMemberUpdateEventTrigger)) {
          return true;
        }
      }
    }
    return false;
  }
  
  private boolean containsEventTrigger(final Rule rule) {
    EList<EventTrigger> _eventtrigger = rule.getEventtrigger();
    for (final EventTrigger trigger : _eventtrigger) {
      if ((trigger instanceof EventEmittedTrigger)) {
        return true;
      }
    }
    return false;
  }
  
  private boolean containsThingStateChangedEventTrigger(final Rule rule) {
    EList<EventTrigger> _eventtrigger = rule.getEventtrigger();
    for (final EventTrigger trigger : _eventtrigger) {
      if ((trigger instanceof ThingStateChangedEventTrigger)) {
        return true;
      }
    }
    return false;
  }
  
  public void infer(final EObject ruleModel, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    if (ruleModel instanceof Script) {
      _infer((Script)ruleModel, acceptor, isPreIndexingPhase);
      return;
    } else if (ruleModel instanceof RuleModel) {
      _infer((RuleModel)ruleModel, acceptor, isPreIndexingPhase);
      return;
    } else if (ruleModel != null) {
      _infer(ruleModel, acceptor, isPreIndexingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(ruleModel, acceptor, isPreIndexingPhase).toString());
    }
  }
}
