/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.binding.ntp.handler;

import java.io.IOException;
import java.math.BigDecimal;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.commons.net.ntp.NTPUDPClient;
import org.apache.commons.net.ntp.TimeInfo;
import org.eclipse.smarthome.config.core.Configuration;
import org.eclipse.smarthome.core.i18n.LocaleProvider;
import org.eclipse.smarthome.core.library.types.DateTimeType;
import org.eclipse.smarthome.core.library.types.StringType;
import org.eclipse.smarthome.core.thing.Channel;
import org.eclipse.smarthome.core.thing.ChannelUID;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingStatus;
import org.eclipse.smarthome.core.thing.ThingStatusDetail;
import org.eclipse.smarthome.core.thing.binding.BaseThingHandler;
import org.eclipse.smarthome.core.types.Command;
import org.eclipse.smarthome.core.types.State;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NtpHandler
extends BaseThingHandler {
    private final Logger logger = LoggerFactory.getLogger(NtpHandler.class);
    private static final int NTP_TIMEOUT = 30000;
    public static final String DATE_PATTERN_WITH_TZ = "yyyy-MM-dd HH:mm:ss z";
    private final DateFormat SDF = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss z");
    private DateTimeFormatter dateTimeFormat = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss z");
    private final LocaleProvider localeProvider;
    ScheduledFuture<?> refreshJob;
    private String hostname;
    private BigDecimal port;
    private BigDecimal refreshInterval;
    private BigDecimal refreshNtp = new BigDecimal(0);
    private TimeZone timeZone;
    private Locale locale;
    private int refreshNtpCount = 0;
    private long timeOffset;
    private ChannelUID dateTimeChannelUID;
    private ChannelUID stringChannelUID;

    public NtpHandler(Thing thing, LocaleProvider localeProvider) {
        super(thing);
        this.localeProvider = localeProvider;
    }

    public void handleCommand(ChannelUID channelUID, Command command) {
        this.refreshNtpCount = 0;
        this.refreshTimeDate();
    }

    public void initialize() {
        try {
            this.logger.debug("Initializing NTP handler for '{}'.", (Object)this.getThing().getUID());
            Configuration config = this.getThing().getConfiguration();
            this.hostname = config.get("hostname").toString();
            this.port = (BigDecimal)config.get("serverPort");
            this.refreshInterval = (BigDecimal)config.get("refreshInterval");
            this.refreshNtp = (BigDecimal)config.get("refreshNtp");
            this.refreshNtpCount = 0;
            try {
                Object timeZoneConfigValue = config.get("timeZone");
                if (timeZoneConfigValue != null) {
                    this.timeZone = TimeZone.getTimeZone(timeZoneConfigValue.toString());
                } else {
                    this.timeZone = TimeZone.getDefault();
                    this.logger.debug("{} using default TZ '{}', because configuration property '{}' is null.", new Object[]{this.getThing().getUID(), this.timeZone, "timeZone"});
                }
            }
            catch (Exception e) {
                this.timeZone = TimeZone.getDefault();
                this.logger.debug("{} using default TZ '{}' due to an occurred exception: ", new Object[]{this.getThing().getUID(), this.timeZone, e});
            }
            try {
                Object localeStringConfigValue = config.get("locale");
                if (localeStringConfigValue != null) {
                    this.locale = new Locale(localeStringConfigValue.toString());
                } else {
                    this.locale = this.localeProvider.getLocale();
                    this.logger.debug("{} using default locale '{}', because configuration property '{}' is null.", new Object[]{this.getThing().getUID(), this.locale, "locale"});
                }
            }
            catch (Exception e) {
                this.locale = this.localeProvider.getLocale();
                this.logger.debug("{} using default locale '{}' due to an occurred exception: ", new Object[]{this.getThing().getUID(), this.locale, e});
            }
            this.dateTimeChannelUID = new ChannelUID(this.getThing().getUID(), "dateTime");
            this.stringChannelUID = new ChannelUID(this.getThing().getUID(), "string");
            try {
                Channel stringChannel = this.getThing().getChannel(this.stringChannelUID.getId());
                if (stringChannel != null) {
                    Configuration cfg = stringChannel.getConfiguration();
                    String dateTimeFormatString = cfg.get("DateTimeFormat").toString();
                    if (dateTimeFormatString != null && !dateTimeFormatString.isEmpty()) {
                        this.dateTimeFormat = DateTimeFormatter.ofPattern(dateTimeFormatString);
                    } else {
                        this.logger.debug("No format set in channel config for {}. Using default format.", (Object)this.stringChannelUID);
                        this.dateTimeFormat = DateTimeFormatter.ofPattern(DATE_PATTERN_WITH_TZ);
                    }
                } else {
                    this.logger.debug("Missing channel: '{}'", (Object)this.stringChannelUID.getId());
                }
            }
            catch (RuntimeException ex) {
                this.logger.debug("No channel config or invalid format for {}. Using default format. ({})", (Object)this.stringChannelUID, (Object)ex.getMessage());
                this.dateTimeFormat = DateTimeFormatter.ofPattern(DATE_PATTERN_WITH_TZ);
            }
            this.SDF.setTimeZone(this.timeZone);
            this.dateTimeFormat.withZone(this.timeZone.toZoneId());
            this.logger.debug("Initialized NTP handler '{}' with configuration: host '{}', refresh interval {}, timezone {}, locale {}.", new Object[]{this.getThing().getUID(), this.hostname, this.refreshInterval, this.timeZone, this.locale});
            this.startAutomaticRefresh();
        }
        catch (Exception ex) {
            this.logger.error("Error occurred while initializing NTP handler: {}", (Object)ex.getMessage(), (Object)ex);
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.CONFIGURATION_ERROR, "@text/offline.conf-error-init-handler");
        }
    }

    public void dispose() {
        this.refreshJob.cancel(true);
        super.dispose();
    }

    private void startAutomaticRefresh() {
        this.refreshJob = this.scheduler.scheduleWithFixedDelay(() -> {
            try {
                this.refreshTimeDate();
            }
            catch (Exception e) {
                this.logger.debug("Exception occurred during refresh: {}", (Object)e.getMessage(), (Object)e);
            }
        }, 0L, this.refreshInterval.intValue(), TimeUnit.SECONDS);
    }

    private synchronized void refreshTimeDate() {
        if (this.timeZone != null && this.locale != null) {
            long networkTimeInMillis;
            if (this.refreshNtpCount <= 0) {
                networkTimeInMillis = this.getTime(this.hostname);
                this.timeOffset = networkTimeInMillis - System.currentTimeMillis();
                this.logger.debug("{} delta system time: {}", (Object)this.getThing().getUID(), (Object)this.timeOffset);
                this.refreshNtpCount = this.refreshNtp.intValue();
            } else {
                networkTimeInMillis = System.currentTimeMillis() + this.timeOffset;
                --this.refreshNtpCount;
            }
            ZonedDateTime zoned = ZonedDateTime.ofInstant(Instant.ofEpochMilli(networkTimeInMillis), this.timeZone.toZoneId());
            this.updateState(this.dateTimeChannelUID, (State)new DateTimeType(zoned));
            this.updateState(this.stringChannelUID, (State)new StringType(this.dateTimeFormat.format(zoned)));
        } else {
            this.logger.debug("Not refreshing, since we do not seem to be initialized yet");
        }
    }

    public long getTime(String hostname) {
        try {
            NTPUDPClient timeClient = new NTPUDPClient();
            timeClient.setDefaultTimeout(30000);
            InetAddress inetAddress = InetAddress.getByName(hostname);
            TimeInfo timeInfo = timeClient.getTime(inetAddress, this.port.intValue());
            timeInfo.computeDetails();
            long serverMillis = timeInfo.getReturnTime() + timeInfo.getOffset();
            this.logger.debug("{} Got time update from host '{}': {}.", new Object[]{this.getThing().getUID(), hostname, this.SDF.format(new Date(serverMillis))});
            this.updateStatus(ThingStatus.ONLINE, ThingStatusDetail.NONE);
            return serverMillis;
        }
        catch (UnknownHostException uhe) {
            this.logger.debug("{} The given hostname '{}' of the timeserver is unknown -> returning current sytem time instead. ({})", new Object[]{this.getThing().getUID(), hostname, uhe.getMessage()});
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.COMMUNICATION_ERROR, "@text/offline.comm-error-unknown-host [\"" + (hostname == null ? "null" : hostname) + "\"]");
        }
        catch (IOException ioe) {
            this.logger.debug("{} Couldn't establish network connection to host '{}' -> returning current sytem time instead. ({})", new Object[]{this.getThing().getUID(), hostname, ioe.getMessage()});
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.COMMUNICATION_ERROR, "@text/offline.comm-error-connection [\"" + (hostname == null ? "null" : hostname) + "\"]");
        }
        return System.currentTimeMillis();
    }

    public void channelLinked(ChannelUID channelUID) {
        this.refreshTimeDate();
    }
}

