/**
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Les Jones <lesojones@gmail.com> - bug 190717
 *     Diego Sandin (Motorola) - Adapted code from org.eclipse.pde.ui
 */
package org.eclipse.mtj.internal.ui.parts;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.forms.widgets.FormToolkit;

public abstract class SharedPartWithButtons extends SharedPart {
    private class SelectionHandler implements SelectionListener {
        public void widgetDefaultSelected(SelectionEvent e) {
            buttonSelected(e);
        }

        public void widgetSelected(SelectionEvent e) {
            buttonSelected(e);
        }

        private void buttonSelected(SelectionEvent e) {
            Integer index = (Integer) e.widget.getData();
            SharedPartWithButtons.this.buttonSelected((Button) e.widget, index
                    .intValue());
        }
    }

    private String[] fButtonLabels;
    private Button[] fButtons;

    protected Composite fButtonContainer;

    public SharedPartWithButtons(String[] buttonLabels) {
        fButtonLabels = buttonLabels;
    }

    /*
     * @see SharedPart#createControl(Composite, FormWidgetFactory)
     */
    @Override
    public void createControl(Composite parent, int style, int span,
            FormToolkit toolkit) {
        createMainLabel(parent, span, toolkit);
        createMainControl(parent, style, span - 1, toolkit);
        createButtons(parent, toolkit);
    }

    /**
     * @param index
     * @return
     */
    public Button getButton(int index) {
        //
        if ((fButtons == null) || (index < 0) || (index >= fButtons.length)) {
            return null;
        }
        //
        return fButtons[index];
    }

    public void setButtonEnabled(int index, boolean enabled) {
        if ((fButtons != null) && (index >= 0) && (fButtons.length > index)) {
            fButtons[index].setEnabled(enabled);
        }
    }

    /**
     * Set the specified button's visibility. Fix for defect 190717.
     * 
     * @param index The index of the button to be changed
     * @param visible true if the button is to be shown, false if hidden
     */
    public void setButtonVisible(int index, boolean visible) {
        if ((fButtons != null) && (index >= 0) && (fButtons.length > index)) {
            fButtons[index].setVisible(visible);
        }
    }

    protected abstract void buttonSelected(Button button, int index);

    protected Button createButton(Composite parent, String label, int index,
            FormToolkit toolkit) {
        Button button;
        if (toolkit != null) {
            button = toolkit.createButton(parent, label, SWT.PUSH);
        } else {
            button = new Button(parent, SWT.PUSH);
            button.setText(label);
        }
        GridData gd = new GridData(GridData.FILL_HORIZONTAL
                | GridData.VERTICAL_ALIGN_BEGINNING);
        button.setLayoutData(gd);
        button.setData(new Integer(index));
        return button;
    }

    protected void createButtons(Composite parent, FormToolkit toolkit) {
        if ((fButtonLabels != null) && (fButtonLabels.length > 0)) {
            fButtonContainer = createComposite(parent, toolkit);
            GridData gd = new GridData(GridData.FILL_VERTICAL);
            fButtonContainer.setLayoutData(gd);
            fButtonContainer.setLayout(createButtonsLayout());
            fButtons = new Button[fButtonLabels.length];
            SelectionHandler listener = new SelectionHandler();
            for (int i = 0; i < fButtonLabels.length; i++) {
                String label = fButtonLabels[i];
                if (label != null) {
                    Button button = createButton(fButtonContainer, label, i,
                            toolkit);
                    button.addSelectionListener(listener);
                    fButtons[i] = button;
                } else {
                    createEmptySpace(fButtonContainer, 1, toolkit);
                }
            }
        }
    }

    protected GridLayout createButtonsLayout() {
        GridLayout layout = new GridLayout();
        layout.marginWidth = layout.marginHeight = 0;
        return layout;
    }

    protected abstract void createMainControl(Composite parent, int style,
            int span, FormToolkit toolkit);

    protected void createMainLabel(Composite parent, int span,
            FormToolkit toolkit) {
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.parts.SharedPart#updateEnabledState()
     */
    @Override
    protected void updateEnabledState() {
        for (Button button : fButtons) {
            button.setEnabled(isEnabled());
        }
    }
}