/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

/**
 * Copyright (c) 2008 Motorola.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Diego Sandin (Motorola) - Adapted code available in JDT
 */
package org.eclipse.mtj.internal.ui.wizards.dialogfields;

import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;

/**
 * Dialog field containing a label and a combo control.
 */
public class ComboDialogField extends DialogField {

    protected static GridData gridDataForCombo(int span) {
        GridData gd = new GridData();
        gd.horizontalAlignment = GridData.FILL;
        gd.grabExcessHorizontalSpace = false;
        gd.horizontalSpan = span;
        return gd;
    }

    private Combo fComboControl;
    private int fFlags;
    private String[] fItems;
    private ModifyListener fModifyListener;
    private int fSelectionIndex;

    private String fText;

    /**
     * @param flags
     */
    public ComboDialogField(int flags) {
        super();
        fText = ""; //$NON-NLS-1$
        fItems = new String[0];
        fFlags = flags;
        fSelectionIndex = -1;
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.testing.dialogfields.DialogField#doFillIntoGrid(org.eclipse.swt.widgets.Composite, int)
     */
    @Override
    public Control[] doFillIntoGrid(Composite parent, int nColumns) {
        assertEnoughColumns(nColumns);

        Label label = getLabelControl(parent);
        label.setLayoutData(gridDataForLabel(1));
        Combo combo = getComboControl(parent);
        combo.setLayoutData(gridDataForCombo(nColumns - 1));

        return new Control[] { label, combo };
    }

    /**
     * Creates or returns the created combo control.
     * 
     * @param parent The parent composite or <code>null</code> when the widget
     *            has already been created.
     */
    public Combo getComboControl(Composite parent) {
        if (fComboControl == null) {
            assertCompositeNotNull(parent);
            fModifyListener = new ModifyListener() {
                public void modifyText(ModifyEvent e) {
                    doModifyText(e);
                }
            };
            SelectionListener selectionListener = new SelectionListener() {
                public void widgetDefaultSelected(SelectionEvent e) {
                }

                public void widgetSelected(SelectionEvent e) {
                    doSelectionChanged(e);
                }
            };

            fComboControl = new Combo(parent, fFlags);
            fComboControl.setItems(fItems);
            if (fSelectionIndex != -1) {
                fComboControl.select(fSelectionIndex);
            } else {
                fComboControl.setText(fText);
            }
            fComboControl.setFont(parent.getFont());
            fComboControl.addModifyListener(fModifyListener);
            fComboControl.addSelectionListener(selectionListener);
            fComboControl.setEnabled(isEnabled());
        }
        return fComboControl;
    }

    // ------- focus methods

    /**
     * Gets the combo items.
     */
    public String[] getItems() {
        return fItems;
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.testing.dialogfields.DialogField#getNumberOfControls()
     */
    @Override
    public int getNumberOfControls() {
        return 2;
    }

    /**
     * @return
     */
    public int getSelectionIndex() {
        return fSelectionIndex;
    }

    /**
     * Gets the text.
     */
    public String getText() {
        return fText;
    }

    /*
     * (non-Javadoc)
     * @see
     * org.eclipse.jdt.internal.ui.wizards.dialogfields.DialogField#refresh()
     */
    @Override
    public void refresh() {
        super.refresh();
        setTextWithoutUpdate(fText);
    }

    /**
     * Selects an item.
     */
    public boolean selectItem(int index) {
        boolean success = false;
        if (isOkToUse(fComboControl)) {
            fComboControl.select(index);
            success = fComboControl.getSelectionIndex() == index;
        } else {
            if ((index >= 0) && (index < fItems.length)) {
                fText = fItems[index];
                fSelectionIndex = index;
                success = true;
            }
        }
        if (success) {
            dialogFieldChanged();
        }
        return success;
    }

    /**
     * Selects an item.
     */
    public boolean selectItem(String name) {
        for (int i = 0; i < fItems.length; i++) {
            if (fItems[i].equals(name)) {
                return selectItem(i);
            }
        }
        return false;
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.testing.dialogfields.DialogField#setFocus()
     */
    @Override
    public boolean setFocus() {
        if (isOkToUse(fComboControl)) {
            fComboControl.setFocus();
        }
        return true;
    }

    /**
     * Sets the combo items. Triggers a dialog-changed event.
     */
    public void setItems(String[] items) {
        fItems = items;
        if (isOkToUse(fComboControl)) {
            fComboControl.setItems(items);
        }
        dialogFieldChanged();
    }

    /**
     * Sets the text. Triggers a dialog-changed event.
     */
    public void setText(String text) {
        fText = text;
        if (isOkToUse(fComboControl)) {
            fComboControl.setText(text);
        } else {
            dialogFieldChanged();
        }
    }

    /**
     * Sets the text without triggering a dialog-changed event.
     */
    public void setTextWithoutUpdate(String text) {
        fText = text;
        if (isOkToUse(fComboControl)) {
            fComboControl.removeModifyListener(fModifyListener);
            fComboControl.setText(text);
            fComboControl.addModifyListener(fModifyListener);
        }
    }

    private void doModifyText(ModifyEvent e) {
        if (isOkToUse(fComboControl)) {
            fText = fComboControl.getText();
            fSelectionIndex = fComboControl.getSelectionIndex();
        }
        dialogFieldChanged();
    }

    private void doSelectionChanged(SelectionEvent e) {
        if (isOkToUse(fComboControl)) {
            fItems = fComboControl.getItems();
            fText = fComboControl.getText();
            fSelectionIndex = fComboControl.getSelectionIndex();
        }
        dialogFieldChanged();
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.testing.dialogfields.DialogField#updateEnableState()
     */
    @Override
    protected void updateEnableState() {
        super.updateEnableState();
        if (isOkToUse(fComboControl)) {
            fComboControl.setEnabled(isEnabled());
        }
    }

}
