/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: RDFTripleAnalyzer.java,v 1.1 2007/03/18 08:39:03 lzhang Exp $
 */

package org.eclipse.eodm.rdf.resource.parser.impl;

import java.net.URISyntaxException;

import org.eclipse.eodm.RDFFactory;
import org.eclipse.eodm.exceptions.AddingTripleException;
import org.eclipse.eodm.exceptions.InvalidLexicalFormException;
import org.eclipse.eodm.exceptions.UnsupportedViewTypeException;
import org.eclipse.eodm.rdf.rdfbase.RDFGraph;
import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.rdfweb.NamespaceDefinition;
import org.eclipse.eodm.rdf.resource.parser.element.RDFBlankNode;
import org.eclipse.eodm.rdf.resource.parser.element.RDFLiteralElement;
import org.eclipse.eodm.rdf.resource.parser.element.RDFResourceElement;
import org.eclipse.eodm.rdf.resource.parser.element.RDFValue;
import org.eclipse.eodm.rdf.resource.parser.element.URIReference;
import org.eclipse.eodm.rdf.resource.parser.exception.ParserException;
import org.eclipse.eodm.util.Triple;
import org.eclipse.eodm.vocabulary.RDF;
import org.eclipse.eodm.vocabulary.RDFS;

/**
 * RDFTripleAnalyzer
 */
public class RDFTripleAnalyzer {
    /* The namespace map */
    private RDFNamespaceMap namespaceMap = new RDFNamespaceMap();

    /**
     * Get the namespaces map
     * 
     * @return the namespaces map
     */
    public RDFNamespaceMap getNamespaceMap() {
        return namespaceMap;
    }

    // the generated document and graph from input triples.
    private Document document;

    private RDFGraph graph;

    /**
     * Get the EODM document.
     * 
     * @return EODM document
     */
    public Document getDocument() {
        return document;
    }

    /**
     * Initialize the analyzer. Add default namespaces.
     */
    public void initialize(String documentURI) {

        try {
            document = RDFFactory.eINSTANCE.createDocument(documentURI);
            graph = RDFFactory.eINSTANCE.createGraph(documentURI);
            document.setComplementalGraph(graph);
        } catch (URISyntaxException e) {
            e.printStackTrace();
            throw new ParserException("RDF Parser errors: " + e.getMessage());
        }
        handleNamespace("rdf", RDF.NAMESPACE);
        handleNamespace("rdfs", RDFS.NAMESPACE);
        handleNamespace("xsd", RDFS.XSD_NAMESPACE);
        handleNamespace("bnode", RDFS.BNODE_NAMESPACE);
    }

    /**
     * Handle namespaces generated during RDF parsing.
     * 
     * @param name
     *            The abbreviation of the namespace
     * @param uri
     *            The namespace full uri
     * @return the Namespace handled
     */
    public NamespaceDefinition handleNamespace(String name, String uri) {
        NamespaceDefinition namespace = namespaceMap.getNamespace(uri);
        /*
         * if the namespace is not found or it's not the BNode namespace &&
         * names are not equivalent, then add it
         */
        if (namespace == null
            || !name.equals(namespace.getNamespacePrefix())
            && !uri.equals(RDFS.BNODE_NAMESPACE)
            && !namespace.getNamespacePrefix().startsWith("ns_")) {
            namespace = createNamespace(name, uri);
            // if the namespace's prefix is null, it will be set a default value
            namespaceMap.addNamespace(namespace);
            document.getNamespaceDefinition().add(namespace);
        } else if (namespace != null
                   && name != null && name.length() > 0
                   && namespace.getNamespacePrefix().startsWith("ns_")
                   && !name.startsWith("ns_"))
            namespace.setNamespacePrefix(name);
        return namespace;
    }

    /**
     * Create a new namespace with the input name and uri.
     * 
     * @param name
     *            The namespace name
     * @param uri
     *            The namespace uri
     * @return The namespace created
     */
    private NamespaceDefinition createNamespace(String name, String uri) {
        NamespaceDefinition namespace = null;
        try {
            namespace = RDFFactory.eINSTANCE.createNamespaceDefinition(name,
                    RDFFactory.eINSTANCE.createNamespace(uri));
        } catch (URISyntaxException e) {
            throw new ParserException("RDF parser exception: " + e.getMessage());
        }
        return namespace;
    }

    /**
     * Get a namespace by a specific uri.
     * 
     * @param uri
     *            The namespace uri
     * @return The namespacedefinition
     */
    public NamespaceDefinition getNamespaceByURI(String uri) {
        NamespaceDefinition namespace = (NamespaceDefinition) namespaceMap
                .getNamespace(uri);
        if (namespace == null)
            namespace = handleNamespace(null, uri);
        return namespace;
    }

    /**
     * Analyze an RDF triple {subject, predicate, object}.
     * 
     * @param subject
     *            The subject of the triple
     * @param predicate
     *            The predicate of the triple
     * @param object
     *            The object of the triple
     */
    public void analyseTriple(RDFResourceElement subject,
            URIReference predicate, RDFValue object) {

        Triple triple = new Triple();

        if (subject instanceof RDFBlankNode)
            triple.setSubjectNodeID(subject.getLocalName());
        else
            triple.setSubjectURI(getNamespaceByURI(subject.getNamespace())
                    .getNamespace(), subject.getLocalName());

        triple.setPredicateUriRef(getNamespaceByURI(predicate.getNamespace())
                .getNamespace(), predicate.getLocalName());

        if (object instanceof RDFBlankNode)
            triple.setObjectNodeID(((RDFBlankNode) object).getLocalName());
        else if (object instanceof URIReference) {
            URIReference uri = (URIReference) object;
            triple.setObjectURI(getNamespaceByURI(uri.getNamespace())
                    .getNamespace(), uri.getLocalName());
        } else {
            RDFLiteralElement literal = (RDFLiteralElement) object;
            URIReference typeRef = literal.getDatatype();
            if (typeRef != null) {
                if (typeRef.getFullURI().equals(
                        RDF.NAMESPACE + RDF.C_XMLLITERAL))
                    try {
                        triple.setObjectLiteral(RDFFactory.eINSTANCE
                                .createRDFXMLLiteral(literal.getText()));
                    } catch (InvalidLexicalFormException e) {
                        e.printStackTrace();
                        throw new ParserException("RDF Parser errors: "
                                                  + e.getMessage());
                    }
                else
                    try {
                        triple.setObjectLiteral(RDFFactory.eINSTANCE
                                .createTypedLiteral(literal.getText(), literal
                                        .getDatatype().getFullURI()));
                    } catch (URISyntaxException e) {
                        e.printStackTrace();
                        throw new ParserException("RDF Parser errors: "
                                                  + e.getMessage());
                    } catch (InvalidLexicalFormException e) {
                        e.printStackTrace();
                        throw new ParserException("RDF Parser errors: "
                                                  + e.getMessage());
                    }
            } else if (literal.getLanguage() != null) {
                triple.setObjectLiteral(RDFFactory.eINSTANCE
                        .createPlainLiteral(literal.getText(), literal
                                .getLanguage()));
            } else {
                if (predicate.getFullURI().equals(RDFS.P_COMMENT_STR)
                    || predicate.getFullURI().equals(RDFS.P_LABEL_STR))
                    triple.setObjectLiteral(RDFFactory.eINSTANCE
                            .createPlainLiteral(literal.getText(), null));
                else
                    triple.setObjectLiteral(RDFFactory.eINSTANCE
                            .createRDFSLiteral(literal.getText()));
            }
        }
        try {
            this.graph.addTriple(triple);
        } catch (UnsupportedViewTypeException e) {
            e.printStackTrace();
            throw new ParserException("RDF Parser errors: " + e.getMessage());
        } catch (URISyntaxException e) {
            e.printStackTrace();
            throw new ParserException("RDF Parser errors: " + e.getMessage());
        } catch (AddingTripleException e) {
            e.printStackTrace();
            throw new ParserException("RDF Parser errors: " + e.getMessage());        	
        }

    }

    /**
     * The analyzing process ended.
     */
    public void endAnalyzing() {
    }

    /**
     * Clear all the temporary resourced created during parsing process.
     */
    public void clearResources() {
        // remove all namespacces
        namespaceMap.removeAllNamespaces();
    }
}