/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWLXMLLoader.java,v 1.2 2007/03/18 09:07:04 lzhang Exp $
 */

package org.eclipse.eodm.owl.resource;

import java.io.InputStream;

import org.eclipse.eodm.owl.resource.parser.OWLDocument;
import org.eclipse.eodm.owl.resource.parser.OWLParser;
import org.eclipse.eodm.owl.resource.parser.exception.OWLParserException;
import org.eclipse.eodm.owl.resource.parser.impl.OWLDocumentImpl;
import org.eclipse.eodm.owl.resource.parser.impl.OWLOntologyGraph;
import org.eclipse.eodm.owl.resource.parser.impl.OWLParserImpl;

/**
 * OWLXMLLoader
 * 
 * Load a file/stream in RDF/XML syntax to construct an ODM model.
 * 
 */
public final class OWLXMLLoader {
    private OWLXMLLoader() {
    }

    private static OWLParser parser = new OWLParserImpl();

    /**
     * Add a file to the loader
     * 
     * @param fileName
     *            The OWL file's name
     * @param publicURI
     *            The file's public uri
     * @param localFile
     *            Whether it is a local file
     * @throws OWLParserException
     *             exception when parsing or wrapping OWL file.
     */
    public static void addFile(String fileName, String publicURI,
            boolean localFile) throws OWLParserException {
        addFile(fileName, publicURI, localFile, null);
    }

    /**
     * 
     * Add a file to the loader
     * 
     * @param fileName
     *            The OWL file's name
     * @param publicURI
     *            The file's public uri
     * @param localFile
     *            Whether it is a local file
     * @param charset
     *            The charset of the file
     * @throws OWLParserException
     *             exception when parsing or wrapping OWL file.
     */
    public static void addFile(String fileName, String publicURI,
            boolean localFile, String charset) throws OWLParserException {
        OWLDocument document = new OWLDocumentImpl(fileName, publicURI,
                localFile);
        document.setCharset(charset);
        parser.addOWLDocument(document);
    }

    /**
     * Load a stream to construct an EODM OWL model
     * 
     * @param inputstream
     *            The input stream
     * @throws OWLParserException
     *             exception when parsing or wrapping OWL file.
     */
    public static OWLOntologyGraph[] loadFromStream(InputStream inputstream)
            throws OWLParserException {
        return (loadFromStream(inputstream, null));
    }

    /**
     * Load a stream to construct an EODM OWL model
     * 
     * @param inputstream
     *            The input stream
     * @param charset
     *            The charset of the stream
     * @throws OWLParserException
     *             exception when parsing or wrapping OWL file.
     */
    public static OWLOntologyGraph[] loadFromStream(InputStream inputstream,
            String charset) throws OWLParserException {
        OWLDocument document = new OWLDocumentImpl(inputstream);
        document.setCharset(charset);
        parser.addOWLDocument(document);
        OWLOntologyGraph[] onto = parser.parseOWLDocument(document);
        parser.clear();
        return onto;
    }

    /**
     * Load an EODM model based on the previously added files
     * 
     * @param filename
     *            The name of the file including ontology to be formed
     * @return Ontology graph
     * @throws OWLParserException
     *             exception when parsing or wrapping OWL file.
     */
    public static OWLOntologyGraph[] loadOntology(String filename)
            throws OWLParserException {
        OWLOntologyGraph[] onto = parser.parseOWLDocument(filename);
        return onto;
    }

    /**
     * Load an EODM OWL model from a file
     * 
     * @param filename
     *            The input file's name
     * @param publicURI
     *            The public URI of the file
     * @param localFile
     *            Whether the input is a local file
     * @param charset
     *            The charset of the input
     * @return Ontology graph
     * @throws OWLParserException
     *             exception when parsing or wrapping OWL file.
     */
    public static OWLOntologyGraph[] loadOntologyFromFile(String filename,
            String publicURI, boolean localFile, String charset)
            throws OWLParserException {
        OWLDocument document = new OWLDocumentImpl(filename, publicURI,
                localFile);
        document.setCharset(charset);
        parser.addOWLDocument(document);
        OWLOntologyGraph[] onto = parser.parseOWLDocument(document);
        parser.clear();
        return onto;
    }

    /**
     * Release the parser and clear all resources used
     * 
     */
    public static void clear() {
        parser.clear();
    }
}