package org.eclipse.incquery.tooling.core.generator.util;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.regex.Matcher;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.patternlanguage.patternLanguage.PatternModel;
import org.eclipse.incquery.patternlanguage.patternLanguage.Variable;
import org.eclipse.incquery.tooling.core.generator.builder.GeneratorIssueCodes;
import org.eclipse.incquery.tooling.core.generator.builder.IErrorFeedback;
import org.eclipse.incquery.tooling.core.generator.util.EMFJvmTypesBuilder;
import org.eclipse.jdt.core.JavaConventions;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.xbase.compiler.TypeReferenceSerializer;
import org.eclipse.xtext.xbase.compiler.output.ITreeAppendable;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.eclipse.xtext.xbase.typing.ITypeProvider;

/**
 * Utility class for the EMFPatternLanguageJvmModelInferrer.
 * 
 * @author Mark Czotter
 */
@SuppressWarnings("all")
public class EMFPatternLanguageJvmModelInferrerUtil {
  @Inject
  private EMFJvmTypesBuilder _eMFJvmTypesBuilder;
  
  private Logger logger = new Function0<Logger>() {
    public Logger apply() {
      Class<? extends Object> _class = EMFPatternLanguageJvmModelInferrerUtil.this.getClass();
      Logger _logger = Logger.getLogger(_class);
      return _logger;
    }
  }.apply();
  
  private String MULTILINE_COMMENT_PATTERN = "(/\\*([^*]|[\\r\\n]|(\\*+([^*/]|[\\r\\n])))*\\*+/)";
  
  @Inject
  private IWorkspaceRoot workspaceRoot;
  
  @Inject
  private ITypeProvider typeProvider;
  
  @Inject
  private TypeReferenceSerializer typeReferenceSerializer;
  
  @Inject
  private IErrorFeedback errorFeedback;
  
  public String bundleName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      Resource _eResource = pattern.eResource();
      URI _uRI = _eResource.getURI();
      String _platformString = _uRI.toPlatformString(true);
      Path _path = new Path(_platformString);
      IFile _file = this.workspaceRoot.getFile(_path);
      final IProject project = _file.getProject();
      String _name = project.getName();
      _xblockexpression = (_name);
    }
    return _xblockexpression;
  }
  
  /**
   * This method returns the pattern name.
   * If the pattern name contains the package (any dot),
   * then removes all segment except the last one.
   */
  public String realPatternName(final Pattern pattern) {
    String name = pattern.getName();
    boolean _contains = name.contains(".");
    if (_contains) {
      int _lastIndexOf = name.lastIndexOf(".");
      int _plus = (_lastIndexOf + 1);
      return name.substring(_plus);
    }
    return name;
  }
  
  public String modelFileName(final EObject object) {
    String _xblockexpression = null;
    {
      Resource _eResource = object.eResource();
      URI _uRI = _eResource==null?(URI)null:_eResource.getURI();
      URI _trimFileExtension = _uRI.trimFileExtension();
      final String name = _trimFileExtension.lastSegment();
      final IStatus status = JavaConventions.validateJavaTypeName(name, JavaCore.VERSION_1_6, JavaCore.VERSION_1_6);
      int _severity = status.getSeverity();
      boolean _equals = (_severity == IStatus.ERROR);
      if (_equals) {
        String _plus = ("The file name " + name);
        String _plus_1 = (_plus + " is not a valid Java type name. Please, rename the file!");
        IllegalArgumentException _illegalArgumentException = new IllegalArgumentException(_plus_1);
        throw _illegalArgumentException;
      }
      _xblockexpression = (name);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the QuerySpecificationClass name based on the Pattern's name
   */
  public String querySpecificationClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      String _plus = (_firstUpper + "QuerySpecification");
      _xblockexpression = (_plus);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the IQuerySpecificationProvider class name based on the Pattern's name
   */
  public String querySpecificationProviderClassName(final Pattern pattern) {
    return "Provider";
  }
  
  /**
   * Returns the holder class name based on the Pattern's name
   */
  public String querySpecificationHolderClassName(final Pattern pattern) {
    return "LazyHolder";
  }
  
  /**
   * Returns the MatcherClass name based on the Pattern's name
   */
  public String matcherClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      String _plus = (_firstUpper + "Matcher");
      _xblockexpression = (_plus);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the MatchClass name based on the Pattern's name
   */
  public String matchClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      String _plus = (_firstUpper + "Match");
      _xblockexpression = (_plus);
    }
    return _xblockexpression;
  }
  
  public String matchImmutableInnerClassName(final Pattern pattern) {
    return "Immutable";
  }
  
  public String matchMutableInnerClassName(final Pattern pattern) {
    return "Mutable";
  }
  
  /**
   * Returns the ProcessorClass name based on the Pattern's name
   */
  public String processorClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      String _plus = (_firstUpper + "Processor");
      _xblockexpression = (_plus);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the EvaluatorClass name based on the Pattern's name
   */
  public String evaluatorClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      String _plus = (_firstUpper + "Evaluator");
      _xblockexpression = (_plus);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns field name for Variable
   */
  public String fieldName(final Variable variable) {
    String _name = variable==null?(String)null:variable.getName();
    String _firstUpper = StringExtensions.toFirstUpper(_name);
    String _plus = ("f" + _firstUpper);
    return _plus;
  }
  
  /**
   * Returns parameter name for Variable
   */
  public String parameterName(final Variable variable) {
    String _name = variable==null?(String)null:variable.getName();
    String _firstUpper = _name==null?(String)null:StringExtensions.toFirstUpper(_name);
    String _plus = ("p" + _firstUpper);
    return _plus;
  }
  
  public String positionConstant(final Variable variable) {
    String _name = variable==null?(String)null:variable.getName();
    String _upperCase = _name==null?(String)null:_name.toUpperCase();
    String _plus = ("POSITION_" + _upperCase);
    return _plus;
  }
  
  /**
   * Returns correct getter method name for variable.
   * For variable with name 'class' returns getValueOfClass, otherwise returns <code>get#variable.name.toFirstUpper#</code>.
   */
  public String getterMethodName(final Variable variable) {
    String _name = variable.getName();
    boolean _equals = Objects.equal(_name, "class");
    if (_equals) {
      return "getValueOfClass";
    } else {
      String _name_1 = variable==null?(String)null:variable.getName();
      String _firstUpper = _name_1==null?(String)null:StringExtensions.toFirstUpper(_name_1);
      return ("get" + _firstUpper);
    }
  }
  
  /**
   * Returns correct setter method name for variable.
   * Currently returns <code>set#variable.name.toFirstUpper#</code>.
   */
  public String setterMethodName(final Variable variable) {
    String _name = variable==null?(String)null:variable.getName();
    String _firstUpper = _name==null?(String)null:StringExtensions.toFirstUpper(_name);
    String _plus = ("set" + _firstUpper);
    return _plus;
  }
  
  /**
   * Calls the typeProvider.
   * See the XBaseUsageCrossReferencer class, possible solution for local variable usage
   * TODO: improve type calculation
   * @return JvmTypeReference pointing the EClass that defines the Variable's type.
   * @see ITypeProvider
   * @see EMFPatternTypeProvider
   */
  public JvmTypeReference calculateType(final Variable variable) {
    JvmTypeReference _typeForIdentifiable = this.typeProvider.getTypeForIdentifiable(variable);
    return _typeForIdentifiable;
  }
  
  /**
   * Serializes the EObject into Java String variable.
   */
  public CharSequence serializeToJava(final EObject eObject) {
    final String parseString = this.serialize(eObject);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(parseString);
    if (_isNullOrEmpty) {
      return "";
    }
    final String[] splits = parseString.split("[\r\n]+");
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("String patternString = \"\"");
    final StringConcatenation stringRep = ((StringConcatenation) _builder);
    stringRep.newLine();
    for (final String s : splits) {
      {
        String _plus = ("+\" " + s);
        String _plus_1 = (_plus + " \"");
        stringRep.append(_plus_1);
        stringRep.newLine();
      }
    }
    stringRep.append(";");
    return stringRep;
  }
  
  /**
   * Serializes the input for Javadoc
   */
  public String serializeToJavadoc(final Pattern pattern) {
    String javadocString = this.serialize(pattern);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(javadocString);
    if (_isNullOrEmpty) {
      return "Serialization error, check Log";
    }
    String _quote = java.util.regex.Pattern.quote("\\\"");
    String _quoteReplacement = Matcher.quoteReplacement("\"");
    String _replaceAll = javadocString.replaceAll(_quote, _quoteReplacement);
    javadocString = _replaceAll;
    String _replaceAll_1 = javadocString.replaceAll("@", "{@literal @}");
    javadocString = _replaceAll_1;
    String _replaceAll_2 = javadocString.replaceAll("<", "{@literal <}");
    javadocString = _replaceAll_2;
    String _replaceAll_3 = javadocString.replaceAll(">", "{@literal >}");
    javadocString = _replaceAll_3;
    return javadocString.trim();
  }
  
  /**
   * Serializes EObject to a String representation. Escapes only the double qoutes.
   */
  private String serialize(final EObject eObject) {
    try {
      final ICompositeNode eObjectNode = NodeModelUtils.getNode(eObject);
      boolean _notEquals = (!Objects.equal(eObjectNode, null));
      if (_notEquals) {
        String _text = eObjectNode.getText();
        return this.escape(_text);
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        boolean _notEquals_1 = (!Objects.equal(this.logger, null));
        if (_notEquals_1) {
          EClass _eClass = eObject.eClass();
          String _name = _eClass.getName();
          String _plus = ("Error when serializing " + _name);
          this.logger.error(_plus, e);
        }
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return null;
  }
  
  private String escape(final String escapable) {
    boolean _equals = Objects.equal(escapable, null);
    if (_equals) {
      return null;
    }
    String escapedString = escapable.replaceAll("\"", "\\\\\"");
    String _replaceAll = escapedString.replaceAll(this.MULTILINE_COMMENT_PATTERN, " ");
    escapedString = _replaceAll;
    return escapedString;
  }
  
  /**
   * Returns the packageName: PatternModel.packageName or "" when nullOrEmpty.
   */
  public String getPackageName(final Pattern pattern) {
    EObject _eContainer = pattern.eContainer();
    String packageName = ((PatternModel) _eContainer).getPackageName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(packageName);
    if (_isNullOrEmpty) {
      packageName = "";
    }
    return packageName.toLowerCase();
  }
  
  public String getUtilPackageName(final Pattern pattern) {
    String _packageName = this.getPackageName(pattern);
    return (_packageName + ".util");
  }
  
  /**
   * Returns the packageName: PatternModel.packageName + Pattern.name, packageName is ignored, when nullOrEmpty.
   */
  public String getPackageNameOld(final Pattern pattern) {
    EObject _eContainer = pattern.eContainer();
    String packageName = ((PatternModel) _eContainer).getPackageName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(packageName);
    if (_isNullOrEmpty) {
      packageName = "";
    } else {
      String _plus = (packageName + ".");
      packageName = _plus;
    }
    String _name = pattern.getName();
    String _plus_1 = (packageName + _name);
    return _plus_1.toLowerCase();
  }
  
  public String getPackagePath(final Pattern pattern) {
    String _packageName = this.getPackageName(pattern);
    String _replace = _packageName.replace(".", "/");
    return _replace;
  }
  
  public ITreeAppendable referClass(final ITreeAppendable appendable, final EObject ctx, final Class<? extends Object> clazz, final JvmTypeReference... typeArgs) {
    ITreeAppendable _xblockexpression = null;
    {
      final JvmTypeReference ref = this._eMFJvmTypesBuilder.newTypeRef(ctx, clazz, typeArgs);
      ITreeAppendable _xifexpression = null;
      boolean _notEquals = (!Objects.equal(ref, null));
      if (_notEquals) {
        this.serialize(appendable, ref, ctx);
      } else {
        ITreeAppendable _xblockexpression_1 = null;
        {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Cannot resolve class ");
          String _canonicalName = clazz.getCanonicalName();
          _builder.append(_canonicalName, "");
          _builder.append(". Check project dependencies.");
          this.errorFeedback.reportError(ctx, _builder.toString(), GeneratorIssueCodes.INVALID_TYPEREF_CODE, Severity.ERROR, IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
          String _canonicalName_1 = clazz.getCanonicalName();
          ITreeAppendable _append = appendable.append(_canonicalName_1);
          _xblockexpression_1 = (_append);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = (_xifexpression);
    }
    return _xblockexpression;
  }
  
  public void serialize(final ITreeAppendable appendable, final JvmTypeReference ref, final EObject ctx) {
    this.typeReferenceSerializer.serialize(ref, ctx, appendable);
  }
}
