/**
 * Copyright (c) 2010-2015, Grill Balázs, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Grill Balázs - initial API and implementation
 */
package org.eclipse.incquery.testing.core;

import com.google.common.base.Objects;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import junit.framework.AssertionFailedError;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.incquery.runtime.api.IMatchProcessor;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IQueryGroup;
import org.eclipse.incquery.runtime.api.IQuerySpecification;
import org.eclipse.incquery.runtime.api.IncQueryEngine;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.runtime.emf.EMFScope;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchRecord;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchSetRecord;
import org.eclipse.incquery.testing.core.IMatchSetModelProvider;
import org.eclipse.incquery.testing.core.SnapshotHelper;
import org.eclipse.incquery.testing.queries.UnexpectedMatchRecordMatch;
import org.eclipse.incquery.testing.queries.UnexpectedMatchRecordMatcher;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * @author Grill Balázs
 */
@SuppressWarnings("all")
public class EIQTestCase {
  public final static String UNEXPECTED_MATCH = "Unexpected match";
  
  public final static String EXPECTED_NOT_FOUND = "Expected match not found";
  
  private final ResourceSet resourceSet;
  
  private final List<IMatchSetModelProvider> modelProviders;
  
  @Extension
  private SnapshotHelper _snapshotHelper = new SnapshotHelper();
  
  public EIQTestCase() {
    ResourceSetImpl _resourceSetImpl = new ResourceSetImpl();
    this.resourceSet = _resourceSetImpl;
    LinkedList<IMatchSetModelProvider> _linkedList = new LinkedList<IMatchSetModelProvider>();
    this.modelProviders = _linkedList;
  }
  
  public Resource loadModel(final URI uri) {
    return this.resourceSet.getResource(uri, true);
  }
  
  public boolean addMatchSetModelProvider(final IMatchSetModelProvider matchSetModelProvider) {
    return this.modelProviders.add(matchSetModelProvider);
  }
  
  public <Match extends IPatternMatch> void assertMatchSetsEqual(final IQuerySpecification<? extends IncQueryMatcher<Match>> querySpecification) {
    int _size = this.modelProviders.size();
    boolean _lessThan = (_size < 2);
    if (_lessThan) {
      throw new IllegalArgumentException("At least two model providers shall be set");
    }
    final IMatchSetModelProvider reference = IterableExtensions.<IMatchSetModelProvider>head(this.modelProviders);
    Iterable<IMatchSetModelProvider> _tail = IterableExtensions.<IMatchSetModelProvider>tail(this.modelProviders);
    final Procedure1<IMatchSetModelProvider> _function = new Procedure1<IMatchSetModelProvider>() {
      @Override
      public void apply(final IMatchSetModelProvider it) {
        final HashSet<String> diff = EIQTestCase.this.<Match>compareMatchSets(querySpecification, reference, it);
        boolean _isEmpty = diff.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          String _string = diff.toString();
          throw new AssertionFailedError(_string);
        }
      }
    };
    IterableExtensions.<IMatchSetModelProvider>forEach(_tail, _function);
  }
  
  public void assertMatchSetsEqual(final IQueryGroup queryGroup) {
    Set<IQuerySpecification<?>> _specifications = queryGroup.getSpecifications();
    final Procedure1<IQuerySpecification<?>> _function = new Procedure1<IQuerySpecification<?>>() {
      @Override
      public void apply(final IQuerySpecification<?> it) {
        EIQTestCase.this.<IPatternMatch>assertMatchSetsEqual(((IQuerySpecification<? extends IncQueryMatcher<IPatternMatch>>) it));
      }
    };
    IterableExtensions.<IQuerySpecification<?>>forEach(_specifications, _function);
  }
  
  private <Match extends IPatternMatch> HashSet<String> compareMatchSets(final IQuerySpecification<? extends IncQueryMatcher<Match>> querySpecification, final IMatchSetModelProvider expectedProvider, final IMatchSetModelProvider actualProvider) {
    try {
      EMFScope _eMFScope = new EMFScope(this.resourceSet);
      final IncQueryEngine engine = IncQueryEngine.on(_eMFScope);
      IQuerySpecification<UnexpectedMatchRecordMatcher> _querySpecification = UnexpectedMatchRecordMatcher.querySpecification();
      final UnexpectedMatchRecordMatcher unexpectedMatcher = _querySpecification.getMatcher(engine);
      final HashSet<String> diff = CollectionLiterals.<String>newHashSet();
      Match filter = null;
      MatchSetRecord expected = expectedProvider.<Match>getMatchSetRecord(this.resourceSet, querySpecification, filter);
      MatchRecord _filter = expected.getFilter();
      boolean _notEquals = (!Objects.equal(_filter, null));
      if (_notEquals) {
        MatchRecord _filter_1 = expected.getFilter();
        Match _createMatchForMatchRecord = this._snapshotHelper.<Match>createMatchForMatchRecord(querySpecification, _filter_1);
        filter = _createMatchForMatchRecord;
      }
      final MatchSetRecord actual = actualProvider.<Match>getMatchSetRecord(this.resourceSet, querySpecification, filter);
      MatchRecord _filter_2 = actual.getFilter();
      boolean _notEquals_1 = (!Objects.equal(_filter_2, null));
      if (_notEquals_1) {
        boolean _notEquals_2 = (!Objects.equal(filter, null));
        if (_notEquals_2) {
          throw new IllegalArgumentException(
            ((("Filter is provided by more than one sources: " + expectedProvider) + ", ") + actualProvider));
        } else {
          MatchRecord _filter_3 = actual.getFilter();
          Match _createMatchForMatchRecord_1 = this._snapshotHelper.<Match>createMatchForMatchRecord(querySpecification, _filter_3);
          filter = _createMatchForMatchRecord_1;
          MatchSetRecord _matchSetRecord = expectedProvider.<Match>getMatchSetRecord(this.resourceSet, querySpecification, filter);
          expected = _matchSetRecord;
        }
      }
      final IMatchProcessor<UnexpectedMatchRecordMatch> _function = new IMatchProcessor<UnexpectedMatchRecordMatch>() {
        @Override
        public void process(final UnexpectedMatchRecordMatch it) {
          String _prettyPrint = it.prettyPrint();
          String _plus = ((EIQTestCase.UNEXPECTED_MATCH + " (") + _prettyPrint);
          String _plus_1 = (_plus + ")");
          diff.add(_plus_1);
        }
      };
      unexpectedMatcher.forEachMatch(actual, expected, null, _function);
      final IMatchProcessor<UnexpectedMatchRecordMatch> _function_1 = new IMatchProcessor<UnexpectedMatchRecordMatch>() {
        @Override
        public void process(final UnexpectedMatchRecordMatch it) {
          String _prettyPrint = it.prettyPrint();
          String _plus = ((EIQTestCase.EXPECTED_NOT_FOUND + " (") + _prettyPrint);
          String _plus_1 = (_plus + ")");
          diff.add(_plus_1);
        }
      };
      unexpectedMatcher.forEachMatch(expected, actual, null, _function_1);
      return diff;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
