/**
 * Copyright (c) 2012 BestSolution.at and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tom Schindl<tom.schindl@bestsolution.at> - initial API and implementation
 */
package org.eclipse.fx.formats.svg.converter;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.fx.formats.svg.converter.Colors;
import org.eclipse.fx.formats.svg.svg.ContentElement;
import org.eclipse.fx.formats.svg.svg.CoreAttributes;
import org.eclipse.fx.formats.svg.svg.Fill_rule;
import org.eclipse.fx.formats.svg.svg.FilterPrimitiveElement;
import org.eclipse.fx.formats.svg.svg.PresentationAttributes;
import org.eclipse.fx.formats.svg.svg.SpreadMethod;
import org.eclipse.fx.formats.svg.svg.Stroke_linecap;
import org.eclipse.fx.formats.svg.svg.Stroke_linejoin;
import org.eclipse.fx.formats.svg.svg.SvgCircleElement;
import org.eclipse.fx.formats.svg.svg.SvgClipPathElement;
import org.eclipse.fx.formats.svg.svg.SvgDefsElement;
import org.eclipse.fx.formats.svg.svg.SvgElement;
import org.eclipse.fx.formats.svg.svg.SvgEllipseElement;
import org.eclipse.fx.formats.svg.svg.SvgFeGaussianBlurElement;
import org.eclipse.fx.formats.svg.svg.SvgFilterElement;
import org.eclipse.fx.formats.svg.svg.SvgGElement;
import org.eclipse.fx.formats.svg.svg.SvgImageElement;
import org.eclipse.fx.formats.svg.svg.SvgLinearGradientElement;
import org.eclipse.fx.formats.svg.svg.SvgPackage;
import org.eclipse.fx.formats.svg.svg.SvgPathElement;
import org.eclipse.fx.formats.svg.svg.SvgPolygonElement;
import org.eclipse.fx.formats.svg.svg.SvgPolylineElement;
import org.eclipse.fx.formats.svg.svg.SvgRadialGradientElement;
import org.eclipse.fx.formats.svg.svg.SvgRectElement;
import org.eclipse.fx.formats.svg.svg.SvgStopElement;
import org.eclipse.fx.formats.svg.svg.SvgSvgElement;
import org.eclipse.fx.formats.svg.svg.SvgUseElement;
import org.eclipse.fx.formats.svg.svg.XLinkAttributes;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class FXMLConverter {
  private SvgSvgElement rootElement;
  
  public FXMLConverter(final SvgSvgElement rootElement) {
    this.rootElement = rootElement;
  }
  
  public CharSequence generate() {
    StringConcatenation _builder = new StringConcatenation();
    final CharSequence content = this.handle(this.rootElement);
    _builder.newLineIfNotEmpty();
    _builder.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("<?import java.lang.*?>");
    _builder.newLine();
    _builder.append("<?import javafx.scene.*?>");
    _builder.newLine();
    _builder.append("<?import javafx.scene.shape.*?>");
    _builder.newLine();
    _builder.append("<?import javafx.scene.paint.*?>");
    _builder.newLine();
    _builder.append("<?import javafx.scene.image.*?>");
    _builder.newLine();
    _builder.append("<?import javafx.scene.transform.*?>");
    _builder.newLine();
    _builder.append("<?import javafx.scene.effect.*?>");
    _builder.newLine();
    _builder.newLine();
    _builder.append(content);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _handle(final EObject o) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<!-- Unhandled type \'");
    String _name = o.eClass().getName();
    _builder.append(_name);
    _builder.append("\' -->");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _handle(final SvgLinearGradientElement o) {
    return null;
  }
  
  protected CharSequence _handle(final SvgRadialGradientElement o) {
    return null;
  }
  
  protected CharSequence _handle(final SvgSvgElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Group id=\'_root\' xmlns:fx=\"http://javafx.com/fxml\" xmlns:fxsvg=\"http://efxclipse.org/fxml-svg\">");
    _builder.newLine();
    _builder.append("\t");
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<children>");
    _builder.newLine();
    {
      EList<SvgElement> _children = element.getChildren();
      for(final SvgElement o : _children) {
        _builder.append("\t\t");
        CharSequence _handle = this.handle(o);
        _builder.append(_handle, "\t\t");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    _builder.append("</children>");
    _builder.newLine();
    _builder.append("</Group>");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handle(final SvgDefsElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<fx:define>");
    _builder.newLine();
    {
      EList<SvgElement> _children = element.getChildren();
      for(final SvgElement o : _children) {
        _builder.append("\t");
        CharSequence _handle = this.handle(o);
        _builder.append(_handle, "\t");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("</fx:define>");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence handleGradient(final SvgLinearGradientElement element, final Double opacity) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<LinearGradient");
    _builder.newLine();
    _builder.append("\t");
    AffineTransform _createAffineTransform = this.createAffineTransform(element.getGradientTransform());
    final AffineTransform t = ((AffineTransform) _createAffineTransform);
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    double x1 = this.parseCoordinate(element.getX1());
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    double y1 = this.parseCoordinate(element.getY1());
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    double x2 = this.parseCoordinate(element.getX2());
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    double y2 = this.parseCoordinate(element.getY2());
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    {
      String _x1 = element.getX1();
      boolean _notEquals = (!Objects.equal(_x1, null));
      if (_notEquals) {
        _builder.append("startX=\"");
        Point2D.Double _double = new Point2D.Double(x1, y1);
        double _x = t.transform(_double, null).getX();
        _builder.append(_x, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _y1 = element.getY1();
      boolean _notEquals_1 = (!Objects.equal(_y1, null));
      if (_notEquals_1) {
        _builder.append("startY=\"");
        Point2D.Double _double_1 = new Point2D.Double(x1, y1);
        double _y = t.transform(_double_1, null).getY();
        _builder.append(_y, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _x2 = element.getX2();
      boolean _notEquals_2 = (!Objects.equal(_x2, null));
      if (_notEquals_2) {
        _builder.append("endX=\"");
        Point2D.Double _double_2 = new Point2D.Double(x2, y2);
        double _x_1 = t.transform(_double_2, null).getX();
        _builder.append(_x_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _y2 = element.getY2();
      boolean _notEquals_3 = (!Objects.equal(_y2, null));
      if (_notEquals_3) {
        _builder.append("endY=\"");
        Point2D.Double _double_3 = new Point2D.Double(x2, y2);
        double _y_1 = t.transform(_double_3, null).getY();
        _builder.append(_y_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      SpreadMethod _spreadMethod = element.getSpreadMethod();
      boolean _notEquals_4 = (!Objects.equal(_spreadMethod, SpreadMethod.PAD));
      if (_notEquals_4) {
        _builder.append("cycleMethod=\"");
        String _fx = this.toFx(element.getSpreadMethod());
        _builder.append(_fx, "\t");
        _builder.append("\"");
      } else {
        _builder.append("cycleMethod=\"NO_CYCLE\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("proportional=\"false\">");
    _builder.newLine();
    _builder.append("\t");
    final ContentElement<?> owner = this.resolveGradientStopElement(element);
    _builder.newLineIfNotEmpty();
    {
      boolean _notEquals_5 = (!Objects.equal(owner, null));
      if (_notEquals_5) {
        _builder.append("\t");
        _builder.append("<stops>");
        _builder.newLine();
        {
          Iterable<SvgStopElement> _filter = Iterables.<SvgStopElement>filter(((ContentElement<?>) owner).getChildren(), SvgStopElement.class);
          for(final SvgStopElement o : _filter) {
            _builder.append("\t");
            CharSequence _handleStop = this.handleStop(o, opacity);
            _builder.append(_handleStop, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("</stops>");
        _builder.newLine();
      }
    }
    _builder.append("</LinearGradient>");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence handleGradient(final SvgRadialGradientElement element, final Double opacity) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<RadialGradient");
    _builder.newLine();
    _builder.append("\t");
    AffineTransform _createAffineTransform = this.createAffineTransform(element.getGradientTransform());
    final AffineTransform t = ((AffineTransform) _createAffineTransform);
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    double cx = this.parseCoordinate(element.getCx());
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    double cy = this.parseCoordinate(element.getCy());
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    double fx = this.parseCoordinate(element.getFx());
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    double fy = this.parseCoordinate(element.getFy());
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    {
      String _cx = element.getCx();
      boolean _notEquals = (!Objects.equal(_cx, null));
      if (_notEquals) {
        _builder.append("centerX=\"");
        Point2D.Double _double = new Point2D.Double(cx, cy);
        double _x = t.transform(_double, null).getX();
        _builder.append(_x, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _cy = element.getCy();
      boolean _notEquals_1 = (!Objects.equal(_cy, null));
      if (_notEquals_1) {
        _builder.append("centerY=\"");
        Point2D.Double _double_1 = new Point2D.Double(cx, cy);
        double _y = t.transform(_double_1, null).getY();
        _builder.append(_y, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _r = element.getR();
      boolean _notEquals_2 = (!Objects.equal(_r, null));
      if (_notEquals_2) {
        _builder.append("radius=\"");
        double _parseLength = this.parseLength(element.getR());
        double _scaleX = t.getScaleX();
        double _multiply = (_parseLength * _scaleX);
        _builder.append(_multiply, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      if (((!Objects.equal(element.getFx(), null)) || (!Objects.equal(element.getFy(), null)))) {
        _builder.append("focusDistance=\"");
        double _calculateFocusDistance = this.calculateFocusDistance(t, cx, cy, fx, fy);
        _builder.append(_calculateFocusDistance, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      if (((!Objects.equal(element.getFx(), null)) || (!Objects.equal(element.getFy(), null)))) {
        _builder.append("focusAngle=\"");
        double _calculateFocusAngle = this.calculateFocusAngle(t, cx, cy, fx, fy);
        _builder.append(_calculateFocusAngle, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      SpreadMethod _spreadMethod = element.getSpreadMethod();
      boolean _notEquals_3 = (!Objects.equal(_spreadMethod, SpreadMethod.PAD));
      if (_notEquals_3) {
        _builder.append("cycleMethod=\"");
        String _fx = this.toFx(element.getSpreadMethod());
        _builder.append(_fx, "\t");
        _builder.append("\"");
      } else {
        _builder.append("cycleMethod=\"NO_CYCLE\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("proportional=\"false\">");
    _builder.newLine();
    _builder.append("\t");
    final ContentElement<?> owner = this.resolveGradientStopElement(element);
    _builder.newLineIfNotEmpty();
    {
      boolean _notEquals_4 = (!Objects.equal(owner, null));
      if (_notEquals_4) {
        _builder.append("\t");
        _builder.append("<stops>");
        _builder.newLine();
        {
          Iterable<SvgStopElement> _filter = Iterables.<SvgStopElement>filter(((ContentElement<?>) owner).getChildren(), SvgStopElement.class);
          for(final SvgStopElement o : _filter) {
            _builder.append("\t");
            CharSequence _handleStop = this.handleStop(o, opacity);
            _builder.append(_handleStop, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("</stops>");
        _builder.newLine();
      }
    }
    _builder.append("</RadialGradient>");
    _builder.newLine();
    return _builder;
  }
  
  public AffineTransform createAffineTransform(final String transformSpec) {
    boolean _equals = Objects.equal(transformSpec, null);
    if (_equals) {
      return new AffineTransform();
    } else {
      int _indexOf = transformSpec.indexOf("(");
      int _plus = (_indexOf + 1);
      final String params = transformSpec.substring(_plus, transformSpec.indexOf(")"));
      boolean _startsWith = transformSpec.startsWith("matrix");
      if (_startsWith) {
        final String[] parts = params.split("\\s*,\\s*|\\s+");
        double _parseDouble = Double.parseDouble(parts[0]);
        double _parseDouble_1 = Double.parseDouble(parts[1]);
        double _parseDouble_2 = Double.parseDouble(parts[2]);
        double _parseDouble_3 = Double.parseDouble(parts[3]);
        double _parseDouble_4 = Double.parseDouble(parts[4]);
        double _parseDouble_5 = Double.parseDouble(parts[5]);
        final AffineTransform rv = new AffineTransform(_parseDouble, _parseDouble_1, _parseDouble_2, _parseDouble_3, _parseDouble_4, _parseDouble_5);
        return rv;
      } else {
        boolean _startsWith_1 = transformSpec.startsWith("translate");
        if (_startsWith_1) {
          final String[] parts_1 = params.split("\\s*,\\s*|\\s+");
          final AffineTransform rv_1 = new AffineTransform();
          rv_1.translate(
            Double.parseDouble(parts_1[0]), 
            Double.parseDouble(parts_1[1]));
          return rv_1;
        } else {
          boolean _startsWith_2 = transformSpec.startsWith("scale");
          if (_startsWith_2) {
            final String[] parts_2 = params.split("\\s*,\\s*|\\s+");
            final AffineTransform rv_2 = new AffineTransform();
            rv_2.scale(
              Double.parseDouble(parts_2[0]), 
              Double.parseDouble(parts_2[1]));
            return rv_2;
          } else {
            boolean _startsWith_3 = transformSpec.startsWith("rotate");
            if (_startsWith_3) {
              final String[] parts_3 = params.split("\\s*,\\s*|\\s+");
              final AffineTransform rv_3 = new AffineTransform();
              rv_3.rotate(
                Double.parseDouble(parts_3[0]), 
                Double.parseDouble(parts_3[1]), 
                Double.parseDouble(parts_3[2]));
              return rv_3;
            }
          }
        }
      }
    }
    return new AffineTransform();
  }
  
  public double calculateFocusDistance(final AffineTransform transform, final double cx, final double cy, final double fx, final double fy) {
    if (((fx == cx) && (fy == cy))) {
      return 0;
    } else {
      Point2D.Double _double = new Point2D.Double(cx, cy);
      final Point2D c = transform.transform(_double, null);
      Point2D.Double _double_1 = new Point2D.Double(fx, fy);
      final Point2D f = transform.transform(_double_1, null);
      double _x = c.getX();
      double _x_1 = f.getX();
      double _minus = (_x - _x_1);
      double _pow = Math.pow(_minus, 2);
      double _y = c.getY();
      double _y_1 = f.getY();
      double _minus_1 = (_y - _y_1);
      double _pow_1 = Math.pow(_minus_1, 2);
      double _plus = (_pow + _pow_1);
      return Math.sqrt(_plus);
    }
  }
  
  public double calculateFocusAngle(final AffineTransform transform, final double cx, final double cy, final double fx, final double fy) {
    if (((fx == cx) && (fy == cy))) {
      return 0;
    } else {
      Point2D.Double _double = new Point2D.Double(cx, cy);
      final Point2D c = transform.transform(_double, null);
      Point2D.Double _double_1 = new Point2D.Double(fx, fy);
      final Point2D f = transform.transform(_double_1, null);
      double _y = c.getY();
      double _y_1 = f.getY();
      double _minus = (_y - _y_1);
      double _pow = Math.pow(_minus, 2);
      double _x = c.getX();
      double _x_1 = f.getX();
      double _minus_1 = (_x - _x_1);
      double _pow_1 = Math.pow(_minus_1, 2);
      double _divide = (_pow / _pow_1);
      return Math.atan(_divide);
    }
  }
  
  public ContentElement<?> resolveGradientStopElement(final SvgElement element) {
    if ((element instanceof ContentElement<?>)) {
      final ContentElement<?> rv = ((ContentElement<?>) element);
      boolean _isEmpty = IterableExtensions.isEmpty(Iterables.<SvgStopElement>filter(rv.getChildren(), SvgStopElement.class));
      if (_isEmpty) {
        if ((element instanceof XLinkAttributes)) {
          final XLinkAttributes v = ((XLinkAttributes) element);
          SvgElement _resolvedInstance = v.getResolvedInstance();
          boolean _notEquals = (!Objects.equal(_resolvedInstance, null));
          if (_notEquals) {
            return this.resolveGradientStopElement(v.getResolvedInstance());
          }
        }
      } else {
        return rv;
      }
    }
    return null;
  }
  
  public SvgElement resolveElement(final String id) {
    final Iterator<SvgElement> treeIt = Iterators.<SvgElement>filter(EcoreUtil.<Object>getAllContents(this.rootElement, true), SvgElement.class);
    while (treeIt.hasNext()) {
      {
        final SvgElement o = treeIt.next();
        if ((o instanceof CoreAttributes)) {
          boolean _equals = id.equals(((CoreAttributes) o).getId());
          if (_equals) {
            return ((SvgElement) o);
          }
        } else {
          final SvgElement eo = ((SvgElement) o);
          final EStructuralFeature f = eo.eClass().getEStructuralFeature("id");
          if (((!Objects.equal(f, null)) && id.equals(eo.eGet(f)))) {
            return o;
          }
        }
      }
    }
    return null;
  }
  
  public CharSequence handleStop(final SvgStopElement element, final Double opacity) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Stop");
    _builder.newLine();
    _builder.append("\t");
    {
      String _offset = element.getOffset();
      boolean _notEquals = (!Objects.equal(_offset, null));
      if (_notEquals) {
        _builder.append("offset=\"");
        double _parsePercentage = this.parsePercentage(element.getOffset());
        _builder.append(_parsePercentage, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(">");
    _builder.newLine();
    {
      String _stop_opacity = element.getStop_opacity();
      boolean _notEquals_1 = (!Objects.equal(_stop_opacity, null));
      if (_notEquals_1) {
        _builder.append("\t");
        _builder.append("<color>");
        _builder.newLine();
        {
          boolean _notEquals_2 = (!Objects.equal(opacity, null));
          if (_notEquals_2) {
            {
              String _stop_color = element.getStop_color();
              boolean _notEquals_3 = (!Objects.equal(_stop_color, null));
              if (_notEquals_3) {
                _builder.append("\t");
                _builder.append("\t");
                String _stop_color_1 = element.getStop_color();
                double _parseDouble = Double.parseDouble(element.getStop_opacity());
                double _multiply = (_parseDouble * (opacity).doubleValue());
                CharSequence _fillPaint = this.fillPaint(_stop_color_1, Double.valueOf(_multiply));
                _builder.append(_fillPaint, "\t\t");
                _builder.newLineIfNotEmpty();
              } else {
                _builder.append("\t");
                _builder.append("\t");
                _builder.append("<opacity>");
                double _parseDouble_1 = Double.parseDouble(element.getStop_opacity());
                double _multiply_1 = (_parseDouble_1 * (opacity).doubleValue());
                _builder.append(_multiply_1, "\t\t");
                _builder.append("</opacity>");
                _builder.newLineIfNotEmpty();
              }
            }
          } else {
            {
              String _stop_color_2 = element.getStop_color();
              boolean _notEquals_4 = (!Objects.equal(_stop_color_2, null));
              if (_notEquals_4) {
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _fillPaint_1 = this.fillPaint(element.getStop_color(), Double.valueOf(Double.parseDouble(element.getStop_opacity())));
                _builder.append(_fillPaint_1, "\t\t");
                _builder.newLineIfNotEmpty();
              } else {
                _builder.append("\t");
                _builder.append("\t");
                _builder.append("<opacity>");
                String _stop_opacity_1 = element.getStop_opacity();
                _builder.append(_stop_opacity_1, "\t\t");
                _builder.append("</opacity>");
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
        _builder.append("\t");
        _builder.append("</color>");
        _builder.newLine();
      } else {
        String _stop_color_3 = element.getStop_color();
        boolean _notEquals_5 = (!Objects.equal(_stop_color_3, null));
        if (_notEquals_5) {
          _builder.append("\t");
          _builder.append("<color>");
          _builder.newLine();
          {
            boolean _notEquals_6 = (!Objects.equal(opacity, null));
            if (_notEquals_6) {
              _builder.append("\t");
              _builder.append("\t");
              CharSequence _fillPaint_2 = this.fillPaint(element.getStop_color(), opacity);
              _builder.append(_fillPaint_2, "\t\t");
              _builder.newLineIfNotEmpty();
            } else {
              _builder.append("\t");
              _builder.append("\t");
              CharSequence _fillPaint_3 = this.fillPaint(element.getStop_color());
              _builder.append(_fillPaint_3, "\t\t");
              _builder.newLineIfNotEmpty();
            }
          }
          _builder.append("\t");
          _builder.append("</color>");
          _builder.newLine();
        }
      }
    }
    _builder.append("</Stop>");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handle(final SvgImageElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<ImageView");
    _builder.newLine();
    _builder.append("\t");
    {
      String _x = element.getX();
      boolean _notEquals = (!Objects.equal(_x, null));
      if (_notEquals) {
        _builder.append("x=\"");
        double _parseLength = this.parseLength(element.getX());
        _builder.append(_parseLength, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _y = element.getY();
      boolean _notEquals_1 = (!Objects.equal(_y, null));
      if (_notEquals_1) {
        _builder.append("y=\"");
        double _parseLength_1 = this.parseLength(element.getY());
        _builder.append(_parseLength_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(">");
    _builder.newLine();
    _builder.append("\t");
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<Image");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("url=\"");
    String _xlink__href = element.getXlink__href();
    _builder.append(_xlink__href, "\t\t");
    _builder.append("\"");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("/>");
    _builder.newLine();
    {
      String _transform = element.getTransform();
      boolean _notEquals_2 = (!Objects.equal(_transform, null));
      if (_notEquals_2) {
        _builder.append("\t");
        _builder.append("<transforms>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _handleTransform = this.handleTransform(element.getTransform());
        _builder.append(_handleTransform, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("</transforms>");
        _builder.newLine();
      }
    }
    {
      String _filter = element.getFilter();
      boolean _notEquals_3 = (!Objects.equal(_filter, null));
      if (_notEquals_3) {
        _builder.append("\t");
        String _filter_1 = element.getFilter();
        int _length = element.getFilter().length();
        int _minus = (_length - 1);
        SvgElement _resolveElement = this.resolveElement(_filter_1.substring(5, _minus).trim());
        final SvgFilterElement e = ((SvgFilterElement) _resolveElement);
        _builder.newLineIfNotEmpty();
        {
          boolean _notEquals_4 = (!Objects.equal(e, null));
          if (_notEquals_4) {
            {
              int _size = IterableExtensions.size(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
              boolean _equals = (_size == 1);
              if (_equals) {
                _builder.append("\t");
                FilterPrimitiveElement _head = IterableExtensions.<FilterPrimitiveElement>head(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
                final SvgElement fiElement = ((SvgElement) _head);
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("<effect>");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _handleFilter = this.handleFilter(fiElement);
                _builder.append(_handleFilter, "\t\t");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("</effect>");
                _builder.newLine();
              } else {
                _builder.append("\t");
                _builder.append("<!-- Multi filter needs different handling -->");
                _builder.newLine();
              }
            }
          }
        }
      }
    }
    {
      if ((((!Objects.equal(element.getClip_path(), null)) && (element.getClip_path().trim().length() > 0)) && (!element.getClip_path().trim().equals("none")))) {
        _builder.append("\t");
        _builder.append("<clip>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _clip_path = element.getClip_path();
        int _length_1 = element.getClip_path().length();
        int _minus_1 = (_length_1 - 1);
        SvgElement _resolveElement_1 = this.resolveElement(_clip_path.substring(5, _minus_1));
        final SvgClipPathElement clipElement = ((SvgClipPathElement) _resolveElement_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("<Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("<children>");
        _builder.newLine();
        {
          EList<SvgElement> _children = clipElement.getChildren();
          for(final SvgElement e_1 : _children) {
            _builder.append("\t");
            _builder.append("\t\t\t");
            CharSequence _handle = this.handle(e_1);
            _builder.append(_handle, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("</children>");
        _builder.newLine();
        {
          if ((((!Objects.equal(clipElement.getTransform(), null)) && (clipElement.getTransform().trim().length() > 0)) && (!element.getClip_path().equals("none")))) {
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("<transforms>");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("\t");
            String _handleTransform_1 = this.handleTransform(clipElement.getTransform());
            _builder.append(_handleTransform_1, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("</transforms>");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("</Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("</clip>");
        _builder.newLine();
      }
    }
    _builder.append("</ImageView>");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handle(final SvgRectElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Rectangle");
    _builder.newLine();
    _builder.append("\t");
    {
      String _x = element.getX();
      boolean _notEquals = (!Objects.equal(_x, null));
      if (_notEquals) {
        _builder.append("x=\"");
        double _parseLength = this.parseLength(element.getX());
        _builder.append(_parseLength, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _y = element.getY();
      boolean _notEquals_1 = (!Objects.equal(_y, null));
      if (_notEquals_1) {
        _builder.append("y=\"");
        double _parseLength_1 = this.parseLength(element.getY());
        _builder.append(_parseLength_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _width = element.getWidth();
      boolean _notEquals_2 = (!Objects.equal(_width, null));
      if (_notEquals_2) {
        _builder.append("width=\"");
        double _parseLength_2 = this.parseLength(element.getWidth());
        _builder.append(_parseLength_2, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _height = element.getHeight();
      boolean _notEquals_3 = (!Objects.equal(_height, null));
      if (_notEquals_3) {
        _builder.append("height=\"");
        double _parseLength_3 = this.parseLength(element.getHeight());
        _builder.append(_parseLength_3, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _rx = element.getRx();
      boolean _notEquals_4 = (!Objects.equal(_rx, null));
      if (_notEquals_4) {
        _builder.append("arcWidth=\"");
        double _parseLength_4 = this.parseLength(element.getRx());
        Double _valueOf = Double.valueOf("2.0");
        double _multiply = (_parseLength_4 * (_valueOf).doubleValue());
        _builder.append(_multiply, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _ry = element.getRy();
      boolean _notEquals_5 = (!Objects.equal(_ry, null));
      if (_notEquals_5) {
        _builder.append("arcHeight=\"");
        double _parseLength_5 = this.parseLength(element.getRy());
        Double _valueOf_1 = Double.valueOf("2.0");
        double _multiply_1 = (_parseLength_5 * (_valueOf_1).doubleValue());
        _builder.append(_multiply_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _opacity = element.getOpacity();
      boolean _notEquals_6 = (!Objects.equal(_opacity, null));
      if (_notEquals_6) {
        _builder.append("opacity=\"");
        String _opacity_1 = element.getOpacity();
        _builder.append(_opacity_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _id = element.getId();
      boolean _notEquals_7 = (!Objects.equal(_id, null));
      if (_notEquals_7) {
        _builder.append("id=\"");
        String _validateId = this.validateId(element.getId());
        _builder.append(_validateId, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleShapePresentationAttributes = this.handleShapePresentationAttributes(element);
    _builder.append(_handleShapePresentationAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(">");
    _builder.newLine();
    _builder.append("\t");
    Object _lookupFeature = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL, element);
    Object _lookupFeature_1 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL_OPACITY, element);
    CharSequence _handlePaint = this.handlePaint("fill", ((String) _lookupFeature), ((String) _lookupFeature_1));
    _builder.append(_handlePaint, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    Object _lookupFeature_2 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE, element);
    Object _lookupFeature_3 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_OPACITY, element);
    CharSequence _handlePaint_1 = this.handlePaint("stroke", ((String) _lookupFeature_2), ((String) _lookupFeature_3));
    _builder.append(_handlePaint_1, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      String _transform = element.getTransform();
      boolean _notEquals_8 = (!Objects.equal(_transform, null));
      if (_notEquals_8) {
        _builder.append("\t");
        _builder.append("<transforms>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _handleTransform = this.handleTransform(element.getTransform());
        _builder.append(_handleTransform, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("</transforms>");
        _builder.newLine();
      }
    }
    {
      String _filter = element.getFilter();
      boolean _notEquals_9 = (!Objects.equal(_filter, null));
      if (_notEquals_9) {
        _builder.append("\t");
        String _filter_1 = element.getFilter();
        int _length = element.getFilter().length();
        int _minus = (_length - 1);
        SvgElement _resolveElement = this.resolveElement(_filter_1.substring(5, _minus).trim());
        final SvgFilterElement e = ((SvgFilterElement) _resolveElement);
        _builder.newLineIfNotEmpty();
        {
          boolean _notEquals_10 = (!Objects.equal(e, null));
          if (_notEquals_10) {
            {
              int _size = IterableExtensions.size(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
              boolean _equals = (_size == 1);
              if (_equals) {
                _builder.append("\t");
                FilterPrimitiveElement _head = IterableExtensions.<FilterPrimitiveElement>head(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
                final SvgElement fiElement = ((SvgElement) _head);
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("<effect>");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _handleFilter = this.handleFilter(fiElement);
                _builder.append(_handleFilter, "\t\t");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("</effect>");
                _builder.newLine();
              } else {
                _builder.append("\t");
                _builder.append("<!-- Multi filter needs different handling -->");
                _builder.newLine();
              }
            }
          }
        }
      }
    }
    {
      if ((((!Objects.equal(element.getClip_path(), null)) && (element.getClip_path().trim().length() > 0)) && (!element.getClip_path().trim().equals("none")))) {
        _builder.append("\t");
        _builder.append("<clip>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _clip_path = element.getClip_path();
        int _length_1 = element.getClip_path().length();
        int _minus_1 = (_length_1 - 1);
        SvgElement _resolveElement_1 = this.resolveElement(_clip_path.substring(5, _minus_1));
        final SvgClipPathElement clipElement = ((SvgClipPathElement) _resolveElement_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("<Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("<children>");
        _builder.newLine();
        {
          EList<SvgElement> _children = clipElement.getChildren();
          for(final SvgElement e_1 : _children) {
            _builder.append("\t");
            _builder.append("\t\t\t");
            CharSequence _handle = this.handle(e_1);
            _builder.append(_handle, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("</children>");
        _builder.newLine();
        {
          if ((((!Objects.equal(clipElement.getTransform(), null)) && (clipElement.getTransform().trim().length() > 0)) && (!element.getClip_path().equals("none")))) {
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("<transforms>");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("\t");
            String _handleTransform_1 = this.handleTransform(clipElement.getTransform());
            _builder.append(_handleTransform_1, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("</transforms>");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("</Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("</clip>");
        _builder.newLine();
      }
    }
    _builder.append("</Rectangle>");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handle(final SvgGElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Group");
    _builder.newLine();
    _builder.append("\t");
    {
      String _x = element.getX();
      boolean _notEquals = (!Objects.equal(_x, null));
      if (_notEquals) {
        _builder.append("translateX=\"");
        double _parseCoordinate = this.parseCoordinate(element.getX());
        _builder.append(_parseCoordinate, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _y = element.getY();
      boolean _notEquals_1 = (!Objects.equal(_y, null));
      if (_notEquals_1) {
        _builder.append("translateY=\"");
        double _parseCoordinate_1 = this.parseCoordinate(element.getY());
        _builder.append(_parseCoordinate_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _opacity = element.getOpacity();
      boolean _notEquals_2 = (!Objects.equal(_opacity, null));
      if (_notEquals_2) {
        _builder.append("opacity=\"");
        double _parseDouble = this.parseDouble(element.getOpacity());
        double _parseDouble_1 = this.parseDouble(element.getFill_opacity());
        double _multiply = (_parseDouble * _parseDouble_1);
        _builder.append(_multiply, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _id = element.getId();
      boolean _notEquals_3 = (!Objects.equal(_id, null));
      if (_notEquals_3) {
        _builder.append("id=\"");
        String _validateId = this.validateId(element.getId());
        _builder.append(_validateId, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(">");
    _builder.newLine();
    _builder.append("\t");
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<children>");
    _builder.newLine();
    {
      EList<SvgElement> _children = element.getChildren();
      for(final SvgElement o : _children) {
        _builder.append("\t\t");
        CharSequence _handle = this.handle(o);
        _builder.append(_handle, "\t\t");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    _builder.append("</children>");
    _builder.newLine();
    {
      String _transform = element.getTransform();
      boolean _notEquals_4 = (!Objects.equal(_transform, null));
      if (_notEquals_4) {
        _builder.append("\t");
        _builder.append("<transforms>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _handleTransform = this.handleTransform(element.getTransform());
        _builder.append(_handleTransform, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("</transforms>");
        _builder.newLine();
      }
    }
    {
      String _filter = element.getFilter();
      boolean _notEquals_5 = (!Objects.equal(_filter, null));
      if (_notEquals_5) {
        _builder.append("\t");
        String _filter_1 = element.getFilter();
        int _length = element.getFilter().length();
        int _minus = (_length - 1);
        SvgElement _resolveElement = this.resolveElement(_filter_1.substring(5, _minus).trim());
        final SvgFilterElement e = ((SvgFilterElement) _resolveElement);
        _builder.newLineIfNotEmpty();
        {
          boolean _notEquals_6 = (!Objects.equal(e, null));
          if (_notEquals_6) {
            {
              int _size = IterableExtensions.size(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
              boolean _equals = (_size == 1);
              if (_equals) {
                _builder.append("\t");
                FilterPrimitiveElement _head = IterableExtensions.<FilterPrimitiveElement>head(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
                final SvgElement fiElement = ((SvgElement) _head);
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("<effect>");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _handleFilter = this.handleFilter(fiElement);
                _builder.append(_handleFilter, "\t\t");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("</effect>");
                _builder.newLine();
              } else {
                _builder.append("\t");
                _builder.append("<!-- Multi filter needs different handling -->");
                _builder.newLine();
              }
            }
          }
        }
      }
    }
    {
      if ((((!Objects.equal(element.getClip_path(), null)) && (element.getClip_path().trim().length() > 0)) && (!element.getClip_path().trim().equals("none")))) {
        _builder.append("\t");
        _builder.append("<clip>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _clip_path = element.getClip_path();
        int _length_1 = element.getClip_path().length();
        int _minus_1 = (_length_1 - 1);
        SvgElement _resolveElement_1 = this.resolveElement(_clip_path.substring(5, _minus_1));
        final SvgClipPathElement clipElement = ((SvgClipPathElement) _resolveElement_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("<Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("<children>");
        _builder.newLine();
        {
          EList<SvgElement> _children_1 = clipElement.getChildren();
          for(final SvgElement e_1 : _children_1) {
            _builder.append("\t");
            _builder.append("\t\t\t");
            CharSequence _handle_1 = this.handle(e_1);
            _builder.append(_handle_1, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("</children>");
        _builder.newLine();
        {
          if ((((!Objects.equal(clipElement.getTransform(), null)) && (clipElement.getTransform().trim().length() > 0)) && (!element.getClip_path().equals("none")))) {
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("<transforms>");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("\t");
            String _handleTransform_1 = this.handleTransform(clipElement.getTransform());
            _builder.append(_handleTransform_1, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("</transforms>");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("</Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("</clip>");
        _builder.newLine();
      }
    }
    _builder.append("</Group>");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handle(final SvgUseElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Group");
    _builder.newLine();
    _builder.append("\t");
    {
      String _x = element.getX();
      boolean _notEquals = (!Objects.equal(_x, null));
      if (_notEquals) {
        _builder.append("translateX=\"");
        double _parseCoordinate = this.parseCoordinate(element.getX());
        _builder.append(_parseCoordinate, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _y = element.getY();
      boolean _notEquals_1 = (!Objects.equal(_y, null));
      if (_notEquals_1) {
        _builder.append("translateY=\"");
        double _parseCoordinate_1 = this.parseCoordinate(element.getY());
        _builder.append(_parseCoordinate_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _opacity = element.getOpacity();
      boolean _notEquals_2 = (!Objects.equal(_opacity, null));
      if (_notEquals_2) {
        _builder.append("opacity=\"");
        double _parseDouble = this.parseDouble(element.getOpacity());
        double _parseDouble_1 = this.parseDouble(element.getFill_opacity());
        double _multiply = (_parseDouble * _parseDouble_1);
        _builder.append(_multiply, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _id = element.getId();
      boolean _notEquals_3 = (!Objects.equal(_id, null));
      if (_notEquals_3) {
        _builder.append("id=\"");
        String _validateId = this.validateId(element.getId());
        _builder.append(_validateId, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(">");
    _builder.newLine();
    _builder.append("\t");
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      SvgElement _resolvedInstance = element.getResolvedInstance();
      boolean _notEquals_4 = (!Objects.equal(_resolvedInstance, null));
      if (_notEquals_4) {
        _builder.append("\t");
        _builder.append("<children>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        CharSequence _handle = this.handle(element.getResolvedInstance());
        _builder.append(_handle, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("</children>");
        _builder.newLine();
      }
    }
    {
      String _transform = element.getTransform();
      boolean _notEquals_5 = (!Objects.equal(_transform, null));
      if (_notEquals_5) {
        _builder.append("\t");
        _builder.append("<transforms>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _handleTransform = this.handleTransform(element.getTransform());
        _builder.append(_handleTransform, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("</transforms>");
        _builder.newLine();
      }
    }
    {
      String _filter = element.getFilter();
      boolean _notEquals_6 = (!Objects.equal(_filter, null));
      if (_notEquals_6) {
        _builder.append("\t");
        String _filter_1 = element.getFilter();
        int _length = element.getFilter().length();
        int _minus = (_length - 1);
        SvgElement _resolveElement = this.resolveElement(_filter_1.substring(5, _minus).trim());
        final SvgFilterElement e = ((SvgFilterElement) _resolveElement);
        _builder.newLineIfNotEmpty();
        {
          boolean _notEquals_7 = (!Objects.equal(e, null));
          if (_notEquals_7) {
            {
              int _size = IterableExtensions.size(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
              boolean _equals = (_size == 1);
              if (_equals) {
                _builder.append("\t");
                FilterPrimitiveElement _head = IterableExtensions.<FilterPrimitiveElement>head(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
                final SvgElement fiElement = ((SvgElement) _head);
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("<effect>");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _handleFilter = this.handleFilter(fiElement);
                _builder.append(_handleFilter, "\t\t");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("</effect>");
                _builder.newLine();
              } else {
                _builder.append("\t");
                _builder.append("<!-- Multi filter needs different handling -->");
                _builder.newLine();
              }
            }
          }
        }
      }
    }
    {
      if ((((!Objects.equal(element.getClip_path(), null)) && (element.getClip_path().trim().length() > 0)) && (!element.getClip_path().trim().equals("none")))) {
        _builder.append("\t");
        _builder.append("<clip>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _clip_path = element.getClip_path();
        int _length_1 = element.getClip_path().length();
        int _minus_1 = (_length_1 - 1);
        SvgElement _resolveElement_1 = this.resolveElement(_clip_path.substring(5, _minus_1));
        final SvgClipPathElement clipElement = ((SvgClipPathElement) _resolveElement_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("<Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("<children>");
        _builder.newLine();
        {
          EList<SvgElement> _children = clipElement.getChildren();
          for(final SvgElement e_1 : _children) {
            _builder.append("\t");
            _builder.append("\t\t\t");
            CharSequence _handle_1 = this.handle(e_1);
            _builder.append(_handle_1, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("</children>");
        _builder.newLine();
        {
          if ((((!Objects.equal(clipElement.getTransform(), null)) && (clipElement.getTransform().trim().length() > 0)) && (!element.getClip_path().equals("none")))) {
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("<transforms>");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("\t");
            String _handleTransform_1 = this.handleTransform(clipElement.getTransform());
            _builder.append(_handleTransform_1, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("</transforms>");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("</Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("</clip>");
        _builder.newLine();
      }
    }
    _builder.append("</Group>");
    _builder.newLine();
    return _builder;
  }
  
  public Object lookupFeature(final EStructuralFeature feature, final EObject object) {
    EObject eo = object;
    do {
      {
        boolean _contains = eo.eClass().getEAllStructuralFeatures().contains(feature);
        if (_contains) {
          boolean _eIsSet = eo.eIsSet(feature);
          if (_eIsSet) {
            return eo.eGet(feature);
          }
        }
        eo = eo.eContainer();
      }
    } while((!Objects.equal(eo, null)));
    return null;
  }
  
  public CharSequence handleShapePresentationAttributes(final PresentationAttributes element) {
    StringConcatenation _builder = new StringConcatenation();
    {
      Object _lookupFeature = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_DASHOFFSET, element);
      boolean _notEquals = (!Objects.equal(_lookupFeature, null));
      if (_notEquals) {
        _builder.append("strokeDashOffset=\"");
        Object _lookupFeature_1 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_DASHOFFSET, element);
        double _parseLength = this.parseLength(((String) _lookupFeature_1));
        _builder.append(_parseLength);
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    {
      Object _lookupFeature_2 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_LINECAP, element);
      boolean _notEquals_1 = (!Objects.equal(_lookupFeature_2, null));
      if (_notEquals_1) {
        _builder.append("strokeLineCap=\"");
        Object _lookupFeature_3 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_LINECAP, element);
        String _fx = this.toFx(((Stroke_linecap) _lookupFeature_3));
        _builder.append(_fx);
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    {
      Object _lookupFeature_4 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_LINEJOIN, element);
      boolean _notEquals_2 = (!Objects.equal(_lookupFeature_4, null));
      if (_notEquals_2) {
        _builder.append("strokeLineJoin=\"");
        Object _lookupFeature_5 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_LINEJOIN, element);
        String _fx_1 = this.toFx(((Stroke_linejoin) _lookupFeature_5));
        _builder.append(_fx_1);
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    {
      Object _lookupFeature_6 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_MITERLIMIT, element);
      boolean _notEquals_3 = (!Objects.equal(_lookupFeature_6, null));
      if (_notEquals_3) {
        _builder.append("strokeMiterLimit=\"");
        Object _lookupFeature_7 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_MITERLIMIT, element);
        double _parseLength_1 = this.parseLength(((String) _lookupFeature_7));
        _builder.append(_parseLength_1);
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    {
      Object _lookupFeature_8 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_WIDTH, element);
      boolean _notEquals_4 = (!Objects.equal(_lookupFeature_8, null));
      if (_notEquals_4) {
        _builder.append("strokeWidth=\"");
        Object _lookupFeature_9 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_WIDTH, element);
        double _parseLength_2 = this.parseLength(((String) _lookupFeature_9));
        _builder.append(_parseLength_2);
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence handleStyleClassAttributes(final String styleClass) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _notEquals = (!Objects.equal(styleClass, null));
      if (_notEquals) {
        _builder.append("<styleClass>");
        _builder.newLine();
        {
          String[] _split = styleClass.split(" ");
          for(final String c : _split) {
            _builder.append("\t");
            _builder.append("<String fx:value=\"");
            String _trim = c.trim();
            _builder.append(_trim, "\t");
            _builder.append("\"/>");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("</styleClass>");
        _builder.newLine();
      }
    }
    return _builder;
  }
  
  protected CharSequence _handle(final SvgPathElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<SVGPath");
    _builder.newLine();
    _builder.append("\t");
    {
      String _d = element.getD();
      boolean _notEquals = (!Objects.equal(_d, null));
      if (_notEquals) {
        _builder.append("content=\"");
        String _d_1 = element.getD();
        _builder.append(_d_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _opacity = element.getOpacity();
      boolean _notEquals_1 = (!Objects.equal(_opacity, null));
      if (_notEquals_1) {
        _builder.append("opacity=\"");
        String _opacity_1 = element.getOpacity();
        _builder.append(_opacity_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      Fill_rule _fill_rule = element.getFill_rule();
      boolean _notEquals_2 = (!Objects.equal(_fill_rule, Fill_rule.NONZERO));
      if (_notEquals_2) {
        _builder.append("fillRule=\"EVEN_ODD\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleShapePresentationAttributes = this.handleShapePresentationAttributes(element);
    _builder.append(_handleShapePresentationAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _id = element.getId();
      boolean _notEquals_3 = (!Objects.equal(_id, null));
      if (_notEquals_3) {
        _builder.append("id=\"");
        String _validateId = this.validateId(element.getId());
        _builder.append(_validateId, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(">");
    _builder.newLine();
    _builder.append("\t");
    Object _lookupFeature = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL, element);
    Object _lookupFeature_1 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL_OPACITY, element);
    CharSequence _handlePaint = this.handlePaint("fill", ((String) _lookupFeature), ((String) _lookupFeature_1));
    _builder.append(_handlePaint, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    Object _lookupFeature_2 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE, element);
    Object _lookupFeature_3 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_OPACITY, element);
    CharSequence _handlePaint_1 = this.handlePaint("stroke", ((String) _lookupFeature_2), ((String) _lookupFeature_3));
    _builder.append(_handlePaint_1, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes, "\t");
    _builder.newLineIfNotEmpty();
    {
      String _transform = element.getTransform();
      boolean _notEquals_4 = (!Objects.equal(_transform, null));
      if (_notEquals_4) {
        _builder.append("\t");
        _builder.append("<transforms>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _handleTransform = this.handleTransform(element.getTransform());
        _builder.append(_handleTransform, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("</transforms>");
        _builder.newLine();
      }
    }
    {
      String _filter = element.getFilter();
      boolean _notEquals_5 = (!Objects.equal(_filter, null));
      if (_notEquals_5) {
        _builder.append("\t");
        String _filter_1 = element.getFilter();
        int _length = element.getFilter().length();
        int _minus = (_length - 1);
        SvgElement _resolveElement = this.resolveElement(_filter_1.substring(5, _minus).trim());
        final SvgFilterElement e = ((SvgFilterElement) _resolveElement);
        _builder.newLineIfNotEmpty();
        {
          boolean _notEquals_6 = (!Objects.equal(e, null));
          if (_notEquals_6) {
            {
              int _size = IterableExtensions.size(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
              boolean _equals = (_size == 1);
              if (_equals) {
                _builder.append("\t");
                FilterPrimitiveElement _head = IterableExtensions.<FilterPrimitiveElement>head(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
                final SvgElement fiElement = ((SvgElement) _head);
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("<effect>");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _handleFilter = this.handleFilter(fiElement);
                _builder.append(_handleFilter, "\t\t");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("</effect>");
                _builder.newLine();
              } else {
                _builder.append("\t");
                _builder.append("<!-- Multi filter needs different handling -->");
                _builder.newLine();
              }
            }
          }
        }
      }
    }
    {
      if ((((!Objects.equal(element.getClip_path(), null)) && (element.getClip_path().trim().length() > 0)) && (!element.getClip_path().trim().equals("none")))) {
        _builder.append("\t");
        _builder.append("<clip>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _clip_path = element.getClip_path();
        int _length_1 = element.getClip_path().length();
        int _minus_1 = (_length_1 - 1);
        SvgElement _resolveElement_1 = this.resolveElement(_clip_path.substring(5, _minus_1));
        final SvgClipPathElement clipElement = ((SvgClipPathElement) _resolveElement_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("<Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("<children>");
        _builder.newLine();
        {
          EList<SvgElement> _children = clipElement.getChildren();
          for(final SvgElement e_1 : _children) {
            _builder.append("\t");
            _builder.append("\t\t\t");
            CharSequence _handle = this.handle(e_1);
            _builder.append(_handle, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("</children>");
        _builder.newLine();
        {
          if ((((!Objects.equal(clipElement.getTransform(), null)) && (clipElement.getTransform().trim().length() > 0)) && (!element.getClip_path().equals("none")))) {
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("<transforms>");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("\t");
            String _handleTransform_1 = this.handleTransform(clipElement.getTransform());
            _builder.append(_handleTransform_1, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("</transforms>");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("</Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("</clip>");
        _builder.newLine();
      }
    }
    _builder.append("</SVGPath>");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handleFilter(final EObject o) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<!-- Unsupported filter type -->");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handleFilter(final SvgFeGaussianBlurElement blur) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<GaussianBlur");
    _builder.newLine();
    _builder.append("\t");
    {
      String _stdDeviation = blur.getStdDeviation();
      boolean _notEquals = (!Objects.equal(_stdDeviation, null));
      if (_notEquals) {
        _builder.append("radius=\"");
        double _parseDouble = this.parseDouble(blur.getStdDeviation());
        double _multiply = (_parseDouble * 2);
        _builder.append(_multiply, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("/>");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence handlePaint(final String type, final String fillDefinition, final String typeOpacity) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _notEquals = (!Objects.equal(fillDefinition, null));
      if (_notEquals) {
        _builder.append("<");
        _builder.append(type);
        _builder.append(">");
        _builder.newLineIfNotEmpty();
        {
          boolean _equals = fillDefinition.equals("none");
          if (_equals) {
            _builder.append("\t");
            _builder.append("TRANSPARENT");
            _builder.newLine();
          } else {
            if (((!Objects.equal(typeOpacity, null)) && (Double.parseDouble(typeOpacity) != (Double.valueOf("1.0")).doubleValue()))) {
              _builder.append("\t");
              CharSequence _fillPaint = this.fillPaint(fillDefinition, Double.valueOf(Double.parseDouble(typeOpacity)));
              _builder.append(_fillPaint, "\t");
              _builder.newLineIfNotEmpty();
            } else {
              _builder.append("\t");
              CharSequence _fillPaint_1 = this.fillPaint(fillDefinition);
              _builder.append(_fillPaint_1, "\t");
              _builder.newLineIfNotEmpty();
            }
          }
        }
        _builder.append("</");
        _builder.append(type);
        _builder.append(">");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
  
  protected CharSequence _handle(final SvgClipPathElement element) {
    return null;
  }
  
  protected CharSequence _handle(final SvgEllipseElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Ellipse");
    _builder.newLine();
    _builder.append("\t");
    {
      String _rx = element.getRx();
      boolean _notEquals = (!Objects.equal(_rx, null));
      if (_notEquals) {
        _builder.append("radiusX=\"");
        double _parseLength = this.parseLength(element.getRx());
        _builder.append(_parseLength, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _ry = element.getRy();
      boolean _notEquals_1 = (!Objects.equal(_ry, null));
      if (_notEquals_1) {
        _builder.append("radiusY=\"");
        double _parseLength_1 = this.parseLength(element.getRy());
        _builder.append(_parseLength_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _cx = element.getCx();
      boolean _notEquals_2 = (!Objects.equal(_cx, null));
      if (_notEquals_2) {
        _builder.append("centerX=\"");
        double _parseCoordinate = this.parseCoordinate(element.getCx());
        _builder.append(_parseCoordinate, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _cy = element.getCy();
      boolean _notEquals_3 = (!Objects.equal(_cy, null));
      if (_notEquals_3) {
        _builder.append("centerY=\"");
        double _parseCoordinate_1 = this.parseCoordinate(element.getCy());
        _builder.append(_parseCoordinate_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _opacity = element.getOpacity();
      boolean _notEquals_4 = (!Objects.equal(_opacity, null));
      if (_notEquals_4) {
        _builder.append("opacity=\"");
        String _opacity_1 = element.getOpacity();
        _builder.append(_opacity_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _id = element.getId();
      boolean _notEquals_5 = (!Objects.equal(_id, null));
      if (_notEquals_5) {
        _builder.append("id=\"");
        String _validateId = this.validateId(element.getId());
        _builder.append(_validateId, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleShapePresentationAttributes = this.handleShapePresentationAttributes(element);
    _builder.append(_handleShapePresentationAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(">");
    _builder.newLine();
    _builder.append("\t");
    Object _lookupFeature = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL, element);
    Object _lookupFeature_1 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL_OPACITY, element);
    CharSequence _handlePaint = this.handlePaint("fill", ((String) _lookupFeature), ((String) _lookupFeature_1));
    _builder.append(_handlePaint, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    Object _lookupFeature_2 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE, element);
    Object _lookupFeature_3 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_OPACITY, element);
    CharSequence _handlePaint_1 = this.handlePaint("stroke", ((String) _lookupFeature_2), ((String) _lookupFeature_3));
    _builder.append(_handlePaint_1, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      String _transform = element.getTransform();
      boolean _notEquals_6 = (!Objects.equal(_transform, null));
      if (_notEquals_6) {
        _builder.append("\t");
        _builder.append("<transforms>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _handleTransform = this.handleTransform(element.getTransform());
        _builder.append(_handleTransform, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("</transforms>");
        _builder.newLine();
      }
    }
    {
      String _filter = element.getFilter();
      boolean _notEquals_7 = (!Objects.equal(_filter, null));
      if (_notEquals_7) {
        _builder.append("\t");
        String _filter_1 = element.getFilter();
        int _length = element.getFilter().length();
        int _minus = (_length - 1);
        SvgElement _resolveElement = this.resolveElement(_filter_1.substring(5, _minus).trim());
        final SvgFilterElement e = ((SvgFilterElement) _resolveElement);
        _builder.newLineIfNotEmpty();
        {
          boolean _notEquals_8 = (!Objects.equal(e, null));
          if (_notEquals_8) {
            {
              int _size = IterableExtensions.size(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
              boolean _equals = (_size == 1);
              if (_equals) {
                _builder.append("\t");
                FilterPrimitiveElement _head = IterableExtensions.<FilterPrimitiveElement>head(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
                final SvgElement fiElement = ((SvgElement) _head);
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("<effect>");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _handleFilter = this.handleFilter(fiElement);
                _builder.append(_handleFilter, "\t\t");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("</effect>");
                _builder.newLine();
              } else {
                _builder.append("\t");
                _builder.append("<!-- Multi filter needs different handling -->");
                _builder.newLine();
              }
            }
          }
        }
      }
    }
    {
      if ((((!Objects.equal(element.getClip_path(), null)) && (element.getClip_path().trim().length() > 0)) && (!element.getClip_path().trim().equals("none")))) {
        _builder.append("\t");
        _builder.append("<clip>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _clip_path = element.getClip_path();
        int _length_1 = element.getClip_path().length();
        int _minus_1 = (_length_1 - 1);
        SvgElement _resolveElement_1 = this.resolveElement(_clip_path.substring(5, _minus_1));
        final SvgClipPathElement clipElement = ((SvgClipPathElement) _resolveElement_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("<Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("<children>");
        _builder.newLine();
        {
          EList<SvgElement> _children = clipElement.getChildren();
          for(final SvgElement e_1 : _children) {
            _builder.append("\t");
            _builder.append("\t\t\t");
            CharSequence _handle = this.handle(e_1);
            _builder.append(_handle, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("</children>");
        _builder.newLine();
        {
          if ((((!Objects.equal(clipElement.getTransform(), null)) && (clipElement.getTransform().trim().length() > 0)) && (!element.getClip_path().equals("none")))) {
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("<transforms>");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("\t");
            String _handleTransform_1 = this.handleTransform(clipElement.getTransform());
            _builder.append(_handleTransform_1, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("</transforms>");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("</Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("</clip>");
        _builder.newLine();
      }
    }
    _builder.append("</Ellipse>");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handle(final SvgCircleElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Circle");
    _builder.newLine();
    _builder.append("\t");
    {
      String _r = element.getR();
      boolean _notEquals = (!Objects.equal(_r, null));
      if (_notEquals) {
        _builder.append("radius=\"");
        double _parseLength = this.parseLength(element.getR());
        _builder.append(_parseLength, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _cx = element.getCx();
      boolean _notEquals_1 = (!Objects.equal(_cx, null));
      if (_notEquals_1) {
        _builder.append("centerX=\"");
        double _parseCoordinate = this.parseCoordinate(element.getCx());
        _builder.append(_parseCoordinate, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _cy = element.getCy();
      boolean _notEquals_2 = (!Objects.equal(_cy, null));
      if (_notEquals_2) {
        _builder.append("centerY=\"");
        double _parseCoordinate_1 = this.parseCoordinate(element.getCy());
        _builder.append(_parseCoordinate_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _opacity = element.getOpacity();
      boolean _notEquals_3 = (!Objects.equal(_opacity, null));
      if (_notEquals_3) {
        _builder.append("opacity=\"");
        String _opacity_1 = element.getOpacity();
        _builder.append(_opacity_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _id = element.getId();
      boolean _notEquals_4 = (!Objects.equal(_id, null));
      if (_notEquals_4) {
        _builder.append("id=\"");
        String _validateId = this.validateId(element.getId());
        _builder.append(_validateId, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleShapePresentationAttributes = this.handleShapePresentationAttributes(element);
    _builder.append(_handleShapePresentationAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(">");
    _builder.newLine();
    _builder.append("\t");
    Object _lookupFeature = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL, element);
    Object _lookupFeature_1 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL_OPACITY, element);
    CharSequence _handlePaint = this.handlePaint("fill", ((String) _lookupFeature), ((String) _lookupFeature_1));
    _builder.append(_handlePaint, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    Object _lookupFeature_2 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE, element);
    Object _lookupFeature_3 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_OPACITY, element);
    CharSequence _handlePaint_1 = this.handlePaint("stroke", ((String) _lookupFeature_2), ((String) _lookupFeature_3));
    _builder.append(_handlePaint_1, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      String _transform = element.getTransform();
      boolean _notEquals_5 = (!Objects.equal(_transform, null));
      if (_notEquals_5) {
        _builder.append("\t");
        _builder.append("<transforms>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _handleTransform = this.handleTransform(element.getTransform());
        _builder.append(_handleTransform, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("</transforms>");
        _builder.newLine();
      }
    }
    {
      String _filter = element.getFilter();
      boolean _notEquals_6 = (!Objects.equal(_filter, null));
      if (_notEquals_6) {
        _builder.append("\t");
        String _filter_1 = element.getFilter();
        int _length = element.getFilter().length();
        int _minus = (_length - 1);
        SvgElement _resolveElement = this.resolveElement(_filter_1.substring(5, _minus).trim());
        final SvgFilterElement e = ((SvgFilterElement) _resolveElement);
        _builder.newLineIfNotEmpty();
        {
          boolean _notEquals_7 = (!Objects.equal(e, null));
          if (_notEquals_7) {
            {
              int _size = IterableExtensions.size(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
              boolean _equals = (_size == 1);
              if (_equals) {
                _builder.append("\t");
                FilterPrimitiveElement _head = IterableExtensions.<FilterPrimitiveElement>head(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
                final SvgElement fiElement = ((SvgElement) _head);
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("<effect>");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _handleFilter = this.handleFilter(fiElement);
                _builder.append(_handleFilter, "\t\t");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("</effect>");
                _builder.newLine();
              } else {
                _builder.append("\t");
                _builder.append("<!-- Multi filter needs different handling -->");
                _builder.newLine();
              }
            }
          }
        }
      }
    }
    {
      if ((((!Objects.equal(element.getClip_path(), null)) && (element.getClip_path().trim().length() > 0)) && (!element.getClip_path().trim().equals("none")))) {
        _builder.append("\t");
        _builder.append("<clip>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        String _clip_path = element.getClip_path();
        int _length_1 = element.getClip_path().length();
        int _minus_1 = (_length_1 - 1);
        SvgElement _resolveElement_1 = this.resolveElement(_clip_path.substring(5, _minus_1));
        final SvgClipPathElement clipElement = ((SvgClipPathElement) _resolveElement_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("<Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("<children>");
        _builder.newLine();
        {
          EList<SvgElement> _children = clipElement.getChildren();
          for(final SvgElement e_1 : _children) {
            _builder.append("\t");
            _builder.append("\t\t\t");
            CharSequence _handle = this.handle(e_1);
            _builder.append(_handle, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append("</children>");
        _builder.newLine();
        {
          if ((((!Objects.equal(clipElement.getTransform(), null)) && (clipElement.getTransform().trim().length() > 0)) && (!element.getClip_path().equals("none")))) {
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("<transforms>");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("\t");
            String _handleTransform_1 = this.handleTransform(clipElement.getTransform());
            _builder.append(_handleTransform_1, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("</transforms>");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("</Group>");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("</clip>");
        _builder.newLine();
      }
    }
    _builder.append("</Circle>");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handle(final SvgPolylineElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Polyline");
    _builder.newLine();
    _builder.append("\t");
    {
      String _points = element.getPoints();
      boolean _notEquals = (!Objects.equal(_points, null));
      if (_notEquals) {
        _builder.append("points=\"");
        String _replaceAll = element.getPoints().replaceAll("\\s+", ",");
        _builder.append(_replaceAll, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _opacity = element.getOpacity();
      boolean _notEquals_1 = (!Objects.equal(_opacity, null));
      if (_notEquals_1) {
        _builder.append("opacity=\"");
        String _opacity_1 = element.getOpacity();
        _builder.append(_opacity_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleShapePresentationAttributes = this.handleShapePresentationAttributes(element);
    _builder.append(_handleShapePresentationAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _id = element.getId();
      boolean _notEquals_2 = (!Objects.equal(_id, null));
      if (_notEquals_2) {
        _builder.append("id=\"");
        String _validateId = this.validateId(element.getId());
        _builder.append(_validateId, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(">");
    _builder.newLine();
    Object _lookupFeature = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL, element);
    Object _lookupFeature_1 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL_OPACITY, element);
    CharSequence _handlePaint = this.handlePaint("fill", ((String) _lookupFeature), ((String) _lookupFeature_1));
    _builder.append(_handlePaint);
    _builder.newLineIfNotEmpty();
    Object _lookupFeature_2 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE, element);
    Object _lookupFeature_3 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_OPACITY, element);
    CharSequence _handlePaint_1 = this.handlePaint("stroke", ((String) _lookupFeature_2), ((String) _lookupFeature_3));
    _builder.append(_handlePaint_1);
    _builder.newLineIfNotEmpty();
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      String _transform = element.getTransform();
      boolean _notEquals_3 = (!Objects.equal(_transform, null));
      if (_notEquals_3) {
        _builder.append("<transforms>");
        _builder.newLine();
        _builder.append("\t");
        String _handleTransform = this.handleTransform(element.getTransform());
        _builder.append(_handleTransform, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("</transforms>");
        _builder.newLine();
      }
    }
    {
      String _filter = element.getFilter();
      boolean _notEquals_4 = (!Objects.equal(_filter, null));
      if (_notEquals_4) {
        _builder.append("\t");
        String _filter_1 = element.getFilter();
        int _length = element.getFilter().length();
        int _minus = (_length - 1);
        SvgElement _resolveElement = this.resolveElement(_filter_1.substring(5, _minus).trim());
        final SvgFilterElement e = ((SvgFilterElement) _resolveElement);
        _builder.newLineIfNotEmpty();
        {
          boolean _notEquals_5 = (!Objects.equal(e, null));
          if (_notEquals_5) {
            {
              int _size = IterableExtensions.size(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
              boolean _equals = (_size == 1);
              if (_equals) {
                _builder.append("\t");
                FilterPrimitiveElement _head = IterableExtensions.<FilterPrimitiveElement>head(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
                final SvgElement fiElement = ((SvgElement) _head);
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("<effect>");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _handleFilter = this.handleFilter(fiElement);
                _builder.append(_handleFilter, "\t\t");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("</effect>");
                _builder.newLine();
              } else {
                _builder.append("\t");
                _builder.append("<!-- Multi filter needs different handling -->");
                _builder.newLine();
              }
            }
          }
        }
      }
    }
    {
      if ((((!Objects.equal(element.getClip_path(), null)) && (element.getClip_path().trim().length() > 0)) && (!element.getClip_path().trim().equals("none")))) {
        _builder.append("<clip>");
        _builder.newLine();
        _builder.append("\t");
        String _clip_path = element.getClip_path();
        int _length_1 = element.getClip_path().length();
        int _minus_1 = (_length_1 - 1);
        SvgElement _resolveElement_1 = this.resolveElement(_clip_path.substring(5, _minus_1));
        final SvgClipPathElement clipElement = ((SvgClipPathElement) _resolveElement_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("<Group>");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("<children>");
        _builder.newLine();
        {
          EList<SvgElement> _children = clipElement.getChildren();
          for(final SvgElement e_1 : _children) {
            _builder.append("\t\t\t");
            CharSequence _handle = this.handle(e_1);
            _builder.append(_handle, "\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t\t");
        _builder.append("</children>");
        _builder.newLine();
        {
          if ((((!Objects.equal(clipElement.getTransform(), null)) && (clipElement.getTransform().trim().length() > 0)) && (!element.getClip_path().equals("none")))) {
            _builder.append("\t\t");
            _builder.append("<transforms>");
            _builder.newLine();
            _builder.append("\t\t");
            _builder.append("\t");
            String _handleTransform_1 = this.handleTransform(clipElement.getTransform());
            _builder.append(_handleTransform_1, "\t\t\t");
            _builder.newLineIfNotEmpty();
            _builder.append("\t\t");
            _builder.append("</transforms>");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("</Group>");
        _builder.newLine();
        _builder.append("</clip>");
        _builder.newLine();
      }
    }
    _builder.append("</Polyline>");
    _builder.newLine();
    return _builder;
  }
  
  protected CharSequence _handle(final SvgPolygonElement element) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Polygon");
    _builder.newLine();
    _builder.append("\t");
    {
      String _points = element.getPoints();
      boolean _notEquals = (!Objects.equal(_points, null));
      if (_notEquals) {
        _builder.append("points=\"");
        String _replaceAll = element.getPoints().replaceAll("\\s+", ",");
        _builder.append(_replaceAll, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _opacity = element.getOpacity();
      boolean _notEquals_1 = (!Objects.equal(_opacity, null));
      if (_notEquals_1) {
        _builder.append("opacity=\"");
        String _opacity_1 = element.getOpacity();
        _builder.append(_opacity_1, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _handleShapePresentationAttributes = this.handleShapePresentationAttributes(element);
    _builder.append(_handleShapePresentationAttributes, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    {
      String _id = element.getId();
      boolean _notEquals_2 = (!Objects.equal(_id, null));
      if (_notEquals_2) {
        _builder.append("id=\"");
        String _validateId = this.validateId(element.getId());
        _builder.append(_validateId, "\t");
        _builder.append("\"");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append(">");
    _builder.newLine();
    Object _lookupFeature = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL, element);
    Object _lookupFeature_1 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__FILL_OPACITY, element);
    CharSequence _handlePaint = this.handlePaint("fill", ((String) _lookupFeature), ((String) _lookupFeature_1));
    _builder.append(_handlePaint);
    _builder.newLineIfNotEmpty();
    Object _lookupFeature_2 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE, element);
    Object _lookupFeature_3 = this.lookupFeature(SvgPackage.Literals.PRESENTATION_ATTRIBUTES__STROKE_OPACITY, element);
    CharSequence _handlePaint_1 = this.handlePaint("stroke", ((String) _lookupFeature_2), ((String) _lookupFeature_3));
    _builder.append(_handlePaint_1);
    _builder.newLineIfNotEmpty();
    CharSequence _handleStyleClassAttributes = this.handleStyleClassAttributes(element.getClass_());
    _builder.append(_handleStyleClassAttributes);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      String _transform = element.getTransform();
      boolean _notEquals_3 = (!Objects.equal(_transform, null));
      if (_notEquals_3) {
        _builder.append("<transforms>");
        _builder.newLine();
        _builder.append("\t");
        String _handleTransform = this.handleTransform(element.getTransform());
        _builder.append(_handleTransform, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("</transforms>");
        _builder.newLine();
      }
    }
    {
      String _filter = element.getFilter();
      boolean _notEquals_4 = (!Objects.equal(_filter, null));
      if (_notEquals_4) {
        _builder.append("\t");
        String _filter_1 = element.getFilter();
        int _length = element.getFilter().length();
        int _minus = (_length - 1);
        SvgElement _resolveElement = this.resolveElement(_filter_1.substring(5, _minus).trim());
        final SvgFilterElement e = ((SvgFilterElement) _resolveElement);
        _builder.newLineIfNotEmpty();
        {
          boolean _notEquals_5 = (!Objects.equal(e, null));
          if (_notEquals_5) {
            {
              int _size = IterableExtensions.size(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
              boolean _equals = (_size == 1);
              if (_equals) {
                _builder.append("\t");
                FilterPrimitiveElement _head = IterableExtensions.<FilterPrimitiveElement>head(Iterables.<FilterPrimitiveElement>filter(e.getChildren(), FilterPrimitiveElement.class));
                final SvgElement fiElement = ((SvgElement) _head);
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("<effect>");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("\t");
                CharSequence _handleFilter = this.handleFilter(fiElement);
                _builder.append(_handleFilter, "\t\t");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("</effect>");
                _builder.newLine();
              } else {
                _builder.append("\t");
                _builder.append("<!-- Multi filter needs different handling -->");
                _builder.newLine();
              }
            }
          }
        }
      }
    }
    {
      if ((((!Objects.equal(element.getClip_path(), null)) && (element.getClip_path().trim().length() > 0)) && (!element.getClip_path().trim().equals("none")))) {
        _builder.append("<clip>");
        _builder.newLine();
        _builder.append("\t");
        String _clip_path = element.getClip_path();
        int _length_1 = element.getClip_path().length();
        int _minus_1 = (_length_1 - 1);
        SvgElement _resolveElement_1 = this.resolveElement(_clip_path.substring(5, _minus_1));
        final SvgClipPathElement clipElement = ((SvgClipPathElement) _resolveElement_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("<Group>");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("<children>");
        _builder.newLine();
        {
          EList<SvgElement> _children = clipElement.getChildren();
          for(final SvgElement e_1 : _children) {
            _builder.append("\t\t\t");
            CharSequence _handle = this.handle(e_1);
            _builder.append(_handle, "\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t\t");
        _builder.append("</children>");
        _builder.newLine();
        {
          if ((((!Objects.equal(clipElement.getTransform(), null)) && (clipElement.getTransform().trim().length() > 0)) && (!element.getClip_path().equals("none")))) {
            _builder.append("\t\t");
            _builder.append("<transforms>");
            _builder.newLine();
            _builder.append("\t\t");
            _builder.append("\t");
            String _handleTransform_1 = this.handleTransform(clipElement.getTransform());
            _builder.append(_handleTransform_1, "\t\t\t");
            _builder.newLineIfNotEmpty();
            _builder.append("\t\t");
            _builder.append("</transforms>");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("</Group>");
        _builder.newLine();
        _builder.append("</clip>");
        _builder.newLine();
      }
    }
    _builder.append("</Polygon>");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence fillPaint(final String fill) {
    Object _xifexpression = null;
    boolean _startsWith = fill.startsWith("#");
    if (_startsWith) {
      return this.hexColor(fill);
    } else {
      Object _xifexpression_1 = null;
      boolean _startsWith_1 = fill.startsWith("rgb");
      if (_startsWith_1) {
        int _indexOf = fill.indexOf("(");
        int _plus = (_indexOf + 1);
        final String c = fill.substring(_plus, fill.indexOf(")"));
        return this.rgbColor(c);
      } else {
        Object _xifexpression_2 = null;
        boolean _startsWith_2 = fill.startsWith("argb");
        if (_startsWith_2) {
          _xifexpression_2 = null;
        } else {
          boolean _startsWith_3 = fill.startsWith("url");
          if (_startsWith_3) {
            int _length = fill.length();
            int _minus = (_length - 1);
            final SvgElement e = this.resolveElement(fill.substring(5, _minus));
            if ((e instanceof SvgLinearGradientElement)) {
              return this.handleGradient(((SvgLinearGradientElement) e), null);
            } else {
              if ((e instanceof SvgRadialGradientElement)) {
                return this.handleGradient(((SvgRadialGradientElement) e), null);
              }
            }
          } else {
            return fill.toUpperCase();
          }
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return ((CharSequence)_xifexpression);
  }
  
  public CharSequence fillPaint(final String fill, final Double opacity) {
    Object _xifexpression = null;
    boolean _startsWith = fill.startsWith("#");
    if (_startsWith) {
      return this.hexColor(fill, opacity);
    } else {
      Object _xifexpression_1 = null;
      boolean _startsWith_1 = fill.toLowerCase().startsWith("rgb");
      if (_startsWith_1) {
        int _indexOf = fill.indexOf("(");
        int _plus = (_indexOf + 1);
        final String c = fill.substring(_plus, fill.indexOf(")"));
        return this.rgbColor(c);
      } else {
        Object _xifexpression_2 = null;
        boolean _startsWith_2 = fill.startsWith("argb");
        if (_startsWith_2) {
          _xifexpression_2 = null;
        } else {
          boolean _startsWith_3 = fill.startsWith("url");
          if (_startsWith_3) {
            int _length = fill.length();
            int _minus = (_length - 1);
            final SvgElement e = this.resolveElement(fill.substring(5, _minus));
            if ((e instanceof SvgLinearGradientElement)) {
              return this.handleGradient(((SvgLinearGradientElement) e), opacity);
            } else {
              if ((e instanceof SvgRadialGradientElement)) {
                return this.handleGradient(((SvgRadialGradientElement) e), opacity);
              }
            }
          } else {
            final Colors c_1 = Colors.findColorByName(fill);
            boolean _notEquals = (!Objects.equal(c_1, null));
            if (_notEquals) {
              return this.hexColor(c_1.hexvalue, opacity);
            }
            return fill.toUpperCase();
          }
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return ((CharSequence)_xifexpression);
  }
  
  public CharSequence rgbColor(final String fill) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Color>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<red>");
    double _rgbRed = this.rgbRed(fill);
    _builder.append(_rgbRed, "\t");
    _builder.append("</red>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<green>");
    double _rgbGreen = this.rgbGreen(fill);
    _builder.append(_rgbGreen, "\t");
    _builder.append("</green>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<blue>");
    double _rgbBlue = this.rgbBlue(fill);
    _builder.append(_rgbBlue, "\t");
    _builder.append("</blue>");
    _builder.newLineIfNotEmpty();
    _builder.append("</Color>");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence rgbColor(final String fill, final Double opacity) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Color>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<red>");
    double _rgbRed = this.rgbRed(fill);
    _builder.append(_rgbRed, "\t");
    _builder.append("</red>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<green>");
    double _rgbGreen = this.rgbGreen(fill);
    _builder.append(_rgbGreen, "\t");
    _builder.append("</green>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<blue>");
    double _rgbBlue = this.rgbBlue(fill);
    _builder.append(_rgbBlue, "\t");
    _builder.append("</blue>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<opacity>");
    _builder.append(opacity, "\t");
    _builder.append("</opacity>");
    _builder.newLineIfNotEmpty();
    _builder.append("</Color>");
    _builder.newLine();
    return _builder;
  }
  
  public double rgbRed(final String color) {
    int _parseInt = Integer.parseInt(color.split(",")[0]);
    double _parseDouble = Double.parseDouble("255");
    return (_parseInt / _parseDouble);
  }
  
  public double rgbGreen(final String color) {
    int _parseInt = Integer.parseInt(color.split(",")[1]);
    double _parseDouble = Double.parseDouble("255");
    return (_parseInt / _parseDouble);
  }
  
  public double rgbBlue(final String color) {
    int _parseInt = Integer.parseInt(color.split(",")[2]);
    double _parseDouble = Double.parseDouble("255");
    return (_parseInt / _parseDouble);
  }
  
  public CharSequence hexColor(final String fill) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Color>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<red>");
    double _hexRed = this.hexRed(fill);
    _builder.append(_hexRed, "\t");
    _builder.append("</red>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<green>");
    double _hexGreen = this.hexGreen(fill);
    _builder.append(_hexGreen, "\t");
    _builder.append("</green>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<blue>");
    double _hexBlue = this.hexBlue(fill);
    _builder.append(_hexBlue, "\t");
    _builder.append("</blue>");
    _builder.newLineIfNotEmpty();
    _builder.append("</Color>");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence hexColor(final String fill, final Double opacity) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<Color>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<red>");
    double _hexRed = this.hexRed(fill);
    _builder.append(_hexRed, "\t");
    _builder.append("</red>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<green>");
    double _hexGreen = this.hexGreen(fill);
    _builder.append(_hexGreen, "\t");
    _builder.append("</green>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<blue>");
    double _hexBlue = this.hexBlue(fill);
    _builder.append(_hexBlue, "\t");
    _builder.append("</blue>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<opacity>");
    _builder.append(opacity, "\t");
    _builder.append("</opacity>");
    _builder.newLineIfNotEmpty();
    _builder.append("</Color>");
    _builder.newLine();
    return _builder;
  }
  
  public double hexRed(final String color) {
    return Colors.hexRed(color);
  }
  
  public double hexGreen(final String color) {
    return Colors.hexGreen(color);
  }
  
  public double hexBlue(final String color) {
    return Colors.hexBlue(color);
  }
  
  public CharSequence transformElement(final String transform) {
    StringConcatenation _builder = new StringConcatenation();
    int _indexOf = transform.indexOf("(");
    int _plus = (_indexOf + 1);
    final String params = transform.substring(_plus, transform.indexOf(")"));
    _builder.newLineIfNotEmpty();
    {
      boolean _startsWith = transform.startsWith("translate");
      if (_startsWith) {
        _builder.append("<Translate");
        _builder.newLine();
        {
          int _indexOf_1 = params.indexOf(",");
          boolean _notEquals = (_indexOf_1 != (-1));
          if (_notEquals) {
            _builder.append("\t");
            final String[] parts = params.split(",");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("x=\"");
            String _get = parts[0];
            _builder.append(_get, "\t");
            _builder.append("\"");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("y=\"");
            String _get_1 = parts[1];
            _builder.append(_get_1, "\t");
            _builder.append("\"");
            _builder.newLineIfNotEmpty();
          } else {
            _builder.append("\t");
            _builder.append("x=\"");
            _builder.append(params, "\t");
            _builder.append("\"");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("y=\"");
            _builder.append(params, "\t");
            _builder.append("\"");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append(">");
        _builder.newLine();
        _builder.append("</Translate>");
        _builder.newLine();
      } else {
        boolean _startsWith_1 = transform.startsWith("scale");
        if (_startsWith_1) {
          _builder.append("<Scale");
          _builder.newLine();
          {
            int _indexOf_2 = params.indexOf(",");
            boolean _notEquals_1 = (_indexOf_2 != (-1));
            if (_notEquals_1) {
              _builder.append("\t");
              final String[] parts_1 = params.split(",");
              _builder.newLineIfNotEmpty();
              _builder.append("\t");
              _builder.append("x=\"");
              String _get_2 = parts_1[0];
              _builder.append(_get_2, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
              _builder.append("\t");
              _builder.append("y=\"");
              String _get_3 = parts_1[1];
              _builder.append(_get_3, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
            } else {
              _builder.append("\t");
              _builder.append("x=\"");
              _builder.append(params, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
              _builder.append("\t");
              _builder.append("y=\"");
              _builder.append(params, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
            }
          }
          _builder.append(">");
          _builder.newLine();
          _builder.append("</Scale>");
          _builder.newLine();
        } else {
          boolean _startsWith_2 = transform.startsWith("rotate");
          if (_startsWith_2) {
            _builder.append("<Rotate");
            _builder.newLine();
            {
              int _indexOf_3 = params.indexOf(",");
              boolean _notEquals_2 = (_indexOf_3 != (-1));
              if (_notEquals_2) {
                _builder.append("\t");
                final String[] parts_2 = params.split(",");
                _builder.newLineIfNotEmpty();
                {
                  int _size = ((List<String>)Conversions.doWrapArray(parts_2)).size();
                  boolean _equals = (_size == 2);
                  if (_equals) {
                    _builder.append("\t");
                    _builder.append("angle=\"");
                    String _get_4 = parts_2[0];
                    _builder.append(_get_4, "\t");
                    _builder.append("\"");
                    _builder.newLineIfNotEmpty();
                    _builder.append("\t");
                    _builder.append("pivotX=\"");
                    String _get_5 = parts_2[1];
                    _builder.append(_get_5, "\t");
                    _builder.append("\"");
                    _builder.newLineIfNotEmpty();
                    _builder.append("\t");
                    _builder.append("pivotY=\"");
                    String _get_6 = parts_2[1];
                    _builder.append(_get_6, "\t");
                    _builder.append("\"");
                    _builder.newLineIfNotEmpty();
                  } else {
                    int _size_1 = ((List<String>)Conversions.doWrapArray(parts_2)).size();
                    boolean _equals_1 = (_size_1 == 3);
                    if (_equals_1) {
                      _builder.append("\t");
                      _builder.append("angle=\"");
                      String _get_7 = parts_2[0];
                      _builder.append(_get_7, "\t");
                      _builder.append("\"");
                      _builder.newLineIfNotEmpty();
                      _builder.append("\t");
                      _builder.append("pivotX=\"");
                      String _get_8 = parts_2[1];
                      _builder.append(_get_8, "\t");
                      _builder.append("\"");
                      _builder.newLineIfNotEmpty();
                      _builder.append("\t");
                      _builder.append("pivotY=\"");
                      String _get_9 = parts_2[2];
                      _builder.append(_get_9, "\t");
                      _builder.append("\"");
                      _builder.newLineIfNotEmpty();
                    }
                  }
                }
              } else {
                _builder.append("\t");
                _builder.append("angle=\"");
                _builder.append(params, "\t");
                _builder.append("\"");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append(">");
            _builder.newLine();
            _builder.append("</Rotate>");
            _builder.newLine();
          } else {
            boolean _startsWith_3 = transform.startsWith("matrix");
            if (_startsWith_3) {
              final String[] parts_3 = params.split("[,\\s+]");
              _builder.newLineIfNotEmpty();
              _builder.append("<Affine");
              _builder.newLine();
              _builder.append("\t");
              _builder.append("mxx=\"");
              String _get_10 = parts_3[0];
              _builder.append(_get_10, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
              _builder.append("\t");
              _builder.append("myx=\"");
              String _get_11 = parts_3[1];
              _builder.append(_get_11, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
              _builder.append("\t");
              _builder.append("mxy=\"");
              String _get_12 = parts_3[2];
              _builder.append(_get_12, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
              _builder.append("\t");
              _builder.append("myy=\"");
              String _get_13 = parts_3[3];
              _builder.append(_get_13, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
              _builder.append("\t");
              _builder.append("tx=\"");
              String _get_14 = parts_3[4];
              _builder.append(_get_14, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
              _builder.append("\t");
              _builder.append("ty=\"");
              String _get_15 = parts_3[5];
              _builder.append(_get_15, "\t");
              _builder.append("\"");
              _builder.newLineIfNotEmpty();
              _builder.append("\t");
              _builder.append(">");
              _builder.newLine();
              _builder.append("</Affine>");
              _builder.newLine();
            }
          }
        }
      }
    }
    return _builder;
  }
  
  public String handleTransform(final String t) {
    String transform = t;
    int endIdx = 0;
    final StringBuilder builder = new StringBuilder();
    while (((endIdx = transform.indexOf(")")) != (-1))) {
      {
        builder.append(this.transformElement(transform.substring(0, (endIdx + 1))));
        int _length = transform.length();
        boolean _greaterThan = ((endIdx + 1) > _length);
        if (_greaterThan) {
          return builder.toString();
        }
        transform = transform.substring((endIdx + 1)).trim();
      }
    }
    return builder.toString();
  }
  
  protected CharSequence _handle(final SvgFilterElement filter) {
    return null;
  }
  
  protected CharSequence _handle(final SvgFeGaussianBlurElement f) {
    return null;
  }
  
  public String toFx(final SpreadMethod m) {
    if (m != null) {
      switch (m) {
        case PAD:
          return "NO_CYCLE";
        case REFLECT:
          return "REFLECT";
        case REPEAT:
          return "REPEAT";
        default:
          break;
      }
    }
    return "NO_CYCLE";
  }
  
  public String toFx(final Stroke_linecap m) {
    if (m != null) {
      switch (m) {
        case BUTT:
          return "BUTT";
        case ROUND:
          return "ROUND";
        case SQUARE:
          return "SQUARE";
        case INHERIT:
          return "INHERIT";
        default:
          break;
      }
    }
    return "BUTT";
  }
  
  public String toFx(final Stroke_linejoin m) {
    if (m != null) {
      switch (m) {
        case BEVEL:
          return "BEVEL";
        case MITER:
          return "MITER";
        case ROUND:
          return "ROUND";
        case INHERIT:
          return "INHERIT";
        default:
          break;
      }
    }
    return "BEVEL";
  }
  
  public double parseLength(final String length) {
    boolean _notEquals = (!Objects.equal(length, null));
    if (_notEquals) {
      boolean _endsWith = length.endsWith("px");
      if (_endsWith) {
        int _length = length.length();
        int _minus = (_length - 2);
        return Double.parseDouble(length.substring(0, _minus));
      } else {
        return Double.parseDouble(length);
      }
    }
    return 0.0;
  }
  
  public double parseCoordinate(final String coordinate) {
    boolean _notEquals = (!Objects.equal(coordinate, null));
    if (_notEquals) {
      return Double.parseDouble(coordinate);
    }
    return 0.0;
  }
  
  public double parsePercentage(final String percentage) {
    boolean _notEquals = (!Objects.equal(percentage, null));
    if (_notEquals) {
      return Double.parseDouble(percentage);
    }
    return 0.0;
  }
  
  public double parseDouble(final String value) {
    boolean _notEquals = (!Objects.equal(value, null));
    if (_notEquals) {
      return Double.parseDouble(value);
    }
    return 0.0;
  }
  
  public String validateId(final String value) {
    return value.replaceAll("\\.", "_dot").replaceAll(" ", "_space").replaceAll(":", "_colon");
  }
  
  public CharSequence handle(final EObject element) {
    if (element instanceof SvgDefsElement) {
      return _handle((SvgDefsElement)element);
    } else if (element instanceof SvgGElement) {
      return _handle((SvgGElement)element);
    } else if (element instanceof SvgSvgElement) {
      return _handle((SvgSvgElement)element);
    } else if (element instanceof SvgCircleElement) {
      return _handle((SvgCircleElement)element);
    } else if (element instanceof SvgClipPathElement) {
      return _handle((SvgClipPathElement)element);
    } else if (element instanceof SvgEllipseElement) {
      return _handle((SvgEllipseElement)element);
    } else if (element instanceof SvgFeGaussianBlurElement) {
      return _handle((SvgFeGaussianBlurElement)element);
    } else if (element instanceof SvgFilterElement) {
      return _handle((SvgFilterElement)element);
    } else if (element instanceof SvgImageElement) {
      return _handle((SvgImageElement)element);
    } else if (element instanceof SvgLinearGradientElement) {
      return _handle((SvgLinearGradientElement)element);
    } else if (element instanceof SvgPathElement) {
      return _handle((SvgPathElement)element);
    } else if (element instanceof SvgPolygonElement) {
      return _handle((SvgPolygonElement)element);
    } else if (element instanceof SvgPolylineElement) {
      return _handle((SvgPolylineElement)element);
    } else if (element instanceof SvgRadialGradientElement) {
      return _handle((SvgRadialGradientElement)element);
    } else if (element instanceof SvgRectElement) {
      return _handle((SvgRectElement)element);
    } else if (element instanceof SvgUseElement) {
      return _handle((SvgUseElement)element);
    } else if (element != null) {
      return _handle(element);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(element).toString());
    }
  }
  
  public CharSequence handleFilter(final EObject blur) {
    if (blur instanceof SvgFeGaussianBlurElement) {
      return _handleFilter((SvgFeGaussianBlurElement)blur);
    } else if (blur != null) {
      return _handleFilter(blur);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(blur).toString());
    }
  }
}
