/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.Query;

import org.eclipse.ebam.model.entities.SbiAlarm;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



public class SbiAlarmTransactions extends EMF {

	/**
	 * Logger.
	 */
	private static Logger logger = 
		LoggerFactory.getLogger(SbiAlarmTransactions.class);


	/** Get all alarms.
	 * 
	 * @return list of alarms
	 * @throws Exception
	 */

	public final List getSbiAlarmList() throws Exception {
		List alarms = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiAlarm.class);	
			Query query = em.createQuery("select a from "
					+ "SbiAlarm a");
			//query.setParameter("code", code);
			alarms = query.getResultList();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return alarms;
	}

	/** get Alarm by id
	 * 
	 * @param id
	 * @return one AlarmContact
	 * @throws Exception
	 */

	public final SbiAlarm getSbiAlarmById(Integer id) 
	throws Exception {
		SbiAlarm alarm = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiAlarm.class);	
			Query query = em.createQuery("select s from "
					+ "SbiAlarm s where s.alarmId = :id");
			query.setParameter("id", id);
			alarm = (SbiAlarm) query.getSingleResult();        

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return alarm;
	}	

	/** get Alarm by label
	 * 
	 * @param label
	 * @return one AlarmContact
	 * @throws Exception
	 */

	public final SbiAlarm getSbiAlarmByLabel(String label) 
	throws Exception {
		SbiAlarm alarm = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiAlarm.class);	
			Query query = em.createQuery("select s from "
					+ "SbiAlarm s where s.label = :label");
			query.setParameter("label", label);
			alarm = (SbiAlarm) query.getSingleResult();        

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return alarm;
	}	

	/**Method that persists <code>SbiAlarm</code> on database.
	 * @param j <code>SbiAlarm</code> to save
	 * @return operation result
	 * @throws Exception error on saving 
	 */
	public final boolean createNewSbiAlarm(final SbiAlarm sbiAlarm) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			em.persist(sbiAlarm);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}



	public final boolean mergeSbiAlarm(final SbiAlarm sbiAlarm) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			//take all contacts

			em.merge(sbiAlarm);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}


	public final boolean deleteSbiAlarm(final SbiAlarm sbiAlarm) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			em.setFlushMode(FlushModeType.COMMIT);

			SbiAlarm toRemove = em.find(SbiAlarm.class, sbiAlarm.getAlarmId());
			toRemove.setSbiAlarmContacts(null);
			em.remove(toRemove);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

	
}
