/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.birt.core.archive.compound.v3;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import org.eclipse.birt.core.archive.cache.CacheListener;
import org.eclipse.birt.core.archive.cache.Cacheable;
import org.eclipse.birt.core.archive.cache.FileCacheManager;
import org.eclipse.birt.core.archive.cache.SystemCacheManager;
import org.eclipse.birt.core.archive.compound.v3.Block;
import org.eclipse.birt.core.archive.compound.v3.DataBlock;
import org.eclipse.birt.core.archive.compound.v3.EntryTable;
import org.eclipse.birt.core.archive.compound.v3.Ext2Block;
import org.eclipse.birt.core.archive.compound.v3.Ext2Entry;
import org.eclipse.birt.core.archive.compound.v3.Ext2File;
import org.eclipse.birt.core.archive.compound.v3.Ext2Node;
import org.eclipse.birt.core.archive.compound.v3.FatBlock;
import org.eclipse.birt.core.archive.compound.v3.FreeBlockTable;
import org.eclipse.birt.core.archive.compound.v3.NodeTable;

public class Ext2FileSystem {
    private RandomAccessFile rf;
    private long length;
    private int maxBlockId;
    private String fileName;
    private boolean readOnly;
    private boolean removeOnExit;
    private HashMap<String, String> properties = new HashMap();
    private boolean propertyDirty = true;
    protected FileCacheManager cacheManager = new FileCacheManager();
    private NodeTable nodeTable = new NodeTable(this);
    private EntryTable entryTable = new EntryTable(this);
    private FreeBlockTable freeTable = new FreeBlockTable(this);
    private HashSet<Ext2File> openedFiles = new HashSet();
    static final int HEADER_SIZE = 1024;
    public static final long EXT2_MAGIC_TAG = 5931333361611265586L;
    static final int EXT2_VERSION_0 = 0;
    static final int BLOCK_SIZE = 4096;
    static final int BLOCK_SIZE_BITS = 12;
    static final int BLOCK_OFFSET_MASK = 4095;

    public Ext2FileSystem(String filePath, String mode) throws IOException {
        this(filePath, null, mode);
    }

    public Ext2FileSystem(String filePath, RandomAccessFile rf, String mode) throws IOException {
        this.fileName = new File(filePath).getCanonicalPath();
        this.rf = rf;
        this.cacheManager.setCacheListener(new Ext2FileSystemCacheListener());
        if ("rw".equals(mode)) {
            this.readOnly = false;
            this.removeOnExit = false;
            this.createFileSystem();
            return;
        }
        if ("rw+".equals(mode)) {
            this.readOnly = false;
            this.removeOnExit = false;
            if (new File(this.fileName).exists()) {
                this.openFileSystem();
            } else {
                this.createFileSystem();
            }
            return;
        }
        if ("r".equals(mode)) {
            this.readOnly = true;
            this.removeOnExit = false;
            this.openFileSystem();
            return;
        }
        if ("rwt".equals(mode)) {
            this.readOnly = false;
            this.removeOnExit = true;
            this.createFileSystem();
            return;
        }
        throw new IOException("unsupported file mode:" + mode);
    }

    private void openFileSystem() throws IOException {
        if (this.rf == null) {
            this.rf = this.readOnly ? new RandomAccessFile(this.fileName, "r") : new RandomAccessFile(this.fileName, "rw");
        }
        this.length = this.rf.length();
        this.maxBlockId = (int)((this.length + 4096L - 1L) / 4096L) + 1;
        this.readHeader();
        this.nodeTable.read();
        this.entryTable.read();
        this.freeTable.read();
        this.readProperties();
    }

    private void ensureParentFolderCreated(String fileName) {
        File parentFile = new File(fileName).getParentFile();
        if (parentFile != null && !parentFile.exists()) {
            parentFile.mkdirs();
        }
    }

    private void createFileSystem() throws IOException {
        if (!this.removeOnExit) {
            if (this.rf == null) {
                this.ensureParentFolderCreated(this.fileName);
                this.rf = new RandomAccessFile(this.fileName, "rw");
            }
            this.rf.setLength(0L);
            this.writeProperties();
            this.entryTable.write();
            this.freeTable.write();
            this.nodeTable.write();
            this.writeHeader();
        }
        this.length = 0L;
        this.maxBlockId = 2;
    }

    public void setRemoveOnExit(boolean mode) {
        this.removeOnExit = mode;
    }

    public synchronized void close() throws IOException {
        try {
            this.closeFiles();
            if (!this.readOnly && !this.removeOnExit) {
                this.writeProperties();
                this.entryTable.write();
                this.nodeTable.write();
                this.freeTable.write();
                this.nodeTable.write(2);
                this.cacheManager.touchAllCaches();
                this.writeHeader();
            }
            this.cacheManager.clear();
        }
        finally {
            if (this.rf != null) {
                this.rf.close();
                this.rf = null;
            }
            if (this.removeOnExit) {
                new File(this.fileName).delete();
            }
        }
    }

    private void closeFiles() throws IOException {
        if (this.openedFiles != null) {
            ArrayList<Ext2File> files = new ArrayList<Ext2File>(this.openedFiles);
            for (Ext2File file : files) {
                if (file == null) continue;
                file.close();
            }
            this.openedFiles.clear();
        }
    }

    public synchronized void flush() throws IOException {
        if (!this.removeOnExit) {
            if (this.readOnly) {
                throw new IOException("file is opened as read only");
            }
            this.ensureFileOpened();
            this.writeProperties();
            this.entryTable.write();
            this.nodeTable.write();
            this.freeTable.write();
            this.nodeTable.write(2);
            this.cacheManager.touchAllCaches(new Ext2FileSystemCacheListener());
        }
    }

    public void refresh() throws IOException {
        throw new UnsupportedOperationException("refresh");
    }

    public boolean isReadOnly() {
        return this.readOnly;
    }

    public boolean isRemoveOnExit() {
        return this.removeOnExit;
    }

    synchronized void registerOpenedFile(Ext2File file) {
        this.openedFiles.add(file);
    }

    synchronized void unregisterOpenedFile(Ext2File file) {
        this.openedFiles.remove(file);
    }

    public void setCacheSize(int cacheSize) {
        this.cacheManager.setMaxCacheSize(cacheSize);
    }

    public void setCacheManager(SystemCacheManager manager) {
        this.cacheManager.setSystemCacheManager(manager);
    }

    public int getUsedCacheSize() {
        return this.cacheManager.getUsedCacheSize();
    }

    public synchronized Ext2File createFile(String name) throws IOException {
        Ext2Node node;
        if (this.readOnly) {
            throw new IOException("the file is opened in read only mode");
        }
        Ext2Entry entry = this.entryTable.getEntry(name);
        if (entry == null) {
            node = this.nodeTable.allocateNode();
            entry = new Ext2Entry(name, node.getNodeId());
            this.entryTable.addEntry(entry);
        }
        node = this.nodeTable.getNode(entry.inode);
        Ext2File file = new Ext2File(this, entry, node);
        file.setLength(0L);
        return file;
    }

    public synchronized Ext2File openFile(String name) throws IOException {
        Ext2Entry entry = this.entryTable.getEntry(name);
        if (entry != null) {
            Ext2Node node = this.nodeTable.getNode(entry.inode);
            return new Ext2File(this, entry, node);
        }
        if (!this.readOnly) {
            return this.createFile(name);
        }
        throw new FileNotFoundException(name);
    }

    public synchronized boolean existFile(String name) {
        return this.entryTable.getEntry(name) != null;
    }

    public synchronized String[] listFiles() {
        return this.entryTable.listEntries();
    }

    public synchronized void removeFile(String name) throws IOException {
        Ext2Entry entry;
        if (this.readOnly) {
            throw new IOException("The file is opend in read only mode");
        }
        if (!this.openedFiles.isEmpty()) {
            ArrayList<Ext2File> removedFiles = new ArrayList<Ext2File>();
            for (Ext2File file : this.openedFiles) {
                if (!name.equals(file.getName())) continue;
                removedFiles.add(file);
            }
            for (Ext2File file : removedFiles) {
                file.close();
            }
        }
        if ((entry = this.entryTable.removeEntry(name)) != null) {
            this.nodeTable.releaseNode(entry.inode);
        }
    }

    public String getFileName() {
        return this.fileName;
    }

    public String getProperty(String name) {
        assert (name != null);
        return this.properties.get(name);
    }

    public void setProperty(String name, String value) {
        assert (name != null);
        if (value == null) {
            this.properties.remove(name);
        } else {
            this.properties.put(name, value);
        }
        this.propertyDirty = true;
    }

    private void readHeader() throws IOException {
        byte[] bytes = new byte[1024];
        this.rf.seek(0L);
        this.rf.readFully(bytes);
        DataInputStream in = new DataInputStream(new ByteArrayInputStream(bytes));
        long magicTag = in.readLong();
        if (magicTag != 5931333361611265586L) {
            throw new IOException("Not a Ext2 archive, the magic code is " + magicTag);
        }
        int version = in.readInt();
        if (version != 0) {
            throw new IOException("Unsupported archive version " + version);
        }
        int blockSize = in.readInt();
        if (blockSize != 4096) {
            throw new IOException("unsupported block size");
        }
    }

    private void readProperties() throws IOException {
        Ext2File file = new Ext2File(this, 1, false);
        try {
            byte[] bytes = new byte[(int)(file.length() - 1024L)];
            file.seek(1024L);
            file.read(bytes, 0, bytes.length);
            DataInputStream in = new DataInputStream(new ByteArrayInputStream(bytes));
            int count = in.readInt();
            int i = 0;
            while (i < count) {
                String name = in.readUTF();
                String value = in.readUTF();
                if (!this.properties.containsKey(name)) {
                    this.properties.put(name, value);
                }
                ++i;
            }
        }
        finally {
            file.close();
        }
        this.propertyDirty = false;
    }

    private void writeHeader() throws IOException {
        ByteArrayOutputStream bytes = new ByteArrayOutputStream(4096);
        DataOutputStream out = new DataOutputStream(bytes);
        out.writeLong(5931333361611265586L);
        out.writeInt(0);
        out.writeInt(4096);
        this.rf.seek(0L);
        this.rf.write(bytes.toByteArray());
    }

    private void writeProperties() throws IOException {
        if (!this.propertyDirty) {
            return;
        }
        this.propertyDirty = false;
        Ext2File file = new Ext2File(this, 1, false);
        try {
            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            DataOutputStream out = new DataOutputStream(buffer);
            out.writeInt(this.properties.size());
            for (Map.Entry<String, String> entry : this.properties.entrySet()) {
                String name = entry.getKey();
                String value = entry.getValue();
                out.writeUTF(name);
                out.writeUTF(value);
            }
            byte[] bytes = buffer.toByteArray();
            file.seek(1024L);
            file.write(bytes, 0, bytes.length);
        }
        finally {
            file.close();
        }
    }

    protected synchronized int allocFreeBlock() throws IOException {
        int blockId = this.freeTable.getFreeBlock();
        if (blockId > 0) {
            return blockId;
        }
        return this.maxBlockId++;
    }

    void releaseFreeBlocks(Ext2Node node) {
        this.freeTable.addFreeBlocks(node);
    }

    protected synchronized FatBlock createFatBlock() throws IOException {
        int blockId = this.allocFreeBlock();
        FatBlock block = new FatBlock(this, blockId);
        this.cacheManager.addCache(block);
        return block;
    }

    protected synchronized DataBlock createDataBlock() throws IOException {
        int blockId = this.allocFreeBlock();
        DataBlock block = new DataBlock(this, blockId);
        this.cacheManager.addCache(block);
        return block;
    }

    protected synchronized void unloadBlock(Block block) throws IOException {
        this.cacheManager.releaseCache(block);
    }

    protected synchronized FatBlock loadFatBlock(int blockId) throws IOException {
        FatBlock block = (FatBlock)this.cacheManager.getCache(blockId);
        if (block == null) {
            block = new FatBlock(this, blockId);
            block.refresh();
            this.cacheManager.addCache(block);
        }
        return block;
    }

    synchronized DataBlock loadDataBlock(int blockId) throws IOException {
        Integer cacheKey = blockId;
        DataBlock block = (DataBlock)this.cacheManager.getCache(cacheKey);
        if (block == null) {
            block = new DataBlock(this, blockId);
            block.refresh();
            this.cacheManager.addCache(block);
        }
        return block;
    }

    void readBlock(int blockId, byte[] buffer, int offset, int size) throws IOException {
        this.readBlock(blockId, offset, buffer, offset, size);
    }

    synchronized void readBlock(int blockId, int blockOff, byte[] buffer, int offset, int size) throws IOException {
        assert (buffer != null);
        assert (blockId >= 0);
        assert (offset >= 0);
        assert (blockOff >= 0);
        assert (offset + size <= buffer.length);
        assert (blockOff + size <= 4096);
        long position = ((long)blockId << 12) + (long)blockOff;
        if (position < this.length) {
            long remainSize = this.length - position;
            this.rf.seek(position);
            if (remainSize < (long)size) {
                size = (int)remainSize;
            }
            this.rf.readFully(buffer, offset, size);
        }
    }

    void writeBlock(int blockId, byte[] buffer, int offset, int size) throws IOException {
        this.writeBlock(blockId, offset, buffer, offset, size);
    }

    synchronized void writeBlock(int blockId, int blockOff, byte[] buffer, int offset, int size) throws IOException {
        assert (buffer != null);
        assert (blockId >= 0);
        assert (offset >= 0);
        assert (blockOff >= 0);
        assert (offset + size <= buffer.length);
        assert (blockOff + size <= 4096);
        this.ensureFileOpened();
        long position = ((long)blockId << 12) + (long)blockOff;
        this.rf.seek(position);
        this.rf.write(buffer, offset, size);
        if ((position += (long)size) > this.length) {
            this.length = position;
        }
    }

    public Ext2Entry getEntry(String name) {
        return this.entryTable.getEntry(name);
    }

    public Ext2Node getNode(int nodeId) {
        return this.nodeTable.getNode(nodeId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void ensureFileOpened() throws IOException {
        if (this.rf != null) {
            return;
        }
        Ext2FileSystem ext2FileSystem = this;
        synchronized (ext2FileSystem) {
            if (this.rf == null) {
                this.ensureParentFolderCreated(this.fileName);
                this.rf = new RandomAccessFile(this.fileName, "rw");
                this.rf.setLength(0L);
            }
        }
    }

    static class Ext2FileSystemCacheListener
    implements CacheListener {
        Ext2FileSystemCacheListener() {
        }

        public void onCacheRelease(Cacheable cache) {
            Ext2Block block = (Ext2Block)cache;
            try {
                block.flush();
            }
            catch (IOException ex) {
                ex.printStackTrace();
            }
        }
    }
}

