"use strict";
/*******************************************************************************
* Copyright (c) 2019, 2020 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v2.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v20.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const FileChangeEventBatchUtil_1 = require("./FileChangeEventBatchUtil");
const CLIState_1 = require("./CLIState");
const log = require("./Logger");
const PathUtils_1 = require("./PathUtils");
/**
 * Information maintained for each project that is being monitored by the watcher.
 * This includes information on what to watch/filter (the ProjectToWatch object),
 * the batch util (one batch util object exists per project), and which watch
 * service (internal/external) is being used for this project.
 */
class ProjectObject {
    constructor(projectId, projectToWatch, watchService, parent) {
        if (!projectId || !projectToWatch || !watchService || !parent) {
            log.severe("Missing parameters to constructor of ProjectObject!");
        }
        this._projectToWatch = projectToWatch;
        this._batchUtil = new FileChangeEventBatchUtil_1.FileChangeEventBatchUtil(projectId, parent);
        this._watchService = watchService;
        //  Here we convert the path to an absolute, canonical OS path for use by cwctl
        this._cliState = new CLIState_1.CLIState(projectId, parent.installerPath, PathUtils_1.convertAbsoluteUnixStyleNormalizedPathToLocalFile(projectToWatch.pathToMonitor));
    }
    updateProjectToWatch(newProjectToWatch) {
        const existingProjectToWatch = this._projectToWatch;
        if (existingProjectToWatch.pathToMonitor !== newProjectToWatch.pathToMonitor) {
            const msg = "The pathToMonitor of a project cannot be changed once it is set, for a particular project id";
            log.severe(msg);
        }
        this._projectToWatch = newProjectToWatch;
    }
    informCwctlOfFileChangesAsync() {
        this._cliState.onFileChangeEvent(this._projectToWatch.projectCreationTimeInAbsoluteMsecs, this._projectToWatch);
    }
    get projectToWatch() {
        return this._projectToWatch;
    }
    get batchUtil() {
        return this._batchUtil;
    }
    get watchService() {
        return this._watchService;
    }
}
exports.ProjectObject = ProjectObject;
