/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.orbits;

import java.io.Serializable;
import java.util.Collection;
import org.apache.commons.math3.analysis.interpolation.HermiteInterpolator;
import org.apache.commons.math3.exception.ConvergenceException;
import org.apache.commons.math3.exception.util.Localizable;
import org.apache.commons.math3.geometry.Vector;
import org.apache.commons.math3.geometry.euclidean.threed.Vector3D;
import org.apache.commons.math3.util.FastMath;
import org.apache.commons.math3.util.MathUtils;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngle;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class KeplerianOrbit
extends Orbit {
    private static final long serialVersionUID = 20141228L;
    private static final double A;
    private static final double B;
    private final double a;
    private final double e;
    private final double i;
    private final double pa;
    private final double raan;
    private final double v;

    public KeplerianOrbit(double a, double e, double i, double pa, double raan, double anomaly, PositionAngle type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        super(frame, date, mu);
        double tmpV;
        if (a * (1.0 - e) < 0.0) {
            throw OrekitException.createIllegalArgumentException(OrekitMessages.ORBIT_A_E_MISMATCH_WITH_CONIC_TYPE, a, e);
        }
        this.a = a;
        this.e = e;
        this.i = i;
        this.pa = pa;
        this.raan = raan;
        switch (type) {
            case MEAN: {
                tmpV = a < 0.0 ? this.hyperbolicEccentricToTrue(this.meanToHyperbolicEccentric(anomaly, e)) : this.ellipticEccentricToTrue(this.meanToEllipticEccentric(anomaly));
                break;
            }
            case ECCENTRIC: {
                tmpV = a < 0.0 ? this.hyperbolicEccentricToTrue(anomaly) : this.ellipticEccentricToTrue(anomaly);
                break;
            }
            case TRUE: {
                tmpV = anomaly;
                break;
            }
            default: {
                throw OrekitException.createInternalError(null);
            }
        }
        if (1.0 + e * FastMath.cos((double)tmpV) <= 0.0) {
            double vMax = FastMath.acos((double)(-1.0 / e));
            throw OrekitException.createIllegalArgumentException(OrekitMessages.ORBIT_ANOMALY_OUT_OF_HYPERBOLIC_RANGE, tmpV, e, -vMax, vMax);
        }
        this.v = tmpV;
    }

    public KeplerianOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        Vector3D momentum = pvCoordinates.getMomentum();
        double m2 = momentum.getNormSq();
        this.i = Vector3D.angle((Vector3D)momentum, (Vector3D)Vector3D.PLUS_K);
        this.raan = Vector3D.crossProduct((Vector3D)Vector3D.PLUS_K, (Vector3D)momentum).getAlpha();
        Vector3D pvP = pvCoordinates.getPosition();
        Vector3D pvV = pvCoordinates.getVelocity();
        double r = pvP.getNorm();
        double V2 = pvV.getNormSq();
        double rV2OnMu = r * V2 / mu;
        this.a = r / (2.0 - rV2OnMu);
        double muA = mu * this.a;
        if (this.a > 0.0) {
            double eSE = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)pvV) / FastMath.sqrt((double)muA);
            double eCE = rV2OnMu - 1.0;
            this.e = FastMath.sqrt((double)(eSE * eSE + eCE * eCE));
            this.v = this.ellipticEccentricToTrue(FastMath.atan2((double)eSE, (double)eCE));
        } else {
            double eSH = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)pvV) / FastMath.sqrt((double)(-muA));
            double eCH = rV2OnMu - 1.0;
            this.e = FastMath.sqrt((double)(1.0 - m2 / muA));
            this.v = this.hyperbolicEccentricToTrue(FastMath.log((double)((eCH + eSH) / (eCH - eSH))) / 2.0);
        }
        Vector3D node = new Vector3D(this.raan, 0.0);
        double px = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)node);
        double py = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)Vector3D.crossProduct((Vector3D)momentum, (Vector3D)node)) / FastMath.sqrt((double)m2);
        this.pa = FastMath.atan2((double)py, (double)px) - this.v;
    }

    public KeplerianOrbit(PVCoordinates pvCoordinates, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(new TimeStampedPVCoordinates(date, pvCoordinates), frame, mu);
    }

    public KeplerianOrbit(Orbit op) {
        this(op.getPVCoordinates(), op.getFrame(), op.getDate(), op.getMu());
    }

    @Override
    public OrbitType getType() {
        return OrbitType.KEPLERIAN;
    }

    @Override
    public double getA() {
        return this.a;
    }

    @Override
    public double getE() {
        return this.e;
    }

    @Override
    public double getI() {
        return this.i;
    }

    public double getPerigeeArgument() {
        return this.pa;
    }

    public double getRightAscensionOfAscendingNode() {
        return this.raan;
    }

    public double getAnomaly(PositionAngle type) {
        return type == PositionAngle.MEAN ? this.getMeanAnomaly() : (type == PositionAngle.ECCENTRIC ? this.getEccentricAnomaly() : this.getTrueAnomaly());
    }

    public double getTrueAnomaly() {
        return this.v;
    }

    public double getEccentricAnomaly() {
        if (this.a < 0.0) {
            double sinhH = FastMath.sqrt((double)(this.e * this.e - 1.0)) * FastMath.sin((double)this.v) / (1.0 + this.e * FastMath.cos((double)this.v));
            return FastMath.asinh((double)sinhH);
        }
        double beta = this.e / (1.0 + FastMath.sqrt((double)((1.0 - this.e) * (1.0 + this.e))));
        return this.v - 2.0 * FastMath.atan((double)(beta * FastMath.sin((double)this.v) / (1.0 + beta * FastMath.cos((double)this.v))));
    }

    private double ellipticEccentricToTrue(double E) {
        double beta = this.e / (1.0 + FastMath.sqrt((double)((1.0 - this.e) * (1.0 + this.e))));
        return E + 2.0 * FastMath.atan((double)(beta * FastMath.sin((double)E) / (1.0 - beta * FastMath.cos((double)E))));
    }

    private double hyperbolicEccentricToTrue(double H) {
        return 2.0 * FastMath.atan((double)(FastMath.sqrt((double)((this.e + 1.0) / (this.e - 1.0))) * FastMath.tanh((double)(H / 2.0))));
    }

    public double getMeanAnomaly() {
        if (this.a < 0.0) {
            double H = this.getEccentricAnomaly();
            return this.e * FastMath.sinh((double)H) - H;
        }
        double E = this.getEccentricAnomaly();
        return E - this.e * FastMath.sin((double)E);
    }

    private double meanToEllipticEccentric(double M) {
        double w;
        double E;
        double reducedM = MathUtils.normalizeAngle((double)M, (double)0.0);
        if (FastMath.abs((double)reducedM) < 0.16666666666666666) {
            E = reducedM + this.e * (FastMath.cbrt((double)(6.0 * reducedM)) - reducedM);
        } else if (reducedM < 0.0) {
            w = Math.PI + reducedM;
            E = reducedM + this.e * (A * w / (B - w) - Math.PI - reducedM);
        } else {
            w = Math.PI - reducedM;
            E = reducedM + this.e * (Math.PI - A * w / (B - w) - reducedM);
        }
        double e1 = 1.0 - this.e;
        boolean noCancellationRisk = e1 + E * E / 6.0 >= 0.1;
        for (int j = 0; j < 2; ++j) {
            double fd;
            double f;
            double fdd = this.e * FastMath.sin((double)E);
            double fddd = this.e * FastMath.cos((double)E);
            if (noCancellationRisk) {
                f = E - fdd - reducedM;
                fd = 1.0 - fddd;
            } else {
                f = this.eMeSinE(E) - reducedM;
                double s = FastMath.sin((double)(0.5 * E));
                fd = e1 + 2.0 * this.e * s * s;
            }
            double dee = f * fd / (0.5 * f * fdd - fd * fd);
            double w2 = fd + 0.5 * dee * (fdd + dee * fddd / 3.0);
            E -= (f - dee * ((fd += dee * (fdd + 0.5 * dee * fddd)) - w2)) / fd;
        }
        return E += M - reducedM;
    }

    private double eMeSinE(double E) {
        double x;
        double mE2 = -E * E;
        double term = E;
        double d = 0.0;
        double x0 = Double.NaN;
        for (x = (1.0 - this.e) * FastMath.sin((double)E); x != x0; x -= (term *= mE2 / ((d += 2.0) * (d + 1.0)))) {
            x0 = x;
        }
        return x;
    }

    private double meanToHyperbolicEccentric(double M, double ecc) {
        double H = ecc < 1.6 ? (-Math.PI < M && M < 0.0 || M > Math.PI ? M - ecc : M + ecc) : (ecc < 3.6 && FastMath.abs((double)M) > Math.PI ? M - FastMath.copySign((double)ecc, (double)M) : M / (ecc - 1.0));
        int iter = 0;
        do {
            double f3 = ecc * FastMath.cosh((double)H);
            double f2 = ecc * FastMath.sinh((double)H);
            double f1 = f3 - 1.0;
            double f0 = f2 - H - M;
            double f12 = 2.0 * f1;
            double d = f0 / f12;
            double fdf = f1 - d * f2;
            double ds = f0 / fdf;
            double shift = f0 / (fdf + ds * ds * f3 / 6.0);
            H -= shift;
            if (!(FastMath.abs((double)shift) <= 1.0E-12)) continue;
            return H;
        } while (++iter < 50);
        throw new ConvergenceException((Localizable)OrekitMessages.UNABLE_TO_COMPUTE_HYPERBOLIC_ECCENTRIC_ANOMALY, new Object[]{iter});
    }

    @Override
    public double getEquinoctialEx() {
        return this.e * FastMath.cos((double)(this.pa + this.raan));
    }

    @Override
    public double getEquinoctialEy() {
        return this.e * FastMath.sin((double)(this.pa + this.raan));
    }

    @Override
    public double getHx() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        return FastMath.cos((double)this.raan) * FastMath.tan((double)(this.i / 2.0));
    }

    @Override
    public double getHy() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        return FastMath.sin((double)this.raan) * FastMath.tan((double)(this.i / 2.0));
    }

    @Override
    public double getLv() {
        return this.pa + this.raan + this.v;
    }

    @Override
    public double getLE() {
        return this.pa + this.raan + this.getEccentricAnomaly();
    }

    @Override
    public double getLM() {
        return this.pa + this.raan + this.getMeanAnomaly();
    }

    @Override
    protected TimeStampedPVCoordinates initPVCoordinates() {
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        double cosPa = FastMath.cos((double)this.pa);
        double sinPa = FastMath.sin((double)this.pa);
        double cosI = FastMath.cos((double)this.i);
        double sinI = FastMath.sin((double)this.i);
        double crcp = cosRaan * cosPa;
        double crsp = cosRaan * sinPa;
        double srcp = sinRaan * cosPa;
        double srsp = sinRaan * sinPa;
        Vector3D p = new Vector3D(crcp - cosI * srsp, srcp + cosI * crsp, sinI * sinPa);
        Vector3D q = new Vector3D(-crsp - cosI * srcp, -srsp + cosI * crcp, sinI * cosPa);
        return this.a > 0.0 ? this.initPVCoordinatesElliptical(p, q) : this.initPVCoordinatesHyperbolic(p, q);
    }

    private TimeStampedPVCoordinates initPVCoordinatesElliptical(Vector3D p, Vector3D q) {
        double uME2 = (1.0 - this.e) * (1.0 + this.e);
        double s1Me2 = FastMath.sqrt((double)uME2);
        double E = this.getEccentricAnomaly();
        double cosE = FastMath.cos((double)E);
        double sinE = FastMath.sin((double)E);
        double x = this.a * (cosE - this.e);
        double y = this.a * sinE * s1Me2;
        double factor = FastMath.sqrt((double)(this.getMu() / this.a)) / (1.0 - this.e * cosE);
        double xDot = -sinE * factor;
        double yDot = cosE * s1Me2 * factor;
        Vector3D position = new Vector3D(x, p, y, q);
        double r2 = x * x + y * y;
        Vector3D velocity = new Vector3D(xDot, p, yDot, q);
        Vector3D acceleration = new Vector3D(-this.getMu() / (r2 * FastMath.sqrt((double)r2)), position);
        return new TimeStampedPVCoordinates(this.getDate(), position, velocity, acceleration);
    }

    private TimeStampedPVCoordinates initPVCoordinatesHyperbolic(Vector3D p, Vector3D q) {
        double sinV = FastMath.sin((double)this.v);
        double cosV = FastMath.cos((double)this.v);
        double f = this.a * (1.0 - this.e * this.e);
        double posFactor = f / (1.0 + this.e * cosV);
        double velFactor = FastMath.sqrt((double)(this.getMu() / f));
        Vector3D position = new Vector3D(posFactor * cosV, p, posFactor * sinV, q);
        Vector3D velocity = new Vector3D(-velFactor * sinV, p, velFactor * (this.e + cosV), q);
        Vector3D acceleration = new Vector3D(-this.getMu() / (posFactor * posFactor * posFactor), position);
        return new TimeStampedPVCoordinates(this.getDate(), position, velocity, acceleration);
    }

    @Override
    public KeplerianOrbit shiftedBy(double dt) {
        return new KeplerianOrbit(this.a, this.e, this.i, this.pa, this.raan, this.getMeanAnomaly() + this.getKeplerianMeanMotion() * dt, PositionAngle.MEAN, this.getFrame(), this.getDate().shiftedBy(dt), this.getMu());
    }

    @Override
    public KeplerianOrbit interpolate(AbsoluteDate date, Collection<Orbit> sample) {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        AbsoluteDate previousDate = null;
        double previousPA = Double.NaN;
        double previousRAAN = Double.NaN;
        double previousM = Double.NaN;
        for (Orbit orbit : sample) {
            double continuousM;
            double continuousRAAN;
            double continuousPA;
            KeplerianOrbit kep = (KeplerianOrbit)OrbitType.KEPLERIAN.convertType(orbit);
            if (previousDate == null) {
                continuousPA = kep.getPerigeeArgument();
                continuousRAAN = kep.getRightAscensionOfAscendingNode();
                continuousM = kep.getMeanAnomaly();
            } else {
                double dt = kep.getDate().durationFrom(previousDate);
                double keplerM = previousM + kep.getKeplerianMeanMotion() * dt;
                continuousPA = MathUtils.normalizeAngle((double)kep.getPerigeeArgument(), (double)previousPA);
                continuousRAAN = MathUtils.normalizeAngle((double)kep.getRightAscensionOfAscendingNode(), (double)previousRAAN);
                continuousM = MathUtils.normalizeAngle((double)kep.getMeanAnomaly(), (double)keplerM);
            }
            previousDate = kep.getDate();
            previousPA = continuousPA;
            previousRAAN = continuousRAAN;
            previousM = continuousM;
            interpolator.addSamplePoint(kep.getDate().durationFrom(date), (double[][])new double[][]{{kep.getA(), kep.getE(), kep.getI(), continuousPA, continuousRAAN, continuousM}});
        }
        double[] interpolated = interpolator.value(0.0);
        return new KeplerianOrbit(interpolated[0], interpolated[1], interpolated[2], interpolated[3], interpolated[4], interpolated[5], PositionAngle.MEAN, this.getFrame(), date, this.getMu());
    }

    @Override
    protected double[][] computeJacobianMeanWrtCartesian() {
        if (this.a > 0.0) {
            return this.computeJacobianMeanWrtCartesianElliptical();
        }
        return this.computeJacobianMeanWrtCartesianHyperbolic();
    }

    private double[][] computeJacobianMeanWrtCartesianElliptical() {
        double factorI1;
        double[][] jacobian = new double[6][6];
        TimeStampedPVCoordinates pvc = this.getPVCoordinates();
        Vector3D position = pvc.getPosition();
        Vector3D velocity = pvc.getVelocity();
        Vector3D momentum = pvc.getMomentum();
        double v2 = velocity.getNormSq();
        double r2 = position.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double r3 = r * r2;
        double px = position.getX();
        double py = position.getY();
        double pz = position.getZ();
        double vx = velocity.getX();
        double vy = velocity.getY();
        double vz = velocity.getZ();
        double mx = momentum.getX();
        double my = momentum.getY();
        double mz = momentum.getZ();
        double mu = this.getMu();
        double sqrtMuA = FastMath.sqrt((double)(this.a * mu));
        double sqrtAoMu = FastMath.sqrt((double)(this.a / mu));
        double a2 = this.a * this.a;
        double twoA = 2.0 * this.a;
        double rOnA = r / this.a;
        double oMe2 = 1.0 - this.e * this.e;
        double epsilon = FastMath.sqrt((double)oMe2);
        double sqrtRec = 1.0 / epsilon;
        double cosI = FastMath.cos((double)this.i);
        double sinI = FastMath.sin((double)this.i);
        double cosPA = FastMath.cos((double)this.pa);
        double sinPA = FastMath.sin((double)this.pa);
        double pv = Vector3D.dotProduct((Vector3D)position, (Vector3D)velocity);
        double cosE = (this.a - r) / (this.a * this.e);
        double sinE = pv / (this.e * sqrtMuA);
        Vector3D vectorAR = new Vector3D(2.0 * a2 / r3, position);
        Vector3D vectorARDot = velocity.scalarMultiply(2.0 * a2 / mu);
        KeplerianOrbit.fillHalfRow(1.0, vectorAR, jacobian[0], 0);
        KeplerianOrbit.fillHalfRow(1.0, vectorARDot, jacobian[0], 3);
        double factorER3 = pv / twoA;
        Vector3D vectorER = new Vector3D(cosE * v2 / (r * mu), position, sinE / sqrtMuA, velocity, -factorER3 * sinE / sqrtMuA, vectorAR);
        Vector3D vectorERDot = new Vector3D(sinE / sqrtMuA, position, cosE * 2.0 * r / mu, velocity, -factorER3 * sinE / sqrtMuA, vectorARDot);
        KeplerianOrbit.fillHalfRow(1.0, vectorER, jacobian[1], 0);
        KeplerianOrbit.fillHalfRow(1.0, vectorERDot, jacobian[1], 3);
        double coefE = cosE / (this.e * sqrtMuA);
        Vector3D vectorEAnR = new Vector3D(-sinE * v2 / (this.e * r * mu), position, coefE, velocity, -factorER3 * coefE, vectorAR);
        Vector3D vectorEAnRDot = new Vector3D(-sinE * 2.0 * r / (this.e * mu), velocity, coefE, position, -factorER3 * coefE, vectorARDot);
        double s1 = -sinE * pz / r - cosE * vz * sqrtAoMu;
        double s2 = -cosE * pz / r3;
        double s3 = -sinE * vz / (2.0 * sqrtMuA);
        double t1 = sqrtRec * (cosE * pz / r - sinE * vz * sqrtAoMu);
        double t2 = sqrtRec * (-sinE * pz / r3);
        double t3 = sqrtRec * (cosE - this.e) * vz / (2.0 * sqrtMuA);
        double t4 = sqrtRec * (this.e * sinI * cosPA * sqrtRec - vz * sqrtAoMu);
        Vector3D s = new Vector3D(cosE / r, Vector3D.PLUS_K, s1, vectorEAnR, s2, position, s3, vectorAR);
        Vector3D sDot = new Vector3D(-sinE * sqrtAoMu, Vector3D.PLUS_K, s1, vectorEAnRDot, s3, vectorARDot);
        Vector3D t = new Vector3D(sqrtRec * sinE / r, Vector3D.PLUS_K).add((Vector)new Vector3D(t1, vectorEAnR, t2, position, t3, vectorAR, t4, vectorER));
        Vector3D tDot = new Vector3D(sqrtRec * (cosE - this.e) * sqrtAoMu, Vector3D.PLUS_K, t1, vectorEAnRDot, t3, vectorARDot, t4, vectorERDot);
        double i1 = factorI1 = -sinI * sqrtRec / sqrtMuA;
        double i2 = -factorI1 * mz / twoA;
        double i3 = factorI1 * mz * this.e / oMe2;
        double i4 = cosI * sinPA;
        double i5 = cosI * cosPA;
        KeplerianOrbit.fillHalfRow(i1, new Vector3D(vy, -vx, 0.0), i2, vectorAR, i3, vectorER, i4, s, i5, t, jacobian[2], 0);
        KeplerianOrbit.fillHalfRow(i1, new Vector3D(-py, px, 0.0), i2, vectorARDot, i3, vectorERDot, i4, sDot, i5, tDot, jacobian[2], 3);
        KeplerianOrbit.fillHalfRow(cosPA / sinI, s, -sinPA / sinI, t, jacobian[3], 0);
        KeplerianOrbit.fillHalfRow(cosPA / sinI, sDot, -sinPA / sinI, tDot, jacobian[3], 3);
        double factorRaanR = 1.0 / (mu * this.a * oMe2 * sinI * sinI);
        KeplerianOrbit.fillHalfRow(-factorRaanR * my, new Vector3D(0.0, vz, -vy), factorRaanR * mx, new Vector3D(-vz, 0.0, vx), jacobian[4], 0);
        KeplerianOrbit.fillHalfRow(-factorRaanR * my, new Vector3D(0.0, -pz, py), factorRaanR * mx, new Vector3D(pz, 0.0, -px), jacobian[4], 3);
        KeplerianOrbit.fillHalfRow(rOnA, vectorEAnR, -sinE, vectorER, jacobian[5], 0);
        KeplerianOrbit.fillHalfRow(rOnA, vectorEAnRDot, -sinE, vectorERDot, jacobian[5], 3);
        return jacobian;
    }

    private double[][] computeJacobianMeanWrtCartesianHyperbolic() {
        double[][] jacobian = new double[6][6];
        TimeStampedPVCoordinates pvc = this.getPVCoordinates();
        Vector3D position = pvc.getPosition();
        Vector3D velocity = pvc.getVelocity();
        Vector3D momentum = pvc.getMomentum();
        double r2 = position.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double r3 = r * r2;
        double x = position.getX();
        double y = position.getY();
        double z = position.getZ();
        double vx = velocity.getX();
        double vy = velocity.getY();
        double vz = velocity.getZ();
        double mx = momentum.getX();
        double my = momentum.getY();
        double mz = momentum.getZ();
        double mu = this.getMu();
        double absA = -this.a;
        double sqrtMuA = FastMath.sqrt((double)(absA * mu));
        double a2 = this.a * this.a;
        double rOa = r / absA;
        double cosI = FastMath.cos((double)this.i);
        double sinI = FastMath.sin((double)this.i);
        double pv = Vector3D.dotProduct((Vector3D)position, (Vector3D)velocity);
        Vector3D vectorAR = new Vector3D(-2.0 * a2 / r3, position);
        Vector3D vectorARDot = velocity.scalarMultiply(-2.0 * a2 / mu);
        KeplerianOrbit.fillHalfRow(-1.0, vectorAR, jacobian[0], 0);
        KeplerianOrbit.fillHalfRow(-1.0, vectorARDot, jacobian[0], 3);
        double m = momentum.getNorm();
        double oOm = 1.0 / m;
        Vector3D dcXP = new Vector3D(0.0, vz, -vy);
        Vector3D dcYP = new Vector3D(-vz, 0.0, vx);
        Vector3D dcZP = new Vector3D(vy, -vx, 0.0);
        Vector3D dcXV = new Vector3D(0.0, -z, y);
        Vector3D dcYV = new Vector3D(z, 0.0, -x);
        Vector3D dcZV = new Vector3D(-y, x, 0.0);
        Vector3D dCP = new Vector3D(mx * oOm, dcXP, my * oOm, dcYP, mz * oOm, dcZP);
        Vector3D dCV = new Vector3D(mx * oOm, dcXV, my * oOm, dcYV, mz * oOm, dcZV);
        double mOMu = m / mu;
        Vector3D dpP = new Vector3D(2.0 * mOMu, dCP);
        Vector3D dpV = new Vector3D(2.0 * mOMu, dCV);
        double p = m * mOMu;
        double moO2ae = 1.0 / (2.0 * absA * this.e);
        double m2OaMu = -p / absA;
        KeplerianOrbit.fillHalfRow(moO2ae, dpP, m2OaMu * moO2ae, vectorAR, jacobian[1], 0);
        KeplerianOrbit.fillHalfRow(moO2ae, dpV, m2OaMu * moO2ae, vectorARDot, jacobian[1], 3);
        double cI1 = 1.0 / (m * sinI);
        double cI2 = cosI * cI1;
        KeplerianOrbit.fillHalfRow(cI2, dCP, -cI1, dcZP, jacobian[2], 0);
        KeplerianOrbit.fillHalfRow(cI2, dCV, -cI1, dcZV, jacobian[2], 3);
        double cP1 = y * oOm;
        double cP2 = -x * oOm;
        double cP3 = -(mx * cP1 + my * cP2);
        double cP4 = cP3 * oOm;
        double cP5 = -1.0 / (r2 * sinI * sinI);
        double cP6 = z * cP5;
        double cP7 = cP3 * cP5;
        Vector3D dacP = new Vector3D(cP1, dcXP, cP2, dcYP, cP4, dCP, oOm, new Vector3D(-my, mx, 0.0));
        Vector3D dacV = new Vector3D(cP1, dcXV, cP2, dcYV, cP4, dCV);
        Vector3D dpoP = new Vector3D(cP6, dacP, cP7, Vector3D.PLUS_K);
        Vector3D dpoV = new Vector3D(cP6, dacV);
        double re2 = r2 * this.e * this.e;
        double recOre2 = (p - r) / re2;
        double resOre2 = pv * mOMu / re2;
        Vector3D dreP = new Vector3D(mOMu, velocity, pv / mu, dCP);
        Vector3D dreV = new Vector3D(mOMu, position, pv / mu, dCV);
        Vector3D davP = new Vector3D(-resOre2, dpP, recOre2, dreP, resOre2 / r, position);
        Vector3D davV = new Vector3D(-resOre2, dpV, recOre2, dreV);
        KeplerianOrbit.fillHalfRow(1.0, dpoP, -1.0, davP, jacobian[3], 0);
        KeplerianOrbit.fillHalfRow(1.0, dpoV, -1.0, davV, jacobian[3], 3);
        double cO0 = cI1 * cI1;
        double cO1 = mx * cO0;
        double cO2 = -my * cO0;
        KeplerianOrbit.fillHalfRow(cO1, dcYP, cO2, dcXP, jacobian[4], 0);
        KeplerianOrbit.fillHalfRow(cO1, dcYV, cO2, dcXV, jacobian[4], 3);
        double s2a = pv / (2.0 * absA);
        double oObux = 1.0 / FastMath.sqrt((double)(m * m + mu * absA));
        double scasbu = pv * oObux;
        Vector3D dauP = new Vector3D(1.0 / sqrtMuA, velocity, -s2a / sqrtMuA, vectorAR);
        Vector3D dauV = new Vector3D(1.0 / sqrtMuA, position, -s2a / sqrtMuA, vectorARDot);
        Vector3D dbuP = new Vector3D(oObux * mu / 2.0, vectorAR, m * oObux, dCP);
        Vector3D dbuV = new Vector3D(oObux * mu / 2.0, vectorARDot, m * oObux, dCV);
        Vector3D dcuP = new Vector3D(oObux, velocity, -scasbu * oObux, dbuP);
        Vector3D dcuV = new Vector3D(oObux, position, -scasbu * oObux, dbuV);
        KeplerianOrbit.fillHalfRow(1.0, dauP, -this.e / (1.0 + rOa), dcuP, jacobian[5], 0);
        KeplerianOrbit.fillHalfRow(1.0, dauV, -this.e / (1.0 + rOa), dcuV, jacobian[5], 3);
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianEccentricWrtCartesian() {
        if (this.a > 0.0) {
            return this.computeJacobianEccentricWrtCartesianElliptical();
        }
        return this.computeJacobianEccentricWrtCartesianHyperbolic();
    }

    private double[][] computeJacobianEccentricWrtCartesianElliptical() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesianElliptical();
        double eccentricAnomaly = this.getEccentricAnomaly();
        double cosE = FastMath.cos((double)eccentricAnomaly);
        double sinE = FastMath.sin((double)eccentricAnomaly);
        double aOr = 1.0 / (1.0 - this.e * cosE);
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = aOr * (anomalyRow[j] + sinE * eRow[j]);
        }
        return jacobian;
    }

    private double[][] computeJacobianEccentricWrtCartesianHyperbolic() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesianHyperbolic();
        double H = this.getEccentricAnomaly();
        double coshH = FastMath.cosh((double)H);
        double sinhH = FastMath.sinh((double)H);
        double absaOr = 1.0 / (this.e * coshH - 1.0);
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = absaOr * (anomalyRow[j] - sinhH * eRow[j]);
        }
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianTrueWrtCartesian() {
        if (this.a > 0.0) {
            return this.computeJacobianTrueWrtCartesianElliptical();
        }
        return this.computeJacobianTrueWrtCartesianHyperbolic();
    }

    private double[][] computeJacobianTrueWrtCartesianElliptical() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesianElliptical();
        double e2 = this.e * this.e;
        double oMe2 = 1.0 - e2;
        double epsilon = FastMath.sqrt((double)oMe2);
        double eccentricAnomaly = this.getEccentricAnomaly();
        double cosE = FastMath.cos((double)eccentricAnomaly);
        double sinE = FastMath.sin((double)eccentricAnomaly);
        double aOr = 1.0 / (1.0 - this.e * cosE);
        double aFactor = epsilon * aOr;
        double eFactor = sinE * aOr / epsilon;
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = aFactor * anomalyRow[j] + eFactor * eRow[j];
        }
        return jacobian;
    }

    private double[][] computeJacobianTrueWrtCartesianHyperbolic() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesianHyperbolic();
        double e2 = this.e * this.e;
        double e2Mo = e2 - 1.0;
        double epsilon = FastMath.sqrt((double)e2Mo);
        double H = this.getEccentricAnomaly();
        double coshH = FastMath.cosh((double)H);
        double sinhH = FastMath.sinh((double)H);
        double aOr = 1.0 / (this.e * coshH - 1.0);
        double aFactor = epsilon * aOr;
        double eFactor = sinhH * aOr / epsilon;
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = aFactor * anomalyRow[j] - eFactor * eRow[j];
        }
        return jacobian;
    }

    @Override
    public void addKeplerContribution(PositionAngle type, double gm, double[] pDot) {
        double absA = FastMath.abs((double)this.a);
        double n = FastMath.sqrt((double)(gm / absA)) / absA;
        switch (type) {
            case MEAN: {
                pDot[5] = pDot[5] + n;
                break;
            }
            case ECCENTRIC: {
                double oMe2 = FastMath.abs((double)(1.0 - this.e * this.e));
                double ksi = 1.0 + this.e * FastMath.cos((double)this.v);
                pDot[5] = pDot[5] + n * ksi / oMe2;
                break;
            }
            case TRUE: {
                double oMe2 = FastMath.abs((double)(1.0 - this.e * this.e));
                double ksi = 1.0 + this.e * FastMath.cos((double)this.v);
                pDot[5] = pDot[5] + n * ksi * ksi / (oMe2 * FastMath.sqrt((double)oMe2));
                break;
            }
            default: {
                throw OrekitException.createInternalError(null);
            }
        }
    }

    public String toString() {
        return new StringBuffer().append("keplerian parameters: ").append('{').append("a: ").append(this.a).append("; e: ").append(this.e).append("; i: ").append(FastMath.toDegrees((double)this.i)).append("; pa: ").append(FastMath.toDegrees((double)this.pa)).append("; raan: ").append(FastMath.toDegrees((double)this.raan)).append("; v: ").append(FastMath.toDegrees((double)this.v)).append(";}").toString();
    }

    private Object writeReplace() {
        return new DTO(this);
    }

    static {
        double k1 = 11.42477796076938;
        double k2 = 2.141592653589793;
        double k3 = 17.84955592153876;
        A = 1.2043347651023166;
        B = 4.64788969626918;
    }

    private static class DTO
    implements Serializable {
        private static final long serialVersionUID = 20140617L;
        private double[] d;
        private final Frame frame;

        private DTO(KeplerianOrbit orbit) {
            TimeStampedPVCoordinates pv = orbit.getPVCoordinates();
            double epoch = FastMath.floor((double)pv.getDate().durationFrom(AbsoluteDate.J2000_EPOCH));
            double offset = pv.getDate().durationFrom(AbsoluteDate.J2000_EPOCH.shiftedBy(epoch));
            this.d = new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.e, orbit.i, orbit.pa, orbit.raan, orbit.v};
            this.frame = orbit.getFrame();
        }

        private Object readResolve() {
            return new KeplerianOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], PositionAngle.TRUE, this.frame, AbsoluteDate.J2000_EPOCH.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
        }
    }
}

