/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.orbits;

import java.io.Serializable;
import java.util.Collection;
import org.apache.commons.math3.analysis.interpolation.HermiteInterpolator;
import org.apache.commons.math3.geometry.euclidean.threed.Vector3D;
import org.apache.commons.math3.util.FastMath;
import org.apache.commons.math3.util.MathUtils;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngle;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class EquinoctialOrbit
extends Orbit {
    private static final long serialVersionUID = 20141228L;
    private final double a;
    private final double ex;
    private final double ey;
    private final double hx;
    private final double hy;
    private final double lv;

    public EquinoctialOrbit(double a, double ex, double ey, double hx, double hy, double l, PositionAngle type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        super(frame, date, mu);
        if (ex * ex + ey * ey >= 1.0) {
            throw OrekitException.createIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        this.a = a;
        this.ex = ex;
        this.ey = ey;
        this.hx = hx;
        this.hy = hy;
        switch (type) {
            case MEAN: {
                this.lv = this.eccentricToTrue(this.meanToEccentric(l));
                break;
            }
            case ECCENTRIC: {
                this.lv = this.eccentricToTrue(l);
                break;
            }
            case TRUE: {
                this.lv = l;
                break;
            }
            default: {
                throw OrekitException.createInternalError(null);
            }
        }
    }

    public EquinoctialOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        Vector3D pvP = pvCoordinates.getPosition();
        Vector3D pvV = pvCoordinates.getVelocity();
        double r = pvP.getNorm();
        double V2 = pvV.getNormSq();
        double rV2OnMu = r * V2 / mu;
        if (rV2OnMu > 2.0) {
            throw OrekitException.createIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        Vector3D w = pvCoordinates.getMomentum().normalize();
        double d = 1.0 / (1.0 + w.getZ());
        this.hx = -d * w.getY();
        this.hy = d * w.getX();
        double cLv = (pvP.getX() - d * pvP.getZ() * w.getX()) / r;
        double sLv = (pvP.getY() - d * pvP.getZ() * w.getY()) / r;
        this.lv = FastMath.atan2((double)sLv, (double)cLv);
        this.a = r / (2.0 - rV2OnMu);
        double eSE = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)pvV) / FastMath.sqrt((double)(mu * this.a));
        double eCE = rV2OnMu - 1.0;
        double e2 = eCE * eCE + eSE * eSE;
        double f = eCE - e2;
        double g = FastMath.sqrt((double)(1.0 - e2)) * eSE;
        this.ex = this.a * (f * cLv + g * sLv) / r;
        this.ey = this.a * (f * sLv - g * cLv) / r;
    }

    public EquinoctialOrbit(PVCoordinates pvCoordinates, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(new TimeStampedPVCoordinates(date, pvCoordinates), frame, mu);
    }

    public EquinoctialOrbit(Orbit op) {
        super(op.getFrame(), op.getDate(), op.getMu());
        this.a = op.getA();
        this.ex = op.getEquinoctialEx();
        this.ey = op.getEquinoctialEy();
        this.hx = op.getHx();
        this.hy = op.getHy();
        this.lv = op.getLv();
    }

    @Override
    public OrbitType getType() {
        return OrbitType.EQUINOCTIAL;
    }

    @Override
    public double getA() {
        return this.a;
    }

    @Override
    public double getEquinoctialEx() {
        return this.ex;
    }

    @Override
    public double getEquinoctialEy() {
        return this.ey;
    }

    @Override
    public double getHx() {
        return this.hx;
    }

    @Override
    public double getHy() {
        return this.hy;
    }

    public double getL(PositionAngle type) {
        return type == PositionAngle.MEAN ? this.getLM() : (type == PositionAngle.ECCENTRIC ? this.getLE() : this.getLv());
    }

    @Override
    public double getLv() {
        return this.lv;
    }

    @Override
    public double getLE() {
        double epsilon = FastMath.sqrt((double)(1.0 - this.ex * this.ex - this.ey * this.ey));
        double cosLv = FastMath.cos((double)this.lv);
        double sinLv = FastMath.sin((double)this.lv);
        double num = this.ey * cosLv - this.ex * sinLv;
        double den = epsilon + 1.0 + this.ex * cosLv + this.ey * sinLv;
        return this.lv + 2.0 * FastMath.atan((double)(num / den));
    }

    private double eccentricToTrue(double lE) {
        double epsilon = FastMath.sqrt((double)(1.0 - this.ex * this.ex - this.ey * this.ey));
        double cosLE = FastMath.cos((double)lE);
        double sinLE = FastMath.sin((double)lE);
        double num = this.ex * sinLE - this.ey * cosLE;
        double den = epsilon + 1.0 - this.ex * cosLE - this.ey * sinLE;
        return lE + 2.0 * FastMath.atan((double)(num / den));
    }

    @Override
    public double getLM() {
        double lE = this.getLE();
        return lE - this.ex * FastMath.sin((double)lE) + this.ey * FastMath.cos((double)lE);
    }

    private double meanToEccentric(double lM) {
        double lE = lM;
        double shift = 0.0;
        double lEmlM = 0.0;
        double cosLE = FastMath.cos((double)lE);
        double sinLE = FastMath.sin((double)lE);
        int iter = 0;
        do {
            double f2 = this.ex * sinLE - this.ey * cosLE;
            double f1 = 1.0 - this.ex * cosLE - this.ey * sinLE;
            double f0 = lEmlM - f2;
            double f12 = 2.0 * f1;
            shift = f0 * f12 / (f1 * f12 - f0 * f2);
            lE = lM + (lEmlM -= shift);
            cosLE = FastMath.cos((double)lE);
            sinLE = FastMath.sin((double)lE);
        } while (++iter < 50 && FastMath.abs((double)shift) > 1.0E-12);
        return lE;
    }

    @Override
    public double getE() {
        return FastMath.sqrt((double)(this.ex * this.ex + this.ey * this.ey));
    }

    @Override
    public double getI() {
        return 2.0 * FastMath.atan((double)FastMath.sqrt((double)(this.hx * this.hx + this.hy * this.hy)));
    }

    @Override
    protected TimeStampedPVCoordinates initPVCoordinates() {
        double lE = this.getLE();
        double hx2 = this.hx * this.hx;
        double hy2 = this.hy * this.hy;
        double factH = 1.0 / (1.0 + hx2 + hy2);
        double ux = (1.0 + hx2 - hy2) * factH;
        double uy = 2.0 * this.hx * this.hy * factH;
        double uz = -2.0 * this.hy * factH;
        double vx = uy;
        double vy = (1.0 - hx2 + hy2) * factH;
        double vz = 2.0 * this.hx * factH;
        double exey = this.ex * this.ey;
        double ex2 = this.ex * this.ex;
        double ey2 = this.ey * this.ey;
        double e2 = ex2 + ey2;
        double eta = 1.0 + FastMath.sqrt((double)(1.0 - e2));
        double beta = 1.0 / eta;
        double cLe = FastMath.cos((double)lE);
        double sLe = FastMath.sin((double)lE);
        double exCeyS = this.ex * cLe + this.ey * sLe;
        double x = this.a * ((1.0 - beta * ey2) * cLe + beta * exey * sLe - this.ex);
        double y = this.a * ((1.0 - beta * ex2) * sLe + beta * exey * cLe - this.ey);
        double factor = FastMath.sqrt((double)(this.getMu() / this.a)) / (1.0 - exCeyS);
        double xdot = factor * (-sLe + beta * this.ey * exCeyS);
        double ydot = factor * (cLe - beta * this.ex * exCeyS);
        Vector3D position = new Vector3D(x * ux + y * vx, x * uy + y * vy, x * uz + y * vz);
        double r2 = position.getNormSq();
        Vector3D velocity = new Vector3D(xdot * ux + ydot * vx, xdot * uy + ydot * vy, xdot * uz + ydot * vz);
        Vector3D acceleration = new Vector3D(-this.getMu() / (r2 * FastMath.sqrt((double)r2)), position);
        return new TimeStampedPVCoordinates(this.getDate(), position, velocity, acceleration);
    }

    @Override
    public EquinoctialOrbit shiftedBy(double dt) {
        return new EquinoctialOrbit(this.a, this.ex, this.ey, this.hx, this.hy, this.getLM() + this.getKeplerianMeanMotion() * dt, PositionAngle.MEAN, this.getFrame(), this.getDate().shiftedBy(dt), this.getMu());
    }

    @Override
    public EquinoctialOrbit interpolate(AbsoluteDate date, Collection<Orbit> sample) {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        AbsoluteDate previousDate = null;
        double previousLm = Double.NaN;
        for (Orbit orbit : sample) {
            double continuousLm;
            EquinoctialOrbit equi = (EquinoctialOrbit)OrbitType.EQUINOCTIAL.convertType(orbit);
            if (previousDate == null) {
                continuousLm = equi.getLM();
            } else {
                double dt = equi.getDate().durationFrom(previousDate);
                double keplerLm = previousLm + equi.getKeplerianMeanMotion() * dt;
                continuousLm = MathUtils.normalizeAngle((double)equi.getLM(), (double)keplerLm);
            }
            previousDate = equi.getDate();
            previousLm = continuousLm;
            interpolator.addSamplePoint(equi.getDate().durationFrom(date), (double[][])new double[][]{{equi.getA(), equi.getEquinoctialEx(), equi.getEquinoctialEy(), equi.getHx(), equi.getHy(), continuousLm}});
        }
        double[] interpolated = interpolator.value(0.0);
        return new EquinoctialOrbit(interpolated[0], interpolated[1], interpolated[2], interpolated[3], interpolated[4], interpolated[5], PositionAngle.MEAN, this.getFrame(), date, this.getMu());
    }

    @Override
    protected double[][] computeJacobianMeanWrtCartesian() {
        double[][] jacobian = new double[6][6];
        Vector3D position = this.getPVCoordinates().getPosition();
        Vector3D velocity = this.getPVCoordinates().getVelocity();
        double r2 = position.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double r3 = r * r2;
        double mu = this.getMu();
        double sqrtMuA = FastMath.sqrt((double)(this.a * mu));
        double a2 = this.a * this.a;
        double e2 = this.ex * this.ex + this.ey * this.ey;
        double oMe2 = 1.0 - e2;
        double epsilon = FastMath.sqrt((double)oMe2);
        double beta = 1.0 / (1.0 + epsilon);
        double ratio = epsilon * beta;
        double hx2 = this.hx * this.hx;
        double hy2 = this.hy * this.hy;
        double hxhy = this.hx * this.hy;
        Vector3D f = new Vector3D(1.0 - hy2 + hx2, 2.0 * hxhy, -2.0 * this.hy).normalize();
        Vector3D g = new Vector3D(2.0 * hxhy, 1.0 + hy2 - hx2, 2.0 * this.hx).normalize();
        Vector3D w = Vector3D.crossProduct((Vector3D)position, (Vector3D)velocity).normalize();
        double x = Vector3D.dotProduct((Vector3D)position, (Vector3D)f);
        double y = Vector3D.dotProduct((Vector3D)position, (Vector3D)g);
        double xDot = Vector3D.dotProduct((Vector3D)velocity, (Vector3D)f);
        double yDot = Vector3D.dotProduct((Vector3D)velocity, (Vector3D)g);
        double c1 = this.a / (sqrtMuA * epsilon);
        double c2 = this.a * sqrtMuA * beta / r3;
        double c3 = sqrtMuA / (r3 * epsilon);
        Vector3D drDotSdEx = new Vector3D(c1 * xDot * yDot - c2 * this.ey * x - c3 * x * y, f, -c1 * xDot * xDot - c2 * this.ey * y + c3 * x * x, g);
        Vector3D drDotSdEy = new Vector3D(c1 * yDot * yDot + c2 * this.ex * x - c3 * y * y, f, -c1 * xDot * yDot + c2 * this.ex * y + c3 * x * y, g);
        Vector3D vectorAR = new Vector3D(2.0 * a2 / r3, position);
        Vector3D vectorARDot = new Vector3D(2.0 * a2 / mu, velocity);
        EquinoctialOrbit.fillHalfRow(1.0, vectorAR, jacobian[0], 0);
        EquinoctialOrbit.fillHalfRow(1.0, vectorARDot, jacobian[0], 3);
        double d1 = -this.a * ratio / r3;
        double d2 = (this.hy * xDot - this.hx * yDot) / (sqrtMuA * epsilon);
        double d3 = (this.hx * y - this.hy * x) / sqrtMuA;
        Vector3D vectorExRDot = new Vector3D((2.0 * x * yDot - xDot * y) / mu, g, -y * yDot / mu, f, -this.ey * d3 / epsilon, w);
        EquinoctialOrbit.fillHalfRow(this.ex * d1, position, -this.ey * d2, w, epsilon / sqrtMuA, drDotSdEy, jacobian[1], 0);
        EquinoctialOrbit.fillHalfRow(1.0, vectorExRDot, jacobian[1], 3);
        Vector3D vectorEyRDot = new Vector3D((2.0 * xDot * y - x * yDot) / mu, f, -x * xDot / mu, g, this.ex * d3 / epsilon, w);
        EquinoctialOrbit.fillHalfRow(this.ey * d1, position, this.ex * d2, w, -epsilon / sqrtMuA, drDotSdEx, jacobian[2], 0);
        EquinoctialOrbit.fillHalfRow(1.0, vectorEyRDot, jacobian[2], 3);
        double h = (1.0 + hx2 + hy2) / (2.0 * sqrtMuA * epsilon);
        EquinoctialOrbit.fillHalfRow(-h * xDot, w, jacobian[3], 0);
        EquinoctialOrbit.fillHalfRow(h * x, w, jacobian[3], 3);
        EquinoctialOrbit.fillHalfRow(-h * yDot, w, jacobian[4], 0);
        EquinoctialOrbit.fillHalfRow(h * y, w, jacobian[4], 3);
        double l = -ratio / sqrtMuA;
        EquinoctialOrbit.fillHalfRow(-1.0 / sqrtMuA, velocity, d2, w, l * this.ex, drDotSdEx, l * this.ey, drDotSdEy, jacobian[5], 0);
        EquinoctialOrbit.fillHalfRow(-2.0 / sqrtMuA, position, this.ex * beta, vectorEyRDot, -this.ey * beta, vectorExRDot, d3, w, jacobian[5], 3);
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianEccentricWrtCartesian() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesian();
        double le = this.getLE();
        double cosLe = FastMath.cos((double)le);
        double sinLe = FastMath.sin((double)le);
        double aOr = 1.0 / (1.0 - this.ex * cosLe - this.ey * sinLe);
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowL = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowL[j] = aOr * (rowL[j] + sinLe * rowEx[j] - cosLe * rowEy[j]);
        }
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianTrueWrtCartesian() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesian();
        double le = this.getLE();
        double cosLe = FastMath.cos((double)le);
        double sinLe = FastMath.sin((double)le);
        double eSinE = this.ex * sinLe - this.ey * cosLe;
        double ecosE = this.ex * cosLe + this.ey * sinLe;
        double e2 = this.ex * this.ex + this.ey * this.ey;
        double epsilon = FastMath.sqrt((double)(1.0 - e2));
        double onePeps = 1.0 + epsilon;
        double d = onePeps - ecosE;
        double cT = (d * d + eSinE * eSinE) / 2.0;
        double cE = ecosE * onePeps - e2;
        double cX = this.ex * eSinE / epsilon - this.ey + sinLe * onePeps;
        double cY = this.ey * eSinE / epsilon + this.ex - cosLe * onePeps;
        double factorLe = (cT + cE) / cT;
        double factorEx = cX / cT;
        double factorEy = cY / cT;
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowL = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowL[j] = factorLe * rowL[j] + factorEx * rowEx[j] + factorEy * rowEy[j];
        }
        return jacobian;
    }

    @Override
    public void addKeplerContribution(PositionAngle type, double gm, double[] pDot) {
        double n = FastMath.sqrt((double)(gm / this.a)) / this.a;
        switch (type) {
            case MEAN: {
                pDot[5] = pDot[5] + n;
                break;
            }
            case ECCENTRIC: {
                double oMe2 = 1.0 - this.ex * this.ex - this.ey * this.ey;
                double ksi = 1.0 + this.ex * FastMath.cos((double)this.lv) + this.ey * FastMath.sin((double)this.lv);
                pDot[5] = pDot[5] + n * ksi / oMe2;
                break;
            }
            case TRUE: {
                double oMe2 = 1.0 - this.ex * this.ex - this.ey * this.ey;
                double ksi = 1.0 + this.ex * FastMath.cos((double)this.lv) + this.ey * FastMath.sin((double)this.lv);
                pDot[5] = pDot[5] + n * ksi * ksi / (oMe2 * FastMath.sqrt((double)oMe2));
                break;
            }
            default: {
                throw OrekitException.createInternalError(null);
            }
        }
    }

    public String toString() {
        return new StringBuffer().append("equinoctial parameters: ").append('{').append("a: ").append(this.a).append("; ex: ").append(this.ex).append("; ey: ").append(this.ey).append("; hx: ").append(this.hx).append("; hy: ").append(this.hy).append("; lv: ").append(FastMath.toDegrees((double)this.lv)).append(";}").toString();
    }

    private Object writeReplace() {
        return new DTO(this);
    }

    private static class DTO
    implements Serializable {
        private static final long serialVersionUID = 20140617L;
        private double[] d;
        private final Frame frame;

        private DTO(EquinoctialOrbit orbit) {
            TimeStampedPVCoordinates pv = orbit.getPVCoordinates();
            double epoch = FastMath.floor((double)pv.getDate().durationFrom(AbsoluteDate.J2000_EPOCH));
            double offset = pv.getDate().durationFrom(AbsoluteDate.J2000_EPOCH.shiftedBy(epoch));
            this.d = new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.hx, orbit.hy, orbit.lv};
            this.frame = orbit.getFrame();
        }

        private Object readResolve() {
            return new EquinoctialOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], PositionAngle.TRUE, this.frame, AbsoluteDate.J2000_EPOCH.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
        }
    }
}

