/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.forces.radiation;

import org.apache.commons.math3.RealFieldElement;
import org.apache.commons.math3.analysis.differentiation.DerivativeStructure;
import org.apache.commons.math3.exception.util.Localizable;
import org.apache.commons.math3.geometry.Vector;
import org.apache.commons.math3.geometry.euclidean.threed.FieldRotation;
import org.apache.commons.math3.geometry.euclidean.threed.FieldVector3D;
import org.apache.commons.math3.geometry.euclidean.threed.Vector3D;
import org.apache.commons.math3.ode.AbstractParameterizable;
import org.apache.commons.math3.util.FastMath;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.forces.ForceModel;
import org.orekit.forces.radiation.RadiationSensitive;
import org.orekit.frames.Frame;
import org.orekit.propagation.SpacecraftState;
import org.orekit.propagation.events.AbstractDetector;
import org.orekit.propagation.events.EventDetector;
import org.orekit.propagation.events.handlers.EventHandler;
import org.orekit.propagation.numerical.TimeDerivativesEquations;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinatesProvider;

public class SolarRadiationPressure
extends AbstractParameterizable
implements ForceModel {
    private static final double D_REF = 1.4959787E11;
    private static final double P_REF = 4.56E-6;
    private final double kRef;
    private final PVCoordinatesProvider sun;
    private final double equatorialRadius;
    private final RadiationSensitive spacecraft;

    public SolarRadiationPressure(PVCoordinatesProvider sun, double equatorialRadius, RadiationSensitive spacecraft) {
        this(1.4959787E11, 4.56E-6, sun, equatorialRadius, spacecraft);
    }

    public SolarRadiationPressure(double dRef, double pRef, PVCoordinatesProvider sun, double equatorialRadius, RadiationSensitive spacecraft) {
        super(new String[]{"absorption coefficient", "reflection coefficient"});
        this.kRef = pRef * dRef * dRef;
        this.sun = sun;
        this.equatorialRadius = equatorialRadius;
        this.spacecraft = spacecraft;
    }

    @Override
    public void addContribution(SpacecraftState s, TimeDerivativesEquations adder) throws OrekitException {
        AbsoluteDate date = s.getDate();
        Frame frame = s.getFrame();
        Vector3D position = s.getPVCoordinates().getPosition();
        Vector3D sunSatVector = position.subtract((Vector)this.sun.getPVCoordinates(date, frame).getPosition());
        double r2 = sunSatVector.getNormSq();
        double rawP = this.kRef * this.getLightningRatio(position, frame, date) / r2;
        Vector3D flux = new Vector3D(rawP / FastMath.sqrt((double)r2), sunSatVector);
        Vector3D acceleration = this.spacecraft.radiationPressureAcceleration(date, frame, position, s.getAttitude().getRotation(), s.getMass(), flux);
        adder.addAcceleration(acceleration, s.getFrame());
    }

    public double getLightningRatio(Vector3D position, Frame frame, AbsoluteDate date) throws OrekitException {
        double[] angle = this.getEclipseAngles(position, frame, date);
        double sunEarthAngle = angle[0];
        double alphaCentral = angle[1];
        double alphaSun = angle[2];
        double result = 1.0;
        if (sunEarthAngle - alphaCentral + alphaSun <= 0.0) {
            result = 0.0;
        } else if (sunEarthAngle - alphaCentral - alphaSun < 0.0) {
            double sEA2 = sunEarthAngle * sunEarthAngle;
            double oo2sEA = 1.0 / (2.0 * sunEarthAngle);
            double aS2 = alphaSun * alphaSun;
            double aE2 = alphaCentral * alphaCentral;
            double aE2maS2 = aE2 - aS2;
            double alpha1 = (sEA2 - aE2maS2) * oo2sEA;
            double alpha2 = (sEA2 + aE2maS2) * oo2sEA;
            double a1oaS = FastMath.min((double)1.0, (double)FastMath.max((double)-1.0, (double)(alpha1 / alphaSun)));
            double aS2ma12 = FastMath.max((double)0.0, (double)(aS2 - alpha1 * alpha1));
            double a2oaE = FastMath.min((double)1.0, (double)FastMath.max((double)-1.0, (double)(alpha2 / alphaCentral)));
            double aE2ma22 = FastMath.max((double)0.0, (double)(aE2 - alpha2 * alpha2));
            double P1 = aS2 * FastMath.acos((double)a1oaS) - alpha1 * FastMath.sqrt((double)aS2ma12);
            double P2 = aE2 * FastMath.acos((double)a2oaE) - alpha2 * FastMath.sqrt((double)aE2ma22);
            result = 1.0 - (P1 + P2) / (Math.PI * aS2);
        }
        return result;
    }

    @Override
    public EventDetector[] getEventsDetectors() {
        return new EventDetector[]{new UmbraDetector(), new PenumbraDetector()};
    }

    @Override
    public FieldVector3D<DerivativeStructure> accelerationDerivatives(AbsoluteDate date, Frame frame, FieldVector3D<DerivativeStructure> position, FieldVector3D<DerivativeStructure> velocity, FieldRotation<DerivativeStructure> rotation, DerivativeStructure mass) throws OrekitException {
        FieldVector3D sunSatVector = position.subtract(this.sun.getPVCoordinates(date, frame).getPosition());
        DerivativeStructure r2 = (DerivativeStructure)sunSatVector.getNormSq();
        double ratio = this.getLightningRatio(position.toVector3D(), frame, date);
        DerivativeStructure rawP = r2.reciprocal().multiply(this.kRef * ratio);
        FieldVector3D flux = new FieldVector3D((RealFieldElement)rawP.divide(r2.sqrt()), sunSatVector);
        return this.spacecraft.radiationPressureAcceleration(date, frame, position, rotation, mass, (FieldVector3D<DerivativeStructure>)flux);
    }

    @Override
    public FieldVector3D<DerivativeStructure> accelerationDerivatives(SpacecraftState s, String paramName) throws OrekitException {
        this.complainIfNotSupported(paramName);
        AbsoluteDate date = s.getDate();
        Frame frame = s.getFrame();
        Vector3D position = s.getPVCoordinates().getPosition();
        Vector3D sunSatVector = position.subtract((Vector)this.sun.getPVCoordinates(date, frame).getPosition());
        double r2 = sunSatVector.getNormSq();
        double rawP = this.kRef * this.getLightningRatio(position, frame, date) / r2;
        Vector3D flux = new Vector3D(rawP / FastMath.sqrt((double)r2), sunSatVector);
        return this.spacecraft.radiationPressureAcceleration(date, frame, position, s.getAttitude().getRotation(), s.getMass(), flux, paramName);
    }

    public double getParameter(String name) throws IllegalArgumentException {
        this.complainIfNotSupported(name);
        if (name.equals("absorption coefficient")) {
            return this.spacecraft.getAbsorptionCoefficient();
        }
        return this.spacecraft.getReflectionCoefficient();
    }

    public void setParameter(String name, double value) throws IllegalArgumentException {
        this.complainIfNotSupported(name);
        if (name.equals("absorption coefficient")) {
            this.spacecraft.setAbsorptionCoefficient(value);
        } else {
            this.spacecraft.setReflectionCoefficient(value);
        }
    }

    private double[] getEclipseAngles(Vector3D position, Frame frame, AbsoluteDate date) throws OrekitException {
        double[] angle = new double[3];
        Vector3D satSunVector = this.sun.getPVCoordinates(date, frame).getPosition().subtract((Vector)position);
        angle[0] = Vector3D.angle((Vector3D)satSunVector, (Vector3D)position.negate());
        double r = position.getNorm();
        if (r <= this.equatorialRadius) {
            throw new OrekitException((Localizable)OrekitMessages.TRAJECTORY_INSIDE_BRILLOUIN_SPHERE, r);
        }
        angle[1] = FastMath.asin((double)(this.equatorialRadius / r));
        angle[2] = FastMath.asin((double)(6.955E8 / satSunVector.getNorm()));
        return angle;
    }

    private class PenumbraDetector
    extends AbstractDetector<PenumbraDetector> {
        private static final long serialVersionUID = 20141228L;

        public PenumbraDetector() {
            super(60.0, 0.001, 100, new EventHandler<PenumbraDetector>(){

                @Override
                public EventHandler.Action eventOccurred(SpacecraftState s, PenumbraDetector detector, boolean increasing) {
                    return EventHandler.Action.RESET_DERIVATIVES;
                }

                @Override
                public SpacecraftState resetState(PenumbraDetector detector, SpacecraftState oldState) {
                    return oldState;
                }
            });
        }

        private PenumbraDetector(double maxCheck, double threshold, int maxIter, EventHandler<PenumbraDetector> handler) {
            super(maxCheck, threshold, maxIter, handler);
        }

        @Override
        protected PenumbraDetector create(double newMaxCheck, double newThreshold, int newMaxIter, EventHandler<PenumbraDetector> newHandler) {
            return new PenumbraDetector(newMaxCheck, newThreshold, newMaxIter, newHandler);
        }

        @Override
        public double g(SpacecraftState s) throws OrekitException {
            double[] angle = SolarRadiationPressure.this.getEclipseAngles(s.getPVCoordinates().getPosition(), s.getFrame(), s.getDate());
            return angle[0] - angle[1] - angle[2];
        }
    }

    private class UmbraDetector
    extends AbstractDetector<UmbraDetector> {
        private static final long serialVersionUID = 20141228L;

        public UmbraDetector() {
            super(60.0, 0.001, 100, new EventHandler<UmbraDetector>(){

                @Override
                public EventHandler.Action eventOccurred(SpacecraftState s, UmbraDetector detector, boolean increasing) {
                    return EventHandler.Action.RESET_DERIVATIVES;
                }

                @Override
                public SpacecraftState resetState(UmbraDetector detector, SpacecraftState oldState) {
                    return oldState;
                }
            });
        }

        private UmbraDetector(double maxCheck, double threshold, int maxIter, EventHandler<UmbraDetector> handler) {
            super(maxCheck, threshold, maxIter, handler);
        }

        @Override
        protected UmbraDetector create(double newMaxCheck, double newThreshold, int newMaxIter, EventHandler<UmbraDetector> newHandler) {
            return new UmbraDetector(newMaxCheck, newThreshold, newMaxIter, newHandler);
        }

        @Override
        public double g(SpacecraftState s) throws OrekitException {
            double[] angle = SolarRadiationPressure.this.getEclipseAngles(s.getPVCoordinates().getPosition(), s.getFrame(), s.getDate());
            return angle[0] - angle[1] + angle[2];
        }
    }
}

