/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.IOException;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.LongStream;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.metrics.Timekeeper;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.exceptions.StateMachineException;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerState;
import org.apache.ratis.server.impl.StateMachineMetrics;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.raftlog.LogProtoUtils;
import org.apache.ratis.server.raftlog.RaftLog;
import org.apache.ratis.server.raftlog.RaftLogIOException;
import org.apache.ratis.server.raftlog.RaftLogIndex;
import org.apache.ratis.server.util.ServerStringUtils;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.statemachine.SnapshotRetentionPolicy;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.util.AwaitForSignal;
import org.apache.ratis.util.Daemon;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.LifeCycle;
import org.apache.ratis.util.MemoizedSupplier;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.UncheckedAutoCloseable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class StateMachineUpdater
implements Runnable {
    static final Logger LOG = LoggerFactory.getLogger(StateMachineUpdater.class);
    private final Consumer<Object> infoIndexChange;
    private final Consumer<Object> debugIndexChange;
    private final String name;
    private final StateMachine stateMachine;
    private final RaftServerImpl server;
    private final RaftLog raftLog;
    private final boolean triggerSnapshotWhenStopEnabled;
    private final boolean triggerSnapshotWhenRemoveEnabled;
    private final Long autoSnapshotThreshold;
    private final boolean purgeUptoSnapshotIndex;
    private final Thread updater;
    private final AwaitForSignal awaitForSignal;
    private final RaftLogIndex appliedIndex;
    private final RaftLogIndex snapshotIndex;
    private final AtomicReference<Long> stopIndex = new AtomicReference();
    private volatile State state = State.RUNNING;
    private final SnapshotRetentionPolicy snapshotRetentionPolicy;
    private final MemoizedSupplier<StateMachineMetrics> stateMachineMetrics;
    private final Consumer<Long> appliedIndexConsumer;
    private volatile boolean isRemoving;

    StateMachineUpdater(StateMachine stateMachine, RaftServerImpl server, ServerState serverState, long lastAppliedIndex, RaftProperties properties, Consumer<Long> appliedIndexConsumer) {
        this.name = ServerStringUtils.generateUnifiedName(serverState.getMemberId(), this.getClass());
        this.appliedIndexConsumer = appliedIndexConsumer;
        this.infoIndexChange = s2 -> LOG.info("{}: {}", (Object)this.name, s2);
        this.debugIndexChange = s2 -> LOG.debug("{}: {}", (Object)this.name, s2);
        this.stateMachine = stateMachine;
        this.server = server;
        this.raftLog = serverState.getLog();
        this.appliedIndex = new RaftLogIndex("appliedIndex", lastAppliedIndex);
        this.snapshotIndex = new RaftLogIndex("snapshotIndex", lastAppliedIndex);
        this.triggerSnapshotWhenStopEnabled = RaftServerConfigKeys.Snapshot.triggerWhenStopEnabled(properties);
        this.triggerSnapshotWhenRemoveEnabled = RaftServerConfigKeys.Snapshot.triggerWhenRemoveEnabled(properties);
        boolean autoSnapshot = RaftServerConfigKeys.Snapshot.autoTriggerEnabled(properties);
        this.autoSnapshotThreshold = autoSnapshot ? Long.valueOf(RaftServerConfigKeys.Snapshot.autoTriggerThreshold(properties)) : null;
        final int numSnapshotFilesRetained = RaftServerConfigKeys.Snapshot.retentionFileNum(properties);
        this.snapshotRetentionPolicy = new SnapshotRetentionPolicy(){

            @Override
            public int getNumSnapshotsRetained() {
                return numSnapshotFilesRetained;
            }
        };
        this.purgeUptoSnapshotIndex = RaftServerConfigKeys.Log.purgeUptoSnapshotIndex(properties);
        this.updater = Daemon.newBuilder().setName(this.name).setRunnable(this).setThreadGroup(server.getThreadGroup()).build();
        this.awaitForSignal = new AwaitForSignal(this.name);
        this.stateMachineMetrics = MemoizedSupplier.valueOf(() -> StateMachineMetrics.getStateMachineMetrics(server, this.appliedIndex, stateMachine));
    }

    void start() {
        this.stateMachineMetrics.get();
        this.updater.start();
        this.notifyAppliedIndex(this.appliedIndex.get());
    }

    private void stop() {
        this.state = State.STOP;
        try {
            LOG.info("{}: closing {}, lastApplied={}", this.name, JavaUtils.getClassSimpleName(this.stateMachine.getClass()), this.stateMachine.getLastAppliedTermIndex());
            this.stateMachine.close();
            if (this.stateMachineMetrics.isInitialized()) {
                this.stateMachineMetrics.get().unregister();
            }
        }
        catch (Throwable t2) {
            LOG.warn(this.name + ": Failed to close " + JavaUtils.getClassSimpleName(this.stateMachine.getClass()) + " " + this.stateMachine, t2);
        }
    }

    void stopAndJoin() throws InterruptedException {
        if (this.state == State.EXCEPTION) {
            this.stop();
            return;
        }
        if (this.stopIndex.compareAndSet(null, this.raftLog.getLastCommittedIndex())) {
            this.notifyUpdater();
            LOG.info("{}: set stopIndex = {}", (Object)this, (Object)this.stopIndex);
        }
        this.updater.join();
    }

    void reloadStateMachine() {
        this.state = State.RELOAD;
        this.notifyUpdater();
    }

    void notifyUpdater() {
        this.awaitForSignal.signal();
    }

    public String toString() {
        return this.name;
    }

    @Override
    public void run() {
        CompletableFuture<Object> applyLogFutures = CompletableFuture.completedFuture(null);
        while (this.state != State.STOP) {
            try {
                this.waitForCommit(applyLogFutures);
                if (this.state == State.RELOAD) {
                    this.reload();
                }
                applyLogFutures = this.applyLog(applyLogFutures);
                this.checkAndTakeSnapshot(applyLogFutures);
                if (!this.shouldStop()) continue;
                applyLogFutures.get();
                this.stop();
            }
            catch (Throwable t2) {
                if (t2 instanceof InterruptedException && this.state == State.STOP) {
                    Thread.currentThread().interrupt();
                    LOG.info("{} was interrupted.  Exiting ...", (Object)this);
                    continue;
                }
                this.state = State.EXCEPTION;
                LOG.error(this + " caught a Throwable.", t2);
                this.server.close();
            }
        }
    }

    private void waitForCommit(CompletableFuture<?> applyLogFutures) throws InterruptedException, ExecutionException {
        long applied = this.getLastAppliedIndex();
        while (applied >= this.raftLog.getLastCommittedIndex() && this.state == State.RUNNING && !this.shouldStop()) {
            if (this.server.getSnapshotRequestHandler().shouldTriggerTakingSnapshot()) {
                this.takeSnapshot(applyLogFutures);
            }
            if (!this.awaitForSignal.await(100L, TimeUnit.MILLISECONDS)) continue;
            return;
        }
    }

    private void reload() throws IOException {
        Preconditions.assertTrue(this.stateMachine.getLifeCycleState() == LifeCycle.State.PAUSED);
        this.stateMachine.reinitialize();
        SnapshotInfo snapshot = this.stateMachine.getLatestSnapshot();
        Objects.requireNonNull(snapshot, "snapshot == null");
        long i = snapshot.getIndex();
        this.snapshotIndex.setUnconditionally(i, this.infoIndexChange);
        this.appliedIndex.setUnconditionally(i, this.infoIndexChange);
        this.notifyAppliedIndex(i);
        this.state = State.RUNNING;
    }

    private CompletableFuture<Void> applyLog(CompletableFuture<Void> applyLogFutures) throws RaftLogIOException {
        long applied;
        long committed = this.raftLog.getLastCommittedIndex();
        while ((applied = this.getLastAppliedIndex()) < committed && this.state == State.RUNNING && !this.shouldStop()) {
            long nextIndex = applied + 1L;
            RaftProtos.LogEntryProto next = this.raftLog.get(nextIndex);
            if (next != null) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace("{}: applying nextIndex={}, nextLog={}", this, nextIndex, LogProtoUtils.toLogEntryString(next));
                } else {
                    LOG.debug("{}: applying nextIndex={}", (Object)this, (Object)nextIndex);
                }
                CompletableFuture<Message> f = this.server.applyLogToStateMachine(next);
                long incremented = this.appliedIndex.incrementAndGet(this.debugIndexChange);
                Preconditions.assertTrue(incremented == nextIndex);
                if (f != null) {
                    CompletionStage exceptionHandledFuture = f.exceptionally(ex -> {
                        LOG.error("Exception while {}: applying txn index={}, nextLog={}", this, nextIndex, LogProtoUtils.toLogEntryString(next), ex);
                        return null;
                    });
                    applyLogFutures = applyLogFutures.thenCombine(exceptionHandledFuture, (v, message) -> null);
                    f.thenAccept(m4 -> this.notifyAppliedIndex(incremented));
                    continue;
                }
                this.notifyAppliedIndex(incremented);
                continue;
            }
            LOG.debug("{}: logEntry {} is null. There may be snapshot to load. state:{}", new Object[]{this, nextIndex, this.state});
            break;
        }
        return applyLogFutures;
    }

    private void checkAndTakeSnapshot(CompletableFuture<?> futures) throws ExecutionException, InterruptedException {
        if (this.shouldTakeSnapshot()) {
            this.takeSnapshot(futures);
        }
    }

    private void takeSnapshot(CompletableFuture<?> applyLogFutures) throws ExecutionException, InterruptedException {
        long i;
        applyLogFutures.get();
        try {
            try (UncheckedAutoCloseable ignored = Timekeeper.start(this.stateMachineMetrics.get().getTakeSnapshotTimer());){
                i = this.stateMachine.takeSnapshot();
            }
            this.server.getSnapshotRequestHandler().completeTakingSnapshot(i);
            long lastAppliedIndex = this.getLastAppliedIndex();
            if (i > lastAppliedIndex) {
                throw new StateMachineException("Bug in StateMachine: snapshot index = " + i + " > appliedIndex = " + lastAppliedIndex + "; StateMachine class=" + this.stateMachine.getClass().getName() + ", stateMachine=" + this.stateMachine);
            }
            this.stateMachine.getStateMachineStorage().cleanupOldSnapshots(this.snapshotRetentionPolicy);
        }
        catch (IOException e) {
            LOG.error(this.name + ": Failed to take snapshot", e);
            return;
        }
        if (i >= 0L) {
            long purgeIndex;
            LOG.info("{}: Took a snapshot at index {}", (Object)this.name, (Object)i);
            this.snapshotIndex.updateIncreasingly(i, this.infoIndexChange);
            if (this.purgeUptoSnapshotIndex) {
                purgeIndex = i;
            } else {
                LongStream commitIndexStream = this.server.getCommitInfos().stream().mapToLong(RaftProtos.CommitInfoProto::getCommitIndex);
                purgeIndex = LongStream.concat(LongStream.of(i), commitIndexStream).min().orElse(i);
            }
            this.raftLog.purge(purgeIndex);
        }
    }

    private boolean shouldStop() {
        return Optional.ofNullable(this.stopIndex.get()).filter(i -> i <= this.getLastAppliedIndex()).isPresent();
    }

    private boolean shouldTakeSnapshot() {
        if (this.state == State.RUNNING && this.server.getSnapshotRequestHandler().shouldTriggerTakingSnapshot()) {
            return true;
        }
        if (this.autoSnapshotThreshold == null) {
            return false;
        }
        if (this.shouldStop()) {
            return this.shouldTakeSnapshotAtStop() && this.getLastAppliedIndex() - this.snapshotIndex.get() > 0L;
        }
        return this.state == State.RUNNING && this.getStateMachineLastAppliedIndex() - this.snapshotIndex.get() >= this.autoSnapshotThreshold;
    }

    private boolean shouldTakeSnapshotAtStop() {
        return this.isRemoving ? this.triggerSnapshotWhenRemoveEnabled : this.triggerSnapshotWhenStopEnabled;
    }

    void setRemoving() {
        this.isRemoving = true;
    }

    private long getLastAppliedIndex() {
        return this.appliedIndex.get();
    }

    private void notifyAppliedIndex(long index) {
        this.appliedIndexConsumer.accept(index);
    }

    long getStateMachineLastAppliedIndex() {
        return Optional.ofNullable(this.stateMachine.getLastAppliedTermIndex()).map(TermIndex::getIndex).orElse(-1L);
    }

    static enum State {
        RUNNING,
        STOP,
        RELOAD,
        EXCEPTION;

    }
}

