/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificate.utils;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.InetAddress;
import java.security.KeyPair;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.validator.routines.DomainValidator;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.x509.exception.CertificateException;
import org.apache.hadoop.ozone.OzoneSecurityUtil;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1Set;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DERUTF8String;
import org.bouncycastle.asn1.pkcs.Attribute;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.BasicConstraints;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.Extensions;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.KeyUsage;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequestBuilder;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemObjectGenerator;
import org.bouncycastle.util.io.pem.PemReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CertificateSignRequest {
    private static final String DISTINGUISHED_NAME_FORMAT = "CN=%s,OU=%s,O=%s";
    private static final String DISTINGUISHED_NAME_WITH_SN_FORMAT = "CN=%s,OU=%s,O=%s,SERIALNUMBER=%s";
    private static final Logger LOG = LoggerFactory.getLogger(CertificateSignRequest.class);
    private final KeyPair keyPair;
    private final SecurityConfig config;
    private final Extensions extensions;
    private String subject;
    private String clusterID;
    private String scmID;

    private CertificateSignRequest(String subject, String scmID, String clusterID, KeyPair keyPair, SecurityConfig config, Extensions extensions) {
        this.subject = subject;
        this.clusterID = clusterID;
        this.scmID = scmID;
        this.keyPair = keyPair;
        this.config = config;
        this.extensions = extensions;
    }

    public static String getDistinguishedNameFormat() {
        return DISTINGUISHED_NAME_FORMAT;
    }

    public static String getDistinguishedNameFormatWithSN() {
        return DISTINGUISHED_NAME_WITH_SN_FORMAT;
    }

    public static X500Name getDistinguishedNameWithSN(String subject, String scmID, String clusterID, String serialID) {
        return new X500Name(String.format(DISTINGUISHED_NAME_WITH_SN_FORMAT, subject, scmID, clusterID, serialID));
    }

    public static X500Name getDistinguishedName(String subject, String scmID, String clusterID) {
        return new X500Name(String.format(CertificateSignRequest.getDistinguishedNameFormat(), subject, scmID, clusterID));
    }

    public static Extensions getPkcs9Extensions(PKCS10CertificationRequest csr) throws CertificateException {
        ASN1Set pkcs9ExtReq = CertificateSignRequest.getPkcs9ExtRequest(csr);
        Object extReqElement = pkcs9ExtReq.getObjects().nextElement();
        if (extReqElement instanceof Extensions) {
            return (Extensions)extReqElement;
        }
        if (extReqElement instanceof ASN1Sequence) {
            return Extensions.getInstance((Object)((ASN1Sequence)extReqElement));
        }
        throw new CertificateException("Unknown element type :" + extReqElement.getClass().getSimpleName());
    }

    public static ASN1Set getPkcs9ExtRequest(PKCS10CertificationRequest csr) throws CertificateException {
        for (Attribute attr : csr.getAttributes()) {
            ASN1ObjectIdentifier oid = attr.getAttrType();
            if (!oid.equals((ASN1Primitive)PKCSObjectIdentifiers.pkcs_9_at_extensionRequest)) continue;
            return attr.getAttrValues();
        }
        throw new CertificateException("No PKCS#9 extension found in CSR");
    }

    private PKCS10CertificationRequest generateCSR() throws OperatorCreationException {
        X500Name dnName = CertificateSignRequest.getDistinguishedName(this.subject, this.scmID, this.clusterID);
        JcaPKCS10CertificationRequestBuilder p10Builder = new JcaPKCS10CertificationRequestBuilder(dnName, this.keyPair.getPublic());
        ContentSigner contentSigner = new JcaContentSignerBuilder(this.config.getSignatureAlgo()).setProvider(this.config.getProvider()).build(this.keyPair.getPrivate());
        if (this.extensions != null) {
            p10Builder.addAttribute(PKCSObjectIdentifiers.pkcs_9_at_extensionRequest, (ASN1Encodable)this.extensions);
        }
        return p10Builder.build(contentSigner);
    }

    public static String getEncodedString(PKCS10CertificationRequest request) throws IOException {
        PemObject pemObject = new PemObject("CERTIFICATE REQUEST", request.getEncoded());
        StringWriter str = new StringWriter();
        try (JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)str);){
            pemWriter.writeObject((PemObjectGenerator)pemObject);
        }
        return str.toString();
    }

    public static PKCS10CertificationRequest getCertificationRequest(String csr) throws IOException {
        try (PemReader reader = new PemReader((Reader)new StringReader(csr));){
            PemObject pemObject = reader.readPemObject();
            if (pemObject.getContent() == null) {
                throw new SCMSecurityException("Invalid Certificate signing request", SCMSecurityException.ErrorCode.INVALID_CSR);
            }
            PKCS10CertificationRequest pKCS10CertificationRequest = new PKCS10CertificationRequest(pemObject.getContent());
            return pKCS10CertificationRequest;
        }
    }

    public static class Builder {
        private String subject;
        private String clusterID;
        private String scmID;
        private KeyPair key;
        private SecurityConfig config;
        private List<GeneralName> altNames;
        private Boolean ca = false;
        private boolean digitalSignature;
        private boolean digitalEncryption;

        public Builder setConfiguration(SecurityConfig configuration) {
            this.config = configuration;
            return this;
        }

        public Builder setKey(KeyPair keyPair) {
            this.key = keyPair;
            return this;
        }

        public Builder setSubject(String subjectString) {
            this.subject = subjectString;
            return this;
        }

        public Builder setClusterID(String s) {
            this.clusterID = s;
            return this;
        }

        public Builder setScmID(String s) {
            this.scmID = s;
            return this;
        }

        public Builder setDigitalSignature(boolean dSign) {
            this.digitalSignature = dSign;
            return this;
        }

        public Builder setDigitalEncryption(boolean dEncryption) {
            this.digitalEncryption = dEncryption;
            return this;
        }

        public Builder addDnsName(String dnsName) {
            Preconditions.checkNotNull((Object)dnsName, (Object)"dnsName cannot be null");
            this.addAltName(2, dnsName);
            return this;
        }

        public boolean hasDnsName() {
            if (this.altNames == null || this.altNames.isEmpty()) {
                return false;
            }
            for (GeneralName name : this.altNames) {
                if (name.getTagNo() != 2) continue;
                return true;
            }
            return false;
        }

        public Builder addIpAddress(String ip) {
            Preconditions.checkNotNull((Object)ip, (Object)"Ip address cannot be null");
            this.addAltName(7, ip);
            return this;
        }

        public Builder addInetAddresses() throws CertificateException {
            try {
                DomainValidator validator = DomainValidator.getInstance();
                List<InetAddress> inetAddresses = OzoneSecurityUtil.getValidInetsForCurrentHost();
                this.addInetAddresses(inetAddresses, validator);
            }
            catch (IOException e) {
                throw new CertificateException("Error while getting Inet addresses for the CSR builder", (Throwable)e, CertificateException.ErrorCode.CSR_ERROR);
            }
            return this;
        }

        public Builder addInetAddresses(List<InetAddress> addresses, DomainValidator validator) {
            addresses.forEach(ip -> {
                this.addIpAddress(ip.getHostAddress());
                if (validator.isValid(ip.getCanonicalHostName())) {
                    this.addDnsName(ip.getCanonicalHostName());
                } else {
                    LOG.error("Invalid domain {}", (Object)ip.getCanonicalHostName());
                }
            });
            return this;
        }

        public Builder addServiceName(String serviceName) {
            Preconditions.checkNotNull((Object)serviceName, (Object)"Service Name cannot be null");
            this.addAltName(0, serviceName);
            return this;
        }

        private Builder addAltName(int tag, String name) {
            if (this.altNames == null) {
                this.altNames = new ArrayList<GeneralName>();
            }
            if (tag == 0) {
                ASN1Object ono = this.addOtherNameAsn1Object(name);
                this.altNames.add(new GeneralName(tag, (ASN1Encodable)ono));
            } else {
                this.altNames.add(new GeneralName(tag, name));
            }
            return this;
        }

        private ASN1Object addOtherNameAsn1Object(String name) {
            String otherNameOID = "2.16.840.1.113730.3.1.34";
            ASN1EncodableVector otherName = new ASN1EncodableVector();
            otherName.add((ASN1Encodable)new ASN1ObjectIdentifier("2.16.840.1.113730.3.1.34"));
            otherName.add((ASN1Encodable)new DERTaggedObject(true, 0, (ASN1Encodable)new DERUTF8String(name)));
            return new DERTaggedObject(false, 0, (ASN1Encodable)new DERSequence(otherName));
        }

        public Builder setCA(Boolean isCA) {
            this.ca = isCA;
            return this;
        }

        private Extension getKeyUsageExtension() throws IOException {
            int keyUsageFlag = 8;
            if (this.digitalEncryption) {
                keyUsageFlag |= 0x30;
            }
            if (this.digitalSignature) {
                keyUsageFlag |= 0x80;
            }
            if (this.ca.booleanValue()) {
                keyUsageFlag |= 6;
            }
            KeyUsage keyUsage = new KeyUsage(keyUsageFlag);
            return new Extension(Extension.keyUsage, true, keyUsage.getEncoded());
        }

        private Optional<Extension> getSubjectAltNameExtension() throws IOException {
            if (this.altNames != null) {
                return Optional.of(new Extension(Extension.subjectAlternativeName, false, (ASN1OctetString)new DEROctetString((ASN1Encodable)new GeneralNames(this.altNames.toArray(new GeneralName[this.altNames.size()])))));
            }
            return Optional.empty();
        }

        private Extension getBasicExtension() throws IOException {
            return new Extension(Extension.basicConstraints, true, (ASN1OctetString)new DEROctetString((ASN1Encodable)new BasicConstraints(this.ca.booleanValue())));
        }

        private Extensions createExtensions() throws IOException {
            ArrayList<Extension> extensions = new ArrayList<Extension>();
            if (this.ca.booleanValue()) {
                extensions.add(this.getBasicExtension());
            }
            extensions.add(this.getKeyUsageExtension());
            Optional<Extension> san = this.getSubjectAltNameExtension();
            if (san.isPresent()) {
                extensions.add(san.get());
            }
            return new Extensions(extensions.toArray(new Extension[extensions.size()]));
        }

        public PKCS10CertificationRequest build() throws SCMSecurityException {
            Preconditions.checkNotNull((Object)this.key, (Object)"KeyPair cannot be null");
            Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)this.subject), (Object)"Subject cannot be blank");
            try {
                CertificateSignRequest csr = new CertificateSignRequest(this.subject, this.scmID, this.clusterID, this.key, this.config, this.createExtensions());
                return csr.generateCSR();
            }
            catch (IOException ioe) {
                throw new CertificateException(String.format("Unable to create extension for certificate sign request for %s.", CertificateSignRequest.getDistinguishedName(this.subject, this.scmID, this.clusterID)), ioe.getCause());
            }
            catch (OperatorCreationException ex) {
                throw new CertificateException(String.format("Unable to create certificate sign request for %s.", CertificateSignRequest.getDistinguishedName(this.subject, this.scmID, this.clusterID)), ex.getCause());
            }
        }
    }
}

