/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container;

import com.google.common.base.Preconditions;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.TextFormat;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import org.apache.hadoop.hdds.client.ECReplicationConfig;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.ContainerReplicaNotFoundException;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.ozone.protocol.commands.CommandForDatanode;
import org.apache.hadoop.ozone.protocol.commands.DeleteContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.slf4j.Logger;

public class AbstractContainerReportHandler {
    private final ContainerManager containerManager;
    private final SCMContext scmContext;
    private final Logger logger;

    AbstractContainerReportHandler(ContainerManager containerManager, SCMContext scmContext, Logger logger) {
        Preconditions.checkNotNull((Object)containerManager);
        Preconditions.checkNotNull((Object)scmContext);
        Preconditions.checkNotNull((Object)logger);
        this.containerManager = containerManager;
        this.scmContext = scmContext;
        this.logger = logger;
    }

    protected void processContainerReplica(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto replicaProto, EventPublisher publisher) throws IOException, InvalidStateTransitionException, TimeoutException {
        ContainerInfo container = this.getContainerManager().getContainer(ContainerID.valueOf((long)replicaProto.getContainerID()));
        this.processContainerReplica(datanodeDetails, container, replicaProto, publisher);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void processContainerReplica(DatanodeDetails datanodeDetails, ContainerInfo containerInfo, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto replicaProto, EventPublisher publisher) throws IOException, InvalidStateTransitionException, TimeoutException {
        ContainerID containerId = containerInfo.containerID();
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("Processing replica of container {} from datanode {}", (Object)containerId, (Object)datanodeDetails);
        }
        ContainerInfo containerInfo2 = containerInfo;
        synchronized (containerInfo2) {
            this.updateContainerStats(datanodeDetails, containerInfo, replicaProto);
            if (!this.updateContainerState(datanodeDetails, containerInfo, replicaProto, publisher)) {
                this.updateContainerReplica(datanodeDetails, containerId, replicaProto);
            }
        }
    }

    private void updateContainerStats(DatanodeDetails datanodeDetails, ContainerInfo containerInfo, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto replicaProto) throws ContainerNotFoundException {
        if (containerInfo.getState() == HddsProtos.LifeCycleState.CLOSED && containerInfo.getSequenceId() < replicaProto.getBlockCommitSequenceId()) {
            this.logger.error("There is a CLOSED container with lower sequence ID than a replica. Container: {}, Container's sequence ID: {}, Replica: {}, Replica's sequence ID: {}, Datanode: {}.", new Object[]{containerInfo, containerInfo.getSequenceId(), TextFormat.shortDebugString((MessageOrBuilder)replicaProto), replicaProto.getBlockCommitSequenceId(), datanodeDetails});
        }
        if (this.isHealthy(() -> ((StorageContainerDatanodeProtocolProtos.ContainerReplicaProto)replicaProto).getState())) {
            if (containerInfo.getSequenceId() < replicaProto.getBlockCommitSequenceId()) {
                containerInfo.updateSequenceId(replicaProto.getBlockCommitSequenceId());
            }
            if (containerInfo.getReplicationConfig().getReplicationType() == HddsProtos.ReplicationType.EC) {
                this.updateECContainerStats(containerInfo, replicaProto, datanodeDetails);
            } else {
                this.updateRatisContainerStats(containerInfo, replicaProto, datanodeDetails);
            }
        }
    }

    private void updateRatisContainerStats(ContainerInfo containerInfo, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto newReplica, DatanodeDetails newSource) throws ContainerNotFoundException {
        List<ContainerReplica> otherReplicas = this.getOtherReplicas(containerInfo.containerID(), newSource);
        long usedBytes = newReplica.getUsed();
        long keyCount = newReplica.getKeyCount();
        for (ContainerReplica r : otherReplicas) {
            usedBytes = this.calculateUsage(containerInfo, usedBytes, r.getBytesUsed());
            keyCount = this.calculateUsage(containerInfo, keyCount, r.getKeyCount());
        }
        this.updateContainerUsedAndKeys(containerInfo, usedBytes, keyCount);
    }

    private void updateECContainerStats(ContainerInfo containerInfo, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto newReplica, DatanodeDetails newSource) throws ContainerNotFoundException {
        int dataNum = ((ECReplicationConfig)containerInfo.getReplicationConfig()).getData();
        if (newReplica.getReplicaIndex() == 1 || newReplica.getReplicaIndex() > dataNum) {
            List<ContainerReplica> otherReplicas = this.getOtherReplicas(containerInfo.containerID(), newSource);
            long usedBytes = newReplica.getUsed();
            long keyCount = newReplica.getKeyCount();
            for (ContainerReplica r : otherReplicas) {
                if (r.getReplicaIndex() > 1 && r.getReplicaIndex() <= dataNum) continue;
                usedBytes = this.calculateUsage(containerInfo, usedBytes, r.getBytesUsed());
                keyCount = this.calculateUsage(containerInfo, keyCount, r.getKeyCount());
            }
            this.updateContainerUsedAndKeys(containerInfo, usedBytes, keyCount);
        }
    }

    private long calculateUsage(ContainerInfo containerInfo, long lastValue, long thisValue) {
        if (containerInfo.getState().equals((Object)HddsProtos.LifeCycleState.OPEN)) {
            return Math.min(lastValue, thisValue);
        }
        return Math.max(lastValue, thisValue);
    }

    private void updateContainerUsedAndKeys(ContainerInfo containerInfo, long usedBytes, long keyCount) {
        if (containerInfo.getUsedBytes() != usedBytes) {
            containerInfo.setUsedBytes(usedBytes);
        }
        if (containerInfo.getNumberOfKeys() != keyCount) {
            containerInfo.setNumberOfKeys(keyCount);
        }
    }

    private List<ContainerReplica> getOtherReplicas(ContainerID containerId, DatanodeDetails exclude) throws ContainerNotFoundException {
        ArrayList<ContainerReplica> filteredReplicas = new ArrayList<ContainerReplica>();
        Set<ContainerReplica> replicas = this.containerManager.getContainerReplicas(containerId);
        for (ContainerReplica r : replicas) {
            if (r.getDatanodeDetails().equals((Object)exclude)) continue;
            filteredReplicas.add(r);
        }
        return filteredReplicas;
    }

    private boolean updateContainerState(DatanodeDetails datanode, ContainerInfo container, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto replica, EventPublisher publisher) throws IOException, InvalidStateTransitionException, TimeoutException {
        ContainerID containerId = container.containerID();
        boolean ignored = false;
        boolean replicaIsEmpty = replica.hasIsEmpty() && replica.getIsEmpty();
        switch (container.getState()) {
            case OPEN: {
                if (replica.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.OPEN) break;
                this.logger.info("Moving OPEN container {} to CLOSING state, datanode {} reported {} replica with index {}.", new Object[]{containerId, datanode, replica.getState(), replica.getReplicaIndex()});
                this.containerManager.updateContainerState(containerId, HddsProtos.LifeCycleEvent.FINALIZE);
                break;
            }
            case CLOSING: {
                if (replica.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED) {
                    this.logger.info("Moving container {} to QUASI_CLOSED state, datanode {} reported QUASI_CLOSED replica.", (Object)containerId, (Object)datanode);
                    this.containerManager.updateContainerState(containerId, HddsProtos.LifeCycleEvent.QUASI_CLOSE);
                }
                if (replica.getState() != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED) break;
                if (container.getReplicationType().equals((Object)HddsProtos.ReplicationType.EC)) {
                    int replicaIndex = replica.getReplicaIndex();
                    int dataNum = ((ECReplicationConfig)container.getReplicationConfig()).getData();
                    if (replicaIndex != 1 && replicaIndex <= dataNum) break;
                }
                if (!this.verifyBcsId(replica.getBlockCommitSequenceId(), container.getSequenceId(), datanode, containerId)) {
                    this.logger.warn("Ignored moving container {} from CLOSING to CLOSED state because replica bcsId ({}) reported by datanode {} does not match sequenceId ({}).", new Object[]{containerId, replica.getBlockCommitSequenceId(), datanode, container.getSequenceId()});
                    return true;
                }
                this.logger.info("Moving container {} from CLOSING to CLOSED state, datanode {} reported CLOSED replica with index {}.", new Object[]{containerId, datanode, replica.getReplicaIndex()});
                this.containerManager.updateContainerState(containerId, HddsProtos.LifeCycleEvent.CLOSE);
                break;
            }
            case QUASI_CLOSED: {
                if (replica.getState() != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED) break;
                if (!this.verifyBcsId(replica.getBlockCommitSequenceId(), container.getSequenceId(), datanode, containerId)) {
                    this.logger.warn("Ignored moving container {} from QUASI_CLOSED to CLOSED state because replica bcsId ({}) reported by datanode {} does not match sequenceId ({}).", new Object[]{containerId, replica.getBlockCommitSequenceId(), datanode, container.getSequenceId()});
                    return true;
                }
                this.logger.info("Moving container {} from QUASI_CLOSED to CLOSED state, datanode {} reported CLOSED replica with index {}.", new Object[]{containerId, datanode, replica.getReplicaIndex()});
                this.containerManager.updateContainerState(containerId, HddsProtos.LifeCycleEvent.FORCE_CLOSE);
                break;
            }
            case CLOSED: {
                break;
            }
            case DELETED: {
                if (replicaIsEmpty) {
                    this.deleteReplica(containerId, datanode, publisher, "DELETED", false);
                    break;
                }
            }
            case DELETING: {
                boolean replicaStateAllowed;
                boolean bl = replicaStateAllowed = replica.getState() != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.INVALID && replica.getState() != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.DELETED;
                if (replicaIsEmpty || !replicaStateAllowed) break;
                this.logger.info("Moving container {} from {} to CLOSED state, datanode {} reported replica with state={}, isEmpty={}, bcsId={}, keyCount={}, and origin={}", new Object[]{container, container.getState(), datanode.getHostName(), replica.getState(), replica.getIsEmpty(), replica.getBlockCommitSequenceId(), replica.getKeyCount(), replica.getOriginNodeId()});
                this.containerManager.transitionDeletingOrDeletedToClosedState(containerId);
                break;
            }
        }
        return ignored;
    }

    private boolean verifyBcsId(long replicaBcsId, long containerBcsId, DatanodeDetails datanode, ContainerID containerId) {
        if (replicaBcsId != containerBcsId) {
            String errMsg = "Unexpected bcsId for container " + containerId + " from datanode " + datanode + ". replica's: " + replicaBcsId + ", SCM's: " + containerBcsId + ". Ignoring container report for " + containerId;
            this.logger.error(errMsg);
            return false;
        }
        return true;
    }

    private void updateContainerReplica(DatanodeDetails datanodeDetails, ContainerID containerId, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto replicaProto) throws ContainerNotFoundException, ContainerReplicaNotFoundException {
        ContainerReplica replica = ContainerReplica.newBuilder().setContainerID(containerId).setContainerState(replicaProto.getState()).setDatanodeDetails(datanodeDetails).setOriginNodeId(UUID.fromString(replicaProto.getOriginNodeId())).setSequenceId(replicaProto.getBlockCommitSequenceId()).setKeyCount(replicaProto.getKeyCount()).setReplicaIndex(replicaProto.getReplicaIndex()).setBytesUsed(replicaProto.getUsed()).setEmpty(replicaProto.getIsEmpty()).build();
        if (replica.getState().equals((Object)StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.DELETED)) {
            this.containerManager.removeContainerReplica(containerId, replica);
        } else {
            this.containerManager.updateContainerReplica(containerId, replica);
        }
    }

    private boolean isHealthy(Supplier<StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State> replicaState) {
        return replicaState.get() != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.UNHEALTHY && replicaState.get() != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.INVALID && replicaState.get() != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.DELETED;
    }

    protected ContainerManager getContainerManager() {
        return this.containerManager;
    }

    protected void deleteReplica(ContainerID containerID, DatanodeDetails dn, EventPublisher publisher, String reason, boolean force) {
        DeleteContainerCommand command = new DeleteContainerCommand(containerID.getId(), force);
        try {
            command.setTerm(this.scmContext.getTermOfLeader());
        }
        catch (NotLeaderException nle) {
            this.logger.warn("Skip sending delete container command, since not leader SCM", (Throwable)nle);
            return;
        }
        publisher.fireEvent(SCMEvents.DATANODE_COMMAND, (Object)new CommandForDatanode(dn.getUuid(), (SCMCommand)command));
        this.logger.info("Sending delete container command for " + reason + " container {} to datanode {}", (Object)containerID.getId(), (Object)dn);
    }
}

