/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.snapshot;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.CacheLoader;
import java.io.IOException;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.hdds.utils.Scheduler;
import org.apache.hadoop.ozone.om.OMMetrics;
import org.apache.hadoop.ozone.om.OmSnapshot;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.snapshot.ReferenceCounted;
import org.apache.hadoop.ozone.om.snapshot.ReferenceCountedCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotCache
implements ReferenceCountedCallback,
AutoCloseable {
    static final Logger LOG = LoggerFactory.getLogger(SnapshotCache.class);
    private final ConcurrentHashMap<UUID, ReferenceCounted<OmSnapshot>> dbMap = new ConcurrentHashMap();
    private final CacheLoader<UUID, OmSnapshot> cacheLoader;
    private final int cacheSizeLimit;
    private final Set<UUID> pendingEvictionQueue;
    private final Scheduler scheduler;
    private static final String SNAPSHOT_CACHE_CLEANUP_SERVICE = "SnapshotCacheCleanupService";
    private final OMMetrics omMetrics;

    public SnapshotCache(CacheLoader<UUID, OmSnapshot> cacheLoader, int cacheSizeLimit, OMMetrics omMetrics, long cleanupInterval) {
        this.cacheLoader = cacheLoader;
        this.cacheSizeLimit = cacheSizeLimit;
        this.omMetrics = omMetrics;
        this.pendingEvictionQueue = ConcurrentHashMap.newKeySet();
        if (cleanupInterval > 0L) {
            this.scheduler = new Scheduler(SNAPSHOT_CACHE_CLEANUP_SERVICE, true, 1);
            this.scheduler.scheduleWithFixedDelay(this::cleanup, cleanupInterval, cleanupInterval, TimeUnit.MILLISECONDS);
        } else {
            this.scheduler = null;
        }
    }

    @VisibleForTesting
    ConcurrentHashMap<UUID, ReferenceCounted<OmSnapshot>> getDbMap() {
        return this.dbMap;
    }

    public int size() {
        return this.dbMap.size();
    }

    public void invalidate(UUID key) {
        this.dbMap.compute(key, (k, v) -> {
            if (v == null) {
                LOG.warn("SnapshotId: '{}' does not exist in snapshot cache.", k);
            } else {
                try {
                    ((OmSnapshot)v.get()).close();
                }
                catch (IOException e) {
                    throw new IllegalStateException("Failed to close snapshotId: " + key, e);
                }
                this.omMetrics.decNumSnapshotCacheSize();
            }
            return null;
        });
    }

    public void invalidateAll() {
        for (UUID key : this.dbMap.keySet()) {
            this.invalidate(key);
        }
    }

    @Override
    public void close() {
        this.invalidateAll();
        if (this.scheduler != null) {
            this.scheduler.close();
        }
    }

    public ReferenceCounted<OmSnapshot> get(UUID key) throws IOException {
        ReferenceCounted rcOmSnapshot;
        if (this.size() > this.cacheSizeLimit) {
            LOG.warn("Snapshot cache size ({}) exceeds configured soft-limit ({}).", (Object)this.size(), (Object)this.cacheSizeLimit);
        }
        if ((rcOmSnapshot = this.dbMap.compute(key, (k, v) -> {
            if (v == null) {
                LOG.info("Loading SnapshotId: '{}'", k);
                try {
                    v = new ReferenceCounted<OmSnapshot>((OmSnapshot)this.cacheLoader.load((Object)key), false, this);
                }
                catch (OMException omEx) {
                    if (!omEx.getResult().equals((Object)OMException.ResultCodes.FILE_NOT_FOUND)) {
                        throw new IllegalStateException(omEx);
                    }
                }
                catch (IOException ioEx) {
                    throw new IllegalStateException(ioEx);
                }
                catch (Exception ex) {
                    throw new IllegalStateException(ex);
                }
                this.omMetrics.incNumSnapshotCacheSize();
            }
            if (v != null) {
                v.incrementRefCount();
            }
            return v;
        })) == null) {
            throw new OMException("SnapshotId: '" + key + "' not found, or the snapshot is no longer active.", OMException.ResultCodes.FILE_NOT_FOUND);
        }
        return rcOmSnapshot;
    }

    public void release(UUID key) {
        ReferenceCounted<OmSnapshot> val = this.dbMap.get(key);
        if (val == null) {
            throw new IllegalArgumentException("Key '" + key + "' does not " + "exist in cache.");
        }
        val.decrementRefCount();
    }

    @VisibleForTesting
    void cleanup() {
        if (this.dbMap.size() > this.cacheSizeLimit) {
            for (UUID evictionKey : this.pendingEvictionQueue) {
                this.dbMap.compute(evictionKey, (k, v) -> {
                    this.pendingEvictionQueue.remove(k);
                    if (v == null) {
                        throw new IllegalStateException("SnapshotId '" + k + "' does not exist in cache. The RocksDB " + "instance of the Snapshot may not be closed properly.");
                    }
                    if (v.getTotalRefCount() > 0L) {
                        LOG.debug("SnapshotId {} is still being referenced ({}), skipping its clean up.", k, (Object)v.getTotalRefCount());
                        return v;
                    }
                    LOG.debug("Closing SnapshotId {}. It is not being referenced anymore.", k);
                    try {
                        ((OmSnapshot)v.get()).close();
                    }
                    catch (IOException ex) {
                        throw new IllegalStateException("Error while closing snapshot DB.", ex);
                    }
                    this.omMetrics.decNumSnapshotCacheSize();
                    return null;
                });
            }
        }
    }

    @Override
    public void callback(ReferenceCounted referenceCounted) {
        if (referenceCounted.getTotalRefCount() == 0L) {
            this.pendingEvictionQueue.add(((OmSnapshot)referenceCounted.get()).getSnapshotID());
        }
    }

    public static enum Reason {
        FS_API_READ,
        SNAP_DIFF_READ,
        DEEP_CLEAN_WRITE,
        GARBAGE_COLLECTION_WRITE;

    }
}

