/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.meecrowave.junit5;

import org.apache.meecrowave.Meecrowave;
import org.apache.meecrowave.configuration.Configuration;
import org.apache.meecrowave.internal.ClassLoaderLock;
import org.apache.meecrowave.testing.Injector;
import org.junit.jupiter.api.extension.AfterAllCallback;
import org.junit.jupiter.api.extension.AfterEachCallback;
import org.junit.jupiter.api.extension.BeforeAllCallback;
import org.junit.jupiter.api.extension.BeforeEachCallback;
import org.junit.jupiter.api.extension.ExtensionContext;

import jakarta.enterprise.context.spi.CreationalContext;
import java.io.File;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Optional;
import java.util.stream.Stream;

import static java.util.Optional.ofNullable;

public class MeecrowaveExtension extends BaseLifecycle
        implements BeforeAllCallback, AfterAllCallback, BeforeEachCallback, AfterEachCallback {

    private static final ExtensionContext.Namespace NAMESPACE = ExtensionContext.Namespace.create(MeecrowaveExtension.class.getName());

    private ClassLoader meecrowaveCL;
    private ClassLoader oldCl;


    private final ScopesExtension scopes = new ScopesExtension() {
        @Override
        protected Optional<Class<? extends Annotation>[]> getScopes(final ExtensionContext context) {
            return context.getElement()
                    .map(e -> findMwConfig(context))
                    .map(MeecrowaveConfig::scopes)
                    .filter(s -> s.length > 0);
        }
    };

    protected ClassLoader createMwClassLoader() {
        if (meecrowaveCL == null) {
            meecrowaveCL = ClassLoaderLock.getUsableContainerLoader();
        }
        return meecrowaveCL;
    }

    @Override
    public void beforeAll(final ExtensionContext context) {
        if (isPerClass(context)) {
            doStart(context);
        }
    }

    @Override
    public void afterAll(final ExtensionContext context) {
        final ExtensionContext.Store store = context.getStore(NAMESPACE);
        ofNullable(store.get(LifecyleState.class, LifecyleState.class))
                .ifPresent(s -> s.afterLastTest(context));
        if (isPerClass(context)) {
            ClassLoader oldClTmp = null;
            try {
                if (meecrowaveCL != null) {
                    oldClTmp = Thread.currentThread().getContextClassLoader();
                    Thread.currentThread().setContextClassLoader(meecrowaveCL);
                }

                store.get(Meecrowave.class, Meecrowave.class).close();
            }
            finally {
                if (oldClTmp != null) {
                    Thread.currentThread().setContextClassLoader(oldClTmp);
                }
            }
        }
    }

    @Override
    public void beforeEach(final ExtensionContext context) {
        if (!isPerClass(context)) {
            doStart(context);
        }

        if (meecrowaveCL != null) {
            oldCl = Thread.currentThread().getContextClassLoader();
            Thread.currentThread().setContextClassLoader(meecrowaveCL);
        }
    }

    @Override
    public void afterEach(final ExtensionContext context) {
        ClassLoader oldClTmp = null;
        try {
            if (!isPerClass(context)) {
                if (meecrowaveCL != null) {
                    oldClTmp = Thread.currentThread().getContextClassLoader();
                    Thread.currentThread().setContextClassLoader(meecrowaveCL);
                }
                doRelease(context);
                context.getStore(NAMESPACE).get(Meecrowave.class, Meecrowave.class).close();
            }
        }
        finally {
            if (oldCl != null) {
                Thread.currentThread().setContextClassLoader(oldCl);
            }
            else if (oldClTmp != null) {
                Thread.currentThread().setContextClassLoader(oldClTmp);
            }
        }
    }

    private void doStart(final ExtensionContext context) {
        final Thread thread = Thread.currentThread();
        ClassLoader oldClTmp = thread.getContextClassLoader();
        boolean unlocked = false;
        doLockContext();

        try {
            ClassLoader newCl = createMwClassLoader();
            if (newCl != null) {
                thread.setContextClassLoader(newCl);
            }

            final Meecrowave.Builder builder = new Meecrowave.Builder();
            final MeecrowaveConfig config = findMwConfig(context);
            final String ctx;
            if (config != null) {
                ctx = config.context();

                for (final Method method : MeecrowaveConfig.class.getMethods()) {
                    if (MeecrowaveConfig.class != method.getDeclaringClass()) {
                        continue;
                    }

                    try {
                        final Object value = method.invoke(config);

                        final Field configField = Configuration.class.getDeclaredField(method.getName());
                        if (!configField.canAccess(builder)) {
                            configField.setAccessible(true);
                        }

                        if (value != null && (!String.class.isInstance(value) || !value.toString().isEmpty())) {
                            if (!configField.canAccess(builder)) {
                                configField.setAccessible(true);
                            }
                            configField.set(builder, File.class == configField.getType() ? /*we use string instead */new File(value.toString()) : value);
                        }
                    }
                    catch (final NoSuchFieldException nsfe) {
                        // ignored
                    }
                    catch (final Exception e) {
                        throw new IllegalStateException(e);
                    }
                }

                if (builder.getHttpPort() < 0) {
                    builder.randomHttpPort();
                }
            }
            else {
                ctx = "";
            }
            final ExtensionContext.Store store = context.getStore(NAMESPACE);
            final Meecrowave meecrowave = new Meecrowave(builder);
            store.put(Meecrowave.class, meecrowave);
            store.put(Meecrowave.Builder.class, builder);
            meecrowave.bake(ctx);

            doInject(context);
            store.put(LifecyleState.class, onInjection(context, null));
        }
        finally {
            doUnlockContext(unlocked);
            if (oldClTmp != null) {
                thread.setContextClassLoader(oldClTmp);
            }
        }
    }

/*X
    private MeecrowaveConfig findConfig(final ExtensionContext context) {
        return findAnnotation(context.getElement(), MeecrowaveConfig.class)
                .orElseGet(() -> context.getParent()
                        .flatMap(ExtensionContext::getElement)
                        .flatMap(it -> findAnnotation(it, MeecrowaveConfig.class))
                        .orElse(null));
    }
*/

    private MeecrowaveConfig findMwConfig(final ExtensionContext context) {
        // if MeecrowaveConfig is directly on the test class
        MeecrowaveConfig mwConfig = findMwConfig(context.getElement().isPresent() ? context.getElement().get(): null);

        // also check parent classes
        if (mwConfig == null && context.getParent().isPresent()) {
            final AnnotatedElement annotatedParent = context.getParent().get().getElement().orElse(null);
            mwConfig = findMwConfig(annotatedParent);
        }


        return mwConfig;
    }

    private MeecrowaveConfig findMwConfig(AnnotatedElement it) {
        MeecrowaveConfig mwConfig = it.getAnnotation(MeecrowaveConfig.class);

        if (mwConfig == null) {
            //find metaconfig
            mwConfig = Arrays.stream(it.getAnnotations()).sequential()
                    .flatMap(an -> findMwConfigMetaAnnotation(an))
                    .findFirst()
                    .orElse(null);
        }

        return  mwConfig;
    }

    private Stream<MeecrowaveConfig> findMwConfigMetaAnnotation(Annotation an) {
        for (Annotation annotation : an.annotationType().getAnnotations()) {
            if (annotation.annotationType().equals(MeecrowaveConfig.class)) {
                return Stream.of((MeecrowaveConfig) annotation);
            }
        }

        return null;
    }


    private void doRelease(final ExtensionContext context) {
        ofNullable(context.getStore(NAMESPACE).get(CreationalContext.class, CreationalContext.class))
                .ifPresent(CreationalContext::release);
        scopes.afterEach(context);
    }

    private void doInject(final ExtensionContext context) {
        scopes.beforeEach(context);
        final ExtensionContext.Store store = context.getStore(NAMESPACE);
        store.put(CreationalContext.class, Injector.inject(context.getTestInstance().orElse(null)));
        Injector.injectConfig(
                store.get(Meecrowave.Builder.class, Meecrowave.Builder.class),
                context.getTestInstance().orElse(null));
    }
}
