﻿using NUnit.Framework;

namespace Lucene.Net.Analysis.Phonetic.Language
{
    /*
     * Licensed to the Apache Software Foundation (ASF) under one or more
     * contributor license agreements.  See the NOTICE file distributed with
     * this work for additional information regarding copyright ownership.
     * The ASF licenses this file to You under the Apache License, Version 2.0
     * (the "License"); you may not use this file except in compliance with
     * the License.  You may obtain a copy of the License at
     *
     *     http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     */

    /// <summary>
    /// Tests <see cref="DoubleMetaphone"/>.
    /// <para/>
    /// The test data was extracted from Stephen Woodbridge's <a
    /// href="http://swoodbridge.com/DoubleMetaPhone/surnames.txt">PHP test program</a>.
    /// </summary>
    public class DoubleMetaphone2Test : StringEncoderAbstractTest<DoubleMetaphone>
    {
        private static readonly int ALTERNATE_INDEX = 2;

        private static readonly int PRIMARY_INDEX = 1;

        /** Test values and their expected primary & alternate Double Metaphone encodings */
        private static readonly string[][]
        TEST_DATA = {
        new string[] {"ALLERTON", "ALRT", "ALRT"},
        new string[] {"Acton", "AKTN", "AKTN"},
        new string[] {"Adams", "ATMS", "ATMS"},
        new string[] {"Aggar", "AKR", "AKR"},
        new string[] {"Ahl", "AL", "AL"},
        new string[] {"Aiken", "AKN", "AKN"},
        new string[] {"Alan", "ALN", "ALN"},
        new string[] {"Alcock", "ALKK", "ALKK"},
        new string[] {"Alden", "ALTN", "ALTN"},
        new string[] {"Aldham", "ALTM", "ALTM"},
        new string[] {"Allen", "ALN", "ALN"},
        new string[] {"Allerton", "ALRT", "ALRT"},
        new string[] {"Alsop", "ALSP", "ALSP"},
        new string[] {"Alwein", "ALN", "ALN"},
        new string[] {"Ambler", "AMPL", "AMPL"},
        new string[] {"Andevill", "ANTF", "ANTF"},
        new string[] {"Andrews", "ANTR", "ANTR"},
        new string[] {"Andreyco", "ANTR", "ANTR"},
        new string[] {"Andriesse", "ANTR", "ANTR"},
        new string[] {"Angier", "ANJ", "ANJR"},
        new string[] {"Annabel", "ANPL", "ANPL"},
        new string[] {"Anne", "AN", "AN"},
        new string[] {"Anstye", "ANST", "ANST"},
        new string[] {"Appling", "APLN", "APLN"},
        new string[] {"Apuke", "APK", "APK"},
        new string[] {"Arnold", "ARNL", "ARNL"},
        new string[] {"Ashby", "AXP", "AXP"},
        new string[] {"Astwood", "ASTT", "ASTT"},
        new string[] {"Atkinson", "ATKN", "ATKN"},
        new string[] {"Audley", "ATL", "ATL"},
        new string[] {"Austin", "ASTN", "ASTN"},
        new string[] {"Avenal", "AFNL", "AFNL"},
        new string[] {"Ayer", "AR", "AR"},
        new string[] {"Ayot", "AT", "AT"},
        new string[] {"Babbitt", "PPT", "PPT"},
        new string[] {"Bachelor", "PXLR", "PKLR"},
        new string[] {"Bachelour", "PXLR", "PKLR"},
        new string[] {"Bailey", "PL", "PL"},
        new string[] {"Baivel", "PFL", "PFL"},
        new string[] {"Baker", "PKR", "PKR"},
        new string[] {"Baldwin", "PLTN", "PLTN"},
        new string[] {"Balsley", "PLSL", "PLSL"},
        new string[] {"Barber", "PRPR", "PRPR"},
        new string[] {"Barker", "PRKR", "PRKR"},
        new string[] {"Barlow", "PRL", "PRLF"},
        new string[] {"Barnard", "PRNR", "PRNR"},
        new string[] {"Barnes", "PRNS", "PRNS"},
        new string[] {"Barnsley", "PRNS", "PRNS"},
        new string[] {"Barouxis", "PRKS", "PRKS"},
        new string[] {"Bartlet", "PRTL", "PRTL"},
        new string[] {"Basley", "PSL", "PSL"},
        new string[] {"Basset", "PST", "PST"},
        new string[] {"Bassett", "PST", "PST"},
        new string[] {"Batchlor", "PXLR", "PXLR"},
        new string[] {"Bates", "PTS", "PTS"},
        new string[] {"Batson", "PTSN", "PTSN"},
        new string[] {"Bayes", "PS", "PS"},
        new string[] {"Bayley", "PL", "PL"},
        new string[] {"Beale", "PL", "PL"},
        new string[] {"Beauchamp", "PXMP", "PKMP"},
        new string[] {"Beauclerc", "PKLR", "PKLR"},
        new string[] {"Beech", "PK", "PK"},
        new string[] {"Beers", "PRS", "PRS"},
        new string[] {"Beke", "PK", "PK"},
        new string[] {"Belcher", "PLXR", "PLKR"},
        new string[] {"benign", "PNN", "PNKN"},
        new string[] {"Benjamin", "PNJM", "PNJM"},
        new string[] {"Benningham", "PNNK", "PNNK"},
        new string[] {"Bereford", "PRFR", "PRFR"},
        new string[] {"Bergen", "PRJN", "PRKN"},
        new string[] {"Berkeley", "PRKL", "PRKL"},
        new string[] {"Berry", "PR", "PR"},
        new string[] {"Besse", "PS", "PS"},
        new string[] {"Bessey", "PS", "PS"},
        new string[] {"Bessiles", "PSLS", "PSLS"},
        new string[] {"Bigelow", "PJL", "PKLF"},
        new string[] {"Bigg", "PK", "PK"},
        new string[] {"Bigod", "PKT", "PKT"},
        new string[] {"Billings", "PLNK", "PLNK"},
        new string[] {"Bimper", "PMPR", "PMPR"},
        new string[] {"Binker", "PNKR", "PNKR"},
        new string[] {"Birdsill", "PRTS", "PRTS"},
        new string[] {"Bishop", "PXP", "PXP"},
        new string[] {"Black", "PLK", "PLK"},
        new string[] {"Blagge", "PLK", "PLK"},
        new string[] {"Blake", "PLK", "PLK"},
        new string[] {"Blanck", "PLNK", "PLNK"},
        new string[] {"Bledsoe", "PLTS", "PLTS"},
        new string[] {"Blennerhasset", "PLNR", "PLNR"},
        new string[] {"Blessing", "PLSN", "PLSN"},
        new string[] {"Blewett", "PLT", "PLT"},
        new string[] {"Bloctgoed", "PLKT", "PLKT"},
        new string[] {"Bloetgoet", "PLTK", "PLTK"},
        new string[] {"Bloodgood", "PLTK", "PLTK"},
        new string[] {"Blossom", "PLSM", "PLSM"},
        new string[] {"Blount", "PLNT", "PLNT"},
        new string[] {"Bodine", "PTN", "PTN"},
        new string[] {"Bodman", "PTMN", "PTMN"},
        new string[] {"BonCoeur", "PNKR", "PNKR"},
        new string[] {"Bond", "PNT", "PNT"},
        new string[] {"Boscawen", "PSKN", "PSKN"},
        new string[] {"Bosworth", "PSR0", "PSRT"},
        new string[] {"Bouchier", "PX", "PKR"},
        new string[] {"Bowne", "PN", "PN"},
        new string[] {"Bradbury", "PRTP", "PRTP"},
        new string[] {"Bradder", "PRTR", "PRTR"},
        new string[] {"Bradford", "PRTF", "PRTF"},
        new string[] {"Bradstreet", "PRTS", "PRTS"},
        new string[] {"Braham", "PRHM", "PRHM"},
        new string[] {"Brailsford", "PRLS", "PRLS"},
        new string[] {"Brainard", "PRNR", "PRNR"},
        new string[] {"Brandish", "PRNT", "PRNT"},
        new string[] {"Braun", "PRN", "PRN"},
        new string[] {"Brecc", "PRK", "PRK"},
        new string[] {"Brent", "PRNT", "PRNT"},
        new string[] {"Brenton", "PRNT", "PRNT"},
        new string[] {"Briggs", "PRKS", "PRKS"},
        new string[] {"Brigham", "PRM", "PRM"},
        new string[] {"Brobst", "PRPS", "PRPS"},
        new string[] {"Brome", "PRM", "PRM"},
        new string[] {"Bronson", "PRNS", "PRNS"},
        new string[] {"Brooks", "PRKS", "PRKS"},
        new string[] {"Brouillard", "PRLR", "PRLR"},
        new string[] {"Brown", "PRN", "PRN"},
        new string[] {"Browne", "PRN", "PRN"},
        new string[] {"Brownell", "PRNL", "PRNL"},
        new string[] {"Bruley", "PRL", "PRL"},
        new string[] {"Bryant", "PRNT", "PRNT"},
        new string[] {"Brzozowski", "PRSS", "PRTS"},
        new string[] {"Buide", "PT", "PT"},
        new string[] {"Bulmer", "PLMR", "PLMR"},
        new string[] {"Bunker", "PNKR", "PNKR"},
        new string[] {"Burden", "PRTN", "PRTN"},
        new string[] {"Burge", "PRJ", "PRK"},
        new string[] {"Burgoyne", "PRKN", "PRKN"},
        new string[] {"Burke", "PRK", "PRK"},
        new string[] {"Burnett", "PRNT", "PRNT"},
        new string[] {"Burpee", "PRP", "PRP"},
        new string[] {"Bursley", "PRSL", "PRSL"},
        new string[] {"Burton", "PRTN", "PRTN"},
        new string[] {"Bushnell", "PXNL", "PXNL"},
        new string[] {"Buss", "PS", "PS"},
        new string[] {"Buswell", "PSL", "PSL"},
        new string[] {"Butler", "PTLR", "PTLR"},
        new string[] {"Calkin", "KLKN", "KLKN"},
        new string[] {"Canada", "KNT", "KNT"},
        new string[] {"Canmore", "KNMR", "KNMR"},
        new string[] {"Canney", "KN", "KN"},
        new string[] {"Capet", "KPT", "KPT"},
        new string[] {"Card", "KRT", "KRT"},
        new string[] {"Carman", "KRMN", "KRMN"},
        new string[] {"Carpenter", "KRPN", "KRPN"},
        new string[] {"Cartwright", "KRTR", "KRTR"},
        new string[] {"Casey", "KS", "KS"},
        new string[] {"Catterfield", "KTRF", "KTRF"},
        new string[] {"Ceeley", "SL", "SL"},
        new string[] {"Chambers", "XMPR", "XMPR"},
        new string[] {"Champion", "XMPN", "XMPN"},
        new string[] {"Chapman", "XPMN", "XPMN"},
        new string[] {"Chase", "XS", "XS"},
        new string[] {"Cheney", "XN", "XN"},
        new string[] {"Chetwynd", "XTNT", "XTNT"},
        new string[] {"Chevalier", "XFL", "XFLR"},
        new string[] {"Chillingsworth", "XLNK", "XLNK"},
        new string[] {"Christie", "KRST", "KRST"},
        new string[] {"Chubbuck", "XPK", "XPK"},
        new string[] {"Church", "XRX", "XRK"},
        new string[] {"Clark", "KLRK", "KLRK"},
        new string[] {"Clarke", "KLRK", "KLRK"},
        new string[] {"Cleare", "KLR", "KLR"},
        new string[] {"Clement", "KLMN", "KLMN"},
        new string[] {"Clerke", "KLRK", "KLRK"},
        new string[] {"Clibben", "KLPN", "KLPN"},
        new string[] {"Clifford", "KLFR", "KLFR"},
        new string[] {"Clivedon", "KLFT", "KLFT"},
        new string[] {"Close", "KLS", "KLS"},
        new string[] {"Clothilde", "KL0L", "KLTL"},
        new string[] {"Cobb", "KP", "KP"},
        new string[] {"Coburn", "KPRN", "KPRN"},
        new string[] {"Coburne", "KPRN", "KPRN"},
        new string[] {"Cocke", "KK", "KK"},
        new string[] {"Coffin", "KFN", "KFN"},
        new string[] {"Coffyn", "KFN", "KFN"},
        new string[] {"Colborne", "KLPR", "KLPR"},
        new string[] {"Colby", "KLP", "KLP"},
        new string[] {"Cole", "KL", "KL"},
        new string[] {"Coleman", "KLMN", "KLMN"},
        new string[] {"Collier", "KL", "KLR"},
        new string[] {"Compton", "KMPT", "KMPT"},
        new string[] {"Cone", "KN", "KN"},
        new string[] {"Cook", "KK", "KK"},
        new string[] {"Cooke", "KK", "KK"},
        new string[] {"Cooper", "KPR", "KPR"},
        new string[] {"Copperthwaite", "KPR0", "KPRT"},
        new string[] {"Corbet", "KRPT", "KRPT"},
        new string[] {"Corell", "KRL", "KRL"},
        new string[] {"Corey", "KR", "KR"},
        new string[] {"Corlies", "KRLS", "KRLS"},
        new string[] {"Corneliszen", "KRNL", "KRNL"},
        new string[] {"Cornelius", "KRNL", "KRNL"},
        new string[] {"Cornwallis", "KRNL", "KRNL"},
        new string[] {"Cosgrove", "KSKR", "KSKR"},
        new string[] {"Count of Brionne", "KNTF", "KNTF"},
        new string[] {"Covill", "KFL", "KFL"},
        new string[] {"Cowperthwaite", "KPR0", "KPRT"},
        new string[] {"Cowperwaite", "KPRT", "KPRT"},
        new string[] {"Crane", "KRN", "KRN"},
        new string[] {"Creagmile", "KRKM", "KRKM"},
        new string[] {"Crew", "KR", "KRF"},
        new string[] {"Crispin", "KRSP", "KRSP"},
        new string[] {"Crocker", "KRKR", "KRKR"},
        new string[] {"Crockett", "KRKT", "KRKT"},
        new string[] {"Crosby", "KRSP", "KRSP"},
        new string[] {"Crump", "KRMP", "KRMP"},
        new string[] {"Cunningham", "KNNK", "KNNK"},
        new string[] {"Curtis", "KRTS", "KRTS"},
        new string[] {"Cutha", "K0", "KT"},
        new string[] {"Cutter", "KTR", "KTR"},
        new string[] {"D'Aubigny", "TPN", "TPKN"},
        new string[] {"DAVIS", "TFS", "TFS"},
        new string[] {"Dabinott", "TPNT", "TPNT"},
        new string[] {"Dacre", "TKR", "TKR"},
        new string[] {"Daggett", "TKT", "TKT"},
        new string[] {"Danvers", "TNFR", "TNFR"},
        new string[] {"Darcy", "TRS", "TRS"},
        new string[] {"Davis", "TFS", "TFS"},
        new string[] {"Dawn", "TN", "TN"},
        new string[] {"Dawson", "TSN", "TSN"},
        new string[] {"Day", "T", "T"},
        new string[] {"Daye", "T", "T"},
        new string[] {"DeGrenier", "TKRN", "TKRN"},
        new string[] {"Dean", "TN", "TN"},
        new string[] {"Deekindaugh", "TKNT", "TKNT"},
        new string[] {"Dennis", "TNS", "TNS"},
        new string[] {"Denny", "TN", "TN"},
        new string[] {"Denton", "TNTN", "TNTN"},
        new string[] {"Desborough", "TSPR", "TSPR"},
        new string[] {"Despenser", "TSPN", "TSPN"},
        new string[] {"Deverill", "TFRL", "TFRL"},
        new string[] {"Devine", "TFN", "TFN"},
        new string[] {"Dexter", "TKST", "TKST"},
        new string[] {"Dillaway", "TL", "TL"},
        new string[] {"Dimmick", "TMK", "TMK"},
        new string[] {"Dinan", "TNN", "TNN"},
        new string[] {"Dix", "TKS", "TKS"},
        new string[] {"Doggett", "TKT", "TKT"},
        new string[] {"Donahue", "TNH", "TNH"},
        new string[] {"Dorfman", "TRFM", "TRFM"},
        new string[] {"Dorris", "TRS", "TRS"},
        new string[] {"Dow", "T", "TF"},
        new string[] {"Downey", "TN", "TN"},
        new string[] {"Downing", "TNNK", "TNNK"},
        new string[] {"Dowsett", "TST", "TST"},
        new string[] {"Duck?", "TK", "TK"},
        new string[] {"Dudley", "TTL", "TTL"},
        new string[] {"Duffy", "TF", "TF"},
        new string[] {"Dunn", "TN", "TN"},
        new string[] {"Dunsterville", "TNST", "TNST"},
        new string[] {"Durrant", "TRNT", "TRNT"},
        new string[] {"Durrin", "TRN", "TRN"},
        new string[] {"Dustin", "TSTN", "TSTN"},
        new string[] {"Duston", "TSTN", "TSTN"},
        new string[] {"Eames", "AMS", "AMS"},
        new string[] {"Early", "ARL", "ARL"},
        new string[] {"Easty", "AST", "AST"},
        new string[] {"Ebbett", "APT", "APT"},
        new string[] {"Eberbach", "APRP", "APRP"},
        new string[] {"Eberhard", "APRR", "APRR"},
        new string[] {"Eddy", "AT", "AT"},
        new string[] {"Edenden", "ATNT", "ATNT"},
        new string[] {"Edwards", "ATRT", "ATRT"},
        new string[] {"Eglinton", "AKLN", "ALNT"},
        new string[] {"Eliot", "ALT", "ALT"},
        new string[] {"Elizabeth", "ALSP", "ALSP"},
        new string[] {"Ellis", "ALS", "ALS"},
        new string[] {"Ellison", "ALSN", "ALSN"},
        new string[] {"Ellot", "ALT", "ALT"},
        new string[] {"Elny", "ALN", "ALN"},
        new string[] {"Elsner", "ALSN", "ALSN"},
        new string[] {"Emerson", "AMRS", "AMRS"},
        new string[] {"Empson", "AMPS", "AMPS"},
        new string[] {"Est", "AST", "AST"},
        new string[] {"Estabrook", "ASTP", "ASTP"},
        new string[] {"Estes", "ASTS", "ASTS"},
        new string[] {"Estey", "AST", "AST"},
        new string[] {"Evans", "AFNS", "AFNS"},
        new string[] {"Fallowell", "FLL", "FLL"},
        new string[] {"Farnsworth", "FRNS", "FRNS"},
        new string[] {"Feake", "FK", "FK"},
        new string[] {"Feke", "FK", "FK"},
        new string[] {"Fellows", "FLS", "FLS"},
        new string[] {"Fettiplace", "FTPL", "FTPL"},
        new string[] {"Finney", "FN", "FN"},
        new string[] {"Fischer", "FXR", "FSKR"},
        new string[] {"Fisher", "FXR", "FXR"},
        new string[] {"Fisk", "FSK", "FSK"},
        new string[] {"Fiske", "FSK", "FSK"},
        new string[] {"Fletcher", "FLXR", "FLXR"},
        new string[] {"Folger", "FLKR", "FLJR"},
        new string[] {"Foliot", "FLT", "FLT"},
        new string[] {"Folyot", "FLT", "FLT"},
        new string[] {"Fones", "FNS", "FNS"},
        new string[] {"Fordham", "FRTM", "FRTM"},
        new string[] {"Forstner", "FRST", "FRST"},
        new string[] {"Fosten", "FSTN", "FSTN"},
        new string[] {"Foster", "FSTR", "FSTR"},
        new string[] {"Foulke", "FLK", "FLK"},
        new string[] {"Fowler", "FLR", "FLR"},
        new string[] {"Foxwell", "FKSL", "FKSL"},
        new string[] {"Fraley", "FRL", "FRL"},
        new string[] {"Franceys", "FRNS", "FRNS"},
        new string[] {"Franke", "FRNK", "FRNK"},
        new string[] {"Frascella", "FRSL", "FRSL"},
        new string[] {"Frazer", "FRSR", "FRSR"},
        new string[] {"Fredd", "FRT", "FRT"},
        new string[] {"Freeman", "FRMN", "FRMN"},
        new string[] {"French", "FRNX", "FRNK"},
        new string[] {"Freville", "FRFL", "FRFL"},
        new string[] {"Frey", "FR", "FR"},
        new string[] {"Frick", "FRK", "FRK"},
        new string[] {"Frier", "FR", "FRR"},
        new string[] {"Froe", "FR", "FR"},
        new string[] {"Frorer", "FRRR", "FRRR"},
        new string[] {"Frost", "FRST", "FRST"},
        new string[] {"Frothingham", "FR0N", "FRTN"},
        new string[] {"Fry", "FR", "FR"},
        new string[] {"Gaffney", "KFN", "KFN"},
        new string[] {"Gage", "KJ", "KK"},
        new string[] {"Gallion", "KLN", "KLN"},
        new string[] {"Gallishan", "KLXN", "KLXN"},
        new string[] {"Gamble", "KMPL", "KMPL"},
        new string[] {"garage", "KRJ", "KRK"},
        new string[] {"Garbrand", "KRPR", "KRPR"},
        new string[] {"Gardner", "KRTN", "KRTN"},
        new string[] {"Garrett", "KRT", "KRT"},
        new string[] {"Gassner", "KSNR", "KSNR"},
        new string[] {"Gater", "KTR", "KTR"},
        new string[] {"Gaunt", "KNT", "KNT"},
        new string[] {"Gayer", "KR", "KR"},
        new string[] {"George", "JRJ", "KRK"},
        new string[] {"Gerken", "KRKN", "JRKN"},
        new string[] {"Gerritsen", "KRTS", "JRTS"},
        new string[] {"Gibbs", "KPS", "JPS"},
        new string[] {"Giffard", "JFRT", "KFRT"},
        new string[] {"Gilbert", "KLPR", "JLPR"},
        new string[] {"Gill", "KL", "JL"},
        new string[] {"Gilman", "KLMN", "JLMN"},
        new string[] {"Glass", "KLS", "KLS"},
        new string[] {"Goddard\\Gifford", "KTRT", "KTRT"},
        new string[] {"Godfrey", "KTFR", "KTFR"},
        new string[] {"Godwin", "KTN", "KTN"},
        new string[] {"Goodale", "KTL", "KTL"},
        new string[] {"Goodnow", "KTN", "KTNF"},
        new string[] {"Gorham", "KRM", "KRM"},
        new string[] {"Goseline", "KSLN", "KSLN"},
        new string[] {"Gott", "KT", "KT"},
        new string[] {"Gould", "KLT", "KLT"},
        new string[] {"Grafton", "KRFT", "KRFT"},
        new string[] {"Grant", "KRNT", "KRNT"},
        new string[] {"Gray", "KR", "KR"},
        new string[] {"Green", "KRN", "KRN"},
        new string[] {"Griffin", "KRFN", "KRFN"},
        new string[] {"Grill", "KRL", "KRL"},
        new string[] {"Grim", "KRM", "KRM"},
        new string[] {"Grisgonelle", "KRSK", "KRSK"},
        new string[] {"Gross", "KRS", "KRS"},
        new string[] {"Guba", "KP", "KP"},
        new string[] {"Gybbes", "KPS", "JPS"},
        new string[] {"Haburne", "HPRN", "HPRN"},
        new string[] {"Hackburne", "HKPR", "HKPR"},
        new string[] {"Haddon?", "HTN", "HTN"},
        new string[] {"Haines", "HNS", "HNS"},
        new string[] {"Hale", "HL", "HL"},
        new string[] {"Hall", "HL", "HL"},
        new string[] {"Hallet", "HLT", "HLT"},
        new string[] {"Hallock", "HLK", "HLK"},
        new string[] {"Halstead", "HLST", "HLST"},
        new string[] {"Hammond", "HMNT", "HMNT"},
        new string[] {"Hance", "HNS", "HNS"},
        new string[] {"Handy", "HNT", "HNT"},
        new string[] {"Hanson", "HNSN", "HNSN"},
        new string[] {"Harasek", "HRSK", "HRSK"},
        new string[] {"Harcourt", "HRKR", "HRKR"},
        new string[] {"Hardy", "HRT", "HRT"},
        new string[] {"Harlock", "HRLK", "HRLK"},
        new string[] {"Harris", "HRS", "HRS"},
        new string[] {"Hartley", "HRTL", "HRTL"},
        new string[] {"Harvey", "HRF", "HRF"},
        new string[] {"Harvie", "HRF", "HRF"},
        new string[] {"Harwood", "HRT", "HRT"},
        new string[] {"Hathaway", "H0", "HT"},
        new string[] {"Haukeness", "HKNS", "HKNS"},
        new string[] {"Hawkes", "HKS", "HKS"},
        new string[] {"Hawkhurst", "HKRS", "HKRS"},
        new string[] {"Hawkins", "HKNS", "HKNS"},
        new string[] {"Hawley", "HL", "HL"},
        new string[] {"Heald", "HLT", "HLT"},
        new string[] {"Helsdon", "HLST", "HLST"},
        new string[] {"Hemenway", "HMN", "HMN"},
        new string[] {"Hemmenway", "HMN", "HMN"},
        new string[] {"Henck", "HNK", "HNK"},
        new string[] {"Henderson", "HNTR", "HNTR"},
        new string[] {"Hendricks", "HNTR", "HNTR"},
        new string[] {"Hersey", "HRS", "HRS"},
        new string[] {"Hewes", "HS", "HS"},
        new string[] {"Heyman", "HMN", "HMN"},
        new string[] {"Hicks", "HKS", "HKS"},
        new string[] {"Hidden", "HTN", "HTN"},
        new string[] {"Higgs", "HKS", "HKS"},
        new string[] {"Hill", "HL", "HL"},
        new string[] {"Hills", "HLS", "HLS"},
        new string[] {"Hinckley", "HNKL", "HNKL"},
        new string[] {"Hipwell", "HPL", "HPL"},
        new string[] {"Hobart", "HPRT", "HPRT"},
        new string[] {"Hoben", "HPN", "HPN"},
        new string[] {"Hoffmann", "HFMN", "HFMN"},
        new string[] {"Hogan", "HKN", "HKN"},
        new string[] {"Holmes", "HLMS", "HLMS"},
        new string[] {"Hoo", "H", "H"},
        new string[] {"Hooker", "HKR", "HKR"},
        new string[] {"Hopcott", "HPKT", "HPKT"},
        new string[] {"Hopkins", "HPKN", "HPKN"},
        new string[] {"Hopkinson", "HPKN", "HPKN"},
        new string[] {"Hornsey", "HRNS", "HRNS"},
        new string[] {"Houckgeest", "HKJS", "HKKS"},
        new string[] {"Hough", "H", "H"},
        new string[] {"Houstin", "HSTN", "HSTN"},
        new string[] {"How", "H", "HF"},
        new string[] {"Howe", "H", "H"},
        new string[] {"Howland", "HLNT", "HLNT"},
        new string[] {"Hubner", "HPNR", "HPNR"},
        new string[] {"Hudnut", "HTNT", "HTNT"},
        new string[] {"Hughes", "HS", "HS"},
        new string[] {"Hull", "HL", "HL"},
        new string[] {"Hulme", "HLM", "HLM"},
        new string[] {"Hume", "HM", "HM"},
        new string[] {"Hundertumark", "HNTR", "HNTR"},
        new string[] {"Hundley", "HNTL", "HNTL"},
        new string[] {"Hungerford", "HNKR", "HNJR"},
        new string[] {"Hunt", "HNT", "HNT"},
        new string[] {"Hurst", "HRST", "HRST"},
        new string[] {"Husbands", "HSPN", "HSPN"},
        new string[] {"Hussey", "HS", "HS"},
        new string[] {"Husted", "HSTT", "HSTT"},
        new string[] {"Hutchins", "HXNS", "HXNS"},
        new string[] {"Hutchinson", "HXNS", "HXNS"},
        new string[] {"Huttinger", "HTNK", "HTNJ"},
        new string[] {"Huybertsen", "HPRT", "HPRT"},
        new string[] {"Iddenden", "ATNT", "ATNT"},
        new string[] {"Ingraham", "ANKR", "ANKR"},
        new string[] {"Ives", "AFS", "AFS"},
        new string[] {"Jackson", "JKSN", "AKSN"},
        new string[] {"Jacob", "JKP", "AKP"},
        new string[] {"Jans", "JNS", "ANS"},
        new string[] {"Jenkins", "JNKN", "ANKN"},
        new string[] {"Jewett", "JT", "AT"},
        new string[] {"Jewitt", "JT", "AT"},
        new string[] {"Johnson", "JNSN", "ANSN"},
        new string[] {"Jones", "JNS", "ANS"},
        new string[] {"Josephine", "JSFN", "HSFN"},
        new string[] {"Judd", "JT", "AT"},
        new string[] {"June", "JN", "AN"},
        new string[] {"Kamarowska", "KMRS", "KMRS"},
        new string[] {"Kay", "K", "K"},
        new string[] {"Kelley", "KL", "KL"},
        new string[] {"Kelly", "KL", "KL"},
        new string[] {"Keymber", "KMPR", "KMPR"},
        new string[] {"Keynes", "KNS", "KNS"},
        new string[] {"Kilham", "KLM", "KLM"},
        new string[] {"Kim", "KM", "KM"},
        new string[] {"Kimball", "KMPL", "KMPL"},
        new string[] {"King", "KNK", "KNK"},
        new string[] {"Kinsey", "KNS", "KNS"},
        new string[] {"Kirk", "KRK", "KRK"},
        new string[] {"Kirton", "KRTN", "KRTN"},
        new string[] {"Kistler", "KSTL", "KSTL"},
        new string[] {"Kitchen", "KXN", "KXN"},
        new string[] {"Kitson", "KTSN", "KTSN"},
        new string[] {"Klett", "KLT", "KLT"},
        new string[] {"Kline", "KLN", "KLN"},
        new string[] {"Knapp", "NP", "NP"},
        new string[] {"Knight", "NT", "NT"},
        new string[] {"Knote", "NT", "NT"},
        new string[] {"Knott", "NT", "NT"},
        new string[] {"Knox", "NKS", "NKS"},
        new string[] {"Koeller", "KLR", "KLR"},
        new string[] {"La Pointe", "LPNT", "LPNT"},
        new string[] {"LaPlante", "LPLN", "LPLN"},
        new string[] {"Laimbeer", "LMPR", "LMPR"},
        new string[] {"Lamb", "LMP", "LMP"},
        new string[] {"Lambertson", "LMPR", "LMPR"},
        new string[] {"Lancto", "LNKT", "LNKT"},
        new string[] {"Landry", "LNTR", "LNTR"},
        new string[] {"Lane", "LN", "LN"},
        new string[] {"Langendyck", "LNJN", "LNKN"},
        new string[] {"Langer", "LNKR", "LNJR"},
        new string[] {"Langford", "LNKF", "LNKF"},
        new string[] {"Lantersee", "LNTR", "LNTR"},
        new string[] {"Laquer", "LKR", "LKR"},
        new string[] {"Larkin", "LRKN", "LRKN"},
        new string[] {"Latham", "LTM", "LTM"},
        new string[] {"Lathrop", "L0RP", "LTRP"},
        new string[] {"Lauter", "LTR", "LTR"},
        new string[] {"Lawrence", "LRNS", "LRNS"},
        new string[] {"Leach", "LK", "LK"},
        new string[] {"Leager", "LKR", "LJR"},
        new string[] {"Learned", "LRNT", "LRNT"},
        new string[] {"Leavitt", "LFT", "LFT"},
        new string[] {"Lee", "L", "L"},
        new string[] {"Leete", "LT", "LT"},
        new string[] {"Leggett", "LKT", "LKT"},
        new string[] {"Leland", "LLNT", "LLNT"},
        new string[] {"Leonard", "LNRT", "LNRT"},
        new string[] {"Lester", "LSTR", "LSTR"},
        new string[] {"Lestrange", "LSTR", "LSTR"},
        new string[] {"Lethem", "L0M", "LTM"},
        new string[] {"Levine", "LFN", "LFN"},
        new string[] {"Lewes", "LS", "LS"},
        new string[] {"Lewis", "LS", "LS"},
        new string[] {"Lincoln", "LNKL", "LNKL"},
        new string[] {"Lindsey", "LNTS", "LNTS"},
        new string[] {"Linher", "LNR", "LNR"},
        new string[] {"Lippet", "LPT", "LPT"},
        new string[] {"Lippincott", "LPNK", "LPNK"},
        new string[] {"Lockwood", "LKT", "LKT"},
        new string[] {"Loines", "LNS", "LNS"},
        new string[] {"Lombard", "LMPR", "LMPR"},
        new string[] {"Long", "LNK", "LNK"},
        new string[] {"Longespee", "LNJS", "LNKS"},
        new string[] {"Look", "LK", "LK"},
        new string[] {"Lounsberry", "LNSP", "LNSP"},
        new string[] {"Lounsbury", "LNSP", "LNSP"},
        new string[] {"Louthe", "L0", "LT"},
        new string[] {"Loveyne", "LFN", "LFN"},
        new string[] {"Lowe", "L", "L"},
        new string[] {"Ludlam", "LTLM", "LTLM"},
        new string[] {"Lumbard", "LMPR", "LMPR"},
        new string[] {"Lund", "LNT", "LNT"},
        new string[] {"Luno", "LN", "LN"},
        new string[] {"Lutz", "LTS", "LTS"},
        new string[] {"Lydia", "LT", "LT"},
        new string[] {"Lynne", "LN", "LN"},
        new string[] {"Lyon", "LN", "LN"},
        new string[] {"MacAlpin", "MKLP", "MKLP"},
        new string[] {"MacBricc", "MKPR", "MKPR"},
        new string[] {"MacCrinan", "MKRN", "MKRN"},
        new string[] {"MacKenneth", "MKN0", "MKNT"},
        new string[] {"MacMael nam Bo", "MKML", "MKML"},
        new string[] {"MacMurchada", "MKMR", "MKMR"},
        new string[] {"Macomber", "MKMP", "MKMP"},
        new string[] {"Macy", "MS", "MS"},
        new string[] {"Magnus", "MNS", "MKNS"},
        new string[] {"Mahien", "MHN", "MHN"},
        new string[] {"Malmains", "MLMN", "MLMN"},
        new string[] {"Malory", "MLR", "MLR"},
        new string[] {"Mancinelli", "MNSN", "MNSN"},
        new string[] {"Mancini", "MNSN", "MNSN"},
        new string[] {"Mann", "MN", "MN"},
        new string[] {"Manning", "MNNK", "MNNK"},
        new string[] {"Manter", "MNTR", "MNTR"},
        new string[] {"Marion", "MRN", "MRN"},
        new string[] {"Marley", "MRL", "MRL"},
        new string[] {"Marmion", "MRMN", "MRMN"},
        new string[] {"Marquart", "MRKR", "MRKR"},
        new string[] {"Marsh", "MRX", "MRX"},
        new string[] {"Marshal", "MRXL", "MRXL"},
        new string[] {"Marshall", "MRXL", "MRXL"},
        new string[] {"Martel", "MRTL", "MRTL"},
        new string[] {"Martha", "MR0", "MRT"},
        new string[] {"Martin", "MRTN", "MRTN"},
        new string[] {"Marturano", "MRTR", "MRTR"},
        new string[] {"Marvin", "MRFN", "MRFN"},
        new string[] {"Mary", "MR", "MR"},
        new string[] {"Mason", "MSN", "MSN"},
        new string[] {"Maxwell", "MKSL", "MKSL"},
        new string[] {"Mayhew", "MH", "MHF"},
        new string[] {"McAllaster", "MKLS", "MKLS"},
        new string[] {"McAllister", "MKLS", "MKLS"},
        new string[] {"McConnell", "MKNL", "MKNL"},
        new string[] {"McFarland", "MKFR", "MKFR"},
        new string[] {"McIlroy", "MSLR", "MSLR"},
        new string[] {"McNair", "MKNR", "MKNR"},
        new string[] {"McNair-Landry", "MKNR", "MKNR"},
        new string[] {"McRaven", "MKRF", "MKRF"},
        new string[] {"Mead", "MT", "MT"},
        new string[] {"Meade", "MT", "MT"},
        new string[] {"Meck", "MK", "MK"},
        new string[] {"Melton", "MLTN", "MLTN"},
        new string[] {"Mendenhall", "MNTN", "MNTN"},
        new string[] {"Mering", "MRNK", "MRNK"},
        new string[] {"Merrick", "MRK", "MRK"},
        new string[] {"Merry", "MR", "MR"},
        new string[] {"Mighill", "ML", "ML"},
        new string[] {"Miller", "MLR", "MLR"},
        new string[] {"Milton", "MLTN", "MLTN"},
        new string[] {"Mohun", "MHN", "MHN"},
        new string[] {"Montague", "MNTK", "MNTK"},
        new string[] {"Montboucher", "MNTP", "MNTP"},
        new string[] {"Moore", "MR", "MR"},
        new string[] {"Morrel", "MRL", "MRL"},
        new string[] {"Morrill", "MRL", "MRL"},
        new string[] {"Morris", "MRS", "MRS"},
        new string[] {"Morton", "MRTN", "MRTN"},
        new string[] {"Moton", "MTN", "MTN"},
        new string[] {"Muir", "MR", "MR"},
        new string[] {"Mulferd", "MLFR", "MLFR"},
        new string[] {"Mullins", "MLNS", "MLNS"},
        new string[] {"Mulso", "MLS", "MLS"},
        new string[] {"Munger", "MNKR", "MNJR"},
        new string[] {"Munt", "MNT", "MNT"},
        new string[] {"Murchad", "MRXT", "MRKT"},
        new string[] {"Murdock", "MRTK", "MRTK"},
        new string[] {"Murray", "MR", "MR"},
        new string[] {"Muskett", "MSKT", "MSKT"},
        new string[] {"Myers", "MRS", "MRS"},
        new string[] {"Myrick", "MRK", "MRK"},
        new string[] {"NORRIS", "NRS", "NRS"},
        new string[] {"Nayle", "NL", "NL"},
        new string[] {"Newcomb", "NKMP", "NKMP"},
        new string[] {"Newcomb(e)", "NKMP", "NKMP"},
        new string[] {"Newkirk", "NKRK", "NKRK"},
        new string[] {"Newton", "NTN", "NTN"},
        new string[] {"Niles", "NLS", "NLS"},
        new string[] {"Noble", "NPL", "NPL"},
        new string[] {"Noel", "NL", "NL"},
        new string[] {"Northend", "NR0N", "NRTN"},
        new string[] {"Norton", "NRTN", "NRTN"},
        new string[] {"Nutter", "NTR", "NTR"},
        new string[] {"Odding", "ATNK", "ATNK"},
        new string[] {"Odenbaugh", "ATNP", "ATNP"},
        new string[] {"Ogborn", "AKPR", "AKPR"},
        new string[] {"Oppenheimer", "APNM", "APNM"},
        new string[] {"Otis", "ATS", "ATS"},
        new string[] {"Oviatt", "AFT", "AFT"},
        new string[] {"PRUST?", "PRST", "PRST"},
        new string[] {"Paddock", "PTK", "PTK"},
        new string[] {"Page", "PJ", "PK"},
        new string[] {"Paine", "PN", "PN"},
        new string[] {"Paist", "PST", "PST"},
        new string[] {"Palmer", "PLMR", "PLMR"},
        new string[] {"Park", "PRK", "PRK"},
        new string[] {"Parker", "PRKR", "PRKR"},
        new string[] {"Parkhurst", "PRKR", "PRKR"},
        new string[] {"Parrat", "PRT", "PRT"},
        new string[] {"Parsons", "PRSN", "PRSN"},
        new string[] {"Partridge", "PRTR", "PRTR"},
        new string[] {"Pashley", "PXL", "PXL"},
        new string[] {"Pasley", "PSL", "PSL"},
        new string[] {"Patrick", "PTRK", "PTRK"},
        new string[] {"Pattee", "PT", "PT"},
        new string[] {"Patten", "PTN", "PTN"},
        new string[] {"Pawley", "PL", "PL"},
        new string[] {"Payne", "PN", "PN"},
        new string[] {"Peabody", "PPT", "PPT"},
        new string[] {"Peake", "PK", "PK"},
        new string[] {"Pearson", "PRSN", "PRSN"},
        new string[] {"Peat", "PT", "PT"},
        new string[] {"Pedersen", "PTRS", "PTRS"},
        new string[] {"Percy", "PRS", "PRS"},
        new string[] {"Perkins", "PRKN", "PRKN"},
        new string[] {"Perrine", "PRN", "PRN"},
        new string[] {"Perry", "PR", "PR"},
        new string[] {"Peson", "PSN", "PSN"},
        new string[] {"Peterson", "PTRS", "PTRS"},
        new string[] {"Peyton", "PTN", "PTN"},
        new string[] {"Phinney", "FN", "FN"},
        new string[] {"Pickard", "PKRT", "PKRT"},
        new string[] {"Pierce", "PRS", "PRS"},
        new string[] {"Pierrepont", "PRPN", "PRPN"},
        new string[] {"Pike", "PK", "PK"},
        new string[] {"Pinkham", "PNKM", "PNKM"},
        new string[] {"Pitman", "PTMN", "PTMN"},
        new string[] {"Pitt", "PT", "PT"},
        new string[] {"Pitts", "PTS", "PTS"},
        new string[] {"Plantagenet", "PLNT", "PLNT"},
        new string[] {"Platt", "PLT", "PLT"},
        new string[] {"Platts", "PLTS", "PLTS"},
        new string[] {"Pleis", "PLS", "PLS"},
        new string[] {"Pleiss", "PLS", "PLS"},
        new string[] {"Plisko", "PLSK", "PLSK"},
        new string[] {"Pliskovitch", "PLSK", "PLSK"},
        new string[] {"Plum", "PLM", "PLM"},
        new string[] {"Plume", "PLM", "PLM"},
        new string[] {"Poitou", "PT", "PT"},
        new string[] {"Pomeroy", "PMR", "PMR"},
        new string[] {"Poretiers", "PRTR", "PRTR"},
        new string[] {"Pote", "PT", "PT"},
        new string[] {"Potter", "PTR", "PTR"},
        new string[] {"Potts", "PTS", "PTS"},
        new string[] {"Powell", "PL", "PL"},
        new string[] {"Pratt", "PRT", "PRT"},
        new string[] {"Presbury", "PRSP", "PRSP"},
        new string[] {"Priest", "PRST", "PRST"},
        new string[] {"Prindle", "PRNT", "PRNT"},
        new string[] {"Prior", "PRR", "PRR"},
        new string[] {"Profumo", "PRFM", "PRFM"},
        new string[] {"Purdy", "PRT", "PRT"},
        new string[] {"Purefoy", "PRF", "PRF"},
        new string[] {"Pury", "PR", "PR"},
        new string[] {"Quinter", "KNTR", "KNTR"},
        new string[] {"Rachel", "RXL", "RKL"},
        new string[] {"Rand", "RNT", "RNT"},
        new string[] {"Rankin", "RNKN", "RNKN"},
        new string[] {"Ravenscroft", "RFNS", "RFNS"},
        new string[] {"Raynsford", "RNSF", "RNSF"},
        new string[] {"Reakirt", "RKRT", "RKRT"},
        new string[] {"Reaves", "RFS", "RFS"},
        new string[] {"Reeves", "RFS", "RFS"},
        new string[] {"Reichert", "RXRT", "RKRT"},
        new string[] {"Remmele", "RML", "RML"},
        new string[] {"Reynolds", "RNLT", "RNLT"},
        new string[] {"Rhodes", "RTS", "RTS"},
        new string[] {"Richards", "RXRT", "RKRT"},
        new string[] {"Richardson", "RXRT", "RKRT"},
        new string[] {"Ring", "RNK", "RNK"},
        new string[] {"Roberts", "RPRT", "RPRT"},
        new string[] {"Robertson", "RPRT", "RPRT"},
        new string[] {"Robson", "RPSN", "RPSN"},
        new string[] {"Rodie", "RT", "RT"},
        new string[] {"Rody", "RT", "RT"},
        new string[] {"Rogers", "RKRS", "RJRS"},
        new string[] {"Ross", "RS", "RS"},
        new string[] {"Rosslevin", "RSLF", "RSLF"},
        new string[] {"Rowland", "RLNT", "RLNT"},
        new string[] {"Ruehl", "RL", "RL"},
        new string[] {"Russell", "RSL", "RSL"},
        new string[] {"Ruth", "R0", "RT"},
        new string[] {"Ryan", "RN", "RN"},
        new string[] {"Rysse", "RS", "RS"},
        new string[] {"Sadler", "STLR", "STLR"},
        new string[] {"Salmon", "SLMN", "SLMN"},
        new string[] {"Salter", "SLTR", "SLTR"},
        new string[] {"Salvatore", "SLFT", "SLFT"},
        new string[] {"Sanders", "SNTR", "SNTR"},
        new string[] {"Sands", "SNTS", "SNTS"},
        new string[] {"Sanford", "SNFR", "SNFR"},
        new string[] {"Sanger", "SNKR", "SNJR"},
        new string[] {"Sargent", "SRJN", "SRKN"},
        new string[] {"Saunders", "SNTR", "SNTR"},
        new string[] {"Schilling", "XLNK", "XLNK"},
        new string[] {"Schlegel", "XLKL", "SLKL"},
        new string[] {"Scott", "SKT", "SKT"},
        new string[] {"Sears", "SRS", "SRS"},
        new string[] {"Segersall", "SJRS", "SKRS"},
        new string[] {"Senecal", "SNKL", "SNKL"},
        new string[] {"Sergeaux", "SRJ", "SRK"},
        new string[] {"Severance", "SFRN", "SFRN"},
        new string[] {"Sharp", "XRP", "XRP"},
        new string[] {"Sharpe", "XRP", "XRP"},
        new string[] {"Sharply", "XRPL", "XRPL"},
        new string[] {"Shatswell", "XTSL", "XTSL"},
        new string[] {"Shattack", "XTK", "XTK"},
        new string[] {"Shattock", "XTK", "XTK"},
        new string[] {"Shattuck", "XTK", "XTK"},
        new string[] {"Shaw", "X", "XF"},
        new string[] {"Sheldon", "XLTN", "XLTN"},
        new string[] {"Sherman", "XRMN", "XRMN"},
        new string[] {"Shinn", "XN", "XN"},
        new string[] {"Shirford", "XRFR", "XRFR"},
        new string[] {"Shirley", "XRL", "XRL"},
        new string[] {"Shively", "XFL", "XFL"},
        new string[] {"Shoemaker", "XMKR", "XMKR"},
        new string[] {"Short", "XRT", "XRT"},
        new string[] {"Shotwell", "XTL", "XTL"},
        new string[] {"Shute", "XT", "XT"},
        new string[] {"Sibley", "SPL", "SPL"},
        new string[] {"Silver", "SLFR", "SLFR"},
        new string[] {"Simes", "SMS", "SMS"},
        new string[] {"Sinken", "SNKN", "SNKN"},
        new string[] {"Sinn", "SN", "SN"},
        new string[] {"Skelton", "SKLT", "SKLT"},
        new string[] {"Skiffe", "SKF", "SKF"},
        new string[] {"Skotkonung", "SKTK", "SKTK"},
        new string[] {"Slade", "SLT", "XLT"},
        new string[] {"Slye", "SL", "XL"},
        new string[] {"Smedley", "SMTL", "XMTL"},
        new string[] {"Smith", "SM0", "XMT"},
        new string[] {"Smythe", "SM0", "XMT"},
        new string[] {"Snow", "SN", "XNF"},
        new string[] {"Soole", "SL", "SL"},
        new string[] {"Soule", "SL", "SL"},
        new string[] {"Southworth", "S0R0", "STRT"},
        new string[] {"Sowles", "SLS", "SLS"},
        new string[] {"Spalding", "SPLT", "SPLT"},
        new string[] {"Spark", "SPRK", "SPRK"},
        new string[] {"Spencer", "SPNS", "SPNS"},
        new string[] {"Sperry", "SPR", "SPR"},
        new string[] {"Spofford", "SPFR", "SPFR"},
        new string[] {"Spooner", "SPNR", "SPNR"},
        new string[] {"Sprague", "SPRK", "SPRK"},
        new string[] {"Springer", "SPRN", "SPRN"},
        new string[] {"St. Clair", "STKL", "STKL"},
        new string[] {"St. Claire", "STKL", "STKL"},
        new string[] {"St. Leger", "STLJ", "STLK"},
        new string[] {"St. Omer", "STMR", "STMR"},
        new string[] {"Stafferton", "STFR", "STFR"},
        new string[] {"Stafford", "STFR", "STFR"},
        new string[] {"Stalham", "STLM", "STLM"},
        new string[] {"Stanford", "STNF", "STNF"},
        new string[] {"Stanton", "STNT", "STNT"},
        new string[] {"Star", "STR", "STR"},
        new string[] {"Starbuck", "STRP", "STRP"},
        new string[] {"Starkey", "STRK", "STRK"},
        new string[] {"Starkweather", "STRK", "STRK"},
        new string[] {"Stearns", "STRN", "STRN"},
        new string[] {"Stebbins", "STPN", "STPN"},
        new string[] {"Steele", "STL", "STL"},
        new string[] {"Stephenson", "STFN", "STFN"},
        new string[] {"Stevens", "STFN", "STFN"},
        new string[] {"Stoddard", "STTR", "STTR"},
        new string[] {"Stodder", "STTR", "STTR"},
        new string[] {"Stone", "STN", "STN"},
        new string[] {"Storey", "STR", "STR"},
        new string[] {"Storrada", "STRT", "STRT"},
        new string[] {"Story", "STR", "STR"},
        new string[] {"Stoughton", "STFT", "STFT"},
        new string[] {"Stout", "STT", "STT"},
        new string[] {"Stow", "ST", "STF"},
        new string[] {"Strong", "STRN", "STRN"},
        new string[] {"Strutt", "STRT", "STRT"},
        new string[] {"Stryker", "STRK", "STRK"},
        new string[] {"Stuckeley", "STKL", "STKL"},
        new string[] {"Sturges", "STRJ", "STRK"},
        new string[] {"Sturgess", "STRJ", "STRK"},
        new string[] {"Sturgis", "STRJ", "STRK"},
        new string[] {"Suevain", "SFN", "SFN"},
        new string[] {"Sulyard", "SLRT", "SLRT"},
        new string[] {"Sutton", "STN", "STN"},
        new string[] {"Swain", "SN", "XN"},
        new string[] {"Swayne", "SN", "XN"},
        new string[] {"Swayze", "SS", "XTS"},
        new string[] {"Swift", "SFT", "XFT"},
        new string[] {"Taber", "TPR", "TPR"},
        new string[] {"Talcott", "TLKT", "TLKT"},
        new string[] {"Tarne", "TRN", "TRN"},
        new string[] {"Tatum", "TTM", "TTM"},
        new string[] {"Taverner", "TFRN", "TFRN"},
        new string[] {"Taylor", "TLR", "TLR"},
        new string[] {"Tenney", "TN", "TN"},
        new string[] {"Thayer", "0R", "TR"},
        new string[] {"Thember", "0MPR", "TMPR"},
        new string[] {"Thomas", "TMS", "TMS"},
        new string[] {"Thompson", "TMPS", "TMPS"},
        new string[] {"Thorne", "0RN", "TRN"},
        new string[] {"Thornycraft", "0RNK", "TRNK"},
        new string[] {"Threlkeld", "0RLK", "TRLK"},
        new string[] {"Throckmorton", "0RKM", "TRKM"},
        new string[] {"Thwaits", "0TS", "TTS"},
        new string[] {"Tibbetts", "TPTS", "TPTS"},
        new string[] {"Tidd", "TT", "TT"},
        new string[] {"Tierney", "TRN", "TRN"},
        new string[] {"Tilley", "TL", "TL"},
        new string[] {"Tillieres", "TLRS", "TLRS"},
        new string[] {"Tilly", "TL", "TL"},
        new string[] {"Tisdale", "TSTL", "TSTL"},
        new string[] {"Titus", "TTS", "TTS"},
        new string[] {"Tobey", "TP", "TP"},
        new string[] {"Tooker", "TKR", "TKR"},
        new string[] {"Towle", "TL", "TL"},
        new string[] {"Towne", "TN", "TN"},
        new string[] {"Townsend", "TNSN", "TNSN"},
        new string[] {"Treadway", "TRT", "TRT"},
        new string[] {"Trelawney", "TRLN", "TRLN"},
        new string[] {"Trinder", "TRNT", "TRNT"},
        new string[] {"Tripp", "TRP", "TRP"},
        new string[] {"Trippe", "TRP", "TRP"},
        new string[] {"Trott", "TRT", "TRT"},
        new string[] {"True", "TR", "TR"},
        new string[] {"Trussebut", "TRSP", "TRSP"},
        new string[] {"Tucker", "TKR", "TKR"},
        new string[] {"Turgeon", "TRJN", "TRKN"},
        new string[] {"Turner", "TRNR", "TRNR"},
        new string[] {"Tuttle", "TTL", "TTL"},
        new string[] {"Tyler", "TLR", "TLR"},
        new string[] {"Tylle", "TL", "TL"},
        new string[] {"Tyrrel", "TRL", "TRL"},
        new string[] {"Ua Tuathail", "AT0L", "ATTL"},
        new string[] {"Ulrich", "ALRX", "ALRK"},
        new string[] {"Underhill", "ANTR", "ANTR"},
        new string[] {"Underwood", "ANTR", "ANTR"},
        new string[] {"Unknown", "ANKN", "ANKN"},
        new string[] {"Valentine", "FLNT", "FLNT"},
        new string[] {"Van Egmond", "FNKM", "FNKM"},
        new string[] {"Van der Beek", "FNTR", "FNTR"},
        new string[] {"Vaughan", "FKN", "FKN"},
        new string[] {"Vermenlen", "FRMN", "FRMN"},
        new string[] {"Vincent", "FNSN", "FNSN"},
        new string[] {"Volentine", "FLNT", "FLNT"},
        new string[] {"Wagner", "AKNR", "FKNR"},
        new string[] {"Waite", "AT", "FT"},
        new string[] {"Walker", "ALKR", "FLKR"},
        new string[] {"Walter", "ALTR", "FLTR"},
        new string[] {"Wandell", "ANTL", "FNTL"},
        new string[] {"Wandesford", "ANTS", "FNTS"},
        new string[] {"Warbleton", "ARPL", "FRPL"},
        new string[] {"Ward", "ART", "FRT"},
        new string[] {"Warde", "ART", "FRT"},
        new string[] {"Ware", "AR", "FR"},
        new string[] {"Wareham", "ARHM", "FRHM"},
        new string[] {"Warner", "ARNR", "FRNR"},
        new string[] {"Warren", "ARN", "FRN"},
        new string[] {"Washburne", "AXPR", "FXPR"},
        new string[] {"Waterbury", "ATRP", "FTRP"},
        new string[] {"Watson", "ATSN", "FTSN"},
        new string[] {"WatsonEllithorpe", "ATSN", "FTSN"},
        new string[] {"Watts", "ATS", "FTS"},
        new string[] {"Wayne", "AN", "FN"},
        new string[] {"Webb", "AP", "FP"},
        new string[] {"Weber", "APR", "FPR"},
        new string[] {"Webster", "APST", "FPST"},
        new string[] {"Weed", "AT", "FT"},
        new string[] {"Weeks", "AKS", "FKS"},
        new string[] {"Wells", "ALS", "FLS"},
        new string[] {"Wenzell", "ANSL", "FNTS"},
        new string[] {"West", "AST", "FST"},
        new string[] {"Westbury", "ASTP", "FSTP"},
        new string[] {"Whatlocke", "ATLK", "ATLK"},
        new string[] {"Wheeler", "ALR", "ALR"},
        new string[] {"Whiston", "ASTN", "ASTN"},
        new string[] {"White", "AT", "AT"},
        new string[] {"Whitman", "ATMN", "ATMN"},
        new string[] {"Whiton", "ATN", "ATN"},
        new string[] {"Whitson", "ATSN", "ATSN"},
        new string[] {"Wickes", "AKS", "FKS"},
        new string[] {"Wilbur", "ALPR", "FLPR"},
        new string[] {"Wilcotes", "ALKT", "FLKT"},
        new string[] {"Wilkinson", "ALKN", "FLKN"},
        new string[] {"Willets", "ALTS", "FLTS"},
        new string[] {"Willett", "ALT", "FLT"},
        new string[] {"Willey", "AL", "FL"},
        new string[] {"Williams", "ALMS", "FLMS"},
        new string[] {"Williston", "ALST", "FLST"},
        new string[] {"Wilson", "ALSN", "FLSN"},
        new string[] {"Wimes", "AMS", "FMS"},
        new string[] {"Winch", "ANX", "FNK"},
        new string[] {"Winegar", "ANKR", "FNKR"},
        new string[] {"Wing", "ANK", "FNK"},
        new string[] {"Winsley", "ANSL", "FNSL"},
        new string[] {"Winslow", "ANSL", "FNSL"},
        new string[] {"Winthrop", "AN0R", "FNTR"},
        new string[] {"Wise", "AS", "FS"},
        new string[] {"Wood", "AT", "FT"},
        new string[] {"Woodbridge", "ATPR", "FTPR"},
        new string[] {"Woodward", "ATRT", "FTRT"},
        new string[] {"Wooley", "AL", "FL"},
        new string[] {"Woolley", "AL", "FL"},
        new string[] {"Worth", "AR0", "FRT"},
        new string[] {"Worthen", "AR0N", "FRTN"},
        new string[] {"Worthley", "AR0L", "FRTL"},
        new string[] {"Wright", "RT", "RT"},
        new string[] {"Wyer", "AR", "FR"},
        new string[] {"Wyere", "AR", "FR"},
        new string[] {"Wynkoop", "ANKP", "FNKP"},
        new string[] {"Yarnall", "ARNL", "ARNL"},
        new string[] {"Yeoman", "AMN", "AMN"},
        new string[] {"Yorke", "ARK", "ARK"},
        new string[] {"Young", "ANK", "ANK"},
        new string[] {"ab Wennonwen", "APNN", "APNN"},
        new string[] {"ap Llewellyn", "APLL", "APLL"},
        new string[] {"ap Lorwerth", "APLR", "APLR"},
        new string[] {"d'Angouleme", "TNKL", "TNKL"},
        new string[] {"de Audeham", "TTHM", "TTHM"},
        new string[] {"de Bavant", "TPFN", "TPFN"},
        new string[] {"de Beauchamp", "TPXM", "TPKM"},
        new string[] {"de Beaumont", "TPMN", "TPMN"},
        new string[] {"de Bolbec", "TPLP", "TPLP"},
        new string[] {"de Braiose", "TPRS", "TPRS"},
        new string[] {"de Braose", "TPRS", "TPRS"},
        new string[] {"de Briwere", "TPRR", "TPRR"},
        new string[] {"de Cantelou", "TKNT", "TKNT"},
        new string[] {"de Cherelton", "TXRL", "TKRL"},
        new string[] {"de Cherleton", "TXRL", "TKRL"},
        new string[] {"de Clare", "TKLR", "TKLR"},
        new string[] {"de Claremont", "TKLR", "TKLR"},
        new string[] {"de Clifford", "TKLF", "TKLF"},
        new string[] {"de Colville", "TKLF", "TKLF"},
        new string[] {"de Courtenay", "TKRT", "TKRT"},
        new string[] {"de Fauconberg", "TFKN", "TFKN"},
        new string[] {"de Forest", "TFRS", "TFRS"},
        new string[] {"de Gai", "TK", "TK"},
        new string[] {"de Grey", "TKR", "TKR"},
        new string[] {"de Guernons", "TKRN", "TKRN"},
        new string[] {"de Haia", "T", "T"},
        new string[] {"de Harcourt", "TRKR", "TRKR"},
        new string[] {"de Hastings", "TSTN", "TSTN"},
        new string[] {"de Hoke", "TK", "TK"},
        new string[] {"de Hooch", "TK", "TK"},
        new string[] {"de Hugelville", "TJLF", "TKLF"},
        new string[] {"de Huntingdon", "TNTN", "TNTN"},
        new string[] {"de Insula", "TNSL", "TNSL"},
        new string[] {"de Keynes", "TKNS", "TKNS"},
        new string[] {"de Lacy", "TLS", "TLS"},
        new string[] {"de Lexington", "TLKS", "TLKS"},
        new string[] {"de Lusignan", "TLSN", "TLSK"},
        new string[] {"de Manvers", "TMNF", "TMNF"},
        new string[] {"de Montagu", "TMNT", "TMNT"},
        new string[] {"de Montault", "TMNT", "TMNT"},
        new string[] {"de Montfort", "TMNT", "TMNT"},
        new string[] {"de Mortimer", "TMRT", "TMRT"},
        new string[] {"de Morville", "TMRF", "TMRF"},
        new string[] {"de Morvois", "TMRF", "TMRF"},
        new string[] {"de Neufmarche", "TNFM", "TNFM"},
        new string[] {"de Odingsells", "TTNK", "TTNK"},
        new string[] {"de Odyngsells", "TTNK", "TTNK"},
        new string[] {"de Percy", "TPRS", "TPRS"},
        new string[] {"de Pierrepont", "TPRP", "TPRP"},
        new string[] {"de Plessetis", "TPLS", "TPLS"},
        new string[] {"de Porhoet", "TPRT", "TPRT"},
        new string[] {"de Prouz", "TPRS", "TPRS"},
        new string[] {"de Quincy", "TKNS", "TKNS"},
        new string[] {"de Ripellis", "TRPL", "TRPL"},
        new string[] {"de Ros", "TRS", "TRS"},
        new string[] {"de Salisbury", "TSLS", "TSLS"},
        new string[] {"de Sanford", "TSNF", "TSNF"},
        new string[] {"de Somery", "TSMR", "TSMR"},
        new string[] {"de St. Hilary", "TSTL", "TSTL"},
        new string[] {"de St. Liz", "TSTL", "TSTL"},
        new string[] {"de Sutton", "TSTN", "TSTN"},
        new string[] {"de Toeni", "TTN", "TTN"},
        new string[] {"de Tony", "TTN", "TTN"},
        new string[] {"de Umfreville", "TMFR", "TMFR"},
        new string[] {"de Valognes", "TFLN", "TFLK"},
        new string[] {"de Vaux", "TF", "TF"},
        new string[] {"de Vere", "TFR", "TFR"},
        new string[] {"de Vermandois", "TFRM", "TFRM"},
        new string[] {"de Vernon", "TFRN", "TFRN"},
        new string[] {"de Vexin", "TFKS", "TFKS"},
        new string[] {"de Vitre", "TFTR", "TFTR"},
        new string[] {"de Wandesford", "TNTS", "TNTS"},
        new string[] {"de Warenne", "TRN", "TRN"},
        new string[] {"de Westbury", "TSTP", "TSTP"},
        new string[] {"di Saluzzo", "TSLS", "TSLT"},
        new string[] {"fitz Alan", "FTSL", "FTSL"},
        new string[] {"fitz Geoffrey", "FTSJ", "FTSK"},
        new string[] {"fitz Herbert", "FTSR", "FTSR"},
        new string[] {"fitz John", "FTSJ", "FTSJ"},
        new string[] {"fitz Patrick", "FTSP", "FTSP"},
        new string[] {"fitz Payn", "FTSP", "FTSP"},
        new string[] {"fitz Piers", "FTSP", "FTSP"},
        new string[] {"fitz Randolph", "FTSR", "FTSR"},
        new string[] {"fitz Richard", "FTSR", "FTSR"},
        new string[] {"fitz Robert", "FTSR", "FTSR"},
        new string[] {"fitz Roy", "FTSR", "FTSR"},
        new string[] {"fitz Scrob", "FTSS", "FTSS"},
        new string[] {"fitz Walter", "FTSL", "FTSL"},
        new string[] {"fitz Warin", "FTSR", "FTSR"},
        new string[] {"fitz Williams", "FTSL", "FTSL"},
        new string[] {"la Zouche", "LSX", "LSK"},
        new string[] {"le Botiller", "LPTL", "LPTL"},
        new string[] {"le Despenser", "LTSP", "LTSP"},
        new string[] {"le deSpencer", "LTSP", "LTSP"},
        new string[] {"of Allendale", "AFLN", "AFLN"},
        new string[] {"of Angouleme", "AFNK", "AFNK"},
        new string[] {"of Anjou", "AFNJ", "AFNJ"},
        new string[] {"of Aquitaine", "AFKT", "AFKT"},
        new string[] {"of Aumale", "AFML", "AFML"},
        new string[] {"of Bavaria", "AFPF", "AFPF"},
        new string[] {"of Boulogne", "AFPL", "AFPL"},
        new string[] {"of Brittany", "AFPR", "AFPR"},
        new string[] {"of Brittary", "AFPR", "AFPR"},
        new string[] {"of Castile", "AFKS", "AFKS"},
        new string[] {"of Chester", "AFXS", "AFKS"},
        new string[] {"of Clermont", "AFKL", "AFKL"},
        new string[] {"of Cologne", "AFKL", "AFKL"},
        new string[] {"of Dinan", "AFTN", "AFTN"},
        new string[] {"of Dunbar", "AFTN", "AFTN"},
        new string[] {"of England", "AFNK", "AFNK"},
        new string[] {"of Essex", "AFSK", "AFSK"},
        new string[] {"of Falaise", "AFFL", "AFFL"},
        new string[] {"of Flanders", "AFFL", "AFFL"},
        new string[] {"of Galloway", "AFKL", "AFKL"},
        new string[] {"of Germany", "AFKR", "AFJR"},
        new string[] {"of Gloucester", "AFKL", "AFKL"},
        new string[] {"of Heristal", "AFRS", "AFRS"},
        new string[] {"of Hungary", "AFNK", "AFNK"},
        new string[] {"of Huntington", "AFNT", "AFNT"},
        new string[] {"of Kiev", "AFKF", "AFKF"},
        new string[] {"of Kuno", "AFKN", "AFKN"},
        new string[] {"of Landen", "AFLN", "AFLN"},
        new string[] {"of Laon", "AFLN", "AFLN"},
        new string[] {"of Leinster", "AFLN", "AFLN"},
        new string[] {"of Lens", "AFLN", "AFLN"},
        new string[] {"of Lorraine", "AFLR", "AFLR"},
        new string[] {"of Louvain", "AFLF", "AFLF"},
        new string[] {"of Mercia", "AFMR", "AFMR"},
        new string[] {"of Metz", "AFMT", "AFMT"},
        new string[] {"of Meulan", "AFML", "AFML"},
        new string[] {"of Nass", "AFNS", "AFNS"},
        new string[] {"of Normandy", "AFNR", "AFNR"},
        new string[] {"of Ohningen", "AFNN", "AFNN"},
        new string[] {"of Orleans", "AFRL", "AFRL"},
        new string[] {"of Poitou", "AFPT", "AFPT"},
        new string[] {"of Polotzk", "AFPL", "AFPL"},
        new string[] {"of Provence", "AFPR", "AFPR"},
        new string[] {"of Ringelheim", "AFRN", "AFRN"},
        new string[] {"of Salisbury", "AFSL", "AFSL"},
        new string[] {"of Saxony", "AFSK", "AFSK"},
        new string[] {"of Scotland", "AFSK", "AFSK"},
        new string[] {"of Senlis", "AFSN", "AFSN"},
        new string[] {"of Stafford", "AFST", "AFST"},
        new string[] {"of Swabia", "AFSP", "AFSP"},
        new string[] {"of Tongres", "AFTN", "AFTN"},
        new string[] {"of the Tributes", "AF0T", "AFTT"},
        new string[] {"unknown", "ANKN", "ANKN"},
        new string[] {"van der Gouda", "FNTR", "FNTR"},
        new string[] {"von Adenbaugh", "FNTN", "FNTN"},
        new string[] {"ARCHITure", "ARKT", "ARKT"},
        new string[] {"Arnoff", "ARNF", "ARNF"},
        new string[] {"Arnow", "ARN", "ARNF"},
        new string[] {"DANGER", "TNJR", "TNKR"},
        new string[] {"Jankelowicz", "JNKL", "ANKL"},
        new string[] {"MANGER", "MNJR", "MNKR"},
        new string[] {"McClellan", "MKLL", "MKLL"},
        new string[] {"McHugh", "MK", "MK"},
        new string[] {"McLaughlin", "MKLF", "MKLF"},
        new string[] {"ORCHEStra", "ARKS", "ARKS"},
        new string[] {"ORCHID", "ARKT", "ARKT"},
        new string[] {"Pierce", "PRS", "PRS"},
        new string[] {"RANGER", "RNJR", "RNKR"},
        new string[] {"Schlesinger", "XLSN", "SLSN"},
        new string[] {"Uomo", "AM", "AM"},
        new string[] {"Vasserman", "FSRM", "FSRM"},
        new string[] {"Wasserman", "ASRM", "FSRM"},
        new string[] {"Womo", "AM", "FM"},
        new string[] {"Yankelovich", "ANKL", "ANKL"},
        new string[] {"accede", "AKST", "AKST"},
        new string[] {"accident", "AKST", "AKST"},
        new string[] {"adelsheim", "ATLS", "ATLS"},
        new string[] {"aged", "AJT", "AKT"},
        new string[] {"ageless", "AJLS", "AKLS"},
        new string[] {"agency", "AJNS", "AKNS"},
        new string[] {"aghast", "AKST", "AKST"},
        new string[] {"agio", "AJ", "AK"},
        new string[] {"agrimony", "AKRM", "AKRM"},
        new string[] {"album", "ALPM", "ALPM"},
        new string[] {"alcmene", "ALKM", "ALKM"},
        new string[] {"alehouse", "ALHS", "ALHS"},
        new string[] {"antique", "ANTK", "ANTK"},
        new string[] {"artois", "ART", "ARTS"},
        new string[] {"automation", "ATMX", "ATMX"},
        new string[] {"bacchus", "PKS", "PKS"},
        new string[] {"bacci", "PX", "PX"},
        new string[] {"bajador", "PJTR", "PHTR"},
        new string[] {"bellocchio", "PLX", "PLX"},
        new string[] {"bertucci", "PRTX", "PRTX"},
        new string[] {"biaggi", "PJ", "PK"},
        new string[] {"bough", "P", "P"},
        new string[] {"breaux", "PR", "PR"},
        new string[] {"broughton", "PRTN", "PRTN"},
        new string[] {"cabrillo", "KPRL", "KPR"},
        new string[] {"caesar", "SSR", "SSR"},
        new string[] {"cagney", "KKN", "KKN"},
        new string[] {"campbell", "KMPL", "KMPL"},
        new string[] {"carlisle", "KRLL", "KRLL"},
        new string[] {"carlysle", "KRLL", "KRLL"},
        new string[] {"chemistry", "KMST", "KMST"},
        new string[] {"chianti", "KNT", "KNT"},
        new string[] {"chorus", "KRS", "KRS"},
        new string[] {"cough", "KF", "KF"},
        new string[] {"czerny", "SRN", "XRN"},
        new string[] {"deffenbacher", "TFNP", "TFNP"},
        new string[] {"dumb", "TM", "TM"},
        new string[] {"edgar", "ATKR", "ATKR"},
        new string[] {"edge", "AJ", "AJ"},
        new string[] {"filipowicz", "FLPT", "FLPF"},
        new string[] {"focaccia", "FKX", "FKX"},
        new string[] {"gallegos", "KLKS", "KKS"},
        new string[] {"gambrelli", "KMPR", "KMPR"},
        new string[] {"geithain", "K0N", "JTN"},
        new string[] {"ghiradelli", "JRTL", "JRTL"},
        new string[] {"ghislane", "JLN", "JLN"},
        new string[] {"gough", "KF", "KF"},
        new string[] {"hartheim", "HR0M", "HRTM"},
        new string[] {"heimsheim", "HMSM", "HMSM"},
        new string[] {"hochmeier", "HKMR", "HKMR"},
        new string[] {"hugh", "H", "H"},
        new string[] {"hunger", "HNKR", "HNJR"},
        new string[] {"hungry", "HNKR", "HNKR"},
        new string[] {"island", "ALNT", "ALNT"},
        new string[] {"isle", "AL", "AL"},
        new string[] {"jose", "HS", "HS"},
        new string[] {"laugh", "LF", "LF"},
        new string[] {"mac caffrey", "MKFR", "MKFR"},
        new string[] {"mac gregor", "MKRK", "MKRK"},
        new string[] {"pegnitz", "PNTS", "PKNT"},
        new string[] {"piskowitz", "PSKT", "PSKF"},
        new string[] {"queen", "KN", "KN"},
        new string[] {"raspberry", "RSPR", "RSPR"},
        new string[] {"resnais", "RSN", "RSNS"},
        new string[] {"rogier", "RJ", "RJR"},
        new string[] {"rough", "RF", "RF"},
        new string[] {"san jacinto", "SNHS", "SNHS"},
        new string[] {"schenker", "XNKR", "SKNK"},
        new string[] {"schermerhorn", "XRMR", "SKRM"},
        new string[] {"schmidt", "XMT", "SMT"},
        new string[] {"schneider", "XNTR", "SNTR"},
        new string[] {"school", "SKL", "SKL"},
        new string[] {"schooner", "SKNR", "SKNR"},
        new string[] {"schrozberg", "XRSP", "SRSP"},
        new string[] {"schulman", "XLMN", "XLMN"},
        new string[] {"schwabach", "XPK", "XFPK"},
        new string[] {"schwarzach", "XRSK", "XFRT"},
        new string[] {"smith", "SM0", "XMT"},
        new string[] {"snider", "SNTR", "XNTR"},
        new string[] {"succeed", "SKST", "SKST"},
        new string[] {"sugarcane", "XKRK", "SKRK"},
        new string[] {"svobodka", "SFPT", "SFPT"},
        new string[] {"tagliaro", "TKLR", "TLR"},
        new string[] {"thames", "TMS", "TMS"},
        new string[] {"theilheim", "0LM", "TLM"},
        new string[] {"thomas", "TMS", "TMS"},
        new string[] {"thumb", "0M", "TM"},
        new string[] {"tichner", "TXNR", "TKNR"},
        new string[] {"tough", "TF", "TF"},
        new string[] {"umbrella", "AMPR", "AMPR"},
        new string[] {"vilshofen", "FLXF", "FLXF"},
        new string[] {"von schuller", "FNXL", "FNXL"},
        new string[] {"wachtler", "AKTL", "FKTL"},
        new string[] {"wechsler", "AKSL", "FKSL"},
        new string[] {"weikersheim", "AKRS", "FKRS"},
        new string[] {"zhao", "J", "J"}};

        private void CheckDoubleMetaphone(int typeIndex, bool alternate)
        {
            for (int i = 0; i < TEST_DATA.Length; i++)
            {
                string value = TEST_DATA[i][0];
                Assert.AreEqual(TEST_DATA[i][typeIndex], this.StringEncoder.GetDoubleMetaphone(value, alternate), "Test [" + i + "]=" + value);
            }
        }

        protected override DoubleMetaphone CreateStringEncoder()
        {
            return new DoubleMetaphone();
        }

        /**
         * Test alternative encoding.
         */
        [Test]
        public void TestDoubleMetaphoneAlternate()
        {
            this.CheckDoubleMetaphone(ALTERNATE_INDEX, true);
        }

        /**
         * Test primary encoding.
         */
        [Test]
        public void TestDoubleMetaphonePrimary()
        {
            this.CheckDoubleMetaphone(PRIMARY_INDEX, false);
        }
    }
}
