/*
 * Decompiled with CFR 0.152.
 */
package org.apache.inlong.tubemq.server.common.zookeeper;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.codec.binary.StringUtils;
import org.apache.inlong.tubemq.server.common.fileconfig.ZKConfig;
import org.apache.inlong.tubemq.server.common.zookeeper.RecoverableZooKeeper;
import org.apache.inlong.tubemq.server.common.zookeeper.ZooKeeperWatcher;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKUtil {
    private static final Logger logger = LoggerFactory.getLogger(ZKUtil.class);
    private static final int RETRY_TIMES = 3;
    private static final int RETRY_INTERVAL = 1000;
    private static final int SOCKET_RETRY_WAIT_MS = 200;
    private static final int DEFAULT_ZOOKEEPER_CLIENT_PORT = 2181;
    private static final char ZNODE_PATH_SEPARATOR = '/';

    public static RecoverableZooKeeper connect(ZKConfig zkConfig, Watcher watcher) throws IOException {
        if (zkConfig.getZkServerAddr() == null) {
            throw new IOException("Unable to determine ZooKeeper Server Address String");
        }
        return new RecoverableZooKeeper(zkConfig.getZkServerAddr(), zkConfig.getZkSessionTimeoutMs(), watcher, 3, 1000);
    }

    public static String joinZNode(String prefix, String suffix) {
        return prefix + '/' + suffix;
    }

    public static String getParent(String node) {
        int idx = node.lastIndexOf(47);
        return idx <= 0 ? null : node.substring(0, idx);
    }

    public static int checkExists(ZooKeeperWatcher zkw, String zkNode) throws KeeperException {
        try {
            Stat s = zkw.getRecoverableZooKeeper().exists(zkNode, null);
            return s != null ? s.getVersion() : -1;
        }
        catch (KeeperException e) {
            logger.warn(zkw.prefix("Unable to set watcher on zkNode (" + zkNode + ")"), (Throwable)e);
            zkw.keeperException(e);
            return -1;
        }
        catch (InterruptedException e) {
            logger.warn(zkw.prefix("Unable to set watcher on zkNode (" + zkNode + ")"), (Throwable)e);
            zkw.interruptedException(e);
            return -1;
        }
    }

    public static byte[] getDataAndWatch(ZooKeeperWatcher zkw, String zkNode) throws KeeperException {
        return ZKUtil.getDataInternal(zkw, zkNode, null, true);
    }

    public static List<String> getChildren(ZooKeeperWatcher zkw, String zkNode) {
        try {
            return zkw.getRecoverableZooKeeper().getChildren(zkNode, false);
        }
        catch (Throwable e) {
            return null;
        }
    }

    public static void delZNode(ZooKeeperWatcher zkw, String zkNode) {
        try {
            zkw.getRecoverableZooKeeper().delete(zkNode, -1);
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    private static byte[] getDataInternal(ZooKeeperWatcher zkw, String zkNode, Stat stat, boolean watcherSet) throws KeeperException {
        try {
            byte[] data = zkw.getRecoverableZooKeeper().getData(zkNode, zkw, stat);
            ZKUtil.logRetrievedMsg(zkw, zkNode, data, watcherSet);
            return data;
        }
        catch (KeeperException.NoNodeException e) {
            if (logger.isDebugEnabled()) {
                logger.debug(zkw.prefix("Unable to get data of zkNode " + zkNode + " because node does not exist (not an error)"));
            }
            return null;
        }
        catch (KeeperException e) {
            logger.warn(zkw.prefix("Unable to get data of zkNode " + zkNode), (Throwable)e);
            zkw.keeperException(e);
            return null;
        }
        catch (InterruptedException e) {
            logger.warn(zkw.prefix("Unable to get data of zkNode " + zkNode), (Throwable)e);
            zkw.interruptedException(e);
            return null;
        }
    }

    public static boolean setData(ZooKeeperWatcher zkw, String zkNode, byte[] data, int expectedVersion) throws KeeperException, KeeperException.NoNodeException {
        try {
            return zkw.getRecoverableZooKeeper().setData(zkNode, data, expectedVersion) != null;
        }
        catch (InterruptedException e) {
            zkw.interruptedException(e);
            return false;
        }
    }

    public static void setData(ZooKeeperWatcher zkw, String zkNode, byte[] data) throws KeeperException {
        ZKUtil.setData(zkw, zkNode, data, -1);
    }

    public static void createSetData(ZooKeeperWatcher zkw, String zkNode, byte[] data) throws KeeperException {
        if (ZKUtil.checkExists(zkw, zkNode) == -1) {
            ZKUtil.createWithParents(zkw, zkNode);
        }
        ZKUtil.setData(zkw, zkNode, data);
    }

    public static boolean isSecureZooKeeper() {
        return System.getProperty("java.security.auth.login.config") != null && System.getProperty("zookeeper.sasl.clientconfig") != null;
    }

    private static ArrayList<ACL> createACL(ZooKeeperWatcher zkw, String node) {
        if (ZKUtil.isSecureZooKeeper()) {
            if (node.equals(zkw.getBaseZNode())) {
                return ZooKeeperWatcher.CREATOR_ALL_AND_WORLD_READABLE;
            }
            return ZooDefs.Ids.CREATOR_ALL_ACL;
        }
        return ZooDefs.Ids.OPEN_ACL_UNSAFE;
    }

    public static void waitForZKConnectionIfAuthenticating(ZooKeeperWatcher zkw) throws InterruptedException {
        if (ZKUtil.isSecureZooKeeper()) {
            if (logger.isDebugEnabled()) {
                logger.debug("Waiting for ZooKeeperWatcher to authenticate");
            }
            zkw.saslLatchAwait();
            if (logger.isDebugEnabled()) {
                logger.debug("Done waiting.");
            }
        }
    }

    public static int createAndWatch(ZooKeeperWatcher zkw, String zkNode, byte[] data) throws KeeperException, KeeperException.NodeExistsException {
        try {
            ZKUtil.waitForZKConnectionIfAuthenticating(zkw);
            zkw.getRecoverableZooKeeper().create(zkNode, data, ZKUtil.createACL(zkw, zkNode), CreateMode.PERSISTENT);
            Stat stat = zkw.getRecoverableZooKeeper().exists(zkNode, zkw);
            if (stat == null) {
                return -1;
            }
            return stat.getVersion();
        }
        catch (InterruptedException e) {
            zkw.interruptedException(e);
            return -1;
        }
    }

    public static void createAndFailSilent(ZooKeeperWatcher zkw, String zkNode) throws KeeperException {
        try {
            RecoverableZooKeeper zk = zkw.getRecoverableZooKeeper();
            ZKUtil.waitForZKConnectionIfAuthenticating(zkw);
            if (zk.exists(zkNode, false) == null) {
                zk.create(zkNode, new byte[0], ZKUtil.createACL(zkw, zkNode), CreateMode.PERSISTENT);
            }
        }
        catch (KeeperException.NodeExistsException zk) {
        }
        catch (KeeperException.NoAuthException nee) {
            try {
                if (null == zkw.getRecoverableZooKeeper().exists(zkNode, false)) {
                    throw nee;
                }
            }
            catch (InterruptedException ie) {
                zkw.interruptedException(ie);
            }
        }
        catch (InterruptedException ie) {
            zkw.interruptedException(ie);
        }
    }

    public static void createWithParents(ZooKeeperWatcher zkw, String zkNode) throws KeeperException {
        try {
            if (zkNode == null) {
                return;
            }
            ZKUtil.waitForZKConnectionIfAuthenticating(zkw);
            zkw.getRecoverableZooKeeper().create(zkNode, new byte[0], ZKUtil.createACL(zkw, zkNode), CreateMode.PERSISTENT);
        }
        catch (KeeperException.NodeExistsException nee) {
            return;
        }
        catch (KeeperException.NoNodeException nne) {
            ZKUtil.createWithParents(zkw, ZKUtil.getParent(zkNode));
            ZKUtil.createWithParents(zkw, zkNode);
        }
        catch (InterruptedException ie) {
            zkw.interruptedException(ie);
        }
    }

    private static void logRetrievedMsg(ZooKeeperWatcher zkw, String zkNode, byte[] data, boolean watcherSet) {
        if (!logger.isDebugEnabled()) {
            return;
        }
        logger.debug(zkw.prefix("Retrieved " + (data == null ? 0 : data.length) + " byte(s) of data from zkNode " + zkNode + (watcherSet ? " and set watcher; " : "; data=") + (data == null ? "null" : (data.length == 0 ? "empty" : new String(data)))));
    }

    public static void createPersistent(ZooKeeperWatcher zkw, String path, boolean createParents) throws KeeperException {
        try {
            ZKUtil.createAndWatch(zkw, path, null);
        }
        catch (KeeperException.NodeExistsException e) {
            if (!createParents) {
                throw e;
            }
        }
        catch (KeeperException.NoNodeException e) {
            if (!createParents) {
                throw e;
            }
            String parentDir = path.substring(0, path.lastIndexOf(47));
            ZKUtil.createPersistent(zkw, parentDir, createParents);
            ZKUtil.createPersistent(zkw, path, createParents);
        }
    }

    public static void createParentPath(ZooKeeperWatcher zkw, String path) throws Exception {
        String parentDir = path.substring(0, path.lastIndexOf(47));
        if (parentDir.length() != 0) {
            ZKUtil.createPersistent(zkw, parentDir, true);
        }
    }

    public static void updatePersistentPath(ZooKeeperWatcher zkw, String path, String data) throws Exception {
        byte[] bytes = data == null ? null : StringUtils.getBytesUtf8((String)data);
        try {
            ZKUtil.setData(zkw, path, bytes);
        }
        catch (KeeperException.NoNodeException e) {
            ZKUtil.createParentPath(zkw, path);
            ZKUtil.createAndWatch(zkw, path, bytes);
        }
        catch (Exception e) {
            throw e;
        }
    }

    public static String readData(ZooKeeperWatcher zkw, String path) throws KeeperException {
        byte[] bytes = ZKUtil.getDataAndWatch(zkw, path);
        if (bytes == null) {
            return null;
        }
        try {
            return new String(bytes, "UTF-8");
        }
        catch (Throwable e) {
            logger.error("readData from " + path + " error! bytes is " + new String(bytes), e);
            return null;
        }
    }

    public static String readDataMaybeNull(ZooKeeperWatcher zkw, String path) throws KeeperException {
        try {
            return ZKUtil.readData(zkw, path);
        }
        catch (KeeperException.NoNodeException e) {
            return null;
        }
    }

    public static String normalizePath(String root) {
        if (root.startsWith("/")) {
            return ZKUtil.removeLastSlash(root);
        }
        return "/" + ZKUtil.removeLastSlash(root);
    }

    private static String removeLastSlash(String root) {
        if (root.endsWith("/")) {
            return root.substring(0, root.lastIndexOf("/"));
        }
        return root;
    }

    public static List<NodeAndData> getChildDataAndWatchForNewChildren(ZooKeeperWatcher zkw, String baseNode) throws KeeperException {
        List<String> nodes = ZKUtil.listChildrenAndWatchForNewChildren(zkw, baseNode);
        ArrayList<NodeAndData> newNodes = new ArrayList<NodeAndData>();
        if (nodes != null) {
            for (String node : nodes) {
                String nodePath = ZKUtil.joinZNode(baseNode, node);
                byte[] data = ZKUtil.getDataAndWatch(zkw, nodePath);
                if (data != null) {
                    newNodes.add(new NodeAndData(nodePath, data));
                    continue;
                }
                logger.error("Get data is null for nodePath " + nodePath);
            }
        }
        return newNodes;
    }

    public static List<String> listChildrenAndWatchForNewChildren(ZooKeeperWatcher zkw, String zNode) throws KeeperException {
        try {
            List<String> children = zkw.getRecoverableZooKeeper().getChildren(zNode, zkw);
            return children;
        }
        catch (KeeperException.NoNodeException ke) {
            if (logger.isDebugEnabled()) {
                logger.debug(zkw.prefix("Unable to list children of zNode " + zNode + " because node does not exist (not an error)"));
            }
            return null;
        }
        catch (KeeperException e) {
            logger.warn(zkw.prefix("Unable to list children of zNode " + zNode + " "), (Throwable)e);
            zkw.keeperException(e);
            return null;
        }
        catch (InterruptedException e) {
            logger.warn(zkw.prefix("Unable to list children of zNode " + zNode + " "), (Throwable)e);
            zkw.interruptedException(e);
            return null;
        }
    }

    public static boolean createEphemeralNodeAndWatch(ZooKeeperWatcher zkw, String zNode, byte[] data) throws KeeperException {
        return ZKUtil.createEphemeralNodeAndWatch(zkw, zNode, data, CreateMode.EPHEMERAL);
    }

    public static boolean createEphemeralNodeAndWatch(ZooKeeperWatcher zkw, String zNode, byte[] data, CreateMode mode) throws KeeperException {
        try {
            ZKUtil.waitForZKConnectionIfAuthenticating(zkw);
            zkw.getRecoverableZooKeeper().create(zNode, data, ZKUtil.createACL(zkw, zNode), mode);
        }
        catch (KeeperException.NodeExistsException nee) {
            if (!ZKUtil.watchAndCheckExists(zkw, zNode)) {
                return ZKUtil.createEphemeralNodeAndWatch(zkw, zNode, data, mode);
            }
            return false;
        }
        catch (InterruptedException e) {
            logger.info("Interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        return true;
    }

    public static boolean watchAndCheckExists(ZooKeeperWatcher zkw, String zNode) throws KeeperException {
        try {
            boolean exists;
            Stat s = zkw.getRecoverableZooKeeper().exists(zNode, zkw);
            boolean bl = exists = s != null;
            if (logger.isDebugEnabled()) {
                if (exists) {
                    logger.debug(zkw.prefix("Set watcher on existing zNode " + zNode));
                } else {
                    logger.debug(zkw.prefix(zNode + " does not exist. Watcher is set."));
                }
            }
            return exists;
        }
        catch (KeeperException e) {
            logger.warn(zkw.prefix("Unable to set watcher on zNode " + zNode), (Throwable)e);
            zkw.keeperException(e);
            return false;
        }
        catch (InterruptedException e) {
            logger.warn(zkw.prefix("Unable to set watcher on zNode " + zNode), (Throwable)e);
            zkw.interruptedException(e);
            return false;
        }
    }

    public static class NodeAndData {
        private final String node;
        private final byte[] data;

        public NodeAndData(String node, byte[] data) {
            this.node = node;
            this.data = data;
        }

        public String getNode() {
            return this.node;
        }

        public byte[] getData() {
            return this.data;
        }

        public boolean isEmpty() {
            return this.data.length == 0;
        }
    }
}

