/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.annotations.NotThreadSafe;
import com.mongodb.annotations.ThreadSafe;
import com.mongodb.assertions.Assertions;
import com.mongodb.connection.ConnectionDescription;
import com.mongodb.connection.ConnectionId;
import com.mongodb.connection.ConnectionPoolSettings;
import com.mongodb.connection.ServerDescription;
import com.mongodb.connection.ServerId;
import com.mongodb.diagnostics.logging.Logger;
import com.mongodb.diagnostics.logging.Loggers;
import com.mongodb.event.ConnectionAddedEvent;
import com.mongodb.event.ConnectionCheckOutFailedEvent;
import com.mongodb.event.ConnectionCheckOutStartedEvent;
import com.mongodb.event.ConnectionCheckedInEvent;
import com.mongodb.event.ConnectionCheckedOutEvent;
import com.mongodb.event.ConnectionClosedEvent;
import com.mongodb.event.ConnectionCreatedEvent;
import com.mongodb.event.ConnectionPoolClearedEvent;
import com.mongodb.event.ConnectionPoolClosedEvent;
import com.mongodb.event.ConnectionPoolCreatedEvent;
import com.mongodb.event.ConnectionPoolListener;
import com.mongodb.event.ConnectionPoolOpenedEvent;
import com.mongodb.event.ConnectionReadyEvent;
import com.mongodb.event.ConnectionRemovedEvent;
import com.mongodb.internal.Timeout;
import com.mongodb.internal.async.ErrorHandlingResultCallback;
import com.mongodb.internal.async.SingleResultCallback;
import com.mongodb.internal.connection.CommandMessage;
import com.mongodb.internal.connection.ConcurrentPool;
import com.mongodb.internal.connection.Connection;
import com.mongodb.internal.connection.ConnectionGenerationSupplier;
import com.mongodb.internal.connection.ConnectionPool;
import com.mongodb.internal.connection.InternalConnection;
import com.mongodb.internal.connection.InternalConnectionFactory;
import com.mongodb.internal.connection.ResponseBuffers;
import com.mongodb.internal.connection.UsageTrackingInternalConnection;
import com.mongodb.internal.event.EventListenerHelper;
import com.mongodb.internal.session.SessionContext;
import com.mongodb.internal.thread.DaemonThreadFactory;
import com.mongodb.lang.NonNull;
import com.mongodb.lang.Nullable;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.StampedLock;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.bson.ByteBuf;
import org.bson.codecs.Decoder;
import org.bson.types.ObjectId;

class DefaultConnectionPool
implements ConnectionPool {
    private static final Logger LOGGER = Loggers.getLogger("connection");
    static final int MAX_CONNECTING = 2;
    private final ConcurrentPool<UsageTrackingInternalConnection> pool;
    private final ConnectionPoolSettings settings;
    private final AtomicInteger generation = new AtomicInteger(0);
    private final ScheduledExecutorService sizeMaintenanceTimer;
    private final AsyncWorkManager asyncWorkManager;
    private final Runnable maintenanceTask;
    private final ConnectionPoolListener connectionPoolListener;
    private final ServerId serverId;
    private final PinnedStatsManager pinnedStatsManager = new PinnedStatsManager();
    private final ServiceStateManager serviceStateManager = new ServiceStateManager();
    private final ConnectionGenerationSupplier connectionGenerationSupplier;
    private volatile boolean closed;
    private final OpenConcurrencyLimiter openConcurrencyLimiter;

    DefaultConnectionPool(ServerId serverId, InternalConnectionFactory internalConnectionFactory, ConnectionPoolSettings settings) {
        this.serverId = Assertions.notNull("serverId", serverId);
        this.settings = Assertions.notNull("settings", settings);
        UsageTrackingInternalConnectionItemFactory connectionItemFactory = new UsageTrackingInternalConnectionItemFactory(internalConnectionFactory);
        this.pool = new ConcurrentPool<UsageTrackingInternalConnection>(settings.getMaxSize(), connectionItemFactory);
        this.connectionPoolListener = EventListenerHelper.getConnectionPoolListener(settings);
        this.maintenanceTask = this.createMaintenanceTask();
        this.sizeMaintenanceTimer = this.createMaintenanceTimer();
        this.connectionPoolCreated(this.connectionPoolListener, serverId, settings);
        this.openConcurrencyLimiter = new OpenConcurrencyLimiter(2);
        this.asyncWorkManager = new AsyncWorkManager();
        this.connectionGenerationSupplier = new ConnectionGenerationSupplier(){

            @Override
            public int getGeneration() {
                return DefaultConnectionPool.this.generation.get();
            }

            @Override
            public int getGeneration(@NonNull ObjectId serviceId) {
                return DefaultConnectionPool.this.serviceStateManager.getGeneration(serviceId);
            }
        };
    }

    @Override
    public void start() {
        if (this.sizeMaintenanceTimer != null) {
            this.sizeMaintenanceTimer.scheduleAtFixedRate(this.maintenanceTask, this.settings.getMaintenanceInitialDelay(TimeUnit.MILLISECONDS), this.settings.getMaintenanceFrequency(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
        }
    }

    @Override
    public InternalConnection get() {
        return this.get(this.settings.getMaxWaitTime(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
    }

    @Override
    public InternalConnection get(long timeoutValue, TimeUnit timeUnit) {
        this.connectionPoolListener.connectionCheckOutStarted(new ConnectionCheckOutStartedEvent(this.serverId));
        Timeout timeout = Timeout.startNow(timeoutValue, timeUnit);
        try {
            PooledConnection connection = this.getPooledConnection(timeout);
            if (!connection.opened()) {
                connection = this.openConcurrencyLimiter.openOrGetAvailable(connection, timeout);
            }
            this.connectionPoolListener.connectionCheckedOut(new ConnectionCheckedOutEvent(this.getId(connection)));
            return connection;
        }
        catch (RuntimeException e) {
            throw (RuntimeException)this.checkOutFailed(e);
        }
    }

    @Override
    public void getAsync(SingleResultCallback<InternalConnection> callback) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace(String.format("Asynchronously getting a connection from the pool for server %s", this.serverId));
        }
        this.connectionPoolListener.connectionCheckOutStarted(new ConnectionCheckOutStartedEvent(this.serverId));
        Timeout timeout = Timeout.startNow(this.settings.getMaxWaitTime(TimeUnit.NANOSECONDS));
        SingleResultCallback<InternalConnection> eventSendingCallback = (result, failure) -> {
            SingleResultCallback<InternalConnection> errHandlingCallback = ErrorHandlingResultCallback.errorHandlingCallback(callback, LOGGER);
            if (failure == null) {
                this.connectionPoolListener.connectionCheckedOut(new ConnectionCheckedOutEvent(this.getId((InternalConnection)result)));
                errHandlingCallback.onResult((InternalConnection)result, null);
            } else {
                errHandlingCallback.onResult(null, this.checkOutFailed(failure));
            }
        };
        this.asyncWorkManager.enqueue(new Task(timeout, t -> {
            if (t != null) {
                eventSendingCallback.onResult((InternalConnection)null, (Throwable)t);
            } else {
                PooledConnection connection;
                try {
                    connection = this.getPooledConnection(timeout);
                }
                catch (RuntimeException e) {
                    eventSendingCallback.onResult(null, e);
                    return;
                }
                if (connection.opened()) {
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.trace(String.format("Pooled connection %s to server %s is already open", this.getId(connection), this.serverId));
                    }
                    eventSendingCallback.onResult(connection, null);
                } else {
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.trace(String.format("Pooled connection %s to server %s is not yet open", this.getId(connection), this.serverId));
                    }
                    this.openConcurrencyLimiter.openAsyncOrGetAvailable(connection, timeout, eventSendingCallback);
                }
            }
        }));
    }

    private Throwable checkOutFailed(Throwable t) {
        Throwable result = t;
        if (t instanceof MongoTimeoutException) {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.TIMEOUT));
        } else if (t instanceof MongoOpenConnectionInternalException) {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR));
            result = Assertions.assertNotNull(t.getCause());
        } else if (t instanceof IllegalStateException && t.getMessage().equals("The pool is closed")) {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.POOL_CLOSED));
        } else {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.UNKNOWN));
        }
        return result;
    }

    @Override
    public void invalidate() {
        LOGGER.debug("Invalidating the connection pool");
        this.generation.incrementAndGet();
        this.connectionPoolListener.connectionPoolCleared(new ConnectionPoolClearedEvent(this.serverId));
    }

    @Override
    public void invalidate(ObjectId serviceId, int generation) {
        if (generation == -1) {
            return;
        }
        if (this.serviceStateManager.incrementGeneration(serviceId, generation)) {
            LOGGER.debug("Invalidating the connection pool for server id " + serviceId);
            this.connectionPoolListener.connectionPoolCleared(new ConnectionPoolClearedEvent(this.serverId, serviceId));
        }
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.pool.close();
            if (this.sizeMaintenanceTimer != null) {
                this.sizeMaintenanceTimer.shutdownNow();
            }
            this.asyncWorkManager.close();
            this.closed = true;
            this.connectionPoolListener.connectionPoolClosed(new ConnectionPoolClosedEvent(this.serverId));
        }
    }

    @Override
    public int getGeneration() {
        return this.generation.get();
    }

    public void doMaintenance() {
        if (this.maintenanceTask != null) {
            this.maintenanceTask.run();
        }
    }

    private PooledConnection getPooledConnection(Timeout timeout) throws MongoTimeoutException {
        try {
            UsageTrackingInternalConnection internalConnection = this.pool.get(timeout.remainingOrInfinite(TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS);
            while (this.shouldPrune(internalConnection)) {
                this.pool.release(internalConnection, true);
                internalConnection = this.pool.get(timeout.remainingOrInfinite(TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS);
            }
            return new PooledConnection(internalConnection);
        }
        catch (MongoTimeoutException e) {
            throw this.createTimeoutException(timeout);
        }
    }

    @Nullable
    private PooledConnection getPooledConnectionImmediately() {
        UsageTrackingInternalConnection internalConnection = this.pool.getImmediately();
        while (internalConnection != null && this.shouldPrune(internalConnection)) {
            this.pool.release(internalConnection, true);
            internalConnection = this.pool.getImmediately();
        }
        return internalConnection == null ? null : new PooledConnection(internalConnection);
    }

    private MongoTimeoutException createTimeoutException(Timeout timeout) {
        int numOtherInUse;
        int numPinnedToCursor = this.pinnedStatsManager.getNumPinnedToCursor();
        int numPinnedToTransaction = this.pinnedStatsManager.getNumPinnedToTransaction();
        if (numPinnedToCursor == 0 && numPinnedToTransaction == 0) {
            return new MongoTimeoutException(String.format("Timed out after %s while waiting for a connection to server %s.", timeout.toUserString(), this.serverId.getAddress()));
        }
        int maxSize = this.settings.getMaxSize();
        int numInUse = this.pool.getInUseCount();
        if (numInUse == 0) {
            numInUse = Math.min(numPinnedToCursor + numPinnedToTransaction, maxSize);
        }
        Assertions.assertTrue((numOtherInUse = numInUse - (numPinnedToCursor = Math.min(numPinnedToCursor, numInUse)) - (numPinnedToTransaction = Math.min(numPinnedToTransaction, numInUse - numPinnedToCursor))) >= 0);
        Assertions.assertTrue(numPinnedToCursor + numPinnedToTransaction + numOtherInUse <= maxSize);
        return new MongoTimeoutException(String.format("Timed out after %s while waiting for a connection to server %s. Details: maxPoolSize: %d, connections in use by cursors: %d, connections in use by transactions: %d, connections in use by other operations: %d", timeout.toUserString(), this.serverId.getAddress(), maxSize, numPinnedToCursor, numPinnedToTransaction, numOtherInUse));
    }

    ConcurrentPool<UsageTrackingInternalConnection> getPool() {
        return this.pool;
    }

    private Runnable createMaintenanceTask() {
        Runnable newMaintenanceTask = null;
        if (this.shouldPrune() || this.shouldEnsureMinSize()) {
            newMaintenanceTask = new Runnable(){

                @Override
                public synchronized void run() {
                    try {
                        if (DefaultConnectionPool.this.shouldPrune()) {
                            if (LOGGER.isDebugEnabled()) {
                                LOGGER.debug(String.format("Pruning pooled connections to %s", DefaultConnectionPool.this.serverId.getAddress()));
                            }
                            DefaultConnectionPool.this.pool.prune();
                        }
                        if (DefaultConnectionPool.this.shouldEnsureMinSize()) {
                            if (LOGGER.isDebugEnabled()) {
                                LOGGER.debug(String.format("Ensuring minimum pooled connections to %s", DefaultConnectionPool.this.serverId.getAddress()));
                            }
                            DefaultConnectionPool.this.pool.ensureMinSize(DefaultConnectionPool.this.settings.getMinSize(), newConnection -> DefaultConnectionPool.this.openConcurrencyLimiter.openImmediately(new PooledConnection((UsageTrackingInternalConnection)newConnection)));
                        }
                    }
                    catch (MongoInterruptedException | MongoTimeoutException mongoException) {
                    }
                    catch (Exception e) {
                        LOGGER.warn("Exception thrown during connection pool background maintenance task", e);
                    }
                }
            };
        }
        return newMaintenanceTask;
    }

    private ScheduledExecutorService createMaintenanceTimer() {
        if (this.maintenanceTask == null) {
            return null;
        }
        return Executors.newSingleThreadScheduledExecutor(new DaemonThreadFactory("MaintenanceTimer"));
    }

    private boolean shouldEnsureMinSize() {
        return this.settings.getMinSize() > 0;
    }

    private boolean shouldPrune() {
        return this.settings.getMaxConnectionIdleTime(TimeUnit.MILLISECONDS) > 0L || this.settings.getMaxConnectionLifeTime(TimeUnit.MILLISECONDS) > 0L;
    }

    private boolean shouldPrune(UsageTrackingInternalConnection connection) {
        return this.fromPreviousGeneration(connection) || this.pastMaxLifeTime(connection) || this.pastMaxIdleTime(connection);
    }

    private boolean pastMaxIdleTime(UsageTrackingInternalConnection connection) {
        return this.expired(connection.getLastUsedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionIdleTime(TimeUnit.MILLISECONDS));
    }

    private boolean pastMaxLifeTime(UsageTrackingInternalConnection connection) {
        return this.expired(connection.getOpenedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionLifeTime(TimeUnit.MILLISECONDS));
    }

    private boolean fromPreviousGeneration(UsageTrackingInternalConnection connection) {
        int generation = connection.getGeneration();
        if (generation == -1) {
            return false;
        }
        ObjectId serviceId = connection.getDescription().getServiceId();
        if (serviceId != null) {
            return this.serviceStateManager.getGeneration(serviceId) > generation;
        }
        return this.generation.get() > generation;
    }

    private boolean expired(long startTime, long curTime, long maxTime) {
        return maxTime != 0L && curTime - startTime > maxTime;
    }

    private void connectionPoolCreated(ConnectionPoolListener connectionPoolListener, ServerId serverId, ConnectionPoolSettings settings) {
        connectionPoolListener.connectionPoolCreated(new ConnectionPoolCreatedEvent(serverId, settings));
        connectionPoolListener.connectionPoolOpened(new ConnectionPoolOpenedEvent(serverId, settings));
    }

    private void connectionCreated(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId) {
        connectionPoolListener.connectionAdded(new ConnectionAddedEvent(connectionId));
        connectionPoolListener.connectionCreated(new ConnectionCreatedEvent(connectionId));
    }

    private void connectionClosed(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId, ConnectionClosedEvent.Reason reason) {
        connectionPoolListener.connectionRemoved(new ConnectionRemovedEvent(connectionId, this.getReasonForRemoved(reason)));
        connectionPoolListener.connectionClosed(new ConnectionClosedEvent(connectionId, reason));
    }

    private ConnectionRemovedEvent.Reason getReasonForRemoved(ConnectionClosedEvent.Reason reason) {
        ConnectionRemovedEvent.Reason removedReason = ConnectionRemovedEvent.Reason.UNKNOWN;
        switch (reason) {
            case STALE: {
                removedReason = ConnectionRemovedEvent.Reason.STALE;
                break;
            }
            case IDLE: {
                removedReason = ConnectionRemovedEvent.Reason.MAX_IDLE_TIME_EXCEEDED;
                break;
            }
            case ERROR: {
                removedReason = ConnectionRemovedEvent.Reason.ERROR;
                break;
            }
            case POOL_CLOSED: {
                removedReason = ConnectionRemovedEvent.Reason.POOL_CLOSED;
                break;
            }
        }
        return removedReason;
    }

    private ConnectionId getId(InternalConnection internalConnection) {
        return internalConnection.getDescription().getConnectionId();
    }

    @NotThreadSafe
    final class Task {
        private final Timeout timeout;
        private final Consumer<RuntimeException> action;
        private boolean completed;

        Task(Timeout timeout, Consumer<RuntimeException> action) {
            this.timeout = timeout;
            this.action = action;
        }

        void execute() {
            this.doComplete(() -> null);
        }

        void failAsClosed() {
            this.doComplete(ConcurrentPool::poolClosedException);
        }

        void failAsTimedOut() {
            this.doComplete(() -> DefaultConnectionPool.this.createTimeoutException(this.timeout));
        }

        private void doComplete(Supplier<RuntimeException> failureSupplier) {
            Assertions.assertFalse(this.completed);
            this.completed = true;
            this.action.accept(failureSupplier.get());
        }

        Timeout timeout() {
            return this.timeout;
        }
    }

    @ThreadSafe
    private static class AsyncWorkManager
    implements AutoCloseable {
        private volatile State state = State.NEW;
        private volatile BlockingQueue<Task> tasks = new LinkedBlockingQueue<Task>();
        private final Lock lock = new StampedLock().asWriteLock();
        @Nullable
        private ExecutorService worker;

        AsyncWorkManager() {
        }

        void enqueue(Task task) {
            this.lock.lock();
            try {
                if (this.initUnlessClosed()) {
                    this.tasks.add(task);
                    return;
                }
            }
            finally {
                this.lock.unlock();
            }
            task.failAsClosed();
        }

        private boolean initUnlessClosed() {
            boolean result = true;
            if (this.state == State.NEW) {
                this.worker = Executors.newSingleThreadExecutor(new DaemonThreadFactory("AsyncGetter"));
                this.worker.submit(() -> this.runAndLogUncaught(this::workerRun));
                this.state = State.INITIALIZED;
            } else if (this.state == State.CLOSED) {
                result = false;
            }
            return result;
        }

        @Override
        public void close() {
            this.lock.lock();
            try {
                if (this.state != State.CLOSED) {
                    this.state = State.CLOSED;
                    if (this.worker != null) {
                        this.worker.shutdownNow();
                    }
                }
            }
            finally {
                this.lock.unlock();
            }
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        private void workerRun() {
            try {
                while (this.state != State.CLOSED) {
                    try {
                        Task task = this.tasks.take();
                        if (task.timeout().expired()) {
                            task.failAsTimedOut();
                            continue;
                        }
                        task.execute();
                    }
                    catch (RuntimeException e) {
                        LOGGER.error(null, e);
                    }
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            this.failAllTasksAfterClosing();
        }

        private void failAllTasksAfterClosing() {
            BlockingQueue<Task> localGets;
            this.lock.lock();
            try {
                Assertions.assertTrue(this.state == State.CLOSED);
                localGets = this.tasks;
                if (!this.tasks.isEmpty()) {
                    this.tasks = new LinkedBlockingQueue<Task>();
                }
            }
            finally {
                this.lock.unlock();
            }
            localGets.forEach(Task::failAsClosed);
            localGets.clear();
        }

        private void runAndLogUncaught(Runnable runnable) {
            try {
                runnable.run();
            }
            catch (Throwable t) {
                LOGGER.error("The pool is not going to work correctly from now on. You may want to recreate the MongoClient", t);
                throw t;
            }
        }

        private static enum State {
            NEW,
            INITIALIZED,
            CLOSED;

        }
    }

    private static final class PinnedStatsManager {
        private final LongAdder numPinnedToCursor = new LongAdder();
        private final LongAdder numPinnedToTransaction = new LongAdder();

        private PinnedStatsManager() {
        }

        void increment(Connection.PinningMode pinningMode) {
            switch (pinningMode) {
                case CURSOR: {
                    this.numPinnedToCursor.increment();
                    break;
                }
                case TRANSACTION: {
                    this.numPinnedToTransaction.increment();
                    break;
                }
                default: {
                    Assertions.fail();
                }
            }
        }

        void decrement(Connection.PinningMode pinningMode) {
            switch (pinningMode) {
                case CURSOR: {
                    this.numPinnedToCursor.decrement();
                    break;
                }
                case TRANSACTION: {
                    this.numPinnedToTransaction.decrement();
                    break;
                }
                default: {
                    Assertions.fail();
                }
            }
        }

        int getNumPinnedToCursor() {
            return this.numPinnedToCursor.intValue();
        }

        int getNumPinnedToTransaction() {
            return this.numPinnedToTransaction.intValue();
        }
    }

    static final class ServiceStateManager {
        private final ConcurrentHashMap<ObjectId, ServiceState> stateByServiceId = new ConcurrentHashMap();

        ServiceStateManager() {
        }

        void addConnection(ObjectId serviceId) {
            this.stateByServiceId.compute(serviceId, (k, v) -> {
                if (v == null) {
                    v = new ServiceState();
                }
                v.incrementConnectionCount();
                return v;
            });
        }

        void removeConnection(ObjectId serviceId) {
            this.stateByServiceId.compute(serviceId, (k, v) -> {
                Assertions.assertNotNull(v);
                return v.decrementAndGetConnectionCount() == 0 ? null : v;
            });
        }

        boolean incrementGeneration(ObjectId serviceId, int expectedGeneration) {
            ServiceState state = this.stateByServiceId.get(serviceId);
            return state == null || state.incrementGeneration(expectedGeneration);
        }

        int getGeneration(ObjectId serviceId) {
            ServiceState state = this.stateByServiceId.get(serviceId);
            return state == null ? 0 : state.getGeneration();
        }

        private static final class ServiceState {
            private final AtomicInteger generation = new AtomicInteger();
            private final AtomicInteger connectionCount = new AtomicInteger();

            private ServiceState() {
            }

            void incrementConnectionCount() {
                this.connectionCount.incrementAndGet();
            }

            int decrementAndGetConnectionCount() {
                return this.connectionCount.decrementAndGet();
            }

            boolean incrementGeneration(int expectedGeneration) {
                return this.generation.compareAndSet(expectedGeneration, expectedGeneration + 1);
            }

            public int getGeneration() {
                return this.generation.get();
            }
        }
    }

    @NotThreadSafe
    private static final class MutableReference<T> {
        @Nullable
        private T reference;

        private MutableReference() {
        }
    }

    @ThreadSafe
    private final class OpenConcurrencyLimiter {
        private final Lock lock = new ReentrantLock(true);
        private final Condition condition = this.lock.newCondition();
        private final int maxPermits;
        private int permits;
        private final Deque<MutableReference<PooledConnection>> desiredConnectionSlots;

        OpenConcurrencyLimiter(int maxConnecting) {
            this.permits = this.maxPermits = maxConnecting;
            this.desiredConnectionSlots = new LinkedList<MutableReference<PooledConnection>>();
        }

        PooledConnection openOrGetAvailable(PooledConnection connection, Timeout timeout) throws MongoTimeoutException {
            return this.openOrGetAvailable(connection, true, timeout);
        }

        void openImmediately(PooledConnection connection) throws MongoTimeoutException {
            PooledConnection result = this.openOrGetAvailable(connection, false, Timeout.immediate());
            Assertions.assertTrue(result == connection);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private PooledConnection openOrGetAvailable(PooledConnection connection, boolean tryGetAvailable, Timeout timeout) throws MongoTimeoutException {
            PooledConnection availableConnection;
            try {
                availableConnection = this.acquirePermitOrGetAvailableOpenedConnection(tryGetAvailable, timeout);
            }
            catch (RuntimeException e) {
                connection.closeSilently();
                throw e;
            }
            if (availableConnection != null) {
                connection.closeSilently();
                return availableConnection;
            }
            try {
                connection.open();
            }
            finally {
                this.releasePermit();
            }
            return connection;
        }

        void openAsyncOrGetAvailable(PooledConnection connection, Timeout timeout, SingleResultCallback<InternalConnection> callback) {
            PooledConnection availableConnection;
            try {
                availableConnection = this.acquirePermitOrGetAvailableOpenedConnection(true, timeout);
            }
            catch (RuntimeException e) {
                connection.closeSilently();
                callback.onResult(null, e);
                return;
            }
            if (availableConnection != null) {
                connection.closeSilently();
                callback.onResult(availableConnection, null);
            } else {
                connection.openAsync((nullResult, failure) -> {
                    this.releasePermit();
                    if (failure != null) {
                        callback.onResult(null, failure);
                    } else {
                        callback.onResult(connection, null);
                    }
                });
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Nullable
        private PooledConnection acquirePermitOrGetAvailableOpenedConnection(boolean tryGetAvailable, Timeout timeout) throws MongoTimeoutException {
            PooledConnection availableConnection = null;
            boolean expressedDesireToGetAvailableConnection = false;
            this.lockInterruptibly(this.lock);
            try {
                if (tryGetAvailable) {
                    availableConnection = DefaultConnectionPool.this.getPooledConnectionImmediately();
                    if (availableConnection != null) {
                        PooledConnection pooledConnection = availableConnection;
                        return pooledConnection;
                    }
                    this.expressDesireToGetAvailableConnection();
                    expressedDesireToGetAvailableConnection = true;
                }
                long remainingNanos = timeout.remainingOrInfinite(TimeUnit.NANOSECONDS);
                while (this.permits == 0) {
                    PooledConnection pooledConnection = availableConnection = tryGetAvailable ? this.tryGetAvailableConnection() : null;
                    if (availableConnection != null) break;
                    if (Timeout.expired(remainingNanos)) {
                        throw DefaultConnectionPool.this.createTimeoutException(timeout);
                    }
                    remainingNanos = this.awaitNanos(this.condition, remainingNanos);
                }
                if (availableConnection == null) {
                    Assertions.assertTrue(this.permits > 0);
                    --this.permits;
                }
                PooledConnection pooledConnection = availableConnection;
                return pooledConnection;
            }
            finally {
                try {
                    if (expressedDesireToGetAvailableConnection && availableConnection == null) {
                        this.giveUpOnTryingToGetAvailableConnection();
                    }
                }
                finally {
                    this.lock.unlock();
                }
            }
        }

        private void releasePermit() {
            this.lock.lock();
            try {
                Assertions.assertTrue(this.permits < this.maxPermits);
                ++this.permits;
                this.condition.signal();
            }
            finally {
                this.lock.unlock();
            }
        }

        private void expressDesireToGetAvailableConnection() {
            this.desiredConnectionSlots.addLast(new MutableReference());
        }

        @Nullable
        private PooledConnection tryGetAvailableConnection() {
            Assertions.assertFalse(this.desiredConnectionSlots.isEmpty());
            PooledConnection result = (PooledConnection)((MutableReference)this.desiredConnectionSlots.peekFirst()).reference;
            if (result != null) {
                this.desiredConnectionSlots.removeFirst();
                Assertions.assertTrue(result.opened());
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Received opened connection [%s] to server %s", DefaultConnectionPool.this.getId(result), DefaultConnectionPool.this.serverId.getAddress()));
                }
            }
            return result;
        }

        private void giveUpOnTryingToGetAvailableConnection() {
            Assertions.assertFalse(this.desiredConnectionSlots.isEmpty());
            PooledConnection connection = (PooledConnection)((MutableReference)this.desiredConnectionSlots.removeLast()).reference;
            if (connection != null) {
                connection.release();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void tryHandOverOrRelease(UsageTrackingInternalConnection openConnection) {
            this.lock.lock();
            try {
                for (MutableReference<PooledConnection> desiredConnectionSlot : this.desiredConnectionSlots) {
                    if (((MutableReference)desiredConnectionSlot).reference != null) continue;
                    ((MutableReference)desiredConnectionSlot).reference = new PooledConnection(openConnection);
                    this.condition.signal();
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.trace(String.format("Handed over opened connection [%s] to server %s", DefaultConnectionPool.this.getId(openConnection), DefaultConnectionPool.this.serverId.getAddress()));
                    }
                    return;
                }
                DefaultConnectionPool.this.pool.release(openConnection);
            }
            finally {
                this.lock.unlock();
            }
        }

        private void lockInterruptibly(Lock lock) throws MongoInterruptedException {
            try {
                lock.lockInterruptibly();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new MongoInterruptedException(null, e);
            }
        }

        private long awaitNanos(Condition condition, long timeoutNanos) throws MongoInterruptedException {
            try {
                if (timeoutNanos < 0L || timeoutNanos == Long.MAX_VALUE) {
                    condition.await();
                    return -1L;
                }
                return Math.max(0L, condition.awaitNanos(timeoutNanos));
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new MongoInterruptedException(null, e);
            }
        }
    }

    private class UsageTrackingInternalConnectionItemFactory
    implements ConcurrentPool.ItemFactory<UsageTrackingInternalConnection> {
        private final InternalConnectionFactory internalConnectionFactory;

        UsageTrackingInternalConnectionItemFactory(InternalConnectionFactory internalConnectionFactory) {
            this.internalConnectionFactory = internalConnectionFactory;
        }

        @Override
        public UsageTrackingInternalConnection create() {
            return new UsageTrackingInternalConnection(this.internalConnectionFactory.create(DefaultConnectionPool.this.serverId, DefaultConnectionPool.this.connectionGenerationSupplier), DefaultConnectionPool.this.serviceStateManager);
        }

        @Override
        public void close(UsageTrackingInternalConnection connection) {
            if (connection.isCloseSilently()) {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Silently closed connection [%s] to server %s", DefaultConnectionPool.this.getId(connection), DefaultConnectionPool.this.serverId.getAddress()));
                }
            } else {
                DefaultConnectionPool.this.connectionClosed(DefaultConnectionPool.this.connectionPoolListener, DefaultConnectionPool.this.getId(connection), this.getReasonForClosing(connection));
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug(String.format("Closed connection [%s] to %s because %s.", DefaultConnectionPool.this.getId(connection), DefaultConnectionPool.this.serverId.getAddress(), this.getReasonStringForClosing(connection)));
                }
            }
            connection.close();
        }

        private String getReasonStringForClosing(UsageTrackingInternalConnection connection) {
            String reason = connection.isClosed() ? "there was a socket exception raised by this connection" : (DefaultConnectionPool.this.fromPreviousGeneration(connection) ? "there was a socket exception raised on another connection from this pool" : (DefaultConnectionPool.this.pastMaxLifeTime(connection) ? "it is past its maximum allowed life time" : (DefaultConnectionPool.this.pastMaxIdleTime(connection) ? "it is past its maximum allowed idle time" : "the pool has been closed")));
            return reason;
        }

        private ConnectionClosedEvent.Reason getReasonForClosing(UsageTrackingInternalConnection connection) {
            ConnectionClosedEvent.Reason reason = connection.isClosed() ? ConnectionClosedEvent.Reason.ERROR : (DefaultConnectionPool.this.fromPreviousGeneration(connection) ? ConnectionClosedEvent.Reason.STALE : (DefaultConnectionPool.this.pastMaxIdleTime(connection) ? ConnectionClosedEvent.Reason.IDLE : ConnectionClosedEvent.Reason.POOL_CLOSED));
            return reason;
        }

        @Override
        public ConcurrentPool.Prune shouldPrune(UsageTrackingInternalConnection usageTrackingConnection) {
            return DefaultConnectionPool.this.shouldPrune(usageTrackingConnection) ? ConcurrentPool.Prune.YES : ConcurrentPool.Prune.NO;
        }
    }

    private static final class MongoOpenConnectionInternalException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        MongoOpenConnectionInternalException(@NonNull Throwable cause) {
            super(null, cause);
        }
    }

    private class PooledConnection
    implements InternalConnection {
        private final UsageTrackingInternalConnection wrapped;
        private final AtomicBoolean isClosed = new AtomicBoolean();
        private Connection.PinningMode pinningMode;

        PooledConnection(UsageTrackingInternalConnection wrapped) {
            this.wrapped = Assertions.notNull("wrapped", wrapped);
        }

        @Override
        public int getGeneration() {
            return this.wrapped.getGeneration();
        }

        @Override
        public void open() {
            Assertions.assertFalse(this.isClosed.get());
            try {
                DefaultConnectionPool.this.connectionCreated(DefaultConnectionPool.this.connectionPoolListener, this.wrapped.getDescription().getConnectionId());
                this.wrapped.open();
            }
            catch (RuntimeException e) {
                this.closeAndHandleOpenFailure();
                throw new MongoOpenConnectionInternalException(e);
            }
            this.handleOpenSuccess();
        }

        @Override
        public void openAsync(SingleResultCallback<Void> callback) {
            Assertions.assertFalse(this.isClosed.get());
            DefaultConnectionPool.this.connectionCreated(DefaultConnectionPool.this.connectionPoolListener, this.wrapped.getDescription().getConnectionId());
            this.wrapped.openAsync((nullResult, failure) -> {
                if (failure != null) {
                    this.closeAndHandleOpenFailure();
                    callback.onResult(null, new MongoOpenConnectionInternalException(failure));
                } else {
                    this.handleOpenSuccess();
                    callback.onResult((Void)nullResult, null);
                }
            });
        }

        @Override
        public void close() {
            if (!this.isClosed.getAndSet(true)) {
                this.unmarkAsPinned();
                DefaultConnectionPool.this.connectionPoolListener.connectionCheckedIn(new ConnectionCheckedInEvent(DefaultConnectionPool.this.getId(this.wrapped)));
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Checked in connection [%s] to server %s", DefaultConnectionPool.this.getId(this.wrapped), DefaultConnectionPool.this.serverId.getAddress()));
                }
                if (this.wrapped.isClosed() || DefaultConnectionPool.this.shouldPrune(this.wrapped)) {
                    DefaultConnectionPool.this.pool.release(this.wrapped, true);
                } else {
                    DefaultConnectionPool.this.openConcurrencyLimiter.tryHandOverOrRelease(this.wrapped);
                }
            }
        }

        void release() {
            if (!this.isClosed.getAndSet(true)) {
                DefaultConnectionPool.this.pool.release(this.wrapped);
            }
        }

        void closeSilently() {
            if (!this.isClosed.getAndSet(true)) {
                this.wrapped.setCloseSilently();
                DefaultConnectionPool.this.pool.release(this.wrapped, true);
            }
        }

        private void closeAndHandleOpenFailure() {
            if (!this.isClosed.getAndSet(true)) {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Pooled connection %s to server %s failed to open", DefaultConnectionPool.this.getId(this), DefaultConnectionPool.this.serverId));
                }
                if (this.wrapped.getDescription().getServiceId() != null) {
                    DefaultConnectionPool.this.invalidate(this.wrapped.getDescription().getServiceId(), this.wrapped.getGeneration());
                }
                DefaultConnectionPool.this.pool.release(this.wrapped, true);
            }
        }

        private void handleOpenSuccess() {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace(String.format("Pooled connection %s to server %s is now open", DefaultConnectionPool.this.getId(this), DefaultConnectionPool.this.serverId));
            }
            DefaultConnectionPool.this.connectionPoolListener.connectionReady(new ConnectionReadyEvent(DefaultConnectionPool.this.getId(this)));
        }

        @Override
        public boolean opened() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.opened();
        }

        @Override
        public boolean isClosed() {
            return this.isClosed.get() || this.wrapped.isClosed();
        }

        @Override
        public ByteBuf getBuffer(int capacity) {
            return this.wrapped.getBuffer(capacity);
        }

        @Override
        public void sendMessage(List<ByteBuf> byteBuffers, int lastRequestId) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessage(byteBuffers, lastRequestId);
        }

        @Override
        public <T> T sendAndReceive(CommandMessage message, Decoder<T> decoder, SessionContext sessionContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.sendAndReceive(message, decoder, sessionContext);
        }

        @Override
        public <T> void send(CommandMessage message, Decoder<T> decoder, SessionContext sessionContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.send(message, decoder, sessionContext);
        }

        @Override
        public <T> T receive(Decoder<T> decoder, SessionContext sessionContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receive(decoder, sessionContext);
        }

        @Override
        public boolean supportsAdditionalTimeout() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.supportsAdditionalTimeout();
        }

        @Override
        public <T> T receive(Decoder<T> decoder, SessionContext sessionContext, int additionalTimeout) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receive(decoder, sessionContext, additionalTimeout);
        }

        @Override
        public boolean hasMoreToCome() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.hasMoreToCome();
        }

        @Override
        public <T> void sendAndReceiveAsync(CommandMessage message, Decoder<T> decoder, SessionContext sessionContext, final SingleResultCallback<T> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendAndReceiveAsync(message, decoder, sessionContext, new SingleResultCallback<T>(){

                @Override
                public void onResult(T result, Throwable t) {
                    callback.onResult(result, t);
                }
            });
        }

        @Override
        public ResponseBuffers receiveMessage(int responseTo) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receiveMessage(responseTo);
        }

        @Override
        public void sendMessageAsync(List<ByteBuf> byteBuffers, int lastRequestId, final SingleResultCallback<Void> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessageAsync(byteBuffers, lastRequestId, new SingleResultCallback<Void>(){

                @Override
                public void onResult(Void result, Throwable t) {
                    callback.onResult(null, t);
                }
            });
        }

        @Override
        public void receiveMessageAsync(int responseTo, final SingleResultCallback<ResponseBuffers> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.receiveMessageAsync(responseTo, new SingleResultCallback<ResponseBuffers>(){

                @Override
                public void onResult(ResponseBuffers result, Throwable t) {
                    callback.onResult(result, t);
                }
            });
        }

        @Override
        public void markAsPinned(Connection.PinningMode pinningMode) {
            Assertions.assertNotNull(pinningMode);
            if (this.pinningMode == null) {
                this.pinningMode = pinningMode;
                DefaultConnectionPool.this.pinnedStatsManager.increment(pinningMode);
            }
        }

        void unmarkAsPinned() {
            if (this.pinningMode != null) {
                DefaultConnectionPool.this.pinnedStatsManager.decrement(this.pinningMode);
            }
        }

        @Override
        public ConnectionDescription getDescription() {
            return this.wrapped.getDescription();
        }

        @Override
        public ServerDescription getInitialServerDescription() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.getInitialServerDescription();
        }
    }
}

