/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.mr.hive;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.io.UncheckedIOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.Warehouse;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.api.Partition;
import org.apache.hadoop.hive.metastore.partition.spec.PartitionSpecProxy;
import org.apache.hadoop.hive.ql.parse.SemanticException;
import org.apache.iceberg.AppendFiles;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.DeleteFiles;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.SerializableTable;
import org.apache.iceberg.Table;
import org.apache.iceberg.Transaction;
import org.apache.iceberg.data.TableMigrationUtil;
import org.apache.iceberg.exceptions.NotFoundException;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.hadoop.HadoopConfigurable;
import org.apache.iceberg.hadoop.HadoopFileIO;
import org.apache.iceberg.hadoop.Util;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.io.OutputFile;
import org.apache.iceberg.mapping.NameMapping;
import org.apache.iceberg.mapping.NameMappingParser;
import org.apache.iceberg.mr.Catalogs;
import org.apache.iceberg.mr.hive.IcebergTableUtil;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.iceberg.util.SerializationUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HiveTableUtil {
    private static final Logger LOG = LoggerFactory.getLogger(HiveTableUtil.class);
    static final String TABLE_EXTENSION = ".table";

    private HiveTableUtil() {
    }

    public static void importFiles(String sourceLocation, String format, PartitionSpecProxy partitionSpecProxy, List<FieldSchema> partitionKeys, Properties icebergTableProperties, Configuration conf) throws MetaException {
        RemoteIterator<LocatedFileStatus> filesIterator = null;
        if (partitionSpecProxy.size() == 0) {
            filesIterator = HiveTableUtil.getFilesIterator(new Path(sourceLocation), conf);
        }
        Table icebergTable = Catalogs.createTable(conf, icebergTableProperties);
        AppendFiles append = icebergTable.newAppend();
        PartitionSpec spec = icebergTable.spec();
        MetricsConfig metricsConfig = MetricsConfig.fromProperties(icebergTable.properties());
        String nameMappingString = icebergTable.properties().get("schema.name-mapping.default");
        NameMapping nameMapping = nameMappingString != null ? NameMappingParser.fromJson(nameMappingString) : null;
        try {
            if (partitionSpecProxy.size() == 0) {
                List<DataFile> dataFiles = HiveTableUtil.getDataFiles(filesIterator, Collections.emptyMap(), format, spec, metricsConfig, nameMapping, conf);
                dataFiles.forEach(append::appendFile);
            } else {
                PartitionSpecProxy.PartitionIterator partitionIterator = partitionSpecProxy.getPartitionIterator();
                ArrayList<Callable<Void>> tasks = Lists.newArrayList();
                while (partitionIterator.hasNext()) {
                    Partition partition = (Partition)partitionIterator.next();
                    Callable<Void> task = () -> {
                        Path partitionPath = new Path(partition.getSd().getLocation());
                        String partitionName = Warehouse.makePartName((List)partitionKeys, (List)partition.getValues());
                        LinkedHashMap partitionSpec = Warehouse.makeSpecFromName((String)partitionName);
                        RemoteIterator<LocatedFileStatus> iterator = HiveTableUtil.getFilesIterator(partitionPath, conf);
                        List<DataFile> dataFiles = HiveTableUtil.getDataFiles(iterator, partitionSpec, format.toLowerCase(), spec, metricsConfig, nameMapping, conf);
                        AppendFiles appendFiles = append;
                        synchronized (appendFiles) {
                            dataFiles.forEach(append::appendFile);
                        }
                        return null;
                    };
                    tasks.add(task);
                }
                int numThreads = HiveConf.getIntVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_SERVER2_ICEBERG_METADATA_GENERATOR_THREADS);
                try (ExecutorService executor = Executors.newFixedThreadPool(numThreads, new ThreadFactoryBuilder().setNameFormat("iceberg-metadata-generator-%d").setDaemon(true).build());){
                    executor.invokeAll(tasks);
                }
            }
            append.commit();
        }
        catch (IOException | InterruptedException e) {
            throw new MetaException("Cannot import hive data into iceberg table.\n" + e.getMessage());
        }
    }

    private static List<DataFile> getDataFiles(RemoteIterator<LocatedFileStatus> fileStatusIterator, Map<String, String> partitionKeys, String format, PartitionSpec spec, MetricsConfig metricsConfig, NameMapping nameMapping, Configuration conf) throws IOException {
        ArrayList<DataFile> dataFiles = Lists.newArrayList();
        while (fileStatusIterator.hasNext()) {
            LocatedFileStatus fileStatus = (LocatedFileStatus)fileStatusIterator.next();
            String fileName = fileStatus.getPath().getName();
            if (fileName.startsWith(".") || fileName.startsWith("_") || fileName.endsWith("metadata.json")) continue;
            dataFiles.addAll(TableMigrationUtil.listPartition(partitionKeys, fileStatus.getPath().toString(), format, spec, conf, metricsConfig, nameMapping));
        }
        return dataFiles;
    }

    public static void appendFiles(URI fromURI, String format, Table icebergTbl, boolean isOverwrite, Map<String, String> partitionSpec, Configuration conf) throws SemanticException {
        try {
            Transaction transaction = icebergTbl.newTransaction();
            if (isOverwrite) {
                DeleteFiles delete = transaction.newDelete();
                if (partitionSpec != null) {
                    Expression partitionExpr = IcebergTableUtil.generateExpressionFromPartitionSpec(icebergTbl, partitionSpec, true);
                    delete.deleteFromRowFilter(partitionExpr);
                } else {
                    delete.deleteFromRowFilter(Expressions.alwaysTrue());
                }
                delete.commit();
            }
            MetricsConfig metricsConfig = MetricsConfig.fromProperties(icebergTbl.properties());
            PartitionSpec spec = icebergTbl.spec();
            String nameMappingStr = icebergTbl.properties().get("schema.name-mapping.default");
            NameMapping nameMapping = null;
            if (nameMappingStr != null) {
                nameMapping = NameMappingParser.fromJson(nameMappingStr);
            }
            AppendFiles append = transaction.newAppend();
            Map<String, String> actualPartitionSpec = Optional.ofNullable(partitionSpec).orElse(Collections.emptyMap());
            String actualFormat = Optional.ofNullable(format).orElse("PARQUET").toLowerCase();
            RemoteIterator<LocatedFileStatus> iterator = HiveTableUtil.getFilesIterator(new Path(fromURI), conf);
            List<DataFile> dataFiles = HiveTableUtil.getDataFiles(iterator, actualPartitionSpec, actualFormat, spec, metricsConfig, nameMapping, conf);
            dataFiles.forEach(append::appendFile);
            append.commit();
            transaction.commitTransaction();
        }
        catch (Exception e) {
            throw new SemanticException("Can not append data files", (Throwable)e);
        }
    }

    public static RemoteIterator<LocatedFileStatus> getFilesIterator(Path path, Configuration conf) throws MetaException {
        try {
            FileSystem fileSystem = FileSystem.get((URI)path.toUri(), (Configuration)conf);
            return fileSystem.listFiles(path, true);
        }
        catch (IOException e) {
            throw new MetaException("Exception happened during the collection of file statuses.\n" + e.getMessage());
        }
    }

    public static String serializeTable(Table table, Configuration config, Properties props, List<String> broadcastConfig) {
        Table serializableTable = SerializableTable.copyOf(table);
        if (broadcastConfig != null) {
            broadcastConfig.forEach(cfg -> serializableTable.properties().computeIfAbsent((String)cfg, props::getProperty));
        }
        HiveTableUtil.checkAndSkipIoConfigSerialization(config, serializableTable);
        return SerializationUtil.serializeToBase64(serializableTable);
    }

    public static Table deserializeTable(Configuration config, String name) {
        Table table = (Table)SerializationUtil.deserializeFromBase64(config.get("iceberg.mr.serialized.table." + name));
        String location = config.get("iceberg.mr.table.location");
        if (table == null && config.getBoolean("created_with_ctas", false) && StringUtils.isNotBlank((CharSequence)location)) {
            table = HiveTableUtil.readTableObjectFromFile(location, config);
        }
        HiveTableUtil.checkAndSetIoConfig(config, table);
        return table;
    }

    private static void checkAndSetIoConfig(Configuration config, Table table) {
        FileIO fileIO;
        if (table != null && config.getBoolean("iceberg.mr.config.serialization.disabled", true) && (fileIO = table.io()) instanceof HadoopConfigurable) {
            HadoopConfigurable fileIO2 = (HadoopConfigurable)((Object)fileIO);
            fileIO2.setConf(config);
        }
    }

    public static void checkAndSkipIoConfigSerialization(Configuration config, Table table) {
        FileIO fileIO;
        if (table != null && config.getBoolean("iceberg.mr.config.serialization.disabled", true) && (fileIO = table.io()) instanceof HadoopConfigurable) {
            HadoopConfigurable fileIO2 = (HadoopConfigurable)((Object)fileIO);
            fileIO2.serializeConfWith(conf -> new NonSerializingConfig(config)::get);
        }
    }

    private static String generateTableObjectLocation(String tableLocation, Configuration conf) {
        return tableLocation + "/temp/" + conf.get(HiveConf.ConfVars.HIVE_QUERY_ID.varname) + TABLE_EXTENSION;
    }

    static void createFileForTableObject(Table table, Configuration conf) {
        String filePath = HiveTableUtil.generateTableObjectLocation(table.location(), conf);
        String bytes = HiveTableUtil.serializeTable(table, conf, null, null);
        OutputFile serializedTableFile = table.io().newOutputFile(filePath);
        try (ObjectOutputStream oos = new ObjectOutputStream(serializedTableFile.createOrOverwrite());){
            oos.writeObject(bytes);
        }
        catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
        LOG.debug("Iceberg table metadata file is created {}", (Object)serializedTableFile);
    }

    static void cleanupTableObjectFile(String location, Configuration configuration) {
        String filePath = HiveTableUtil.generateTableObjectLocation(location, configuration);
        Path toDelete = new Path(filePath);
        try {
            FileSystem fs = Util.getFs(toDelete, configuration);
            fs.delete(toDelete, true);
        }
        catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static Table readTableObjectFromFile(String location, Configuration config) {
        String filePath = HiveTableUtil.generateTableObjectLocation(location, config);
        try (HadoopFileIO io = new HadoopFileIO(config);){
            Table table;
            try (ObjectInputStream ois = new ObjectInputStream(io.newInputFile(filePath).newStream());){
                table = (Table)SerializationUtil.deserializeFromBase64((String)ois.readObject());
            }
            return table;
        }
        catch (IOException | ClassNotFoundException e) {
            throw new NotFoundException("Can not read or parse table object file: %s", filePath);
        }
    }

    public static boolean isCtas(Properties properties) {
        return Boolean.parseBoolean(properties.getProperty("created_with_ctas"));
    }

    private static class NonSerializingConfig
    implements Serializable {
        private final transient Configuration conf;

        NonSerializingConfig(Configuration conf) {
            this.conf = conf;
        }

        public Configuration get() {
            if (this.conf == null) {
                throw new IllegalStateException("Configuration was not serialized on purpose but was not set manually either");
            }
            return this.conf;
        }
    }
}

