/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.service.cli.operation;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.hadoop.hive.common.ServerUtils;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.ql.QueryInfo;
import org.apache.hadoop.util.StringUtils;
import org.apache.hive.com.google.common.annotations.VisibleForTesting;
import org.apache.hive.org.apache.commons.io.FileUtils;
import org.apache.hive.service.cli.operation.Operation;
import org.apache.hive.service.cli.operation.OperationManager;
import org.apache.hive.service.cli.operation.SQLOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OperationLogManager {
    private static final Logger LOG = LoggerFactory.getLogger(OperationLogManager.class);
    private static final String HISTORIC_DIR_SUFFIX = "_historic";
    private static String historicLogRootDir;
    private static long maxBytesToFetch;
    private final HiveConf hiveConf;
    private final OperationManager operationManager;
    private OperationLogDirCleaner cleaner;
    private String historicParentLogDir;
    private String serverInstance;

    public OperationLogManager(OperationManager operationManager, HiveConf hiveConf) {
        this.operationManager = operationManager;
        this.hiveConf = hiveConf;
        if (HiveConf.getBoolVar(hiveConf, HiveConf.ConfVars.HIVE_SERVER2_HISTORIC_OPERATION_LOG_ENABLED) && hiveConf.getBoolVar(HiveConf.ConfVars.HIVE_SERVER2_LOGGING_OPERATION_ENABLED) && hiveConf.isWebUiQueryInfoCacheEnabled()) {
            this.initHistoricOperationLogRootDir();
            maxBytesToFetch = HiveConf.getSizeVar(hiveConf, HiveConf.ConfVars.HIVE_SERVER2_HISTORIC_OPERATION_LOG_FETCH_MAXBYTES);
            if (historicLogRootDir != null && !HiveConf.getBoolVar(hiveConf, HiveConf.ConfVars.HIVE_IN_TEST)) {
                this.cleaner = new OperationLogDirCleaner();
                this.cleaner.start();
            }
        }
    }

    private String getServerInstance() {
        String hostname;
        try {
            hostname = ServerUtils.hostname();
        }
        catch (Exception e) {
            hostname = UUID.randomUUID().toString();
        }
        int serverPort = this.hiveConf.getIntVar(HiveConf.ConfVars.HIVE_SERVER2_THRIFT_PORT);
        return hostname + "_" + serverPort;
    }

    private void initHistoricOperationLogRootDir() {
        String origLogLoc = this.hiveConf.getVar(HiveConf.ConfVars.HIVE_SERVER2_LOGGING_OPERATION_LOG_LOCATION);
        File logLocation = new File(origLogLoc);
        this.historicParentLogDir = logLocation.getAbsolutePath() + HISTORIC_DIR_SUFFIX;
        this.serverInstance = this.getServerInstance();
        String logRootDir = this.historicParentLogDir + "/" + this.serverInstance + "_" + System.currentTimeMillis();
        File operationLogRootDir = new File(logRootDir);
        if (operationLogRootDir.exists() && !operationLogRootDir.isDirectory()) {
            LOG.warn("The historic operation log root directory exists, but it is not a directory: " + operationLogRootDir.getAbsolutePath());
            return;
        }
        if (!operationLogRootDir.exists() && !operationLogRootDir.mkdirs()) {
            LOG.warn("Unable to create historic operation log root directory: " + operationLogRootDir.getAbsolutePath());
            return;
        }
        historicLogRootDir = logRootDir;
    }

    public void deleteHistoricQueryLogs() {
        if (historicLogRootDir == null) {
            return;
        }
        File logDir = new File(historicLogRootDir);
        if (!logDir.exists() || !logDir.isDirectory()) {
            return;
        }
        File[] subDirs = logDir.listFiles();
        if (subDirs == null || subDirs.length == 0) {
            return;
        }
        Set sessionIds = this.operationManager.getHistoricalQueryInfos().stream().map(QueryInfo::getSessionId).collect(Collectors.toSet());
        Set queryIds = this.operationManager.getHistoricalQueryInfos().stream().map(queryInfo -> queryInfo.getQueryDisplay().getQueryId()).collect(Collectors.toSet());
        for (File dir : subDirs) {
            if (!dir.isDirectory()) continue;
            if (sessionIds.contains(dir.getName())) {
                for (File f : dir.listFiles()) {
                    if (queryIds.contains(f.getName())) continue;
                    LOG.debug("delete file not in hist: " + f.getName());
                    FileUtils.deleteQuietly((File)f);
                }
                continue;
            }
            FileUtils.deleteQuietly((File)dir);
        }
    }

    private void deleteElderLogRootDirs() {
        File[] children = new File(this.historicParentLogDir).listFiles(new FileFilter(){

            @Override
            public boolean accept(File child) {
                return child.isDirectory() && child.getName().startsWith(OperationLogManager.this.serverInstance) && !child.getAbsolutePath().equals(historicLogRootDir);
            }
        });
        if (children == null || children.length == 0) {
            return;
        }
        for (File f : children) {
            FileUtils.deleteQuietly((File)f);
        }
    }

    public void stop() {
        if (this.cleaner != null) {
            this.cleaner.shutDown();
        }
    }

    private static boolean isHistoricOperationLogEnabled(String logLocation) {
        if (logLocation == null || historicLogRootDir == null) {
            return false;
        }
        return logLocation.startsWith(historicLogRootDir);
    }

    public static void closeOperation(Operation operation) {
        String queryId = operation.getQueryId();
        File originOpLogFile = new File(operation.parentSession.getOperationLogSessionDir(), queryId);
        if (!originOpLogFile.exists()) {
            return;
        }
        HiveConf hiveConf = operation.queryState.getConf();
        if (hiveConf.getBoolVar(HiveConf.ConfVars.HIVE_SERVER2_HISTORIC_OPERATION_LOG_ENABLED) && operation instanceof SQLOperation) {
            String sessionHandle = operation.getParentSession().getSessionHandle().getHandleIdentifier().toString();
            String histOpLogFileLocation = historicLogRootDir + "/" + sessionHandle + "/" + queryId;
            try {
                FileUtils.moveFile((File)originOpLogFile, (File)new File(histOpLogFileLocation));
                QueryInfo queryInfo = ((SQLOperation)operation).getQueryInfo();
                queryInfo.setOperationLogLocation(histOpLogFileLocation);
                LOG.info("The operation log location changes from {} to {}.", (Object)originOpLogFile, (Object)histOpLogFileLocation);
            }
            catch (IOException e) {
                LOG.error("Failed to move operation log location from {} to {}: {}.", originOpLogFile, histOpLogFileLocation, e.getMessage());
            }
        } else {
            FileUtils.deleteQuietly((File)originOpLogFile);
        }
        LOG.debug(queryId + ": closeOperation");
    }

    public static String getOperationLog(QueryInfo queryInfo) {
        String logLocation = queryInfo.getOperationLogLocation();
        StringBuilder builder = new StringBuilder();
        if (logLocation == null) {
            return "Operation log is disabled, please set hive.server2.logging.operation.enabled = true to enable it";
        }
        if (historicLogRootDir == null) {
            builder.append("Operation Log - will be deleted after query completes, ").append("set hive.server2.historic.operation.log.enabled = true ").append("and hive.server2.webui.max.historic.queries > 0 to disable it").append(System.lineSeparator());
        }
        try (RandomAccessFile raf = new RandomAccessFile(logLocation, "r");){
            long fileLen = raf.length();
            long seekPos = 0L;
            if (fileLen > maxBytesToFetch) {
                seekPos = fileLen - maxBytesToFetch;
            }
            ByteBuffer buffer = ByteBuffer.allocate((int)maxBytesToFetch);
            int read = raf.getChannel().read(buffer, seekPos);
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(buffer.array(), 0, read)));){
                String line;
                while ((line = reader.readLine()) != null) {
                    builder.append(line).append(System.lineSeparator());
                }
            }
        }
        catch (Exception e) {
            builder.append(StringUtils.stringifyException(e));
        }
        return builder.toString();
    }

    @VisibleForTesting
    public static String getHistoricLogDir() {
        return historicLogRootDir;
    }

    private class OperationLogDirCleaner
    extends Thread {
        private final long interval;
        private boolean shutdown = false;
        private final Object monitor = new Object();

        OperationLogDirCleaner() {
            long checkInterval = HiveConf.getTimeVar(OperationLogManager.this.hiveConf, HiveConf.ConfVars.HIVE_SERVER2_HISTORIC_OPERATION_LOG_CHECK_INTERVAL, TimeUnit.MILLISECONDS);
            this.interval = Math.max(checkInterval, 3000L);
            this.setName("Historic-OperationLogDir-Cleaner");
            this.setDaemon(true);
        }

        @Override
        public void run() {
            OperationLogManager.this.deleteElderLogRootDirs();
            this.sleepFor(this.interval);
            while (!this.shutdown) {
                try {
                    OperationLogManager.this.deleteHistoricQueryLogs();
                    this.sleepFor(this.interval);
                }
                catch (Exception e) {
                    LOG.warn("OperationLogDir cleaner caught exception: " + e.getMessage(), e);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void sleepFor(long interval) {
            Object object = this.monitor;
            synchronized (object) {
                if (this.shutdown) {
                    return;
                }
                try {
                    this.monitor.wait(interval);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void shutDown() {
            Object object = this.monitor;
            synchronized (object) {
                this.shutdown = true;
                this.monitor.notifyAll();
            }
        }
    }
}

