/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.druid.msq.dart.worker.http;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.druid.msq.dart.controller.http.DartQueryInfo;
import org.apache.druid.msq.dart.worker.WorkerId;
import org.joda.time.DateTime;

import java.util.Objects;

/**
 * Class included in {@link GetWorkersResponse}.
 */
public class DartWorkerInfo
{
  private final String dartQueryId;
  private final WorkerId workerId;
  private final String controllerHost;
  private final DateTime startTime;

  public DartWorkerInfo(
      @JsonProperty("dartQueryId") final String dartQueryId,
      @JsonProperty("workerId") final WorkerId workerId,
      @JsonProperty("controllerHost") final String controllerHost,
      @JsonProperty("startTime") final DateTime startTime
  )
  {
    this.dartQueryId = dartQueryId;
    this.workerId = workerId;
    this.controllerHost = controllerHost;
    this.startTime = startTime;
  }

  /**
   * Dart query ID generated by the system. Globally unique.
   */
  @JsonProperty
  public String getDartQueryId()
  {
    return dartQueryId;
  }

  /**
   * Worker ID for this query.
   */
  @JsonProperty
  public WorkerId getWorkerId()
  {
    return workerId;
  }

  /**
   * Controller host:port that manages this query.
   */
  @JsonProperty
  public String getControllerHost()
  {
    return controllerHost;
  }

  /**
   * Time this query was accepted by this worker. May be somewhat later than the {@link DartQueryInfo#getStartTime()}
   * on the controller.
   */
  @JsonProperty
  public DateTime getStartTime()
  {
    return startTime;
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    DartWorkerInfo that = (DartWorkerInfo) o;
    return Objects.equals(dartQueryId, that.dartQueryId)
           && Objects.equals(workerId, that.workerId)
           && Objects.equals(controllerHost, that.controllerHost)
           && Objects.equals(startTime, that.startTime);
  }

  @Override
  public int hashCode()
  {
    return Objects.hash(dartQueryId, workerId, controllerHost, startTime);
  }
}
