/*
 * baseop.c
 *
 * AUTHORS: Robert Fahy <rfahy@ymail.com>
 * CREDITS: n/a
 * VERSION: 1.01
 *
 */

/*
 * TASK:
 * - define basic operations which are not intrinsic to XOR-FD
 *
 */

/*
 * TODO:
 * - n/a
 *
 */

// standard include files
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

// hack, hack, hack your code gently with a scream
// NOTE: these will cause some Warnings during compilation, ignore them
#define __MSVCRT__
#define __MSVCRT_VERSION__	0x0601
// environment-specific include files
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>

// custom include files
#include "globals.h"
#include "defaults.h"
#include "error.h"
#include "baseop.h"

// definitions that are used internally by `print_usage()'
// WARN: don't set them to 0 (zero) or other stupid values
#define DEFAULT_PROGRAM_FILENAME_SIZE	128
#define DEFAULT_MESSAGE_SIZE			256

void signals_setup()
// ABOUT: sets up the signals' "Rules of Engagement"
{
	signal(SIGINT, signal_handler);
	signal(SIGTERM, signal_handler);
} // ### void signals_setup()

void signal_handler(int signal)
// ABOUT: modifies `_global_run_flag_' so that the program can quit cleanly
{
	switch (signal)
	{
		case SIGINT:
		case SIGTERM:
			_global_run_flag_ = XORFD_ABORT;
			break;

		default: fprintf(stderr, "\nThis should have never happened.\n");
	}
} // ### void signal_handler(int signal)

void print_usage(const char *program_filename, const char *message)
// ABOUT: prints the usage information and `message' to Standard Error stream (stderr)
{
	char // FIXME: what if this fails?
		program_filename_[DEFAULT_PROGRAM_FILENAME_SIZE],
		message_[DEFAULT_MESSAGE_SIZE];

	// program filename sanity check
	if (program_filename != NULL &&
		*program_filename != '\0')
		strncpy(program_filename_, program_filename, sizeof program_filename_ - 1);
	else
		strncpy(program_filename_, DEFAULT_PROGRAM_FILENAME, sizeof program_filename_ - 1);

	// ensure NIL-termination of array
	program_filename_[sizeof program_filename_ - 1] = '\0';

	// message sanity check
	if (message != NULL &&
		*message != '\0')
		strncpy(message_, message, sizeof message_ - 1);
	else
		strncpy(message_, DEFAULT_ERROR_MESSAGE, sizeof message_ - 1);

	// ensure NIL-termination of array
	message_[sizeof message_ - 1] = '\0';

	fprintf(stderr, "\n\n%s usage:\n\n %s [file_to_be_encrypted] <[resulting_file] [file_used_as_key]>\n"
		"Also accepting standard input (stdin) data as key if `file_used_as_key'\n isn't provided.\n"
		"(Maximum stdin key length is %d bytes.)\n\nERROR: %s\n\n\n",
		DEFAULT_NAME_VERSION, program_filename_, STDIN_BUFFER_SIZE, message_);
} // ### void print_usage(const char *program_filename, const char *message)

unsigned char get_percentage(const off64_t full, const off64_t current)
// ABOUT: computes how much percent `current' represents from `full'
// NOTE: sanity of parameters not verified, so this can return values of over 100 (%)
//  this is so that users will notice if anything goes crazy, instead of getting stuck
//  at an innocuous "100%"
{
	if (full) // prevent division by zero
	{
		const off64_t result_ = current * 100 / full;

		// verify if `result_' does not fit into a byte
		if (result_ > 255 ||
			result_ < 0)
			return 255;

		return result_;
	}

	return 255;
} // ### unsigned char get_percentage(const off64_t full, const off64_t current)

off64_t get_file_size_tame(FILE *file)
// ABOUT: same as `get_file_size()' but doesn't `exit()' on error and doesn't try to close the `file'
// NOTE: this function might not be portable to non-Windows operating systems
{
	int file_descriptor_;
	struct __stat64 file_stat_;

	// check for error at file descriptor retrieval
	if ((file_descriptor_ = fileno(file))
		== -1)
	{
		PRINT_ERROR("fileno()");
		return -1;
	}

	// check for error at file status completion
	if (_fstat64(file_descriptor_, &file_stat_) == -1)
	{
		PRINT_ERROR("_fstat64()");
		return -1;
	}

	return file_stat_.st_size;
} // ### off64_t get_file_size_tame(FILE *file)

off64_t get_file_size(FILE *file)
// ABOUT: retrieve the file size of `file'
// NOTE: this function might not be portable to non-Windows operating systems
{
	int file_descriptor_;
	struct __stat64 file_stat_;
	bool failed_ = false;

	// check for error at file descriptor retrieval
	if ((file_descriptor_ = fileno(file))
		== -1)
	{
		PRINT_ERROR("fileno()");
		failed_ = true;
	}

	// check for error at file status completion
	if (_fstat64(file_descriptor_, &file_stat_) == -1)
	{
		PRINT_ERROR("_fstat64()");
		failed_ = true;
	}

	// check for function failure, in which case close the file and exit
	if (failed_ == true)
	{
		FCLOSE_VERBOSE(file); // check for error at file closing attempt
		exit(EXIT_FAILURE);
	}

	return file_stat_.st_size;
} // ### off64_t get_file_size(FILE *file)
