/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* BUTTON.C --- Buttons and selectors. */

#include <stdlib.h>
#include <assert.h>
#include <graphics.h>
#include <time.h>
#include "window.h"
#include "key.h"
#include "resource.h"

enum {

button_light_color = cWHITE,
button_dark_color = cDARKGRAY,
button_no_focus_border_color = cBLACK,
button_focus_border_color = cLIGHTRED,
button_button = 0,
selector_border_width = 1,

};

/* Draw a button with colors based on its status. */
void draw_button(BUTTON b)
{
  void *image;
  int i, fg, bg, top_left, bot_rght, sw;
  WINDOW w = &b->window;

  if (!w_status_p(w, wsVISIBLE))
    return;

  if (b_status_p(b, bENABLED)) {
    if (b_status_p(b, bON)) {
      fg = b->bg_color;
      bg = b->fg_color;
      top_left = button_dark_color;
      bot_rght = button_light_color;
    }
    else {
      fg = b->fg_color;
      bg = b->bg_color;
      top_left = button_light_color;
      bot_rght = button_dark_color;
    }
    set_window_border_color(w, b_status_p(b, bHAVE_FOCUS) ?
				 button_focus_border_color :
				 button_no_focus_border_color);
  }
  else {
    fg = b->fg_color;
    top_left = bot_rght = bg = b->bg_color;
  }

  sw = b->shadow_width;
  push_graphics_state(w, 0);
  setlinestyle(SOLID_LINE, 0, 1);
  protect_cursor(w);

  for (i = 0; i < sw; ++i) {
    int x1 = w->width - i - 1;
    int y1 = w->height - i - 1;
    setcolor(top_left);
    line(i, i, x1, i);
    line(i, i, i, y1);
    setcolor(bot_rght);
    line(i, y1, x1, y1);
    line(x1, i, x1, y1);
  }

  if (b_status_p(b, bBITMAP)) {
    image = make_image(b->icon, b->width, b->height, fg, bg);
    putimage(sw, sw, image, COPY_PUT);
    free(image);
  }
  else {
    setfillstyle(SOLID_FILL, bg);
    bar(sw, sw, sw + b->width - 1, sw + b->height - 1);
    setcolor(fg);                         
    settextjustify(CENTER_TEXT, CENTER_TEXT);
    out_hot_textxy(b->width/2 + sw, b->height/2 + sw, b->icon);
  }

  unprotect_cursor();
  pop_graphics_state();
}

/* Handler called when button window is mapped. */
void handle_button_map(EVENT e)
{
  draw_button((BUTTON)e->map.window);
}

/* Clear all buttons in a selector. */
LOCAL(void) clear_selector(BUTTON b)
{
  int i;
  SELECTOR s = (SELECTOR)b->window.parent;
  for (i = 0; i < s->n_buttons; ++i) {
    BUTTON sb = s->buttons + i;
    if (sb != b && b_status_p(sb, bON)) {
      sb->status &= notbit(bON);
      draw_button(sb);
    }
  }
}

/* Handler called with generalized press or
   release occur on button. */
LOCAL(void) handle_button_button(EVENT e)
{
  BUTTON b = (BUTTON)e->mouse.window;

  if (e->mouse.button != button_button)
    return;

  if (e->type == eBUTTON_PRESS) {
    if (b_status_p(b, bENABLED) && !b_status_p(b, bON)) {
      if (b_status_p(b, bSELECTOR)) 
	clear_selector(b);
      b->status |= bit(bON);
      draw_button(b);
      if (b_status_p(b, bTOGGLE))
	(*b->action.code)(b->action.env);
      else if (b_status_p(b, bREPEAT))
	begin_repeat(b->action.code, b->action.env);
    }
  }
  else if (e->type == eBUTTON_RELEASE && !b_status_p(b, bTOGGLE) && b_status_p(b, bON)) {
    b->status &= notbit(bON);
    draw_button(b);
    if (b_status_p(b, bREPEAT))
      end_repeat();
    else if (b_status_p(b, bENABLED))
      (*b->action.code)(b->action.env);
  }
}

LOCAL(void) handle_button_gain_focus(EVENT e)
{
  BUTTON b = (BUTTON)e->mouse.window;
  b->status |= bit(bHAVE_FOCUS);
  draw_button(b);
}

LOCAL(void) handle_button_lose_focus(EVENT e)
{
  BUTTON b = (BUTTON)e->mouse.window;
  b->status &= notbit(bHAVE_FOCUS);
  draw_button(b);
}

/* Make believe mouse pressed button, when
   a key really did it.  We don't honor
   repeats here because we count on the
   keyboard.  We momentarily draw the button
   depressed, too. */
void feign_button_press(BUTTON b)
{
  if (b == NULL)
    return;
  if (b_status_p(b, bSELECTOR)) 
    clear_selector(b);
  b->status |= bit(bON);
  draw_button(b);
  /* Set flag before call because handler may 
     unwind stack and return elsewhere. */
  if (!b_status_p(b, bTOGGLE)) 
    b->status &= notbit(bON);
  (*b->action.code)(b->action.env);
  if (!b_status_p(b, bTOGGLE)) 
    draw_button(b);
}

LOCAL(void) handle_button_key(EVENT e)
{
  BUTTON b = (BUTTON)e->keystroke.window;
  switch (e->keystroke.key) {

    case '\t':
      unset_focus(e->keystroke.window);
      break;

    case ' ':
    case '\r':
      feign_button_press(b);
      break;

    default:
      refuse_keystroke(e);
      break;
  }
}

LOCAL(void) handle_button_hotkey(EVENT e)
{
  BUTTON b = (BUTTON)e->keystroke.window;
  if (hot_p(e->keystroke.key, b->hot_key) || 
      !b_status_p(b, bBITMAP) && key_hot_p(e->keystroke.key, b->icon))
    feign_button_press(b);
}

/* Return the width a button will be when created. */
int button_width(BUTTON b)
{
  return(2*b->shadow_width + b->width);
}

/* Return the height of a button without creating it. */
int button_height(BUTTON b)
{
  return(2*b->shadow_width + b->height);
}

/* Event dispatcher. */
BeginDefDispatch(button)
  Dispatch(eMAP, handle_button_map)
  Dispatch(eBUTTON_PRESS And eBUTTON_RELEASE, handle_button_button)
  Dispatch(eGAIN_FOCUS, handle_button_gain_focus)
  Dispatch(eLOSE_FOCUS, handle_button_lose_focus)
  Dispatch(eKEYSTROKE, handle_button_key)
  Dispatch(eVISIBLE_HOTKEY, handle_button_hotkey)
EndDefDispatch(button)

/* Make a new button. */
void open_button(BUTTON b, int x, int y, WINDOW parent)
{
  unsigned event_mask;

  event_mask = bit(eMAP);
  if (b_status_p(b, bENABLED))
    event_mask |= bit(eBUTTON_PRESS)|bit(eBUTTON_RELEASE)
		 |bit(eGAIN_FOCUS)|bit(eLOSE_FOCUS)
		 |bit(eKEYSTROKE)|bit(eVISIBLE_HOTKEY);

  open_window(&b->window, parent, 
	      x, y, button_width(b), button_height(b),
	      b->border_width, button_no_focus_border_color, NO_COLOR, 
	      event_mask);
  SetDispatch(&b->window, button);
}

/* Enable a button for pressing. */
void enable_button(BUTTON b)
{
  if (b_status_p(b, bENABLED))
    return;
  b->window.event_mask |= bit(eBUTTON_PRESS)|bit(eBUTTON_RELEASE)
			 |bit(eKEYSTROKE)|bit(eVISIBLE_HOTKEY)
			 |bit(eGAIN_FOCUS)|bit(eLOSE_FOCUS);
  b->status |= bit(bENABLED);
  draw_button(b);
}

LOCAL(void) advance_focus(SELECTOR s)
{
  int f;

# define ENABLED_OR_ON (bit(bENABLED)|bit(bON))

  /* Set focus to next button that's enabled. */  
  for (f = s->focus + 1; f < s->n_buttons; ++f) 
    /* If button is enabled and not on. */
    if ((s->buttons[f].status & ENABLED_OR_ON) == bit(bENABLED)) {
      s->focus = f;
      set_focus(&s->buttons[f].window);
      return;
    }
  /* Refuse focus. */
  s->focus = -1;
  unset_focus((WINDOW)s);
}

/* Disable a button so it can't be pressed. */
void disable_button(BUTTON b)
{
  if (!b_status_p(b, bENABLED))
    return;
  if (b_status_p(b, bON) && b_status_p(b, bREPEAT))
    end_repeat();
  b->window.event_mask &= notbit(eBUTTON_PRESS)&notbit(eBUTTON_RELEASE)& 
			  notbit(eKEYSTROKE)&notbit(eVISIBLE_HOTKEY)&notbit(eGAIN_FOCUS)&notbit(eLOSE_FOCUS);
  b->status &= notbit(bENABLED)&notbit(bON);
  unset_focus(&b->window);
#define FOCUS_SELECTOR (bit(bSELECTOR)|bit(bHAVE_FOCUS))
  if ((b->status & FOCUS_SELECTOR) == FOCUS_SELECTOR)
    advance_focus((SELECTOR)b->window.parent);
  draw_button(b);
}

/* DEBUG: This is currently correct only for buttons in selectors. */
void set_button_toggle(BUTTON b, unsigned mask)
{
  int selector_and_toggle = mask & (bit(bSELECTOR) | bit(bTOGGLE));

  if (mask & bit(bON)) 
    b->status |= selector_and_toggle;
  else {
    b->status &= ~selector_and_toggle;
    if (b_status_p(b, bON)) {
      b->status &= notbit(bON);
      draw_button(b);
    }
  }
}

BeginDefDispatch(selector)
  DispatchAction(eGAIN_FOCUS, advance_focus((SELECTOR)e->focus.window));
EndDefDispatch(selector)

/* Selectors.  These are aggregates of buttons arranged in
   either a vertical or horizontal stack.  If the Selector
   bit is set in a button and it is clicked, all other buttons
   that are currently toggled down are popped back up. */

/* Make a new selector. */
void open_selector(SELECTOR s, int x, int y, WINDOW parent)
{
  int bx, by, width, height;
  BUTTON b, end;

  /* Make the window too big then shrink it to surround the buttons. */
  open_window(&s->window, parent, x, y, 20000, 20000,
	      selector_border_width, button_no_focus_border_color, NO_COLOR, bit(eGAIN_FOCUS));
  SetDispatch(&s->window, selector);
  width = height = 0;
  for (b = s->buttons, end = b + s->n_buttons, bx = by = 0; b < end; ++b) {
    open_button(b, bx, by, &s->window);
    map_window(&b->window);
    if (s_status_p(s, sVERTICAL)) {
      by += b->border_width + button_height(b);
      height = by;
      width = max(width, button_width(b));
    }
    else {
      bx += b->border_width + button_width(b);
      width = bx;
      height = max(height, button_height(b));
    }
  }
  if (s_status_p(s, sVERTICAL))
    height -= (end - 1)->border_width;
  else
    width -= (end - 1)->border_width;
  s->window.width = width;
  s->window.height = height;
  s->focus = -1;
}
