/*
    REFERENCES -- bibliographic software
    Copyright (C) 1995-2007  Volker Kiefel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA
*/

/*
   search.c: queries for REFERENCES database
   version 4.2
*/

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include "search.h"
#include "refs.h"
#include "str_fun.h"
#include "interf.h"
// #include "rglobal.h"
#include "fl_ut.h"
#include "ix01_fun.h"
#include "sortbbf1.h"
#include "bbf_fun.h"
#include "initial.h"


#define SEARCHCMD_LEN    800
#define MELDUNG_LEN     5000


 char zahl_ex[ZAHLSTRING_LEN +1];
 char zahl_exex[ZAHLSTRING_LEN +1];



 extern journal_rec one_journal_rec;
 extern buch_rec one_buch_rec;
 extern quell_rec one_quell_rec;
 extern keylist_rec one_key_rec;
 extern refspix_rec ix01rec;
 extern quellpix_rec ix02rec;
 extern keylistpix_rec ix03rec;

 extern config_set one_config_set;
 extern char wrtxtbu[WRTXTBU_LEN+1];

 char fpjournalbuf[sizeof(one_journal_rec)];
 char fpbuchbuf[sizeof(one_buch_rec)];

 extern char textpfad[PFAD_LEN+1];
 extern char rdbpfad[PFAD_LEN+1];

 int c_insens; /* case-insensitive search? 1 yes, 0 no */

 int max_oder; /* max number of OR (German ODER) conditions. If max_oder==1:
                  not required that bbt_datei_name be sorted!  */

 extern char journal_pix_name[PFAD_LEN + 1];
 extern char journal_name[PFAD_LEN + 1];
 extern char desc_name[PFAD_LEN + 1];
 extern char book_name[PFAD_LEN + 1];
 extern char autorjl_name[PFAD_LEN + 1];
 extern char autorbu_name[PFAD_LEN + 1];
 extern char quell_pix_name[PFAD_LEN + 1];
 extern char quell_name[PFAD_LEN + 1];
 extern char stichw_name[PFAD_LEN + 1];
 extern char stichw_pix_name[PFAD_LEN + 1];
 extern char archiv_datei_name[PFAD_LEN + 1];

 char bbt_datei_name[PFAD_LEN + 1];  /* File (output) with result of search */
 char bbt_in_datei_name[PFAD_LEN + 1]; /* File with records to be searched 
                                          in */
 char tmpix_name[PFAD_LEN + 1];
 char newix_name[PFAD_LEN + 1];


 extern FILE * ix01file;

 extern FILE * fpjournal;
 extern FILE * fpbuch;
 extern FILE * fpdesc;
 extern FILE * fpautorjl;
 extern FILE * fpautorbu;
 extern FILE * fparchiv;
  static FILE * tmpix;
  static FILE * newix;
 extern FILE * fpquell;
 extern FILE * fpstichw;


 char * in_zeile;
 char * ex_zeile;
 char * exex_zeile;
 char * exexex_zeile;
 char * indat_zeile;
 char * alloc_test_zeile;
 
 static int search_initialize(void);
 static int search_aufraeumen(void);
 static int search_datenbankdateien_oeffnen(void);
 static int search_datenbankdateien_schliessen(void);
 static int search_textdatei_lesen(char * suchkommando, char * bbt_ausgabedatei);
 static int search_kommando_eingeben(char * suchkommando, char * bbt_ausgabedatei);
 static void search_text_zeigen(char * txt);
 static int check_command(char * zeile);
 static void journal_search(const char * fieldlabel);
 static int jahreszahlok(char * suchstring, char * feldwert);
 static void buch_search(const char * fieldlabel);
 static void autorjl_search(const char * fieldlabel);
 static void autorbu_search(const char * fieldlabel);
 static void desc_search(const char * fieldlabel);
 static void ix2bbt(const char * ix_name, const char * bbt_name);
 static int bbt2ix(char * dateiname);
 static void int2fn(int zahl, char * dateiname);
 static int steuerung(char * zeile);


int search_last_bbt(char * out_bbt_file)
{
   int rueckgabewert;

   rueckgabewert = 1;
   search_initialize();
   if (search_textdatei_lesen(indat_zeile, bbt_datei_name)==1)
   {
       if (!isfile(bbt_datei_name))
       {
         sprintf(wrtxtbu,"\nFile `%s' does not exist\n",bbt_datei_name);
         wrtxt(wrtxtbu);
         rueckgabewert = 0;
       }
       else
       {
         strlimcpy(out_bbt_file,bbt_datei_name,PFAD_LEN);
       }
   }
   else
   {
       rueckgabewert = 0;
   }
   search_aufraeumen();
   return rueckgabewert;
}
 
 
 /* out_bbt_file is the filename with complete path */

int search_main(char * out_bbt_file)
{
   int rueckgabewert;

   rueckgabewert = 1;
   
   strlimcpy(out_bbt_file,rdbpfad,PFAD_LEN);
   strlimcat(out_bbt_file,"",PFAD_LEN);

   search_initialize();
   if (search_textdatei_lesen(indat_zeile, bbt_datei_name)==1 ||
       search_kommando_eingeben(indat_zeile, bbt_datei_name))
   {
     static char ein[INPUT_BU_LEN+1];
     int option;
     
     option = 0;
     strcpy(ein,"menu");
     
     wrtxt("\n");
     wrtxt("Search command:\n  ");
     wrtxt(indat_zeile);
     wrtxt("\n\n");
     wrtxt("Output file:\n  ");
     wrtxt(bbt_datei_name);
     wrtxt("\n");

     strlimcpy(out_bbt_file,bbt_datei_name,PFAD_LEN);

     wrtxt("\nMenu: search complete database/search from bbt-file [c/b/q]\n");
     option = menu_wahl(ein,"[c] search complete database\n"
                            "[b] search records referred to in bbt-file\n"
                            "[q] quit");
     if (option==3)
     {
        rueckgabewert = 0;
        search_aufraeumen();
        return rueckgabewert;
     }
     if (option==2)
     {
       if (!datei_suchen_mit_pfad("*.bbt",rdbpfad,bbt_in_datei_name))
       {
         wrtxt("\n(No bbt-file selected - searching complete database)\n");
         strcpy(bbt_in_datei_name,"");
       }
     }
     else
     {
        strcpy(bbt_in_datei_name,"");
     }
     strcpy(ein,"menu");
     wrtxt("\nMenu: Search case-sensitive/insensitive [s/i/q]\n");
     option = menu_wahl(ein,"[s] search case-sensitive\n"
                            "[i] search case-insensitive\n"
                            "[q] quit");
     if (option==3)
     {
        rueckgabewert = 0;
        search_aufraeumen();
        return rueckgabewert;
     }
     if (option==2)
     {
        c_insens = 1;
     }
     else
     {
        c_insens = 0;
     }
     if (!check_command(indat_zeile))
     {
        rueckgabewert = 0;
     }
     else
     {
       search_datenbankdateien_oeffnen();
       /* neu 22.6.2002 naechste Zeile */
       max_oder = 0;
       rueckgabewert = steuerung(indat_zeile);
       if (rueckgabewert)
       {
          logdatei_oeffnen();
          sprintf(wrtxtbu,"Search command `%s'",indat_zeile);
          logdatei_schreiben(wrtxtbu);
          sprintf(wrtxtbu,"Result written into `%s'",bbt_datei_name);
          logdatei_schreiben(wrtxtbu);
          logdatei_schliessen();
       }
       search_datenbankdateien_schliessen();
     }
     
   }
   else
   {
      rueckgabewert = 0;
   }
   search_aufraeumen();

   if (!isfile(out_bbt_file))
   {
      sprintf(wrtxtbu,"\nFile `%s' not written\n",out_bbt_file);
      wrtxt(wrtxtbu);
      rueckgabewert=0;
   }
   return rueckgabewert;
}



static int search_initialize(void)
{
   // Name der Textdatei mit dem Suchkommando
   if  ((in_zeile = (char *) malloc((SEARCHCMD_LEN+1) * sizeof(char)))==NULL)
   {
      fprintf(stderr,"\nERROR in search_initialize(): "
        "cannot allocate memory for in_zeile\n");
      exit(1);
   }
   if  ((ex_zeile = (char *) malloc((SEARCHCMD_LEN+1) * sizeof(char)))==NULL)
   {
      fprintf(stderr,"\nERROR in search_initialize(): "
        "cannot allocate memory for ex_zeile\n");
      exit(1);
   }
   if  ((exex_zeile = (char *) malloc((SEARCHCMD_LEN+1) * sizeof(char)))==NULL)
   {
      fprintf(stderr,"\nERROR in search_initialize(): "
        "cannot allocate memory for exex_zeile\n");
      exit(1);
   }
   if  ((exexex_zeile = (char *) malloc((SEARCHCMD_LEN+1) * sizeof(char)))==NULL)
   {
      fprintf(stderr,"\nERROR in search_initialize(): "
        "cannot allocate memory for exexex_zeile\n");
      exit(1);
   }
   if  ((indat_zeile = (char *) malloc((SEARCHCMD_LEN+1) * sizeof(char)))==NULL)
   {
      fprintf(stderr,"\nERROR in search_initialize(): "
        "cannot allocate memory for indat_zeile\n");
      exit(1);
   }

   return 1;
}

static int search_aufraeumen(void)
{
   free(in_zeile);
   free(ex_zeile);
   free(exex_zeile);
   free(exexex_zeile);
   free(indat_zeile);

   return 1;
}


static int search_datenbankdateien_oeffnen(void)
{
   if ((fpjournal=fopen(journal_name,"rb"))==NULL)
   {
      fprintf(stderr,"\nERROR: Cannot read %s\n",journal_name);
      exit(1);
   }

   if (setvbuf(fpjournal,fpjournalbuf,_IOFBF,sizeof(one_journal_rec))!=0)
   {
      fprintf(stderr,"\nERROR: cannot allocate memory "
         "for fpjournalbuf\n");
      exit(1);
   }

   if ((fpdesc=fopen(desc_name,"rb"))==NULL)
   {
      fprintf(stderr,"\nERROR: Cannot read %s\n",desc_name);
      exit(1);
   }


   if ((fpbuch=fopen(book_name,"rb"))==NULL)
   {
      fprintf(stderr,"\nERROR: Cannot read %s\n",book_name);
      exit(1);
   }
   if (setvbuf(fpbuch,fpbuchbuf,_IOFBF,sizeof(one_buch_rec))!=0)
   {
      fprintf(stderr,"\nERROR: cannot allocate memory for "
         "fpbuchbuf\n");
      exit(1);
   }

   if ((fpautorjl=fopen(autorjl_name,"rb"))==NULL)
   {
      fprintf(stderr,"\nERROR: Cannot read %s\n",autorjl_name);
      exit(1);
   }
   if ((fpautorbu=fopen(autorbu_name,"rb"))==NULL)
   {
       fprintf(stderr,"\nERROR: Cannot read %s\n",
          autorbu_name);
      exit(1);
   }
 return 1;
 
}

static int search_datenbankdateien_schliessen(void)
{
   fclose(fpjournal);
   fclose(fpdesc);
   fclose(fpbuch);
   fclose(fpautorjl);
   fclose(fpautorbu);

   return 1;
}

static int search_textdatei_lesen(char * suchkommando, char * bbt_ausgabedatei)
{
   char such_textdatei_name[PFAD_LEN+1];
   FILE * fptxt;
   int ret_value;
   char zeile[SEARCHCMD_LEN+1];

   ret_value=1;
   
   strlimcpy(such_textdatei_name,textpfad,PFAD_LEN);
   strlimcat(such_textdatei_name,one_config_set.search_form_name,PFAD_LEN);

   /* read search command */
   if ((fptxt=fopen(such_textdatei_name,"r"))==NULL)
   {
      sprintf(wrtxtbu,"\nSearch command file (%s) cannot be opened\n",
        such_textdatei_name); 
      wrtxt(wrtxtbu);
      ret_value = 0;
      return ret_value;
   }

   /* read output file name */
   if (fgets(zeile,SEARCHCMD_LEN,fptxt)!=NULL)
   {
      rtrim(zeile,'\n');
      trim(zeile,' ');
      strlimcpy(suchkommando,zeile,SEARCHCMD_LEN);
      if (strlen(suchkommando)==0)
      {
         wrtxt("\nSearch command line in search command file is empty\n"
           "  References is unable to process search command file\n");
         ret_value = 0;
      }
   }
   else 
   {
      sprintf(wrtxtbu,"\nSearch command in search command file cannot be read\n"
        "  References is unable to process search command file\n"); 
      wrtxt(wrtxtbu);
      ret_value = 0;
   }
   if ((ret_value==1) && (fgets(zeile,SEARCHCMD_LEN,fptxt)!=NULL))
   {
      rtrim(zeile,'\n');
      trim(zeile,' ');
      if (strlen(zeile) == 0)
      {
         strcpy(zeile,"answer");
      }
      if (isvalidfilename(zeile)==0)
      {
         sprintf(wrtxtbu,"\nProblem: `%s' is not a valid filename\n",zeile); 
         wrtxt(wrtxtbu);
         ret_value = 0;
      }
   }
   else
   {
      strcpy(zeile,"answer");
   }
   fclose(fptxt);

   strlimcpy(bbt_ausgabedatei,rdbpfad,PFAD_LEN); 
   strlimcat(bbt_ausgabedatei,zeile,PFAD_LEN);
   strlimcat(bbt_ausgabedatei,".bbt",PFAD_LEN);
   return ret_value;
}

static int search_kommando_eingeben(char * suchkommando, char * bbt_ausgabedatei)
{
   int ret_value;
   char zeile[SEARCHCMD_LEN+1];

   ret_value=1;

   wrtxt("\nPlease enter valid search command directly\n");
   ein_zeile(zeile,"Search command",SEARCHCMD_LEN);
   if (strlen(zeile) < 1)
   {
      ret_value = 0;
      wrtxt("\n(Searching procedure aborted)\n");
      return ret_value;
   }
   
   strlimcpy(suchkommando,zeile,SEARCHCMD_LEN);
   
   wrtxt("\nPlease enter name for (bbt) output file, empty line for default: `answer'\n");
   ein_zeile(zeile,"Output file, `.bbt' will be appended",SEARCHCMD_LEN);
   
   if (strlen(zeile) == 0)
   {
      strcpy(zeile,"answer");
   }
   if (isvalidfilename(zeile)==0)
   {
      sprintf(wrtxtbu,"\nProblem: `%s' is not a valid filename\n",zeile); 
      wrtxt(wrtxtbu);
      ret_value = 0;
      return ret_value;
   }
   strlimcpy(bbt_ausgabedatei,rdbpfad,PFAD_LEN); 
   strlimcat(bbt_ausgabedatei,zeile,PFAD_LEN);
   strlimcat(bbt_ausgabedatei,".bbt",PFAD_LEN);
   return ret_value;
   
}



static void search_text_zeigen(char * txt)
{
     wrtxt("\n");
     wrtxt(txt);
     wrtxt("\n");
}


static int check_command(char * zeile)
{
   char meldung[MELDUNG_LEN+1];
   int anzahl_und, anzahl_oder, anzahl_at;
   int i, j;
   /* length of message in string meldung will never be longer than 
      strlen(WARNUNGNN) + 256 
    */
   assert((strlen(WARNUNG02) + 256) < MELDUNG_LEN);
   assert((strlen(WARNUNG01) + 256) < MELDUNG_LEN);
   assert((strlen(WARNUNG03) + 256) < MELDUNG_LEN);
   assert((strlen(WARNUNG04) + 256) < MELDUNG_LEN);
   assert((strlen(WARNUNG02F) + 256) < MELDUNG_LEN);
   assert((strlen(WARNUNG02A) + 256) < MELDUNG_LEN);
   assert((strlen(WARNUNG02B) + 256) < MELDUNG_LEN);
   assert((strlen(WARNUNG02C) + 256) < MELDUNG_LEN);
   assert((strlen(WARNUNG02D) + 256) < MELDUNG_LEN);
   assert((strlen(WARNUNG02E) + 256) < MELDUNG_LEN);

   if (strlen(zeile) < 1)
   {
      search_text_zeigen(WARNUNG00);
      return 0;
   }
   
   anzahl_und = countitem(zeile,"&");

   for (i = 1;i<=anzahl_und;i++)
   {
      extractitem(i,zeile,"&",ex_zeile);
      anzahl_at = countitem(ex_zeile,"=");
      if (anzahl_at!=2)
      {
         sprintf(meldung,"%s [%i.'&-period']",WARNUNG01,i);
         wwrap(meldung,SCREENLINE_LEN);
         search_text_zeigen(meldung);
         /// SCROLL2;
         return 0;
      }
      extractitem(1,ex_zeile,"=",exex_zeile);
      trim(exex_zeile,' ');
      upper(exex_zeile);
      if ( strcmp(exex_zeile,"KEYW")
          && strcmp(exex_zeile,"AUTH")
          && strcmp(exex_zeile,"INAU")
          && strcmp(exex_zeile,"EDTN")
          && strcmp(exex_zeile,"EDIT")
          && strcmp(exex_zeile,"INED")
          && strcmp(exex_zeile,"TITA")
          && strcmp(exex_zeile,"TITB")
          && strcmp(exex_zeile,"HOWP")
          && strcmp(exex_zeile,"DTYP")
          && strcmp(exex_zeile,"YEAR")
          && strcmp(exex_zeile,"JRNL")
          && strcmp(exex_zeile,"IDNR")
          && strcmp(exex_zeile,"STAT")
          && strcmp(exex_zeile,"VOLJ")
          && strcmp(exex_zeile,"PUBL")
          && strcmp(exex_zeile,"PLAC")
          )
      {
         sprintf(meldung,"%s [%i.'&-period']",WARNUNG02,i);
         wwrap(meldung,SCREENLINE_LEN);
         search_text_zeigen(meldung);
         /// SCROLL2;
         return 0;
      }

      extractitem(2,ex_zeile,"=",exex_zeile);
      trim(exex_zeile,' ');
      if (!strlen(exex_zeile))
      {
         sprintf(meldung,"%s [%i.'&-period']",WARNUNG03,i);
         wwrap(meldung,SCREENLINE_LEN);
         search_text_zeigen(meldung);
         /// SCROLL2;
         return 0;
      }

      /* ein leeres oder-Kriterium abfangen... */
      anzahl_oder = countitem(exex_zeile,"|");
      for (j=1;j<=anzahl_oder;j++)
      {
          extractitem(j,exex_zeile,"|",exexex_zeile);
          trim(exexex_zeile,' ');
          if (!strlen(exexex_zeile))
          {
             sprintf(meldung,"%s [%i.'&-period', %i.'|-period']",WARNUNG04,i,j);
             wwrap(meldung,SCREENLINE_LEN);
             search_text_zeigen(meldung);
             /// SCROLL2;
             return 0;
          }
      }

      extractitem(1,ex_zeile,"=",exex_zeile);
      trim(exex_zeile,' ');
      upper(exex_zeile);
      if (!strcmp(exex_zeile,"DTYP"))
      {
          extractitem(2,ex_zeile,"=",exex_zeile);
          anzahl_oder=countitem(exex_zeile,"|");
          trim(exex_zeile,' ');
          for (j=1;j<=anzahl_oder;j++)
          {
              extractitem(j,exex_zeile,"|",exexex_zeile);
              trim(exexex_zeile,' ');
              upper(exexex_zeile);
              if (
                   strcmp(exexex_zeile,"J1") &&
                   strcmp(exexex_zeile,"J2") &&
                   strcmp(exexex_zeile,"B1") &&
                   strcmp(exexex_zeile,"B2") &&
                   strcmp(exexex_zeile,"B3") &&
                   strcmp(exexex_zeile,"M1") &&
                   strcmp(exexex_zeile,"M2")
                 )
              {
                  sprintf(meldung,"%s [%i.'&-period', %i.'|-period']",
                     WARNUNG02F,i,j);
                  wwrap(meldung,SCREENLINE_LEN);
                  search_text_zeigen(meldung);
                  /// SCROLL2;
                  return 0;
              }
          }
      }

      extractitem(1,ex_zeile,"=",exex_zeile);
      trim(exex_zeile,' ');
      upper(exex_zeile);

      if (!strcmp(exex_zeile,"YEAR"))
      {
          extractitem(2,ex_zeile,"=",exex_zeile);
          anzahl_oder=countitem(exex_zeile,"|");
          trim(exex_zeile,' ');
          for (j=1;j<=anzahl_oder;j++)
          {
              extractitem(j,exex_zeile,"|",exexex_zeile);
              trim(exexex_zeile,' ');
              if (strlen(exexex_zeile) >= ZAHLSTRING_LEN)
              {
                  sprintf(meldung,"%s [max %i characters allowed: %i.'&-period', %i.'|-period']",
                     WARNUNG02A,ZAHLSTRING_LEN,i,j);
                  wwrap(meldung,SCREENLINE_LEN);
                  search_text_zeigen(meldung);
                  /// SCROLL2;
                  return 0;
              }
              strcpy(zahl_ex,exexex_zeile);
              trim(zahl_ex,' ');
              if (countitem(zahl_ex,"-")==1)
              {
                 int ok;
                 long jahr;
                 string_in_long(zahl_ex,&jahr,&ok);
                 if (!ok)
                 {
                   sprintf(meldung,"%s [%i.'&-period', %i.'|-period']",
                      WARNUNG02B,i,j);
                   wwrap(meldung,SCREENLINE_LEN);
                   search_text_zeigen(meldung);
                   /// SCROLL2;
                   return 0;
                 }
              }

              if (countitem(zahl_ex,"-")==2)
              {
                 int ok;
                 long jahr1, jahr2;
                 extractitem(1,zahl_ex,"-",zahl_exex);
                 trim(zahl_exex,' ');
                 string_in_long(zahl_exex,&jahr1,&ok);
                 if (!ok)
                 {
                   sprintf(meldung,"%s [%i.'&-period', %i.'|-period']",
                      WARNUNG02C,i,j);
                   wwrap(meldung,SCREENLINE_LEN);
                   search_text_zeigen(meldung);
                   /// SCROLL2;
                   return 0;
                 }
                 extractitem(2,zahl_ex,"-",zahl_exex);
                 trim(zahl_exex,' ');
                 string_in_long(zahl_exex,&jahr2,&ok);
                 if (!ok)
                 {
                   sprintf(meldung,"%s [%i.'&-period', %i.'|-period']",
                      WARNUNG02D,i,j);
                   wwrap(meldung,SCREENLINE_LEN);
                   search_text_zeigen(meldung);
                   /// SCROLL2;
                   return 0;
                 }
                 if (jahr1 >= jahr2)
                 {
                   sprintf(meldung,"%s [%i.'&-period', %i.'|-period']",
                      WARNUNG02E,i,j);
                   wwrap(meldung,SCREENLINE_LEN);
                   search_text_zeigen(meldung);
                   /// SCROLL2;
                   return 0;
                 }
              }
          }
      }
   }
   return 1;
}

static void journal_search(const char * fieldlabel)
{
  /* Es wird in exex_zeile gesucht */
  int j, anzahl_oder;
  long byte_zeiger, rec_zeiger, anzahl_ixrecords, k;
  refspix_rec ix01rec;
  journal_rec one_journal_rec;
  char feld[MAXFELD_LEN + 1];

  assert(MAXFELD_LEN >= TITEL_LEN);
  assert(MAXFELD_LEN >= BAND_LEN);
  assert(MAXFELD_LEN >= HOWPUBLISHED_LEN);
  assert(MAXFELD_LEN >= INSTAU_LEN);
  assert(MAXFELD_LEN >= REFTYP_LEN);
  assert(MAXFELD_LEN >= IDNR_LEN);
  assert(MAXFELD_LEN >= JAHRGANG_LEN);
  assert(MAXFELD_LEN >= KUERZEL_LEN);
  assert(MAXFELD_LEN >= STATUS_LEN);

  fseek(tmpix,0L,SEEK_END);
  byte_zeiger=ftell(tmpix);
  anzahl_ixrecords = byte_zeiger / sizeof(ix01rec);
  anzahl_ixrecords--;
  fseek(newix,0L,SEEK_END);

  anzahl_oder = countitem(exex_zeile,"|");
  if (max_oder < anzahl_oder) max_oder = anzahl_oder;
  for (j=1;j<=anzahl_oder;j++)
  {
     rewind(tmpix);
     extractitem(j,exex_zeile,"|",exexex_zeile);
     trim(exexex_zeile,' ');
     for (k=0;k<=anzahl_ixrecords;k++)
     {
        fread(&ix01rec,sizeof(ix01rec),1,tmpix);
        rec_zeiger = ix01rec.journal_rptr;
        byte_zeiger = rec_zeiger * sizeof(one_journal_rec);
        fseek(fpjournal,byte_zeiger,SEEK_SET);
        fread(&one_journal_rec,sizeof(one_journal_rec),1,fpjournal);

        if (!strcmp(fieldlabel,"TITA"))
        {
           strcpy(feld,one_journal_rec.titel);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
        if (!strcmp(fieldlabel,"EDTN"))
        {
           if ((strcmp(one_journal_rec.reftyp,"b1")==0) ||
               (strcmp(one_journal_rec.reftyp,"b2")==0) ||
               (strcmp(one_journal_rec.reftyp,"b3")==0))
           {
             strcpy(feld,one_journal_rec.band);
             if (c_insens) upper(feld);
             if (strcmp(exexex_zeile,feld)==0)
             {
               if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
             }
           }
        }
        if (!strcmp(fieldlabel,"VOLJ"))
        {
           if ((strcmp(one_journal_rec.reftyp,"j1")==0) ||
               (strcmp(one_journal_rec.reftyp,"j2")==0))
           {
             strcpy(feld,one_journal_rec.band);
             if (c_insens) upper(feld);
             if (strcmp(exexex_zeile,feld)==0)
             {
               if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
             }
           }
        }
        if (!strcmp(fieldlabel,"HOWP"))
        {
           strcpy(feld,one_journal_rec.howpublished);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
        if (!strcmp(fieldlabel,"INAU"))
        {
           strcpy(feld,one_journal_rec.instau);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
        if (!strcmp(fieldlabel,"DTYP"))
        {
           strcpy(feld,one_journal_rec.reftyp);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
        if (!strcmp(fieldlabel,"YEAR"))
        {
           strcpy(feld,one_journal_rec.jahrgang);
           if (jahreszahlok(exexex_zeile,feld))   /* Vorsicht bei Kopieren !*/
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
        if (!strcmp(fieldlabel,"JRNL"))
        {
           strcpy(feld,one_journal_rec.kuerzel);
           if (c_insens) upper(feld);
           if (strcmp(exexex_zeile,feld)==0)
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
        if (!strcmp(fieldlabel,"IDNR"))
        {
           strcpy(feld,one_journal_rec.idnr);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
        if (!strcmp(fieldlabel,"STAT"))
        {
           strcpy(feld,one_journal_rec.status);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }

     }
  }
}

static int jahreszahlok(char * suchstring, char * feldwert)
{
    long jahrfeld;
    int okfeld;

    trim(feldwert,' ');
    string_in_long(feldwert,&jahrfeld,&okfeld);
    if (!okfeld) return 0;

    trim(suchstring,' ');
    strncpy(zahl_ex,suchstring,ZAHLSTRING_LEN);
    trim(zahl_ex,' ');
    if (countitem(zahl_ex,"-")==1)
    {
       int ok;
       long jahr1;
       string_in_long(zahl_ex,&jahr1,&ok);
       if (!ok) return 0;
       if (jahr1==jahrfeld) return 1;
    }
    if (countitem(zahl_ex,"-")==2)
    {
       int ok;
       long jahr1, jahr2;
       extractitem(1,zahl_ex,"-",zahl_exex);
       trim(zahl_exex,' ');
       string_in_long(zahl_exex,&jahr1,&ok);
       if (!ok) return 0;
       extractitem(2,zahl_ex,"-",zahl_exex);
       trim(zahl_exex,' ');
       string_in_long(zahl_exex,&jahr2,&ok);
       if (!ok) return 0;
       if ((jahrfeld >= jahr1) && (jahrfeld <= jahr2)) return 1;
    }
    return 0;
}

static void buch_search(const char * fieldlabel)
{
  /* Es wird in exex_zeile gesucht */
  int j, anzahl_oder;
  long byte_zeiger, rec_zeiger, anzahl_ixrecords, k;
  refspix_rec ix01rec;
  buch_rec one_buch_rec;
  char feld[MAXFELD_LEN + 1];

  assert(MAXFELD_LEN >= TITELBUCH_LEN);
  assert(MAXFELD_LEN >= INSTED_LEN);
  assert(MAXFELD_LEN >= VERLAG_LEN);
  assert(MAXFELD_LEN >= PUBLORT_LEN);


  fseek(tmpix,0L,SEEK_END);
  byte_zeiger=ftell(tmpix);
  anzahl_ixrecords = byte_zeiger / sizeof(ix01rec);
  anzahl_ixrecords--;
  fseek(newix,0L,SEEK_END);

  anzahl_oder = countitem(exex_zeile,"|");
  if (max_oder < anzahl_oder) max_oder = anzahl_oder;
  for (j=1;j<=anzahl_oder;j++)
  {
     rewind(tmpix);
     extractitem(j,exex_zeile,"|",exexex_zeile);
     trim(exexex_zeile,' ');
     for (k=0;k<=anzahl_ixrecords;k++)
     {
        fread(&ix01rec,sizeof(ix01rec),1,tmpix);
        rec_zeiger = ix01rec.buch_rptr;
        if (ix01rec.buch_rptr < 0) continue;
        byte_zeiger = rec_zeiger * sizeof(one_buch_rec);
        fseek(fpbuch,byte_zeiger,SEEK_SET);
        fread(&one_buch_rec,sizeof(one_buch_rec),1,fpbuch);
        if (!strcmp(fieldlabel,"TITB"))
        {
           strcpy(feld,one_buch_rec.titel_bu);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
        if (!strcmp(fieldlabel,"INED"))
        {
           strcpy(feld,one_buch_rec.insted);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }

        if (!strcmp(fieldlabel,"PUBL"))
        {
           strcpy(feld,one_buch_rec.verlag);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
        if (!strcmp(fieldlabel,"PLAC"))
        {
           strcpy(feld,one_buch_rec.publ_ort);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
           }
        }
     }
  }
}

static void autorjl_search(const char * fieldlabel)
{
  /* Es wird in exex_zeile gesucht */
  int j, anzahl_oder;
  long byte_zeiger, rec_zeiger, anzahl_ixrecords, k, l, n_aued;
  refspix_rec ix01rec;
  author_rec one_aued_rec;
  char feld[MAXFELD_LEN + 1];

  assert(MAXFELD_LEN >= AUNAME_LEN);

  fseek(tmpix,0L,SEEK_END);
  byte_zeiger=ftell(tmpix);
  anzahl_ixrecords = byte_zeiger / sizeof(ix01rec);
  anzahl_ixrecords--;
  fseek(newix,0L,SEEK_END);

  anzahl_oder = countitem(exex_zeile,"|");
  if (max_oder < anzahl_oder) max_oder = anzahl_oder;
  for (j=1;j<=anzahl_oder;j++)
  {
     rewind(tmpix);
     extractitem(j,exex_zeile,"|",exexex_zeile);
     trim(exexex_zeile,' ');
     for (k=0;k<=anzahl_ixrecords;k++)
     {
        fread(&ix01rec,sizeof(ix01rec),1,tmpix);
        rec_zeiger = ix01rec.autorjl_rptr;
        if (ix01rec.autorjl_rptr < 0) continue;
        byte_zeiger = rec_zeiger * sizeof(one_aued_rec);
        fseek(fpautorjl,byte_zeiger,SEEK_SET);
        fread(&one_aued_rec,sizeof(one_aued_rec),1,fpautorjl);
        if (!strcmp(fieldlabel,"AUTH"))
        {
           strcpy(feld,one_aued_rec.au_nachname);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
             continue;
           }
        }
        n_aued = one_aued_rec.au_n;
        if (n_aued==1) continue;
        for (l=2;l<=n_aued;l++)
        {

           fread(&one_aued_rec,sizeof(one_aued_rec),1,fpautorjl);
           if (!strcmp(fieldlabel,"AUTH"))
           {
              strcpy(feld,one_aued_rec.au_nachname);
              if (c_insens) upper(feld);
              if (is_substring(exexex_zeile,feld))
              {
                if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
                break;
              }
           }
        }
     }
  }
}


static void autorbu_search(const char * fieldlabel)
{
  /* Es wird in exex_zeile gesucht */
  int j, anzahl_oder;
  long byte_zeiger, rec_zeiger, anzahl_ixrecords, k, l, n_aued;
  refspix_rec ix01rec;
  author_rec one_aued_rec;
  char feld[MAXFELD_LEN + 1];

  assert(MAXFELD_LEN >= AUNAME_LEN);

  fseek(tmpix,0L,SEEK_END);
  byte_zeiger=ftell(tmpix);
  anzahl_ixrecords = byte_zeiger / sizeof(ix01rec);
  anzahl_ixrecords--;
  fseek(newix,0L,SEEK_END);

  anzahl_oder = countitem(exex_zeile,"|");
  if (max_oder < anzahl_oder) max_oder = anzahl_oder;
  for (j=1;j<=anzahl_oder;j++)
  {
     rewind(tmpix);
     extractitem(j,exex_zeile,"|",exexex_zeile);
     trim(exexex_zeile,' ');
     for (k=0;k<=anzahl_ixrecords;k++)
     {
        fread(&ix01rec,sizeof(ix01rec),1,tmpix);
        rec_zeiger = ix01rec.autorbu_rptr;
        if (ix01rec.autorbu_rptr < 0) continue;
        byte_zeiger = rec_zeiger * sizeof(one_aued_rec);
        fseek(fpautorbu,byte_zeiger,SEEK_SET);
        fread(&one_aued_rec,sizeof(one_aued_rec),1,fpautorbu);
        if (!strcmp(fieldlabel,"EDIT"))
        {
           strcpy(feld,one_aued_rec.au_nachname);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
             continue;
           }
        }
        n_aued = one_aued_rec.au_n;
        if (n_aued==1) continue;
        for (l=2;l<=n_aued;l++)
        {

           fread(&one_aued_rec,sizeof(one_aued_rec),1,fpautorbu);
           if (!strcmp(fieldlabel,"EDIT"))
           {
              strcpy(feld,one_aued_rec.au_nachname);
              if (c_insens) upper(feld);
              if (is_substring(exexex_zeile,feld))
              {
                if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
                break;
              }
           }
        }
     }
  }
}



static void desc_search(const char * fieldlabel)
{
  /* Es wird in exex_zeile gesucht */
  int j, anzahl_oder;
  long byte_zeiger, rec_zeiger, anzahl_ixrecords, k, l, n_desc;
  refspix_rec ix01rec;
  desc_rec dr;
  char feld[MAXFELD_LEN + 1];

  assert(MAXFELD_LEN >= KEYWORDS_LEN);

  fseek(tmpix,0L,SEEK_END);
  byte_zeiger=ftell(tmpix);
  anzahl_ixrecords = byte_zeiger / sizeof(ix01rec);
  anzahl_ixrecords--;
  fseek(newix,0L,SEEK_END);

  anzahl_oder = countitem(exex_zeile,"|");
  if (max_oder < anzahl_oder) max_oder = anzahl_oder;
  for (j=1;j<=anzahl_oder;j++)
  {
     rewind(tmpix);
     extractitem(j,exex_zeile,"|",exexex_zeile);
     trim(exexex_zeile,' ');
     for (k=0;k<=anzahl_ixrecords;k++)
     {
        fread(&ix01rec,sizeof(ix01rec),1,tmpix);
        rec_zeiger = ix01rec.desc_rptr;
        if (ix01rec.desc_rptr < 0) continue;
        byte_zeiger = rec_zeiger * sizeof(dr);
        fseek(fpdesc,byte_zeiger,SEEK_SET);
        fread(&dr,sizeof(dr),1,fpdesc);
        if (!strcmp(fieldlabel,"KEYW"))
        {
           strcpy(feld,dr.keywordsfield);
           if (c_insens) upper(feld);
           if (is_substring(exexex_zeile,feld))
           {
             if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
             continue;
           }
        }
        n_desc = dr.desc_n;
        if (n_desc==1) continue;
        for (l=2;l<=n_desc;l++)
        {

           fread(&dr,sizeof(dr),1,fpdesc);
           if (!strcmp(fieldlabel,"KEYW"))
           {
              strcpy(feld,dr.keywordsfield);
              if (c_insens) upper(feld);
              if (is_substring(exexex_zeile,feld))
              {
                if (fwrite(&ix01rec,sizeof(ix01rec),1,newix) < 1) schreibfehler();
                break;
              }
           }
        }
     }
  }

}


static void ix2bbt(const char * ix_name, const char * bbt_name)
{
    FILE * fpbbt;
    long anzahl_records, i,byte_zeiger;
    refspix_rec ix01rec;
    bbatchtable_rec one_bbt_rec;

    memset(&one_bbt_rec,0,sizeof(one_bbt_rec));

    if ((fpbbt= fopen(bbt_name,"wb"))==NULL)
    {
       fprintf(stderr,"\nERROR: cannot create %s\n",bbt_name);
       exit(1);
    }
    if ((newix = fopen(ix_name,"rb"))==NULL)
    {
       fprintf(stderr,"\nERROR: cannot read %s\n",ix_name);
       exit(1);
    }
    fseek(newix,0L,SEEK_END);
    byte_zeiger = ftell(newix);
    rewind(newix);
    anzahl_records = byte_zeiger/sizeof(ix01rec);
    anzahl_records--;
    for (i=0;i<=anzahl_records;i++)
    {
        if (fread(&ix01rec,sizeof(ix01rec),1,newix)==0)
        {
           fprintf(stderr,"\nERROR: cannot read record in %s\n",
             ix_name);
           exit(1);
        }
        strncpy(one_bbt_rec.kennziffer,ix01rec.kennziffer,KENNZIFFER_LEN);
        if (fwrite(&one_bbt_rec,sizeof(one_bbt_rec),1,fpbbt) < 1) schreibfehler();
    }

    fclose(newix);
    fclose(fpbbt);
}

static int bbt2ix(char * dateiname)
{
   FILE * fpbbt;     /* die zu lesende BBT-Datei */
   FILE * fppix1;    /* die neu zu schreibende Datei */

   long anzahl_records, i, byte_zeiger, ix_zeiger;
   refspix_rec ix01rec;
   bbatchtable_rec one_bbt_rec;

   if (!read_ix01(journal_pix_name))
   {
     fprintf(stderr,"\nERROR: cannot read %s\n",journal_pix_name);
     exit(1);
   }
   if ((fpbbt = fopen(bbt_in_datei_name,"rb"))==NULL)
   {
     fprintf(stderr,"\nERROR: cannot read %s\n",bbt_in_datei_name);
     exit(1);
   }
   if ((fppix1 = fopen(dateiname,"wb"))==NULL)
   {
     fprintf(stderr,"\nERROR: cannot create/write into %s\n",
          dateiname);
     exit(1);
   }
   fseek(fpbbt,0L,SEEK_END);
   byte_zeiger = ftell(fpbbt);
   rewind(fpbbt);
   anzahl_records = byte_zeiger / sizeof(one_bbt_rec);
   anzahl_records--;

   sprintf(wrtxtbu,"\n(Extracting %li records from %s)",
       anzahl_records +1, bbt_in_datei_name);
   wrtxt(wrtxtbu);

   for (i=0;i<=anzahl_records;i++)
   {
      fread(&one_bbt_rec,sizeof(one_bbt_rec),1,fpbbt);
      ix_zeiger = locate_ix01(one_bbt_rec.kennziffer,&ix01rec);
      if (ix_zeiger > -1)
      {
          if (fwrite(&ix01rec,sizeof(ix01rec),1,fppix1) < 1) schreibfehler();
      }
   }
   close_ix01();
   fclose(fpbbt);
   fclose(fppix1);
   if (filesize(dateiname) < sizeof(ix01rec))
   {
      remove(dateiname);
      wrtxt("\n0 record extracted, therefore SEARCH in complete "
                "database...\n");
      return 0;
   }
   else
   {
      wrtxt("\n");
      return 1;
   }
}


static void int2fn(int zahl, char * dateiname)
{
   char namenselement[PFAD_LEN + 1]; 
   // sprintf(dateiname,"%07i.ix$",zahl);
   
   sprintf(namenselement,"%07i.ix$",zahl);
   strlimcpy(dateiname,rdbpfad,PFAD_LEN);
   strlimcat(dateiname,namenselement,PFAD_LEN);
   return;
}

static int steuerung(char * zeile)
{
   int anzahl_und;
   time_t zeit1, zeit2;
   int i;
   char zkopie[SEARCHCMD_LEN+1];
   char pix1_name[PFAD_LEN+1];
   int rueckgabewert;

   rueckgabewert = 1;
  /* Resultate auf 0 abfragen */
   time(&zeit1);
   anzahl_und = countitem(zeile,"&");
   int2fn(1,pix1_name);       /* Diese Datei aus BBT-Datei erzeugen */

   for (i=1;i<=anzahl_und;i++)
   {
     if (i==1)
     {
        if (strlen(bbt_in_datei_name) > 0)
        {
            if (bbt2ix(pix1_name))
            {
                strcpy(tmpix_name,pix1_name);
            }
            else
            {
               strcpy(bbt_in_datei_name,"");
               strcpy(tmpix_name,journal_pix_name);
            }
        }
        else
        {
          strcpy(tmpix_name,journal_pix_name);
        }
     }
     else
     {
        int2fn(i,tmpix_name);
     }
     int2fn(i+1,newix_name);
     if ((tmpix=fopen(tmpix_name,"rb"))==NULL)
     {
        fprintf(stderr,"\nERROR: Cannot read %s\n",tmpix_name);
        exit(1);
     }
     if (setvbuf(tmpix,NULL,_IOFBF,9000)!=0)
     {
        fprintf(stderr,"\nERROR: cannot alloacate memory "
           "for tmpix-buffer\n");
        exit(1);
     }

     if ((newix=fopen(newix_name,"wb"))==NULL)
     {
        fprintf(stderr,"\nERROR: Cannot read %s\n",newix_name);
        exit(1);
     }
     if (setvbuf(newix,NULL,_IOFBF,3000)!=0)
     {
        fprintf(stderr,"\nERROR: cannot alloacate memory "
           "for newix-buffer\n");
        exit(1);
     }
     /* Aufrufe */
     strcpy(zkopie,zeile);
     if (c_insens) upper(zkopie);
        /* den originalen Eingabestring nicht kapitalisieren! */

     extractitem(i,zkopie,"&",ex_zeile);
     extractitem(1,ex_zeile,"=",exex_zeile);
     trim(exex_zeile,' ');
     upper(exex_zeile);

     if (!strcmp(exex_zeile,"KEYW"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        desc_search("KEYW");
     }
     else if (!strcmp(exex_zeile,"INAU"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("INAU");
     }
     else if (!strcmp(exex_zeile,"EDTN"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("EDTN");
     }
     else if (!strcmp(exex_zeile,"VOLJ"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("VOLJ");
     }
     else if (!strcmp(exex_zeile,"TITA"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("TITA");
     }
     else if (!strcmp(exex_zeile,"HOWP"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("HOWP");
     }
     else if (!strcmp(exex_zeile,"DTYP"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("DTYP");
     }
     else if (!strcmp(exex_zeile,"YEAR"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("YEAR");
     }
     else if (!strcmp(exex_zeile,"JRNL"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("JRNL");
     }
     else if (!strcmp(exex_zeile,"IDNR"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("IDNR");
     }
     else if (!strcmp(exex_zeile,"STAT"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        journal_search("STAT");
     }
     else if (!strcmp(exex_zeile,"INED"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        buch_search("INED");
     }
     else if (!strcmp(exex_zeile,"TITB"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        buch_search("TITB");
     }

     else if (!strcmp(exex_zeile,"PUBL"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        buch_search("PUBL");
     }
     else if (!strcmp(exex_zeile,"PLAC"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        buch_search("PLAC");
     }

     else if (!strcmp(exex_zeile,"AUTH"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        autorjl_search("AUTH");
     }

     else if (!strcmp(exex_zeile,"EDIT"))
     {
        extractitem(2,ex_zeile,"=",exex_zeile);
        trim(exex_zeile,' ');
        autorbu_search("EDIT");
     }

     /* Aufrufe Ende */
     fclose(tmpix);
     fclose(newix);
     if (i!=1)  /* Hauptindex nicht loeschen! */
     {
         remove(tmpix_name);
     }
     else
     {
         if (isfile(pix1_name)) remove (pix1_name);
     }
     if (filesize(newix_name)==0)
     {
        time(&zeit2);
        sprintf(wrtxtbu,"\n(No references found, %.0f seconds required)\n",
              difftime(zeit2,zeit1));
        wrtxt(wrtxtbu);
        remove(newix_name);
        rueckgabewert = 0;
        return rueckgabewert;
     }
   }
   if (filesize(newix_name)>=sizeof(refspix_rec))
   {
       long rec_zeiger, byte_zeiger;
       char dollar1bbt[PFAD_LEN +1]; /* name is $1.bbt*/
       char dollar2bbt[PFAD_LEN +1]; /* name is $2.bbt*/

       strlimcpy(dollar1bbt,rdbpfad,PFAD_LEN);
       strlimcat(dollar1bbt,"$1.bbt",PFAD_LEN);
       strlimcpy(dollar2bbt,rdbpfad,PFAD_LEN);
       strlimcat(dollar2bbt,"$2.bbt",PFAD_LEN);

       ix2bbt(newix_name,dollar1bbt);
       remove(newix_name);
       if (max_oder > 1)
       {
          wrtxt("\n");
          domergebbf1(dollar1bbt,dollar2bbt);
          wrtxt("\n");
          remove(dollar1bbt);
          wrtxt("(Removing duplicate records, ");
          dupl_ex_bbf(dollar2bbt,bbt_datei_name);
          remove(dollar2bbt);
          wrtxt("done)");
       }
       else
       {
          if (isfile(bbt_datei_name))
              remove(bbt_datei_name);
          rename(dollar1bbt,bbt_datei_name);
       }
       byte_zeiger=filesize(bbt_datei_name);
       rec_zeiger=byte_zeiger/sizeof(bbatchtable_rec);
       time(&zeit2);
       sprintf(wrtxtbu,"\n(%.0f seconds required)\n",
           difftime(zeit2,zeit1));
       wrtxt(wrtxtbu);
   }
   return rueckgabewert;
}

