//
// MODTOOL
//
// Copyright (c) !998 David Lindauer (LADSOFT)
//
// see license.txt for licensing info
//
// ==============================================================
//
// NNTPSND.CPP
//
// news poster, posts one message.  Newsgroups must be specified in
// the newsgroups line of the header
//
// This is a state machine.  Rather than comment every state I give a brief
// summary of transactions here:
//
// local:  issues winsock connect command
// remote: returns code 200, posting allowed
// local:  sends "POST"
// remote: returns code 340
//
// local:  sends headers and body
//
// local:  sends a ','
// remote: returns code 250
// local:  sends 'QUIT'
// remote: some reply which I discard
// disconnect from network
//
// All commands must be on a single line.  All lines are terminated with
// the sequence "\r\n".  When we send the text we could send as much
// data as we want in a packet, but, it is convenient to send a line at
// a time.  
//
// Note:  the mail message is terminated by the sequence
// "\r\n.\r\n" after which the remote starts responding again.
// For transparency, any other line in the message which begins with
// a dot gets another dot added.  It is the responsibility of the eventual
// receiver to strip this out.
//
// 

#define STRICT
#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <stdarg.h>

#include "profile.h"
#include "mailfile.h"
#include "network.h"
#include "effects.h"
#include "progress.h"
#include "send.h"

// ==============================================================
//
// states
#define MS_EXPCONNECT 0
#define MS_SENDPOST   1
#define MS_RECEIVEPOST 2 
#define MS_SENDLINES   3
#define MS_SENDDOT     4
#define MS_RECEIVEDOT  5
#define MS_ERROR       6
#define MS_ABORT       7
#define MS_IGNORE      8

// flags to keep track of winsock status
#define FL_SEND 1
#define FL_RECEIVE 2

extern HINSTANCE hInst;
extern HWND hWndProgress,hWndMain;
extern int networkinuse;
extern int batchflags;

long currentsize;		// size sent
long totalsize;			// total size to send
int isaoneshot;			// set if doing from command line
static int flags;		// winsock flags

static FILE *workfil;
static int state;
static SOCKET worksocket;
static int nntprejectedit;

// ==============================================================
//
// state machine
static int callback(HWND hWnd, int mode, int error)
{
	static char string[10000];
	int t;

	// handle winsock errors
	if (error == SOCKET_ERROR)  {
		state = MS_ERROR;
	}
	// handle cancel button
	if (!hWndProgress) {
		NetworkMessageBox(MB_ICONINFORMATION,"User canceled transfer");
		state = MS_ABORT;
	}
	// ignore close events
	if (mode == FD_CLOSE) {
		return 0;
	}
	// check for valid connection to host
	if (mode == FD_CONNECT) {
		if (error == SOCKET_ERROR) {
			NoConnect(nntppostserver);
			networkinuse = 0;
		}
		state = MS_EXPCONNECT;
		return 0;
	}

	// collect winsock flags
	if (mode == FD_WRITE)
		flags |= FL_SEND;
	if (mode == FD_READ)
		flags |= FL_RECEIVE;

	// actual state machine
	switch (state) {
		case MS_EXPCONNECT:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 200) {
				nntprejectedit = TRUE;
				NetworkMessageBox(MB_ICONERROR,"SMTP host refused connection for posting");
				goto abort;
			}
			state = MS_SENDPOST;
		case MS_SENDPOST:
			if (!(flags & FL_SEND))
				break;
			flags &= ~FL_SEND;
			if (StreamedNetworkSend(worksocket,"POST") != NETWORK_OK)
				goto doerror;
			state = MS_RECEIVEPOST;
			break;
		case MS_RECEIVEPOST:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 340) {
				nntprejectedit = TRUE;
				NetworkMessageBox(MB_ICONERROR,"NNTP Host failed POST command");
				goto abort;
			}
			state = MS_SENDLINES;
		case MS_SENDLINES:
			while (!feof(workfil)) {
				char buf[512];
				buf[0] = 0;

				if (!GetMailString(buf,workfil)) {
					if (StreamedNetworkSend(worksocket,buf)) {
						ExtendedMessageBox("File error",MB_ICONERROR,"Can't read from temp mail file, aborting");
						goto abort;
					}
					currentsize += strlen(buf);
					SetProgress(1000*currentsize/totalsize,"Sending news %d%% of %dK",100*currentsize/totalsize,totalsize/1024+1);
				}
			}
			state = MS_SENDDOT;
		case MS_SENDDOT:
			if (StreamedNetworkSend(worksocket,".") != NETWORK_OK)
				goto doerror;
			state = MS_RECEIVEDOT;
			break;
		case MS_RECEIVEDOT:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 240) {
				nntprejectedit = TRUE;
				if (t == 441) {
					NetworkMessageBox(MB_ICONERROR,"NNTP host rejected message for the following reason: %s; messages left in outbox",string+4);
				}
				else {
					NetworkMessageBox(MB_ICONERROR,"NNTP host failed to accept message");
					goto abort;
				}
			}
			goto abort;
doerror:
		case MS_ERROR:
			nntprejectedit = TRUE;
			NetworkMessageBox(MB_ICONERROR,"WINSOCK error, aborting");
abort:
		case MS_ABORT:
			state = MS_IGNORE;
			StreamedNetworkSend(worksocket,"QUIT");
			StreamedNetworkDisconnect(worksocket);
			fclose(workfil);
			if (isaoneshot)
				PostQuitMessage(0);
			else
				if (nntprejectedit)
					SendMessage(hWndMain,WM_USENDFAILURE,0,0);
				else
					SendMessage(hWndMain,WM_USENDSUCCESS,0,0);
			break;
		case MS_IGNORE:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			StreamedNetworkSend(worksocket,"QUIT");
			StreamedNetworkDisconnect(worksocket);
			break;
	}
}
// ==============================================================
// main routine for posts
//

int PostNews(char *file, int oneshot)
{
	// set mode flag
	isaoneshot = oneshot;

	// open input file
	if (!(workfil = fopen(file,"r"))) {
		ExtendedMessageBox("Post Error",MB_ICONERROR,"News file %s can't be opened",file);
		return 0;
	}
	else if (oneshot) {
		currentsize = 0;
		fseek(workfil,0,2);
		fgetpos(workfil,&totalsize);
		fseek(workfil,0,0);
		SetProgress(1000*currentsize/totalsize,"Sending news %d%% of %dK",100*currentsize/totalsize,totalsize/1024+1);
	}
	// set callback
	NetworkCallback = callback;
			  
	nntprejectedit = FALSE;
	// initiate connection
	switch(StreamedNetworkConnect(nntppostserver,NNTPPort,&worksocket)) {
		case NETWORK_OK:
			flags = 0;
			break;
		case NETWORK_NOHOST:
			NoHost(smtpserver);
			break;
	}
}