/*****************************************************************************
 * sapserver.cpp : SAP discovery service mini-server
 ****************************************************************************
 * Copyright (C) 1998-2002 VideoLAN
 * $Id: sapserver.cpp,v 1.16 2003/09/23 16:47:15 nitrox Exp $ 
 *
 * Authors: Arnaud Schauly <gitan@via.ecp.fr>
 *          Clment Stenac <zorglub@via.ecp.fr>
 *          Damien Lucas <nitrox@videolan.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string>
#include <vector>

using namespace std;

#include "config.h"

#include "program.h"
#include "message.h"
#include "broadcast.h"
#include "parser.h"
#include "sapserver.h"

#ifdef CONFIG_SLP
    #include <slp.h>
    #include "lslp.h"
#endif

/*************************************************
 * Display the help
 *************************************************/
void help(void)
{
  printf("Options:\n" \
  "  -d      Use this to daemonize the process (run in the background)\n" \
  "  -f      Use this to give a configuration file (default is %s)\n" \
  "  -c      Same as -f\n" \
  "  -t      \"slp\" or \"sap\". sap is default"  \
  "  -h      Display this help\n" \
  "  -s      Display a dot for each packet sent\n",DEFAULT_CONF);
}

/*************************************************
 * The main function
 *************************************************/
int main(int argc, char *argv[])
{
    int result=0;
    int i_init_status=0;
    char getopt_string[128];
    Broadcast* broadcast;
    Message *message;
    Config *config;
#ifdef CONFIG_SLP
    SLP *slp;
#endif 

    fprintf(stdout, "%s Version %s - (c)1999-2003 VideoLAN\n\n",
                    PACKAGE,VERSION);
    config=new Config(DEFAULT_CONF);

    /* Parse the command line */
    sprintf(getopt_string,"dhsf:c:t:u");
    while((result = getopt(argc,argv,getopt_string))>0)
    {
        switch(result)
        {
            case 'c':
            case 'f':
                config->SetFile(optarg);
                break;
            case 's':
                config->SetDotMode(true);
                break;
            case 'd':
                config->SetDaemonMode(true);
                break;
	    case 't':
		config->SetType(optarg);
		break;
	    case 'u':
		config->SetReverse(true);
		break;
            case 'h':
                help();
                return ( 0 );
            default:
                return ( -1 );
        }
    }

    /* Get the programs */
    fprintf(stdout, "+ Parsing configuration file\n");
    if(config->Parse())
    {
        fprintf(stderr, "+ Parsing failed\n");
        return(-1);
    }
    /*TODO check configuration */
    fprintf(stdout, "+ %d programs loaded\n",(int)config->Programs.size());

    if(config->GetType() == TYPE_SAP)
    {
        fprintf(stdout, "+ Packet TTL set to %i\n",config->GetTTL());
        if(config->GetDaemonMode()) fprintf(stdout, "+ Running as daemon.\n");
        else fprintf(stdout, "+ Running as program.\n");

        /* Create the broadcast object */
        /* IPv4 is the default behavior */
        broadcast = new Broadcast(config->GetTTL());
        if(config->GetIPVersion() == 6)
        {
            fprintf(stdout,"+ Using IPv6\n");
            i_init_status=broadcast->Initv6(config->GetScope());
        }
        else
        {
            fprintf(stdout,"+ Using IPv4\n");
            i_init_status=broadcast->Initv4(); 
        }


        if(i_init_status)
        {
            fprintf(stderr,"- Network initialization failed. Aborting\n");
            return -1;
        }

        /* Forking if necessary */
        if(config->GetDaemonMode())
        {
            fprintf(stdout, "+ Forking ... \n");
            daemon(0,0);
        }


        while(1)
        {
            for (unsigned int i=0; i<config->Programs.size(); i++)
            {
                /* Create a new message */
                message=new Message(SAP_ANNOUNCE, 0x4212+i, "1.2.3.4");
                /* Build the message header */
                message->BuildHeader("application/sdp");
                /* Add the program */
                message->AddProgram(config->Programs[i]);
                /* Send the message */
                if(broadcast->Send( message , config->GetIPVersion() ))
                fprintf(stderr,"- Message send failed\n");
                /* Delete the message */
                delete message;
                if(config->GetDotMode())
                   fprintf(stdout,".");

                /* Wait for next sending */
                usleep(SAP_DELAY*1000000/config->Programs.size());
            }
            fflush(stdout);
        }
    }
    else 
    {
	#ifndef CONFIG_SLP
	    fprintf(stderr,"- SLP not compiled in...Aborting\n");
	    return -1;
	#endif
	#ifdef CONFIG_SLP

	slp = new SLP();
	
	for(unsigned int i=0 ; i<config->Programs.size(); i++)
	{
	    if(!config->GetReverse())
	    {
		slp->Register(config->Programs[i]);
	    }
	    else
 	    {
		slp->UnRegister(config->Programs[i]);
	    }
	}
	if(!config->GetReverse())
	{
	    fprintf(stdout,"+ Programs registered. To unregister them, run" \
		       "me with -u\n");
	}
	else
	{
	    fprintf(stdout,"+ Programs unregistered\n");
	}

	fflush(stdout);
	delete slp;
	#endif
    }
    
fprintf(stdout,"Done");

    return 0;
}
