/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System.IO;
using System.Collections.Generic;

namespace LocaleInpector
{

// General I/O methods

internal static class BPIO
{

// dont't use System.Text.Encoding.UTF8 (it emits BOM)
public static readonly System.Text.Encoding UTF8 = new System.Text.UTF8Encoding(false);

private static Dictionary<string, List<string>>  m_dirListCache = new Dictionary<string, List<string>>();
private static Dictionary<string, NameValueFile> m_loadCache    = new Dictionary<string, NameValueFile>();


public static bool HasBom(string name)
{
    FileStream fs = File.OpenRead(name);
    int[] bytes = new int[] {
        fs.ReadByte(),
        fs.ReadByte(),
        fs.ReadByte()
    };
    fs.Close();
    return (bytes[0] == 0xef) &&
           (bytes[1] == 0xbb) &&
           (bytes[2] == 0xbf);
}


public static bool Exists(string path)
{
    return IsFile(path) || IsDir(path);
}


public static bool IsDir(string path)
{
    return Directory.Exists(path);
}


public static bool IsFile(string path)
{
    return File.Exists(path);
}


public static void DeleteFile(string path)
{
    m_loadCache.Remove(path);//[path] = null;
    m_dirListCache.Clear();
    System.IO.File.Delete(path);
}


public static NameValueFile GetFileFromLoadCache(string k)
{
    if (m_loadCache[k] == null) {
        throw new System.NullReferenceException(k);
    }
    return m_loadCache[k];
}


public static void SetFileInLoadCache(string k, NameValueFile f)
{
    if (f == null) {
        throw new System.NullReferenceException(k);
    }
    m_loadCache[k] = f;
}


public static bool ExistsInLoadCache(string k)
{
    return m_loadCache.ContainsKey(k);
}


public static void Save(string txt, string path, System.Text.Encoding encoding)
{
    m_loadCache.Remove(path); //[path] = null;
    m_dirListCache.Clear();
    string created = Path.GetDirectoryName(path);
    if (created.Length > 0) {
        if (BPIO.IsDir(created) == false) {
            DirectoryInfo dir = Directory.CreateDirectory(created);
        }
    }
    using (StreamWriter sw = new StreamWriter(path, false, encoding)) {
        sw.Write(txt);
    }
}


public static Stream ExtractZip(Stream streamZip)
{
    MemoryStream mem = new MemoryStream((int) streamZip.Length);
    byte[] dataBuffer = new byte[2048];
    int lenBuffer = dataBuffer.Length;
    while (true) {
        int size = streamZip.Read(dataBuffer, 0, lenBuffer);
        if (size == 0) {
            break;
        }
        mem.Write(dataBuffer, 0, size);
    }

    return mem;
}


// c:\foo\bar.zip|chrome/locale/en-US/ui.dtd
public static string LoadText(string fullName, System.Text.Encoding enc)
{
    if (fullName.IndexOf('|') > -1) {
        return XpiList.LoadTextFromArchive(fullName, enc);
    }

    if (IsFile(fullName) == false) {
        return string.Empty;
    }

    StreamReader sreader = new StreamReader(fullName, enc);
    string txt = sreader.ReadToEnd();
    sreader.Close();
    return txt;
}


public static List<string> GetFileList(string parentDir, string filter)
{
    string key = parentDir + "|" + filter;
    if (m_dirListCache.ContainsKey(key)) {
        return m_dirListCache[key];
    }
    List<string> obj = GetFileList2(parentDir, filter);
    m_dirListCache[key] = obj;
    return obj;
}


private static List<string> GetFileList2(string parentDir, string filter)
{
    List<string> subDirs = GetSubDirectoriesStructure(parentDir);
    subDirs.Add(string.Empty); // root parentDir files
    List<string> paths = new List<string>();
    for (int idx = 0; idx < subDirs.Count; idx++) {
        string[] files = Directory.GetFiles(parentDir + subDirs[idx], filter);
        for (int idx2 = 0; idx2 < files.Length; idx2++) {
            // 012345678901234567890123456789012345
            // E:\Moz-L10n\cvs\trunk.l10n\l10n\ca\browser\chrome\branding\brand.properties
            //              (35chars)             ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            paths.Add(files[idx2].Substring(parentDir.Length));
        }
    }
    return paths;
}


public static List<string> GetSubDirectoriesStructure(string dir)
{
    List<string> allSubdirs = new List<string>();
    GetSubDirectoriesStructureCore(dir, allSubdirs);
    for (int idx = 0; idx < allSubdirs.Count; idx++) {
        allSubdirs[idx] = allSubdirs[idx].Substring(dir.Length);
    }
    return allSubdirs;
}

private static void GetSubDirectoriesStructureCore(string dir, List<string> allSubdirs)
{
    string[] subDirs = Directory.GetDirectories(dir);
    for (int idx = 0; idx < subDirs.Length; idx++) {
        DirectoryInfo di = new DirectoryInfo(subDirs[idx]);
        FileAttributes attr = di.Attributes;
        if ((attr | FileAttributes.Hidden) != attr) {
            allSubdirs.Add(subDirs[idx]);
            GetSubDirectoriesStructureCore(subDirs[idx], allSubdirs);
        }
    }
}


public static string GetStringFromResource(string id)
{
    System.IO.Stream stream = GetStreamFromResource(id);
    if (stream == null) {
        // nao existe
        return null;
    }
    string txt = StreamToString(stream);
    stream.Close();
    stream.Dispose();
    return txt;
}


private static System.IO.Stream GetStreamFromResource(string id)
{
    System.Reflection.Assembly assembly = System.Reflection.Assembly.GetExecutingAssembly();
    System.IO.Stream stream = assembly.GetManifestResourceStream(id);

    // nao existe ==> null
    return stream;
}


private static string StreamToString(System.IO.Stream stream)
{
    if (stream == null) {
        return null;
    }
    System.IO.StreamReader sr = new System.IO.StreamReader(stream, System.Text.Encoding.UTF8);
    string txt = sr.ReadToEnd();
    sr.Close();
    sr.Dispose();
    return txt;
}


}//class
}//ns