/* Tests for 64bit AIO in librt.
   Copyright (C) 1998, 1999 Free Software Foundation, Inc.
   Contributed by Ulrich Drepper <drepper@cygnus.com>, 1998.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#define _LARGEFILE_SOURCE 1
#include <aio.h>
#include <errno.h>
#include <error.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>


/* Prototype for our test function.  */
extern void do_prepare (int argc, char *argv[]);
extern int do_test (int argc, char *argv[]);

/* We have a preparation function.  */
#define PREPARE do_prepare

/* We might need a bit longer timeout.  */
#define TIMEOUT 20 /* sec */

/* This defines the `main' function and some more.  */
#include <test-skeleton.c>


/* These are for the temporary file we generate.  */
char *name;
int fd;

void
do_prepare (int argc, char *argv[])
{
  char name_len;

  name_len = strlen (test_dir);
  name = malloc (name_len + sizeof ("/aioXXXXXX"));
  mempcpy (mempcpy (name, test_dir, name_len),
	   "/aioXXXXXX", sizeof ("/aioXXXXXX"));
  add_temp_file (name);

  /* Open our test file.   */
  fd = mkstemp (name);
  if (fd == -1)
    error (EXIT_FAILURE, errno, "cannot open test file `%s'", name);
}


int
test_file (const void *buf, size_t size, int fd, const char *msg)
{
  struct stat st;
  char tmp[size];

  errno = 0;
  if (fstat (fd, &st) < 0)
    {
      error (0, errno, "%s: failed stat", msg);
      return 1;
    }

  if (st.st_size != size)
    {
      error (0, errno, "%s: wrong size: %lu, should be %lu",
	     msg, (unsigned long int) st.st_size, (unsigned long int) size);
      return 1;
    }

  if (pread (fd, tmp, size, 0) != size)
    {
      error (0, errno, "%s: failed stat", msg);
      return 1;
    }

  if (memcmp (buf, tmp, size) != 0)
    {
      error (0, errno, "%s: failed comparison", msg);
      return 1;
    }

  printf ("%s test ok\n", msg);

  return 0;
}


void
do_wait (struct aiocb64 **cbp, size_t nent)
{
  int go_on;
  do
    {
      size_t cnt;

      aio_suspend64 ((const struct aiocb64 *const *) cbp, nent, NULL);
      go_on = 0;
      for (cnt = 0; cnt < nent; ++cnt)
	if (cbp[cnt] != NULL && aio_error64 (cbp[cnt]) == EINPROGRESS)
	  go_on = 1;
	else
	  cbp[cnt] = NULL;
    }
  while (go_on);
}


int
do_test (int argc, char *argv[])
{
  struct aiocb64 cbs[10];
  struct aiocb64 *cbp[10];
  char buf[1000];
  size_t cnt;
  int result = 0;

  /* Preparation.  */
  for (cnt = 0; cnt < 10; ++cnt)
    {
      cbs[cnt].aio_fildes = fd;
      cbs[cnt].aio_reqprio = 0;
      cbs[cnt].aio_buf = memset (&buf[cnt * 100], '0' + cnt, 100);
      cbs[cnt].aio_nbytes = 100;
      cbs[cnt].aio_offset = cnt * 100;
      cbs[cnt].aio_sigevent.sigev_notify = SIGEV_NONE;

      cbp[cnt] = &cbs[cnt];
    }

  /* First a simple test.  */
  for (cnt = 10; cnt > 0; )
    aio_write64 (cbp[--cnt]);
  /* Wait 'til the results are there.  */
  do_wait (cbp, 10);
  /* Test this.  */
  result |= test_file (buf, sizeof (buf), fd, "aio_write");

  /* Read now as we've written it.  */
  memset (buf, '\0', sizeof (buf));
  /* Issue the commands.  */
  for (cnt = 10; cnt > 0; )
    {
      --cnt;
      cbp[cnt] = &cbs[cnt];
      aio_read64 (cbp[cnt]);
    }
  /* Wait 'til the results are there.  */
  do_wait (cbp, 10);
  /* Test this.  */
  for (cnt = 0; cnt < 1000; ++cnt)
    if (buf[cnt] != '0' + (cnt / 100))
      {
	result = 1;
	error (0, 0, "comparison failed for aio_read test");
	break;
      }

  if (cnt == 1000)
    puts ("aio_read test ok");

  /* Remove the test file contents.  */
  if (ftruncate64 (fd, 0) < 0)
    {
      error (0, errno, "ftruncate failed\n");
      result = 1;
    }

  /* Test lio_listio.  */
  for (cnt = 0; cnt < 10; ++cnt)
    {
      cbs[cnt].aio_lio_opcode = LIO_WRITE;
      cbp[cnt] = &cbs[cnt];
    }
  /* Issue the command.  */
  lio_listio64 (LIO_WAIT, cbp, 10, NULL);
  /* ...and immediately test it since we started it in wait mode.  */
  result |= test_file (buf, sizeof (buf), fd, "lio_listio (write)");

  return result;
}
