/*******************************************************************************
 * Copyright (c) 2023 Dawid Pakuła and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Dawid Pakuła <zulus@w3des.net> - initial implementation
 *******************************************************************************/
package org.eclipse.wildwebdeveloper.tests;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.Arrays;
import java.util.Optional;
import java.util.stream.Collectors;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.lsp4e.operations.completion.LSContentAssistProcessor;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.tests.harness.util.DisplayHelper;
import org.eclipse.wildwebdeveloper.embedder.node.NodeJSManager;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class TestVue {
	static IProject project;
	static IFolder componentFolder;

	@BeforeAll
	public static void setUp() throws Exception {
		AllCleanRule.closeIntro();
		AllCleanRule.enableLogging();

		project = Utils.provisionTestProject("vue-app");
		ProcessBuilder builder = new ProcessBuilder(NodeJSManager.getNpmLocation().getAbsolutePath(), "install",
				"--no-bin-links", "--ignore-scripts").directory(project.getLocation().toFile());
		Process process = builder.start();
		System.out.println(builder.command().toString());
		String result = new BufferedReader(new InputStreamReader(process.getErrorStream())).lines()
				.collect(Collectors.joining("\n"));
		System.out.println("Error Stream: >>>\n" + result + "\n<<<");

		result = new BufferedReader(new InputStreamReader(process.getInputStream())).lines()
				.collect(Collectors.joining("\n"));
		System.out.println("Output Stream: >>>\n" + result + "\n<<<");

		assertEquals(0, process.waitFor(), "npm install didn't complete property");

		project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
		assertTrue(project.exists());
		componentFolder = project.getFolder("src").getFolder("components");
		assertTrue(componentFolder.exists());
	}

	@BeforeEach
	public void setUpTestCase() {
		AllCleanRule.enableLogging();
	}

	@AfterAll
	public static void tearDown() throws Exception {
		new AllCleanRule().afterEach(null);
	}

	@Test
	void testVueApp() throws Exception {
		IFile appComponentFile = project.getFile("src/App.vue");
		TextEditor editor = (TextEditor) IDE
				.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage(), appComponentFile);
		DisplayHelper.sleep(4000); // Give time for LS to initialize enough before making edit and sending a
									// didChange
		assertTrue(new DisplayHelper() {
			@Override
			protected boolean condition() {
				try {
					return Arrays
							.stream(appComponentFile.findMarkers("org.eclipse.lsp4e.diagnostic", true,
									IResource.DEPTH_ZERO))
							.anyMatch(marker -> marker.getAttribute(IMarker.MESSAGE, "").contains("never read"));
				} catch (CoreException e) {
					e.printStackTrace();
					return false;
				}
			}
		}.waitForCondition(editor.getSite().getShell().getDisplay(), 30000),
				"Diagnostic not published in standalone component file");
		IDocument document = editor.getDocumentProvider().getDocument(editor.getEditorInput());
		LSContentAssistProcessor contentAssistProcessor = new LSContentAssistProcessor();
		ICompletionProposal[] proposals = contentAssistProcessor.computeCompletionProposals(Utils.getViewer(editor),
				document.get().indexOf(" }}"));
		Optional<ICompletionProposal> proposal = Arrays.stream(proposals)
				.filter(item -> item.getDisplayString().equals("appParameter")).findFirst();

		assertTrue(proposal.isPresent(), "Proposal not exists");
		proposal.get().apply(document);

		assertTrue(document.get().contains("{{ appParameter }}"), "Incorrect completion insertion");

		editor.close(false);
	}

	@Test
	void testVueTemplate() throws Exception {
		TextEditor editor = (TextEditor) IDE.openEditor(
				PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage(),
				componentFolder.getFile("HelloWorld.vue"));
		DisplayHelper.sleep(4000); // Give time for LS to initialize enough before making edit and sending a
									// didChange
		IFile appComponentHTML = componentFolder.getFile("HelloWorld.vue");
		editor = (TextEditor) IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage(),
				appComponentHTML);
		// Give some time for LS to load
		DisplayHelper.sleep(2000);
		// then make an edit
		IDocument document = editor.getDocumentProvider().getDocument(editor.getEditorInput());
		document.set(document.get() + "\n");
		assertTrue(new DisplayHelper() {
			@Override
			protected boolean condition() {
				IMarker[] markers;
				try {
					markers = appComponentHTML.findMarkers("org.eclipse.lsp4e.diagnostic", true, IResource.DEPTH_ZERO);
					return Arrays.stream(markers).anyMatch(
							marker -> marker.getAttribute(IMarker.MESSAGE, "").contains("Element is missing end tag."));
				} catch (CoreException e) {
					e.printStackTrace();
					return false;
				}
			}
		}.waitForCondition(editor.getSite().getShell().getDisplay(), 30000),
				"No error found on erroneous HTML component file");
		// test completion
		LSContentAssistProcessor contentAssistProcessor = new LSContentAssistProcessor();
		ICompletionProposal[] proposals = contentAssistProcessor.computeCompletionProposals(Utils.getViewer(editor),
				document.get().indexOf("<only-start>") + " <only-start>".length() - 1);
		proposals[0].apply(document);
		assertEquals(new String(componentFolder.getFile("HelloWorldCorrect.vue").getContents().readAllBytes()).trim(),
				document.get().trim(), "Incorrect completion insertion");
	}
}
