
package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link ContainerChange}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableContainerChange.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableContainerChange implements ContainerChange {
  private final String path;
  private final Integer kind;

  private ImmutableContainerChange(String path, Integer kind) {
    this.path = path;
    this.kind = kind;
  }

  /**
   * @return The value of the {@code path} attribute
   */
  @JsonProperty("Path")
  @Override
  public String path() {
    return path;
  }

  /**
   * @return The value of the {@code kind} attribute
   */
  @JsonProperty("Kind")
  @Override
  public Integer kind() {
    return kind;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerChange#path() path} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for path
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainerChange withPath(String value) {
    String newValue = Objects.requireNonNull(value, "path");
    if (this.path.equals(newValue)) return this;
    return new ImmutableContainerChange(newValue, this.kind);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerChange#kind() kind} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for kind
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainerChange withKind(Integer value) {
    Integer newValue = Objects.requireNonNull(value, "kind");
    if (this.kind.equals(newValue)) return this;
    return new ImmutableContainerChange(this.path, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableContainerChange} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableContainerChange
        && equalsByValue((ImmutableContainerChange) another);
  }

  private boolean equalsByValue(ImmutableContainerChange another) {
    return path.equals(another.path)
        && kind.equals(another.kind);
  }

  /**
   * Computes a hash code from attributes: {@code path}, {@code kind}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + path.hashCode();
    h += (h << 5) + kind.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code ContainerChange} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ContainerChange{"
        + "path=" + path
        + ", kind=" + kind
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ContainerChange} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ContainerChange instance
   */
  public static ImmutableContainerChange copyOf(ContainerChange instance) {
    if (instance instanceof ImmutableContainerChange) {
      return (ImmutableContainerChange) instance;
    }
    return ImmutableContainerChange.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableContainerChange ImmutableContainerChange}.
   * <pre>
   * ImmutableContainerChange.builder()
   *    .path(String) // required {@link ContainerChange#path() path}
   *    .kind(Integer) // required {@link ContainerChange#kind() kind}
   *    .build();
   * </pre>
   * @return A new ImmutableContainerChange builder
   */
  public static ImmutableContainerChange.Builder builder() {
    return new ImmutableContainerChange.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableContainerChange ImmutableContainerChange}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ContainerChange.Builder {
    private static final long INIT_BIT_PATH = 0x1L;
    private static final long INIT_BIT_KIND = 0x2L;
    private long initBits = 0x3L;

    private String path;
    private Integer kind;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ContainerChange} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ContainerChange instance) {
      Objects.requireNonNull(instance, "instance");
      this.path(instance.path());
      this.kind(instance.kind());
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerChange#path() path} attribute.
     * @param path The value for path 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Path")
    public final Builder path(String path) {
      this.path = Objects.requireNonNull(path, "path");
      initBits &= ~INIT_BIT_PATH;
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerChange#kind() kind} attribute.
     * @param kind The value for kind 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Kind")
    public final Builder kind(Integer kind) {
      this.kind = Objects.requireNonNull(kind, "kind");
      initBits &= ~INIT_BIT_KIND;
      return this;
    }

    /**
     * Builds a new {@link ImmutableContainerChange ImmutableContainerChange}.
     * @return An immutable instance of ContainerChange
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableContainerChange build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableContainerChange(path, kind);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_PATH) != 0) attributes.add("path");
      if ((initBits & INIT_BIT_KIND) != 0) attributes.add("kind");
      return "Cannot build ContainerChange, some of required attributes are not set " + attributes;
    }
  }
}
