package org.eclipse.incquery.testing.core;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Maps;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.eclipse.incquery.runtime.api.AdvancedIncQueryEngine;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IQueryGroup;
import org.eclipse.incquery.runtime.api.IQuerySpecification;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.runtime.api.scope.IncQueryScope;
import org.eclipse.incquery.runtime.exception.IncQueryException;
import org.eclipse.incquery.runtime.extensibility.QueryBackendRegistry;
import org.eclipse.incquery.runtime.matchers.backend.IQueryBackend;
import org.eclipse.incquery.runtime.matchers.backend.QueryEvaluationHint;
import org.eclipse.incquery.runtime.util.IncQueryLoggingUtil;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IntegerRange;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.junit.Test;

/**
 * This abstract test class can be used to measure the steady-state memory requirements of the base index and
 * Rete networks of individual queries on a given IncQueryScope and with a given query group.
 * 
 * <p/>
 * This test case prepares an IncQueryEngine on the given scope and with the provided query group.
 * After the initial preparation is done, the engine is wiped (deletes the Rete network but keeps the base index).
 * Next, the following is performed for each query in the group:
 * <p/>
 * <ol>
 *   <li> Wipe the engine </li>
 *   <li> Create the matcher and count matches </li>
 *   <li> Wipe the engine </li>
 * </ol>
 * 
 * After each step, the used, total and free heap space is logged in MBytes after 5 GC calls and 1 second of waiting.
 * Note that even this does not always provide an absolute steady state or a precise result, but can be useful for
 * finding problematic queries.
 */
@SuppressWarnings("all")
public abstract class QueryPerformanceTest {
  @Extension
  protected static Logger logger = IncQueryLoggingUtil.getLogger(QueryPerformanceTest.class);
  
  private AdvancedIncQueryEngine incQueryEngine;
  
  private Map<String, Long> results = Maps.<String, Long>newTreeMap();
  
  /**
   * This method shall return a scope that identifies the input artifact used for performance testing the queries.
   */
  public abstract IncQueryScope getScope() throws IncQueryException;
  
  /**
   * This method shall return the query group that contains the set of queries to evaluate.
   */
  public abstract IQueryGroup getQueryGroup() throws IncQueryException;
  
  /**
   * This method shall return the query backend class that will be used for evaluation.
   * The backend must be already registered in the {@link QueryBackendRegistry}.
   * 
   * Default implementation returns the registered default backend class.
   */
  public Class<? extends IQueryBackend> getQueryBackend() {
    QueryBackendRegistry _instance = QueryBackendRegistry.getInstance();
    return _instance.getDefaultBackendClass();
  }
  
  protected void prepare() {
    try {
      QueryPerformanceTest.logger.info("Preparing query performance test");
      final IncQueryScope preparedScope = this.getScope();
      QueryPerformanceTest.logMemoryProperties("Scope prepared");
      AdvancedIncQueryEngine _createUnmanagedEngine = AdvancedIncQueryEngine.createUnmanagedEngine(preparedScope);
      this.incQueryEngine = _createUnmanagedEngine;
      IQueryGroup _queryGroup = this.getQueryGroup();
      _queryGroup.prepare(this.incQueryEngine);
      QueryPerformanceTest.logMemoryProperties("Base index created");
      this.incQueryEngine.wipe();
      QueryPerformanceTest.logMemoryProperties("IncQuery engine wiped");
      QueryPerformanceTest.logger.info("Prepared query performance test");
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * This test case executes the performance evaluation on the given scope and with the provided query group.
   */
  @Test
  public void queryPerformance() {
    try {
      QueryPerformanceTest.logger.setLevel(Level.DEBUG);
      this.prepare();
      QueryPerformanceTest.logger.info("Starting query performance test");
      IQueryGroup _queryGroup = this.getQueryGroup();
      Set<IQuerySpecification<?>> _specifications = _queryGroup.getSpecifications();
      for (final IQuerySpecification<?> _specification : _specifications) {
        {
          final IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>> specification = ((IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>>) _specification);
          String _fullyQualifiedName = specification.getFullyQualifiedName();
          String _plus = ("Measuring query " + _fullyQualifiedName);
          QueryPerformanceTest.logger.debug(_plus);
          this.incQueryEngine.wipe();
          final long usedHeapBefore = QueryPerformanceTest.logMemoryProperties("Wiped engine before building");
          QueryPerformanceTest.logger.debug("Building Rete");
          final Stopwatch watch = Stopwatch.createStarted();
          Class<? extends IQueryBackend> _queryBackend = this.getQueryBackend();
          HashMap<String, Object> _newHashMap = CollectionLiterals.<String, Object>newHashMap();
          QueryEvaluationHint _queryEvaluationHint = new QueryEvaluationHint(_queryBackend, _newHashMap);
          final IncQueryMatcher<? extends IPatternMatch> matcher = this.incQueryEngine.getMatcher(specification, _queryEvaluationHint);
          watch.stop();
          final int countMatches = matcher.countMatches();
          final long usedHeapAfter = QueryPerformanceTest.logMemoryProperties("Matcher created");
          final long usedHeap = (usedHeapAfter - usedHeapBefore);
          String _fullyQualifiedName_1 = specification.getFullyQualifiedName();
          this.results.put(_fullyQualifiedName_1, Long.valueOf(usedHeap));
          String _fullyQualifiedName_2 = specification.getFullyQualifiedName();
          String _plus_1 = ("Query " + _fullyQualifiedName_2);
          String _plus_2 = (_plus_1 + "( ");
          String _plus_3 = (_plus_2 + Integer.valueOf(countMatches));
          String _plus_4 = (_plus_3 + " matches, used ");
          String _plus_5 = (_plus_4 + Long.valueOf(usedHeap));
          String _plus_6 = (_plus_5 + 
            " kByte heap, took ");
          long _elapsed = watch.elapsed(TimeUnit.MILLISECONDS);
          String _plus_7 = (_plus_6 + Long.valueOf(_elapsed));
          String _plus_8 = (_plus_7 + " ms)");
          QueryPerformanceTest.logger.info(_plus_8);
          this.incQueryEngine.wipe();
          QueryPerformanceTest.logMemoryProperties("Wiped engine after building");
          QueryPerformanceTest.logger.debug("\n-------------------------------------------\n");
        }
      }
      QueryPerformanceTest.logger.info("Finished query performance test");
      this.printResults();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  protected void printResults() {
    final StringBuilder resultSB = new StringBuilder("\n\nPerformance test results:\n");
    Set<Map.Entry<String, Long>> _entrySet = this.results.entrySet();
    final Procedure1<Map.Entry<String, Long>> _function = new Procedure1<Map.Entry<String, Long>>() {
      @Override
      public void apply(final Map.Entry<String, Long> entry) {
        String _key = entry.getKey();
        String _plus = ("  " + _key);
        String _plus_1 = (_plus + ",");
        Long _value = entry.getValue();
        String _plus_2 = (_plus_1 + _value);
        String _plus_3 = (_plus_2 + "\n");
        resultSB.append(_plus_3);
      }
    };
    IterableExtensions.<Map.Entry<String, Long>>forEach(_entrySet, _function);
    QueryPerformanceTest.logger.info(resultSB);
  }
  
  /**
   * Calls garbage collector 5 times, sleeps 1 second and logs the used, total and free heap sizes in MByte.
   * 
   * @param logger
   * @return The amount of used heap memory in kBytes
   */
  protected static long logMemoryProperties(final String status) {
    long _xblockexpression = (long) 0;
    {
      IntegerRange _upTo = new IntegerRange(0, 4);
      final Procedure1<Integer> _function = new Procedure1<Integer>() {
        @Override
        public void apply(final Integer it) {
          Runtime _runtime = Runtime.getRuntime();
          _runtime.gc();
        }
      };
      IterableExtensions.<Integer>forEach(_upTo, _function);
      try {
        Thread.sleep(1000);
      } catch (final Throwable _t) {
        if (_t instanceof InterruptedException) {
          final InterruptedException e = (InterruptedException)_t;
          QueryPerformanceTest.logger.trace("Sleep after GC interrupted");
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      Runtime _runtime = Runtime.getRuntime();
      long _talMemory = _runtime.totalMemory();
      final long totalHeapKB = (_talMemory / 1024);
      Runtime _runtime_1 = Runtime.getRuntime();
      long _freeMemory = _runtime_1.freeMemory();
      final long freeHeapKB = (_freeMemory / 1024);
      final long usedHeapKB = (totalHeapKB - freeHeapKB);
      QueryPerformanceTest.logger.debug((((((((status + ": Used Heap size: ") + Long.valueOf((usedHeapKB / 1024))) + " MByte (Total: ") + Long.valueOf((totalHeapKB / 1024))) + " MByte, Free: ") + Long.valueOf((freeHeapKB / 1024))) + " MByte)"));
      _xblockexpression = usedHeapKB;
    }
    return _xblockexpression;
  }
}
