/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.config;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.net.impl.pkcs1.PrivateKeyParser;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.eclipse.hono.config.FileFormat;
import org.eclipse.hono.config.PemReader;
import org.eclipse.hono.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KeyLoader {
    private static final Logger LOG = LoggerFactory.getLogger(KeyLoader.class);
    private final PrivateKey privateKey;
    private final List<Certificate> certs = new ArrayList<Certificate>();

    private KeyLoader(PrivateKey privateKey, List<Certificate> certs) {
        this.privateKey = privateKey;
        if (certs != null) {
            this.certs.addAll(certs);
        }
    }

    public PrivateKey getPrivateKey() {
        return this.privateKey;
    }

    public Certificate[] getCertificateChain() {
        if (this.certs.isEmpty()) {
            return null;
        }
        return this.certs.toArray(new Certificate[this.certs.size()]);
    }

    public PublicKey getPublicKey() {
        if (this.certs.isEmpty()) {
            return null;
        }
        return this.certs.get(0).getPublicKey();
    }

    public static KeyLoader fromKeyStore(Vertx vertx, String keyStorePath, char[] password) {
        Objects.requireNonNull(vertx);
        Objects.requireNonNull(keyStorePath);
        if (!vertx.fileSystem().existsBlocking(Objects.requireNonNull(keyStorePath))) {
            throw new IllegalArgumentException("key store does not exist");
        }
        FileFormat format = FileFormat.detect(keyStorePath);
        return KeyLoader.loadKeysFromStore(vertx, switch (format) {
            case FileFormat.JKS -> "JKS";
            case FileFormat.PKCS12 -> "PKCS12";
            default -> throw new IllegalArgumentException("key store must be JKS or PKCS format but is: " + format);
        }, keyStorePath, password);
    }

    public static KeyLoader fromFiles(Vertx vertx, String keyPath, String certPath) {
        PrivateKey privateKey = null;
        List<Certificate> certChain = null;
        if (!Strings.isNullOrEmpty(keyPath)) {
            privateKey = KeyLoader.loadPrivateKeyFromFile(vertx, keyPath);
        }
        if (!Strings.isNullOrEmpty(certPath)) {
            certChain = KeyLoader.loadCertsFromFile(vertx, certPath);
        }
        return new KeyLoader(privateKey, certChain);
    }

    private static PrivateKey generatePrivateKey(String algorithm, KeySpec keySpec) throws GeneralSecurityException {
        return KeyFactory.getInstance(algorithm).generatePrivate(keySpec);
    }

    @SuppressFBWarnings(value={"PATH_TRAVERSAL_IN"}, justification="The path that the file is read from is determined from configuration properties that\nare supposed to be passed in during startup of the component only.\n")
    private static <R> R processFile(Vertx vertx, String pathName, PemProcessor<R> processor) {
        Path path = Paths.get(pathName, new String[0]);
        if (!vertx.fileSystem().existsBlocking(pathName)) {
            throw new IllegalArgumentException(String.format("%s: PEM file does not exist", path));
        }
        try {
            List<PemReader.Entry> pems = PemReader.readAllBlocking(vertx, path);
            if (pems.isEmpty()) {
                throw new IllegalArgumentException(String.format("%s: File is empty", path));
            }
            return processor.process(pems);
        }
        catch (IllegalArgumentException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IllegalArgumentException(String.format("%s: Failed to load PEM file: ", pathName), e);
        }
    }

    private static PrivateKey loadPrivateKeyFromFile(Vertx vertx, String keyPath) {
        return KeyLoader.processFile(vertx, keyPath, pems -> {
            PemReader.Entry pem = (PemReader.Entry)pems.get(0);
            switch (pem.getType()) {
                case "PRIVATE KEY": {
                    String algorithm = PrivateKeyParser.getPKCS8EncodedKeyAlgorithm(pem.getPayload());
                    if ("RSA".equals(algorithm)) {
                        return KeyLoader.generatePrivateKey(algorithm, new PKCS8EncodedKeySpec(pem.getPayload()));
                    }
                    if ("EC".equals(algorithm)) {
                        return KeyLoader.generatePrivateKey(algorithm, new PKCS8EncodedKeySpec(pem.getPayload()));
                    }
                    throw new IllegalArgumentException(String.format("%s: Unsupported key algorithm: %s", keyPath, algorithm));
                }
                case "EC PRIVATE KEY": {
                    return KeyLoader.generatePrivateKey("EC", PrivateKeyParser.getECKeySpec(pem.getPayload()));
                }
                case "RSA PRIVATE KEY": {
                    return KeyLoader.generatePrivateKey("RSA", PrivateKeyParser.getRSAKeySpec(pem.getPayload()));
                }
            }
            throw new IllegalArgumentException(String.format("%s: Unsupported key type: %s", keyPath, pem.getType()));
        });
    }

    private static List<Certificate> loadCertsFromFile(Vertx vertx, String certPath) {
        return KeyLoader.processFile(vertx, certPath, pems -> pems.stream().filter(entry -> "CERTIFICATE".equals(entry.getType())).map(entry -> {
            try {
                CertificateFactory factory = CertificateFactory.getInstance("X.509");
                return factory.generateCertificate(new ByteArrayInputStream(entry.getPayload()));
            }
            catch (CertificateException e) {
                return null;
            }
        }).filter(s -> s != null).collect(Collectors.toList()));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static KeyLoader loadKeysFromStore(Vertx vertx, String type, String path, char[] password) {
        PrivateKey privateKey = null;
        Buffer buffer = vertx.fileSystem().readFileBlocking(path);
        try (ByteArrayInputStream is = new ByteArrayInputStream(buffer.getBytes());){
            KeyStore store = KeyStore.getInstance(type);
            store.load(is, password);
            LOG.debug("loading keys from key store containing {} entries", (Object)store.size());
            Enumeration<String> e = store.aliases();
            while (e.hasMoreElements()) {
                String alias = e.nextElement();
                LOG.debug("current alias: {}", (Object)alias);
                if (store.isKeyEntry(alias)) {
                    LOG.debug("loading private key [{}]", (Object)alias);
                    privateKey = (PrivateKey)store.getKey(alias, password);
                    LOG.debug("loading public key [{}]", (Object)alias);
                    Certificate[] chain = store.getCertificateChain(alias);
                    List<Certificate> certChain = Optional.of(chain).map(c -> Arrays.asList(c)).orElse(Collections.emptyList());
                    KeyLoader keyLoader = new KeyLoader(privateKey, certChain);
                    return keyLoader;
                }
                LOG.debug("skipping non-private key entry");
            }
            throw new IllegalArgumentException(String.format("%s: Key store doesn't contain private key", path));
        }
        catch (IOException | GeneralSecurityException e) {
            LOG.error("cannot load keys", e);
        }
        throw new IllegalArgumentException(String.format("%s: Key store doesn't contain private key", path));
    }

    @FunctionalInterface
    private static interface PemProcessor<R> {
        public R process(List<PemReader.Entry> var1) throws Exception;
    }
}

