"use strict";
/*! Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Response = void 0;
const mona_dish_1 = require("mona-dish");
const ResponseProcessor_1 = require("./ResponseProcessor");
const Const_1 = require("../core/Const");
const ResonseDataResolver_1 = require("./ResonseDataResolver");
const ExtDomQuery_1 = require("../util/ExtDomQuery");
var Response;
(function (Response) {
    /**
     * Standardized faces.ts response
     * this one is called straight from faces.ts.response
     *
     * The processing follows the spec by going for the responseXML
     * and processing its tags
     *
     * @param {XMLHttpRequest} request (xhrRequest) - xhr request object
     * @param context {Context} context (Map) - AJAX context
     *
     */
    function processResponse(request, context) {
        let req = ExtDomQuery_1.ExtConfig.fromNullable(request);
        let { externalContext, internalContext } = (0, ResonseDataResolver_1.resolveContexts)(context);
        let responseXML = (0, ResonseDataResolver_1.resolveResponseXML)(req);
        let responseProcessor = new ResponseProcessor_1.ResponseProcessor(req, externalContext, internalContext);
        internalContext.assign(Const_1.RESPONSE_XML).value = responseXML;
        // we now process the partial tags, or in none given raise an error
        responseXML.querySelectorAll(Const_1.XML_TAG_PARTIAL_RESP)
            .each(item => processPartialTag(item, responseProcessor, internalContext));
        // We now process the viewStates, client windows and the elements to be evaluated are delayed.
        // The reason for this is that often it is better
        // to wait until the document has caught up before
        // doing any evaluations even on embedded scripts.
        // Usually this does not matter, the client window comes in almost last always anyway
        // we maybe drop this deferred assignment in the future, but myfaces did it until now.
        responseProcessor.updateNamedViewRootState();
        responseProcessor.fixViewStates();
        responseProcessor.fixClientWindow();
        responseProcessor.globalEval();
        responseProcessor.done();
    }
    Response.processResponse = processResponse;
    /**
     * highest node partial-response from there the main operations are triggered
     */
    function processPartialTag(node, responseProcessor, internalContext) {
        /*
        https://javaee.github.io/javaserverfaces/docs/2.2/javadocs/web-partialresponse.html#ns_xsd
        The "partial-response" element is the root of the partial response information hierarchy,
        and contains nested elements for all possible elements that can exist in the response.
        This element must have an "id" attribute whose value is the return from calling getContainerClientId()
        on the UIViewRoot to which this response pertains.
         */
        // we can determine whether we are in a naming container scenario by checking whether the passed view id is present in the page
        // under or in body as identifier
        var _a;
        let partialId = (_a = node === null || node === void 0 ? void 0 : node.id) === null || _a === void 0 ? void 0 : _a.value;
        internalContext.assignIf(!!partialId, Const_1.NAMING_CONTAINER_ID).value = partialId; // second case mojarra
        // there must be at least one container viewstate element resembling the viewroot that we know
        // this is named
        responseProcessor.updateNamedViewRootState();
        const SEL_SUB_TAGS = [Const_1.XML_TAG_ERROR, Const_1.XML_TAG_REDIRECT, Const_1.XML_TAG_CHANGES].join(",");
        // now we can process the main operations
        node.querySelectorAll(SEL_SUB_TAGS).each((node) => {
            switch (node.tagName.value) {
                case Const_1.XML_TAG_ERROR:
                    responseProcessor.error(node);
                    break;
                case Const_1.XML_TAG_REDIRECT:
                    responseProcessor.redirect(node);
                    break;
                case Const_1.XML_TAG_CHANGES:
                    processChangesTag(node, responseProcessor);
                    break;
            }
        });
    }
    let processInsert = function (responseProcessor, node) {
        // path1 insert after as child tags
        if (node.querySelectorAll([Const_1.XML_TAG_BEFORE, Const_1.XML_TAG_AFTER].join(",")).length) {
            responseProcessor.insertWithSubTags(node);
        }
        else { // insert before after with id
            responseProcessor.insert(node);
        }
    };
    /**
     * next level changes tag
     *
     * @param node
     * @param responseProcessor
     */
    function processChangesTag(node, responseProcessor) {
        const ALLOWED_TAGS = [Const_1.XML_TAG_UPDATE, Const_1.XML_TAG_EVAL, Const_1.XML_TAG_INSERT, Const_1.XML_TAG_DELETE, Const_1.XML_TAG_ATTRIBUTES, Const_1.XML_TAG_EXTENSION].join(", ");
        node.querySelectorAll(ALLOWED_TAGS).each((node) => {
            switch (node.tagName.value) {
                case Const_1.XML_TAG_UPDATE:
                    processUpdateTag(node, responseProcessor);
                    break;
                case Const_1.XML_TAG_EVAL:
                    responseProcessor.eval(node);
                    break;
                case Const_1.XML_TAG_INSERT:
                    processInsert(responseProcessor, node);
                    break;
                case Const_1.XML_TAG_DELETE:
                    responseProcessor.delete(node);
                    break;
                case Const_1.XML_TAG_ATTRIBUTES:
                    responseProcessor.attributes(node);
                    break;
                case Const_1.XML_TAG_EXTENSION:
                    break;
            }
        });
        return true;
    }
    /**
     * checks and stores a state update for delayed processing
     *
     * @param responseProcessor the response processor to perform the store operation
     * @param node the xml node to check for the state
     *
     * @private
     */
    function storeState(responseProcessor, node) {
        return responseProcessor.processViewState(node) || responseProcessor.processClientWindow(node);
    }
    /**
     * branch tag update. drill further down into the updates
     * special case viewState in that case it is a leaf
     * and the viewState must be processed
     *
     * @param node
     * @param responseProcessor
     */
    function processUpdateTag(node, responseProcessor) {
        // early state storing, if no state we perform a normal update cycle
        if (!storeState(responseProcessor, node)) {
            handleElementUpdate(node, responseProcessor);
        }
    }
    /**
     * element update
     *
     * @param node
     * @param responseProcessor
     */
    function handleElementUpdate(node, responseProcessor) {
        let cdataBlock = node.cDATAAsString;
        switch (node.id.value) {
            case (0, Const_1.$nsp)(Const_1.P_VIEWROOT):
                responseProcessor.replaceViewRoot(mona_dish_1.DQ.fromMarkup(cdataBlock.substring(cdataBlock.indexOf("<html"))));
                break;
            case (0, Const_1.$nsp)(Const_1.P_VIEWHEAD):
                responseProcessor.replaceHead(mona_dish_1.DQ.fromMarkup(cdataBlock));
                break;
            case (0, Const_1.$nsp)(Const_1.P_VIEWBODY):
                responseProcessor.replaceBody(mona_dish_1.DQ.fromMarkup(cdataBlock));
                break;
            case (0, Const_1.$nsp)(Const_1.P_RESOURCE):
                responseProcessor.addToHead(mona_dish_1.DQ.fromMarkup(cdataBlock));
                break;
            default: // htmlItem replacement
                responseProcessor.update(node, cdataBlock);
                break;
        }
    }
})(Response || (exports.Response = Response = {}));
//# sourceMappingURL=Response.js.map